/**
 * nameserver.c
 *
 * Contains some functions to deal with lists of
 * nameservers
 *
 * (c) NLnet Labs 2004
 *
 * See the file LICENSE for the license
 *
 */

#include <stdlib.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

#include "common.h"

/**
 * Adds an ip address string to the nameserver list
 */
struct t_rr *
ns_add_ip(struct t_rr *n, const uint8_t *ip_str, uint16_t type)
{
	struct t_rr *new_ns;
	assert(n != NULL);
	/* for nameserver we are only interested in ip's
	 * the rest is of minor importance. Use the
	 * name of n to add the others.
	 */
	if (type != TYPE_A && type != TYPE_AAAA)
		return NULL;

	/* create an RR out of the IP */
	new_ns = rr_create(n->name,
			type,
			0,
			SEC_ALL);
	rr_add_rdata(rdata_create(ip_str, strlen((char *) ip_str)), new_ns);
	return rr_add_rr(n, new_ns);
}

/**
 * Adds a t_rr a struct to a nameserver list 
 */
struct t_rr *
ns_add_rr(struct t_rr *n, struct t_rr *r)
{
	/* nameserver can only have A or AAAA records */
	assert(n != NULL);
	assert(r != NULL);
	/* name check.... I think we don't care... */

	if (r->type != TYPE_A && r->type != TYPE_AAAA)
		return NULL;

	return rr_add_rr(n, r);
}

/**
 * Create a new nameserver list rr structure
 */
struct t_rr *
ns_create(struct t_rdata *name, const uint8_t *ip_str, uint16_t type)
{
	struct t_rr *new;
	struct t_rdata *root_name;
	
	if (type != TYPE_A && type != TYPE_AAAA)
		return NULL;

	if (name) {
		new = rr_create(name, type, 0, SEC_ALL);
	} else {
		root_name = rdata_create((uint8_t *) ".",1);
		new = rr_create(root_name,
					type,
					0,
					SEC_ALL);
		rdata_destroy(root_name);
	}
	rr_add_rdata(rdata_create(ip_str, strlen((char *) ip_str)), new);
	return new;
}

/**
 * Try to resolv all the NS rr's in the packet to 
 * A records.  Return the addresses found
 *
 * TODO: ipv6
 */
struct t_rr *
get_ns_addresses(struct t_dpacket *p, int protocol, int section)
{
	struct t_rr *nslist = NULL;
	struct t_rr *nsrec = NULL;
	struct t_rr *addr = NULL;
	struct t_rr *query = NULL;

	nslist = dpacket_get_rrset(NULL, TYPE_NS, p, section);
	nsrec = nslist;
	while (nsrec) {
		query = rr_create(nsrec->rdata[0], TYPE_A, DEF_TTL, SEC_QUESTION);
		addr = rr_add_rr(addr, (do_trace(query, protocol, NO_PRINT)));
		nsrec = nsrec->next;
	}
	return addr;
}

/**
 * this function primes to root ns's. It takes the root nameserver
 * list preconfigured - and then checks one (or more) servers for
 * a more up2date list of servers
 *
 * return the new list or if something goes wrong; return the 
 * original list
 */
struct t_rr *
prime_root_ns(int print)
{
	struct t_rdata *q_rd;
	struct t_dpacket *p;
	struct t_rr *new_roots;

	/* get a query ready */
	q_rd = rdata_create((uint8_t*)".", 1);
	/* perform the query */
	p = do_query(q_rd, TYPE_NS, root_servers, PROTO_UDP);
	/* you are not needed anymore */
	rdata_destroy(q_rd);

	if (!p)
		return root_servers;
	
	if (print)
		print_packet_dense(p);

	/* the nameservers we're interested live in the
	 * answer section - because we're not asking for
	 * a referral
	 */
	new_roots = get_ns_addresses(p, PROTO_UDP, SEC_ANSWER);

	return new_roots;
}
