/* 
 * key.c
 *
 * read a key from a public key file
 * and store the thing in a RR
 *
 * (c) NLnet Labs, 2004
 *
 * See the file LICENSE for the license
 *
 */

#include <stdio.h>

#include "common.h"
#include "wirerdata.h"

/**
 * Reads a public key from the given file
 */
struct t_rr *
pubkey_fromfile(const char *filename)
{
	FILE *f;
	
	/* key file
	 * owner IN DNSKEY flag type alg data
	 */
        if ((f=fopen(filename,"r")) == NULL) {
		warning("Could not open key file: %s\n", filename);
		return NULL;
	}
	return pubkey_fromfilepointer(f);
}

/**
 * Reads a public key from the given file handle
 */
struct t_rr *
pubkey_fromfilepointer(FILE *f)
{
	struct t_rdata *rd;
	struct t_rr *key;

	/* owner name */
	if ((rd = get_next_rdata(f))) {
		key = rr_create(rd, TYPE_DNSKEY, DEF_TTL, SEC_ALL);
		rdata_destroy(rd);
	} else {
		return NULL;
	}
	
	/* class, discard it */
	if ((rd = get_next_rdata(f))) {
		rdata_destroy(rd);
	} else {
		rr_destroy(key, FOLLOW);
		return NULL;
	}
	
	/* dnskey type, discard it */
	if ((rd = get_next_rdata(f))) {
		rdata_destroy(rd);
	} else {
		rr_destroy(key, FOLLOW);
		return NULL;
	}

	/* flags */
	if ((rd = get_next_rdata(f)))
		rr_add_rdata(rd, key);
	else {
		rr_destroy(key, FOLLOW);
		return NULL;
	}
	
	/* key type */
	if ((rd = get_next_rdata(f)))
		rr_add_rdata(rd, key);
	else {
		rr_destroy(key, FOLLOW);
		return NULL;
	}
	
	/* alg */
	if ((rd = get_next_rdata(f)))
		rr_add_rdata(rd, key);
	else {
		rr_destroy(key, FOLLOW);
		return NULL;
	}

	/* base64 keydata */
	/* base64 may contain spaces (which is the delimiter for rdatas
	 * , but this is last rdata anyway, so we use get_last_rdata, which
	 * does not stop on a space
	 */
	if ((rd = get_last_rdata(f)))
		rr_add_rdata(rd, key);
	else {
		rr_destroy(key, FOLLOW);
		return NULL;
	}
	return key;
}

/**
 * Reads a public key from the given string buffer
 */
struct t_rr *
pubkey_fromstring(char *str)
{
	struct t_rdata *rd;
	struct t_rr *key;

	/* owner name */
	if ((str += get_next_rdata_string(str, &rd)))
		key = rr_create(rd, TYPE_DNSKEY, DEF_TTL, SEC_ALL);
	else
		return NULL;

	/* class, discard it */
	if (!(str += get_next_rdata_string(str, &rd)))
		return NULL;

	/* dnskey type, discard it */
	if (!(str += get_next_rdata_string(str, &rd)))
		return NULL;

	/* flags */
	if ((str += get_next_rdata_string(str, &rd)))
		rr_add_rdata(rd, key);
	else
		return NULL;

	/* key type */
	if ((str += get_next_rdata_string(str, &rd)))
		rr_add_rdata(rd, key);
	else
		return NULL;

	/* alg */
	if ((str += get_next_rdata_string(str, &rd)))
		rr_add_rdata(rd, key);
	else
		return NULL;

	/* base64 keydata */
	if ((str += get_next_rdata_string(str, &rd)))
		rr_add_rdata(rd, key);
	else
		return NULL;

	return key;
}

/**
 * Reads a DS record from the given file
 */
struct t_rr *
ds_fromfile(const char *filename)
{
	struct t_rdata *rd;
	struct t_rr *ds;
	FILE *f;

	/* ds file
	 * owner IN DS keyid alg hash data
	 */
        if ((f=fopen(filename,"r")) == NULL) {
		warning("Could not open ds file: %s\n", filename);
		return NULL;
	}
	/* owner name */
	if ((rd = get_next_rdata(f)))
		ds = rr_create(rd, TYPE_DS, DEF_TTL, SEC_ALL);
	else
		return NULL;

	/* class, discard it */
	if (!(rd = get_next_rdata(f)))
		return NULL;
	/* ds type, discard it */
	if (!(rd = get_next_rdata(f)))
		return NULL;

	/* key id */
	if ((rd = get_next_rdata(f)))
		rr_add_rdata(rd, ds);
	else
		return NULL;

	/* algorithm */
	if ((rd = get_next_rdata(f)))
		rr_add_rdata(rd, ds);
	else
		return NULL;

	/* hash function */
	if ((rd = get_next_rdata(f)))
		rr_add_rdata(rd, ds);
	else
		return NULL;

	/* hex ds data */
	if ((rd = get_next_rdata(f)))
		rr_add_rdata(rd, ds);
	else
		return NULL;

	return ds;
}

/* calculate the keytag from the key rr */
uint16_t
keytag(struct t_rr * keyrr)
{
	size_t bufsize = 1024;
	uint8_t *buf = xmalloc(bufsize);
	int offset = 0;
	unsigned int result = 0;

	assert(keyrr->type == TYPE_DNSKEY);

	/* for the key tag we need
	 * flags + proto + alg + pubkey format
	 * in a buffer and give to to our helper
	 */

	/* flags */
	offset += rdata2wire_int16(keyrr->rdata[0], buf, (unsigned long) offset, bufsize);
	/* proto */
	offset += rdata2wire_int8(keyrr->rdata[1], buf, (unsigned long) offset, bufsize);
	/* algorithm */
	offset += rdata2wire_int8(keyrr->rdata[2], buf, (unsigned long) offset, bufsize);
	/* keydata */
	offset += rdata2wire_base64(keyrr->rdata[3], buf, (unsigned long) offset, bufsize);

	if (rdata2uint8(keyrr->rdata[2]) == ALG_RSAMD5) {
		result = keytag_helper_rsamd5(buf, offset);
	} else {
		result = keytag_helper(buf, offset);
	}

	xfree(buf);
	return (uint16_t) result;
}

/**
 * Calculates the keytag from the given dnskey RR
 * (Copied from 2535bis), for algorithms != RSAMD5
 */
unsigned int
keytag_helper(uint8_t *key, int keysize)
{
	uint32_t ac;     /* assumed to be 32 bits or larger */
	int i;                /* loop index */

	for ( ac = 0, i = 0; i < keysize; ++i )
		ac += (i & 1) ? key[i] : key[i] << 8;
	ac += (ac >> 16) & 0xFFFF;
	return (unsigned int) (ac & 0xFFFF);
}

/**
 * Calculates the keytag from the given dnskey RR
 * (Copied from 2535bis), for algorithm = RSAMD5
 */
unsigned int 
keytag_helper_rsamd5(uint8_t *key, int keysize)
{
	uint16_t ac = 0;

	if (keysize > 4) {
		memcpy(&ac, &key[keysize-3], 2);
	}
	ac = ntohs(ac);
	
	return (unsigned int) ac;
}

/**
 * Returns a DS representation for the given key
 */
struct t_rr *
key2ds(struct t_rr *key)
{
	struct t_rr *ds;
	unsigned long datalength = 0;
	uint8_t *data = xmalloc(MAX_PACKET); /* scratch space */
	uint8_t *digest = xmalloc(20); /* SHA1 */
	
	ds = rr_create(key->name, TYPE_DS, DEF_TTL, key->section);
	ds->ttl = key->ttl; /* rr_create mangles TTL */

	/* keytag */
	snprintf((char *) data, 8, "%d", (int) keytag(key));
	rr_add_rdata(rdata_create(data, strlen((char *) data)), ds);
	
	/* algorithm */
	rr_add_rdata(rdata_clone(key->rdata[2]), ds);
	
	/* digest type,  only SHA-1 supported atm */
	rr_add_rdata(rdata_create((uint8_t *) "1", 1), ds);
	
	/* digest */
	datalength = name2wire(key->name, data, datalength, MAX_PACKET);
	datalength += rdata2wire_int16(key->rdata[0], data, datalength, MAX_PACKET);
	datalength += rdata2wire_int8(key->rdata[1], data, datalength, MAX_PACKET);
	datalength += rdata2wire_int8(key->rdata[2], data, datalength, MAX_PACKET);
	datalength += rdata2wire_base64(key->rdata[3], data, datalength, MAX_PACKET);

	(void) SHA1((unsigned char *) data, datalength, (unsigned char *) digest);
	(void) wire2rdata_hex(digest, 0, ds, 20);
	
	xfree(data);
	return ds;
}
