/** 
 * dns.h
 *
 * modelled slightly after dns.h of NSD
 * Everything we know about the DNS
 * (c) NLnet Labs, 2004
 *  
 * See the file LICENSE for the license
 *
 */

#ifndef _DNSTYPES_H_
#define _DNSTYPES_H_

#define DNS_PORT 53

/* RFC1035 */
#define CLASS_IN        1       /* Class IN */
#define CLASS_CH        3       /* Class CHAOS */
#define CLASS_HS        4       /* Class HS */
#define CLASS_ANY       255     /* Class IN */

#define TYPE_A          1       /* a host address */
#define TYPE_NS         2       /* an authoritative name server */
#define TYPE_MD         3       /* a mail destination (Obsolete - use MX) */
#define TYPE_MF         4       /* a mail forwarder (Obsolete - use MX) */
#define TYPE_CNAME      5       /* the canonical name for an alias */
#define TYPE_SOA        6       /* marks the start of a zone of authority */
#define TYPE_MB         7       /* a mailbox domain name (EXPERIMENTAL) */
#define TYPE_MG         8       /* a mail group member (EXPERIMENTAL) */
#define TYPE_MR         9       /* a mail rename domain name (EXPERIMENTAL) */
#define TYPE_NULL       10      /* a null RR (EXPERIMENTAL) */
#define TYPE_WKS        11      /* a well known service description */
#define TYPE_PTR        12      /* a domain name pointer */
#define TYPE_HINFO      13      /* host information */
#define TYPE_MINFO      14      /* mailbox or mail list information */
#define TYPE_MX         15      /* mail exchange */
#define TYPE_TXT        16      /* text strings */
#define TYPE_RP         17      /* RFC1183 */
#define TYPE_AFSDB      18      /* RFC1183 */
#define TYPE_X25	19	/* RFC1183 */
#define TYPE_ISDN	20	/* RFC1183 */
#define TYPE_RT		21	/* RFC1183 */
#define TYPE_NSAP	22	/* RFC1647 */

/* some newer types */
#define TYPE_SIG	24	/* DNSSEC SIG (rfc2535) */
#define TYPE_KEY	25	/* DNSSEC KEY (rfc2535) */
#define TYPE_PX		26	/* RFC 2163 */
#define TYPE_AAAA       28      /* ipv6 address */
#define TYPE_LOC        29      /* LOC record  RFC1876 */
#define TYPE_NXT	30	/* DNSSEC NXT (rfc2535) */
#define TYPE_SRV        33      /* SRV record RFC2782 */
#define TYPE_NAPTR      35      /* NAPTR record RFC2915 */
#define TYPE_KX		36	/* RFC2230 */
#define TYPE_CERT	37	/* CERT record RFC 2538 */
#define TYPE_DNAME	39	/* RFC2672 */
#define TYPE_OPT        41      /* Pseudo OPT record... */
#define TYPE_APL	42	/* APL, RFC3123 */
#define TYPE_DS         43      /* draft-ietf-dnsext-delegation */
#define TYPE_SSHFP      44      /* SSH Key Fingerprint */
#define TYPE_RRSIG	46      /* dnssecbis */
#define TYPE_NSEC   	47      /* dnssecbis */
#define TYPE_DNSKEY    	48      /* dnssecbis */

#define TYPE_TSIG	250	/* rfc2845 */

#define TYPE_IXFR       251
#define TYPE_AXFR       252
#define TYPE_MAILB      253     /* A request for mailbox-related records (MB, MG or MR) */
#define TYPE_MAILA      254     /* A request for mail agent RRs (Obsolete - see MX) */
#define TYPE_ANY        255     /* any type (wildcard) */

/* rcodes */
#define RCODE_NOERROR	1	/* +1 from what the real one is */
#define RCODE_FORMERR	2	/* ---------------------------- */
#define RCODE_SERVFAIL	3
#define RCODE_NAMEERR	4
#define RCODE_NOTIMPL	5
#define RCODE_REFUSED	6
#define RCODE_NOTAUTH	10	/* See rfc2845 */
#define RCODE_BADSIG	16
#define	RCODE_BADKEY	17
#define	RCODE_BADTIME	18

#define MAX_LABELLEN     63
#define MAX_DOMAINLEN    255

#define MAX_RDLENGTH	65535

/* Maximum size of a single RR.  */
/*
#define MAX_RR_SIZE \
	(MAXDOMAINLEN + sizeof(uint32_t) + 4*sizeof(uint16_t) + MAX_RDLENGTH)
*/

/** 
 * Generic lookup table.
 * This is used to map internal symbols to human-readable names
 * (for instance the value 1 to RCODE_NOERROR) 
 */
struct gtab {
	/**
	 * symbol
	 */
	uint16_t sym;
	/**
	 * name
	 */
	const char *name;
};

#define Z_CLASSES {             \
	{CLASS_IN, "IN"},       \
	{CLASS_CH, "CH"},       \
	{CLASS_HS, "HS"},       \
	{0, NULL}               \
}

#define Z_TYPES {               \
        {TYPE_A, "A"},          \
        {TYPE_NS, "NS"},        \
        {TYPE_MD, "MD"},        \
        {TYPE_MF, "MF"},        \
        {TYPE_CNAME, "CNAME"},  \
        {TYPE_SOA, "SOA"},      \
        {TYPE_MB, "MB"},        \
        {TYPE_MG, "MG"},        \
        {TYPE_MR, "MR"},        \
        {TYPE_NULL, "NULL"},    \
        {TYPE_WKS, "WKS"},      \
        {TYPE_PTR, "PTR"},      \
        {TYPE_HINFO, "HINFO"},  \
        {TYPE_MINFO, "MINFO"},  \
        {TYPE_MX, "MX"},        \
        {TYPE_TXT, "TXT"},      \
        {TYPE_SIG, "SIG"},	\
        {TYPE_KEY, "KEY"},	\
        {TYPE_NXT, "NXT"},	\
        {TYPE_PX, "PX"},	\
        {TYPE_AAAA, "AAAA"},    \
        {TYPE_SRV, "SRV"},      \
        {TYPE_NAPTR, "NAPTR"},  \
        {TYPE_KX, "KX"},	\
        {TYPE_CERT, "CERT"},    \
        {TYPE_DNAME, "DNAME"},    \
        {TYPE_LOC, "LOC"},      \
        {TYPE_AFSDB, "AFSDB"},  \
        {TYPE_X25, "X25"},	\
        {TYPE_ISDN, "ISDN"},	\
        {TYPE_RT, "RT"},	\
        {TYPE_NSAP, "NSAP"},	\
        {TYPE_RP, "RP"},        \
        {TYPE_APL, "APL"},        \
        {TYPE_DS, "DS"},        \
        {TYPE_SSHFP, "SSHFP"},  \
        {TYPE_RRSIG, "RRSIG"},  \
        {TYPE_NSEC, "NSEC"},    \
        {TYPE_DNSKEY, "DNSKEY"},\
	{TYPE_TSIG, "TSIG"},	\
        {TYPE_ANY, "ANY"},      \
        {TYPE_AXFR, "AXFR"},	\
        {0, NULL}               \
}

#define R_CODES {			\
        {RCODE_NOERROR, "NOERROR"},	\
        {RCODE_FORMERR, "FORMERR"},	\
        {RCODE_SERVFAIL, "SERVFAIL"},	\
        {RCODE_NAMEERR, "NAMEERR"},	\
        {RCODE_NOTIMPL, "NOTIMPL"},	\
        {RCODE_REFUSED, "REFUSED"},	\
        {RCODE_NOTAUTH, "NOTAUTH"},	\
        {RCODE_BADSIG, "BADSIG"},	\
        {RCODE_BADKEY, "BADKEY"},	\
        {RCODE_BADTIME, "BADTIME"},	\
        {0, NULL}			\
}

/*#define RDATA_TYPE_STRING 1*/
#define RDATA_TYPE_DNAME	1
#define RDATA_TYPE_INT32	2
#define RDATA_TYPE_INT16	3
#define RDATA_TYPE_INT8		4
#define RDATA_TYPE_A		5
#define RDATA_TYPE_AAAA		6
#define RDATA_TYPE_ANY		7
#define RDATA_TYPE_STR		8
#define RDATA_TYPE_BASE64	9
#define RDATA_TYPE_HEX		10
#define RDATA_TYPE_INT48	11

/* special rdata formats */
#define RDATA_TYPE_NSEC		12

/* extra subdivision for printing */
#define RDATA_TYPE_TYPE		13
#define RDATA_TYPE_CLASS	14
#define RDATA_TYPE_ALGO 	15
#define RDATA_TYPE_CERTTYPE	16
#define RDATA_TYPE_WKS		17
#define RDATA_TYPE_NSAP		18
#define RDATA_TYPE_LOC		19
#define RDATA_TYPE_APL		20

/* contains info on rdata types */

#define RR_RDATA_TYPES {		\
	{0, 0},				/* no type zero 	*/\
	{RDATA_TYPE_A},			/* A		1	*/\
	{RDATA_TYPE_DNAME},		/* NS		2	*/\
	{RDATA_TYPE_DNAME},		/* MD		3	*/\
	{RDATA_TYPE_DNAME},		/* MF		4	*/\
	{RDATA_TYPE_DNAME},		/* CNAME	5	*/\
	{RDATA_TYPE_DNAME,		/* SOA		6	*/\
		RDATA_TYPE_DNAME,	\
		RDATA_TYPE_INT32,	\
		RDATA_TYPE_INT32,	\
		RDATA_TYPE_INT32,	\
		RDATA_TYPE_INT32,	\
		RDATA_TYPE_INT32},	\
	{RDATA_TYPE_DNAME},		/* MB		7	*/\
	{RDATA_TYPE_DNAME},		/* MG		8	*/\
	{RDATA_TYPE_DNAME},		/* MR		9	*/\
	{RDATA_TYPE_HEX},		/* NULL (TODO)10	*/\
	{RDATA_TYPE_A,			/* WKS		11	*/\
		RDATA_TYPE_WKS},	\
	{RDATA_TYPE_DNAME},		/* PTR		12	*/\
	{RDATA_TYPE_STR,		/* HINFO	13	*/\
		RDATA_TYPE_STR},	\
	{RDATA_TYPE_DNAME,		/* MINFO	14	*/\
		RDATA_TYPE_DNAME},	\
	{RDATA_TYPE_INT16,		/* MX		15	*/\
		RDATA_TYPE_DNAME},	\
	{RDATA_TYPE_STR},		/* TXT (TODO)	16	*/\
	{RDATA_TYPE_DNAME,		/* RP		17	*/\
		RDATA_TYPE_DNAME},	\
	{RDATA_TYPE_INT16,		/* AFSDB	18	*/\
		RDATA_TYPE_DNAME},	\
	{RDATA_TYPE_STR},		/* X25		19	*/\
	{RDATA_TYPE_STR,		\
		RDATA_TYPE_STR},	/* ISDN		20	*/\
	{RDATA_TYPE_INT16,		\
		RDATA_TYPE_DNAME},	/* RT		21	*/\
	{RDATA_TYPE_NSAP},		/* NSAP		21	*/\
	{},				/* skip 23		*/\
	{RDATA_TYPE_TYPE,		/* SIG		24	*/\
		RDATA_TYPE_ALGO,	\
		RDATA_TYPE_INT8,	\
		RDATA_TYPE_INT32,	\
		RDATA_TYPE_INT32,	/*TODO <- make _DATE	*/\
		RDATA_TYPE_INT32,	\
		RDATA_TYPE_INT16,	\
		RDATA_TYPE_DNAME,	\
		RDATA_TYPE_BASE64},	\
	{RDATA_TYPE_INT16,		/* KEY 		25	*/\
		RDATA_TYPE_INT8,	\
		RDATA_TYPE_INT8,	\
		RDATA_TYPE_BASE64},	\
	{RDATA_TYPE_INT16,		/* PX		26	*/\
		RDATA_TYPE_DNAME,	\
		RDATA_TYPE_DNAME},	\
	{},				/* skip 27		*/\
	{RDATA_TYPE_AAAA},		/* AAAA 	28	*/\
	{RDATA_TYPE_LOC},		/* LOC		29	*/\
	{RDATA_TYPE_DNAME,		/* NXT		30	*/\
		RDATA_TYPE_HEX},	\
	{}, {},				/* skip 31-32		*/\
	{RDATA_TYPE_INT16,		/* SRV		33	*/\
		RDATA_TYPE_INT16,	\
		RDATA_TYPE_INT16,	\
		RDATA_TYPE_DNAME},	\
	{},				/* skip 34		*/\
	{RDATA_TYPE_INT16,		/* NAPTR	35	*/\
		RDATA_TYPE_INT16,	\
		RDATA_TYPE_STR,		\
		RDATA_TYPE_STR,		\
		RDATA_TYPE_STR,		\
		RDATA_TYPE_DNAME},	\
	{RDATA_TYPE_INT16,		/* KX		36	*/\
		RDATA_TYPE_DNAME},	\
	{RDATA_TYPE_CERTTYPE,		/* CERT 	37	*/\
		RDATA_TYPE_INT16,	\
		RDATA_TYPE_INT8,	\
		RDATA_TYPE_BASE64},	\
	{}, 				/* skip 38		*/\
	{RDATA_TYPE_DNAME},		/* DNAME	39	*/\
	{}, 				/* skip 40		*/\
	{},				/* OPT		41	*/\
	{RDATA_TYPE_APL},		/* APL		42	*/\
	{RDATA_TYPE_INT16,		/* DS		43	*/\
		RDATA_TYPE_INT8,	\
		RDATA_TYPE_INT8,	\
		RDATA_TYPE_HEX},	\
	{},				/* SSHFP TODO 	44		*/\
	{},				/* skip 	45		*/\
	{RDATA_TYPE_TYPE,		/* RRSIG	46		*/\
		RDATA_TYPE_ALGO,	\
		RDATA_TYPE_INT8,	\
		RDATA_TYPE_INT32,	\
		RDATA_TYPE_INT32,	\
		RDATA_TYPE_INT32,	\
		RDATA_TYPE_INT16,	\
		RDATA_TYPE_DNAME,	\
		RDATA_TYPE_BASE64},	\
	{RDATA_TYPE_DNAME,		/* NSEC		47	*/\
		RDATA_TYPE_NSEC},	\
	{RDATA_TYPE_INT16,		/* DNSKEY	48	*/\
		RDATA_TYPE_INT8,	\
		RDATA_TYPE_ALGO,	\
		RDATA_TYPE_BASE64},	\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 49-58	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 59-68	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 69-78	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 79-88	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 89-98	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 99-108	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 109-118	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 119-128	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 129-138	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 139-148	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 149-158	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 159-168	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 169-178	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 179-188	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 189-198	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 199-208	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 209-218	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 219-228	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 229-238	*/\
	{}, {}, {}, {}, {}, {}, {}, {}, {}, {}, /* skip 239-248	*/\
	{}, 					/* skip 249	*/\
	{RDATA_TYPE_DNAME,			/* TSIG 250	*/\
		RDATA_TYPE_INT48,		\
		RDATA_TYPE_INT16,		\
		RDATA_TYPE_INT16,		\
		RDATA_TYPE_HEX,			\
		RDATA_TYPE_INT16,		\
		RDATA_TYPE_INT16,		\
		RDATA_TYPE_HEX},		\
	{},					/* IXFR		*/\
	{},					/* AXFR		*/\
	{},					/* MAILB	*/\
	{},					/* MAILA	*/\
	{}					/* ANY		*/\
}

/* set or clr packet flags */
#define SET_QR(x)	((x)->flags.qr) = 1
#define CLR_QR(x)	((x)->flags.qr) = 0
#define SET_RD(x)	((x)->flags.rd) = 1
#define CLR_RD(x)	((x)->flags.rd) = 0
#define SET_RA(x)	((x)->flags.ra) = 1
#define CLR_RA(x)	((x)->flags.ra) = 0
#define SET_AD(x)	((x)->flags.ad) = 1
#define CLR_AD(x)	((x)->flags.ad) = 0
#define SET_CD(x)	((x)->flags.cd) = 1
#define CLR_CD(x)	((x)->flags.cd) = 0
#define SET_AA(x)	((x)->flags.aa) = 1
#define CLR_AA(x)	((x)->flags.aa) = 0
/* get macros */
#define GET_TC(x)	(x)->flags.tc
			
/* more pkt specifics */
#define SET_UDPSIZE(x, y)	((x)->udppacketsize) = (y)
#define SET_RCODE(x, y) 	((x)->opt.xrcode) = (y)
#define SET_DNSSEC(x)		((x)->opt.dnssec_ok) = 1
#define SET_VERSION(x,y)	((x)->opt.dnssec_ok) = (y)

#define GET_UDPSIZE(x)	((x)->udppacketsize)
#define GET_RCODE(x) 	((x)->opt.xrcode)
#define GET_DNSSEC(x)	((x)->opt.dnssec_ok)
#define GET_VERSION(x)	((x)->opt.version)

extern struct gtab ztypes[];
extern struct gtab zclasses[];
extern struct gtab rcodes[];
extern int have_drill_opt;
#define MAX_RD_COUNT 25
extern int rr_rdata_types[65535][MAX_RD_COUNT];

uint16_t intbyname(const char *, struct gtab *);
uint16_t typebyname(const char *, struct gtab *);
const char * namebyint(uint16_t, struct gtab *);
char * typebyint(uint16_t, struct gtab *);
const char * classbyint(uint16_t, struct gtab *);
uint8_t dpacket_type(struct t_dpacket *);
uint16_t signature_covered_type(struct t_rr *);

#define PKT_REFERRAL 1
#define PKT_NXDOMAIN 2
#define PKT_NODATA   3
#define PKT_ANSWER   4
#define PKT_ERROR    8  /* no answer, no connection, empty packet */

#endif /* _DNSTYPES_H_ */
