/**
 dns.h
 * This file contains our internal representation of
 * dns rr's and packets
 * And other DNS related defines
 *
 * (c) NLnet Labs, 2004
 * 
 * See the file LICENSE for the license
 *   
 */

#ifndef _DNS_H
#define _DNS_H

/* default size of buffer */
#define BUFSIZE  512	
/* max id in 16 bits (2^16-1)*/
#define MAX_ID   65535	
/* maximum numbers of nameservers */
#define MAX_NS	 16	
/* default TTL */
#define DEF_TTL	 3600	

#define OFF	0
#define ON	1

enum enum_section {SEC_QUESTION, SEC_ANSWER, SEC_AUTH, SEC_ADD, SEC_MAX, SEC_ALL};
typedef enum enum_section t_section;

/**
 * Resource record rdata.
 */
struct t_rdata {
	/**
	 * Length of the data.
	 */
	uint16_t length;
	/**
	 * Data, formatted as 8-bit clean string.
	 */
	uint8_t	 *data;
};

/**
 * Resource record.
 */
struct t_rr {
	/**
	 * Owner name of the resource record.
	 */
	struct t_rdata *name;
	/**
	 * RR type of the resource record (A, MX etc).
	 */
	uint16_t type;
	/**
	 * Class of the resource record (IN, CH etc).
	 */
	uint16_t class;
	/**
	 * Time to live.
	 */
	uint32_t ttl;
	/**
	 * Section of the packet this resource record belongs to
	 * (Answer, Authority, Additional or Qeustion).
	 */
	t_section section;
	/**
	 * Number of rdata fields this resource records has.
	 */
	uint16_t rdcount;

	/**
	 * Array of rdata fields.
	 */
	struct t_rdata **rdata;

	/**
	 * Pointer to the next t_rr structure (to make linked lists,
	 * for instance for rrsets).
	 */
	struct t_rr *next;
};

/**
 * DNS Packet.
 * Represents a dns packet (answer or query).
 */
struct t_dpacket {
	/**
	 * Packet ID number.
	 */
	uint16_t id;

	struct {
		/** Query (0) or response (1). */
		unsigned qr		:1; 
		/** Query type (standard, inverse, etc.). */
		unsigned opcode		:4; 
		/** Answer is atuhoritative. */
		unsigned aa		:1; 
		/** Packet was truncated. */
		unsigned tc		:1; 
		/** recursion disired. */
		unsigned rd		:1; 
		/** recursion available. */
		unsigned ra		:1; 
		/** the infamous Z-bit. */
		unsigned z		:1; 
		/** Authenticated data. */
		unsigned ad		:1; 
		/** Checking disabled. */
		unsigned cd		:1; 
		/** Response code. */
		unsigned rcode		:4; 
	} flags; /* dns flags. */

	/**
	 * Number of resource records in each section.
	 */
	uint8_t count[SEC_MAX];

	/** 
	 * Pointer to first RR structure in this packet.
	 * All resource records are put in the packet in this linked list.
	 */
	struct t_rr *rrs;

	/**
	 * Size of the packet on the wire if it is an answer packet.
	 * Expected size of the answer if it is a query (for EDNS0 requests).
	 */
	uint16_t udppacketsize;

	/**
	 * OPT RR data.
	 */
	struct {
		/** Extended rcode. */
		unsigned xrcode         :8;
		/** EDNS version. */
		unsigned version        :8;
		/** DO bit ('do' is reserved in c). */
		unsigned dnssec_ok      :1;
		/** Reserved. */
		unsigned z              :15;
	} opt;

	/**
	 * Optional string that contains source of packet if it is an answer.
	 */
	char *serverip;
	
	/**
	 * Optional long that contains the time it took to do the query.
	 */
 	long querytime;
 	
 	/**
 	 * Date the query was done.
 	 */
	time_t date;
};

/* prototypes of dns.c */
struct t_rdata * rdata_create(const uint8_t *, size_t );
struct t_rdata * rdata_clone(struct t_rdata * );
void rdata_destroy(struct t_rdata *rdata);
struct t_rr * rr_create(struct t_rdata *, uint16_t, uint16_t, t_section);
int rr_add_rdata(struct t_rdata *, struct t_rr *);
int rr_set_name(struct t_rdata *, struct t_rr *);
int rr_set_class(uint16_t, struct t_rr *);
struct t_dpacket * dpacket_create(void);
void dpacket_destroy(struct t_dpacket *);
int dpacket_add_rr(struct t_rr *, t_section, struct t_dpacket *);
struct t_rr * dpacket_get_rrset(struct t_rdata *, uint16_t, struct t_dpacket *, t_section);
void rrset_set_ttl(struct t_rr *, uint32_t);
void rrset_sort(struct t_rr **);
uint8_t rdata_cmp(struct t_rdata *, struct t_rdata *);
struct t_rr * rr_clone(struct t_rr *, unsigned int);
char * rdata2str(struct t_rdata *);
uint8_t rdata2uint8(struct t_rdata *);
uint16_t rdata2uint16(struct t_rdata *);
uint32_t rdata2uint32(struct t_rdata *);
uint64_t rdata2uint64(struct t_rdata *);
uint8_t packet_is_opt(struct t_dpacket *);
int rr_cmp(struct t_rr *, struct t_rr *);
size_t rr_size(struct t_rr *);
int rr_compare(struct t_rr *, struct t_rr *);
struct t_rr *rr_add_rr(struct t_rr *, struct t_rr *);
void rr_destroy(struct t_rr *r, unsigned int);
struct t_rr *rr_get_rr(struct t_rr *, struct t_rr *);
uint8_t label_cnt(struct t_rr *);
struct t_rr *chop_labels_left(struct t_rr *, uint8_t);

#endif /* _DNS_H */
