/*
 * bits.c
 *
 * Convenience functions for bit manipulations
 *
 * (c) NLnet Labs, 2004
 *
 * See the file LICENSE for the license
 *
 */
 
#include "common.h"

#include <assert.h>

/**
 * Returns the value of the byte with only the bit at the given position set
 * (bitnr: 7=first bit, 0=last bit)
 */
uint8_t
bitval(unsigned short bitnr)
{
	switch (bitnr) {
		case 0:
			return 1;
		case 1:
			return 2;
		case 2:
			return 4;
		case 3:
			return 8;
		case 4:
			return 16;
		case 5:
			return 32;
		case 6:
			return 64;
		case 7:
			return 128;
		default:
			return 0;
	}
}

/**
 * Swaps the bit in the given byte at the given position
 * (bitnr: 7=first bit, 0=last bit)
 */
void
swap_bit(uint8_t *byte, unsigned short bitnr)
{
	assert(bitnr < 8);
	*byte = *byte ^ bitval(bitnr);
}

/**
 * Returns the value of the bit in the given byte at the given position
 * (bitnr: 7=first bit, 0=last bit)
 */
uint8_t
get_bit (uint8_t *byte, unsigned short bitnr)
{
	assert(bitnr < 8);
	return *byte & bitval(bitnr);
}

/**
 * Returns 1 if the specified bit is set, 0 if not
 */
unsigned int
bit_set (uint8_t *byte, unsigned short bitnr)
{
	assert(bitnr < 8);
	if ((*byte & bitval(bitnr)) > 0) {
		return 1;
	} else {
		return 0;
	}
}

/**
 * Sets the bit in the given byte at the given to the given value
 * (bitnr: 7=first bit, 0=last bit)
 * (value: 0=0 otherwise=1)
 */
void 
set_bit (uint8_t *byte, unsigned short bitnr, unsigned int value) 
{
	assert(bitnr < 8);
	if (value) {
		if (!get_bit(byte, bitnr)) {
			swap_bit(byte, bitnr);
		}
	} else {
		if (get_bit(byte, bitnr)) {
			swap_bit(byte, bitnr);
		}
	}
}

/**
 * Returns the uint8 value of the specified bits (where 'from' is least
 * significant bit, 7=first bit, 0=last bit, 'from' bit specifies 2^0 value)
 */
uint8_t
get_bits (uint8_t *byte, unsigned short from, unsigned short to)
{
	uint8_t result = 0;
	unsigned short i;
	assert(from < 8);
	assert(to < 8);
	for (i=from; i<=to; i++) {
		result += *byte & bitval(i);
	}
	return result >> from;
}

/**
 * sets the bits in byte '*byte' at positions between 'from' and 'to' to the corresponding
 * values from 'set_to'
 * (bitnr: 7=first bit, 0=last bit)
 */
void
set_bits (uint8_t *byte, uint8_t set_to, unsigned short from, unsigned short to)
{
	unsigned short i;
	assert(from < 8);
	assert(to < 8);
 	for (i=from; i<=to; i++) {
 		if (get_bit(&set_to, i)) {
 			if(!get_bit(byte, i)) {
 				swap_bit(byte, i);
 			}
 		} else {
 			if(get_bit(byte, i)) {
 				swap_bit(byte, i);
 			}
	 	}
	}
}

/**
 * Prints the bits in the given buffer (from offset, size bytes, len bits per line)
 */
void print_bits (uint8_t *byte, int offset, int size, int len)
{
	int i;
	int curpos = 0;
	short j;
	  
	for (i=0; i<size; i++) {
		for (j=7; j>=0; j--) {
			if (get_bit(&byte[offset+i], (unsigned short) j)) {
				printf("1");
			} else {
				printf("0");
			}

			curpos++;
			if (curpos % len == 0) {
				curpos = 0;
				printf("\n");
			}
		}
	}
	printf("\n");
}
	   
/**
 * Prints the (int-value of the) bytes in the given buffer (from offset, size bytes, len bytes per line) to stdout
 * If len=0, print all on the same line
 */
void
print_bytes (uint8_t *byte, int offset, int size, int len)
{
	int i;
	int curpos = 0;
	
	assert(byte != NULL);

	printf("Printing %d bytes from the buffer at %ld\n", size, (long) byte);

	if (len > 0) {
		printf("\t");
		for (i=0; i<len; i++) {
			printf("%3d ", i);
		}
	}
	
	for (i=0; i<size; i++) {
		if (curpos == 0) {
			printf("\n%d:\t", i);			
		}

		printf ("%3d ", (int) byte[i+offset]);
	
		curpos++;
		
		if (len != 0 && curpos % len == 0) {
			curpos = 0;
		}
	}
	printf("\n");
}

/**
 * Prints the hex-value of the bytes in the given buffer (from offset, size bytes, len bytes per line) to stdout
 * If len=0, print all on the same line
 */
void
print_bytes_hex (uint8_t *byte, int offset, int size, int len)
{
	printf("Printing %d bytes from the buffer at %p as hex values\n", size,
		byte);

	fprint_bytes_hex(stdout, byte, offset, size, len);
}

/**
 * Prints the hex-value of the bytes in the given buffer (from offset, size bytes, len bytes per line) to the given file handle
 * If len=0, print all on the same line
 */
void
fprint_bytes_hex (FILE *f, uint8_t *byte, int offset, int size, int len)
{
	int i, previ;
	int curpos = 0;
	
	assert(byte != NULL);

	if (len > 0) {
		fprintf(f, ";");
		for (i=0; i<len; i++) {
			fprintf(f, "%2d ", i);
		}
		fprintf(f, "\n;");
		for (i=0; i<len; i++) {
			fprintf(f, "-- ");
		}
		fprintf(f, "\n ");
	}
	
	previ=0;
	for (i=0; i<size; i++) {
		fprintf (f, "%02x ", (unsigned int) byte[i+offset]);
	
		curpos++;
		
		if (len != 0 && curpos % len == 0) {
			fprintf(f, "\t; %3d-%3d\n ", previ, i);
			previ = i+1;
			curpos = 0;
		}
	}
	
	if (len != 0 && curpos != 0) {
		for(; curpos<len; curpos++) {
			fprintf(f, "   ");
		}
		fprintf(f, "\t; %3d-%3d", previ, i-1);
	}
	
	fprintf(f, "\n");
}

/**
 * This function converts a base64 encoded string to a character string
 * len is the number of bytes in the string, and after the function
 * completes, len contains the number of bytes in the result
 */
char *base64_decode(unsigned char *bbuf, int strlen, size_t *result_len)
{

	BIO *bio, *bio_64;
	size_t blen; 
	char *str;

	/* malloc could be a bit smaller. It could? Explain */
	str = xmalloc((size_t) strlen);

	bio_64 = BIO_new(BIO_f_base64());
	BIO_set_flags(bio_64,BIO_FLAGS_BASE64_NO_NL);

	bio = BIO_new_mem_buf((void *) bbuf, strlen);
	bio = BIO_push(bio_64, bio);

	/* if correct, should now read and decode the b64 stuff in bbuf */
	blen = (size_t) BIO_read(bio, str, strlen);

	*result_len = blen;

	BIO_free_all(bio);
	return str;
}

/**
 * This function converts a character string to a base64 encoded string
 * len is the size of the buffer
 */
unsigned char *base64_encode(unsigned char *buf, int len) {
	unsigned char *ret;
	int b64_len;
	b64_len = (((len + 2) / 3) * 4) + 1;
	ret = (unsigned char *) xmalloc((size_t) b64_len);
	EVP_EncodeBlock(ret,buf,len);
	ret[b64_len - 1] = 0;
	return ret;
}

/**
 * Converts a hex string to binary data
 * len is the length of the string
 * buf is the buffer to store the result in
 * offset is the starting position in the result buffer
 *
 * This function returns the length of the result
 */
size_t
hexstr2bin(char *hexstr, int len, uint8_t *buf, size_t offset, size_t buf_len)
{
	char c;
	int i; 
	uint8_t int8 = 0;
	int sec = 0;
	size_t bufpos = 0;
	
	if (len % 2 != 0) {
		return 0;
	}

	for (i=0; i<len; i++) {
		c = hexstr[i];

		/* case insensitive, skip spaces */
		if (c != ' ') {
			if (c >= '0' && c <= '9') {
				int8 += c & 0x0f;  
			} else if (c >= 'a' && c <= 'z') {
				int8 += (c & 0x0f) + 9;   
			} else if (c >= 'A' && c <= 'Z') {
				int8 += (c & 0x0f) + 9;   
			} else {
				warning("Error in reading hex data: \n");
				warning("%s ('%c' at %d, should read %d bytes)\n", hexstr, c, i, len);
				return 0;
			}
			 
			if (sec == 0) {
				int8 = int8 << 4;
				sec = 1;
			} else {
				if (bufpos + offset + 1 <= buf_len) {
					buf[bufpos+offset] = int8;
					int8 = 0;
					sec = 0; 
					bufpos++;
				} else {
					fprintf(stderr, "Buffer too small in hexstr2bin\n");
					exit(1);
				}
			}
		}
                 
        }
        return bufpos;        
}

/**
 * Converts an IPv4 string to 16 bytes in network order, and places the
 * bytes at the specified pointer (remember to malloc :p)
 */
int
ipv42bin(const char *string, uint8_t *dest)
{
	char *str, *strp;
	int a,b,c,d;

	str = xmalloc(strlen(string));
	strncpy(str, string, strlen(string)+1);

	strp = str;
	a = atoi(str);
	str = index(str, '.')+1;
	b = atoi(str);
	str = index(str, '.')+1;
	c = atoi(str);
	str = index(str, '.')+1;
	d = atoi(str);
	
	if (a < 0 || a > 255 || b < 0 || b > 255 || c < 0 || c > 255 || d < 0 || d > 255) {
		warning("Bad IPv4 addres: %s\n", string);
		return RET_FAIL;
	}

	dest[0] = (uint8_t) a;
	dest[1] = (uint8_t) b;
	dest[2] = (uint8_t) c;
	dest[3] = (uint8_t) d;

	xfree(strp);
	
	return RET_SUC;

}


/**
 * Converts an IPv6 string to 16 bytes in network order, and places the
 * bytes at the specified pointer (remember to malloc :p)
 */
int
ipv62bin(const char *string, uint8_t *dest)
{
	/* remove colons and remember position of :: */
	char *address_str;
	size_t dp, len;
	size_t i, j;
	size_t result;

	address_str = xmalloc(33);
	
	dp = -1;
	len = 0;
	
	/* leading zeroes? */
	i=0;
	j=1;
	while(j < strlen(string) && string[j] != ':') {
		j++;
	}
	while(j-i < 4) {
		address_str[len] = '0';
		len++;
		j++;
	}
	
	for (i=0; i<=strlen(string); i++) {
		if (string[i] == ':') {
			/* check for double colon */
			if (i<strlen(string) && string[i+1] == ':') {
				dp = len;
			}
			/* pad with zeroes if number is not 4 chars */
			j = i + 1;
			while(j < strlen(string) && string[j] != ':') {
				j++;
			}
			while(j-i-1 < 4) {
				address_str[len] = '0';
				len++;
				j++;
			}
		} else {
			address_str[len] = string[i];
			len++;
		}
	}

	if (dp >= 0) {
		memcpy(&address_str[32-(len-dp)+1], &address_str[dp], len-dp);
		for (i=0; i<32-len+1; i++) {
			address_str[dp+i] = '0';
		}
	}

	/* todo: 16 should be given buf_len? */
	result = hexstr2bin(address_str, 32, dest, 0, 16);
	
	xfree(address_str);

	return RET_SUC;
}

/**
 * Converts the given 4 bytes to a string with the ipv4 address
 * This is not the way---XXX TODO
 */
char *
bin2ipv4(uint8_t *bin)
{
	char *result;
	
	result = xmalloc(16);
	snprintf(result, 16, "%d.%d.%d.%d", (int) bin[0], (int) bin[1], (int) bin[2], (int) bin[3]);
	return result;
}

/**
 * Converts the given 16 bytes to a string with the ipv6 address
 */
char *
bin2ipv6(uint8_t *buf)
{
	char *data;
	int stroffset, i;
	uint16_t int16;
	int offset = 0;
	data = xmalloc(48);

	i = 0;
	memcpy(&int16, &buf[offset+ (2 * i)], 2);
	stroffset = snprintf(data, 48, "%04x", (unsigned int) ntohs(int16));

	for(i = 1; i < 8; i++) {
		memcpy(&int16, &buf[offset + (2 * i)], 2);
		stroffset += snprintf(&data[stroffset], (size_t) (48 - stroffset), ":%04x", (unsigned int) ntohs(int16));
	}
	return data;
}
