/* serverside.c   Handles the server side of dopewars                   */
/* copyright (c)  1998-2000  ben webb                                   */
/*                Email: ben@bellatrix.pcl.ox.ac.uk                     */
/*                WWW: http://bellatrix.pcl.ox.ac.uk/~ben/dopewars/     */

/* This program is free software; you can redistribute it and/or        */
/* modify it under the terms of the GNU General Public License          */
/* as published by the Free Software Foundation; either version 2       */
/* of the License, or (at your option) any later version.               */

/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */

/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to the Free Software          */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston,               */
/*                   MA  02111-1307, USA.                               */


#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include "dopeos.h"
#include "serverside.h"
#include "dopewars.h"
#include "message.h"

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#define SD_SEND 1

/* Maximum time, in seconds, between reports from this server to the   */
/* metaserver. Setting this to be too small will most likely annoy the */
/* metaserver maintainer, and result in your host being blocked... ;)  */
#define METATIME (1200)

int TerminateRequest,ReregisterRequest;

int MetaTimeout;

#define LOGF(x, args...) if (Network) printf(x, ## args)
/* Prints a status message to standard output, unless the server is */
/* running locally as part of an antique or single-player game.     */

struct PLAYER *FirstServer;

/* Pointer to the filename of a pid file (if non-NULL) */
char *PidFile;

int SendSingleHighScore(struct PLAYER *Play,struct HISCORE *Score,
                        int index,char Bold);

int SendToMetaServer(char Up,int MetaSock,char *data,
                     struct sockaddr_in *MetaAddr) {
/* Sends server details, and any additional data, to the metaserver */
   char buf[2000];
   int numbytes;
   sprintf(buf,"R:%d\n%d\n%s\n%s",
           METAVERSION,Port,MetaServer.LocalName,MetaServer.Password);
   if (data) { strcat(buf,"\n"); strcat(buf,data); }
   numbytes=sendto(MetaSock,buf,strlen(buf),0,
                   (struct sockaddr *)MetaAddr,sizeof(struct sockaddr));
   if (numbytes==-1) {
      printf("Error: cannot send data to metaserver\n");
      return 0;
   }
   return 1;
}

void RegisterWithMetaServer(char Up,char SendData) {
/* Sends server details to the metaserver, if specified. If "Up" is  */
/* TRUE, informs the metaserver that the server is now accepting     */
/* connections - otherwise tells the metaserver that this server is  */
/* about to go down. If "SendData" is TRUE, then also sends game     */
/* data (e.g. scores) to the metaserver. If networking is disabled,  */
/* does nothing.                                                     */
#if NETWORKING
   struct HISCORE MultiScore[NUMHISCORE],AntiqueScore[NUMHISCORE];
   struct sockaddr_in MetaAddr;
   struct hostent *he;
   int MetaSock;
   char buf[900],prstr[100];
   int i,numbytes;
   if (!MetaServer.Active || !NotifyMetaServer) return;
   printf("%s metaserver at %s\n",SendData ? "Sending data to" : "Notifying",
                                  MetaServer.Name);
   if ((he=gethostbyname(MetaServer.Name))==NULL) {
      printf("Error: cannot locate metaserver\n");
      return;
   }
   MetaSock=socket(AF_INET,SOCK_DGRAM,0);
   if (MetaSock==-1) {
      printf("Error: cannot create socket for metaserver communication\n");
      return;
   }
   memset(&MetaAddr,0,sizeof(struct sockaddr_in));
   MetaAddr.sin_family=AF_INET;
   MetaAddr.sin_port=htons(MetaServer.UdpPort);
   MetaAddr.sin_addr=*((struct in_addr *)he->h_addr);

   sprintf(buf,"report\n%d\n%s\n%d\n%d\n%s",
           Up ? 1 : 0,VERSION,CountPlayers(FirstServer),
           MaxClients,MetaServer.Comment);
   if (!SendToMetaServer(Up,MetaSock,buf,&MetaAddr)) return;

   if (SendData) {
      if (HighScoreRead(MultiScore,AntiqueScore)) {
         for (i=0;i<NUMHISCORE;i++) {
            sprintf(buf,"multi\n%d\n%s^%s^%s^%c",
                    i,FormatPrice(MultiScore[i].Money,prstr),MultiScore[i].Time,
                    MultiScore[i].Name,MultiScore[i].Dead ? '1':'0');
            if (!SendToMetaServer(Up,MetaSock,buf,&MetaAddr)) return;
         }
      } else { printf("Error: cannot read high score file\n"); }
   }
   CloseSocket(MetaSock);
   MetaTimeout=time(NULL)+METATIME;
#endif /* NETWORKING */
}

void SendGameData(struct PLAYER *To) {
/* Sends all the game data required by a client (names of locations,  */
/* drugs, guns, etc.) This can be skipped if the server is virtual    */
/* (i.e. the network is down) because the client and server share the */
/* same game data already in this case.                               */
   char text[BUFLEN];
   char price[2][40];
   int i;
   if (Network) {
      sprintf(text,"%s^%d^%d^%d^%s^%s^%s^%s^%s^%s^%s^%s^",
              VERSION,NumLocation,NumGun,NumDrug,
              Names.Bitch,Names.Bitches,Names.Gun,Names.Guns,
              Names.Drug,Names.Drugs,Names.Month,Names.Year);
      SendServerMessage(NULL,C_NONE,C_INIT,To,text);
      sprintf(text,"0^%c%s^%s^",DT_PRICES,pricetostr(Prices.Spy,price[0],40),
                            pricetostr(Prices.Tipoff,price[1],40));
      SendServerMessage(NULL,C_NONE,C_DATA,To,text);
      for (i=0;i<NumGun;i++) {
         sprintf(text,"%d^%c%s^%s^%d^%d^",i,DT_GUN,Gun[i].Name,
                 pricetostr(Gun[i].Price,price[0],40),
                 Gun[i].Space,Gun[i].Damage);
         SendServerMessage(NULL,C_NONE,C_DATA,To,text);
      }
      for (i=0;i<NumDrug;i++) {
         sprintf(text,"%d^%c%s^%s^%s^",i,DT_DRUG,Drug[i].Name,
                 pricetostr(Drug[i].MinPrice,price[0],40),
                 pricetostr(Drug[i].MaxPrice,price[1],40));
         SendServerMessage(NULL,C_NONE,C_DATA,To,text);
      }
      for (i=0;i<NumLocation;i++) {
         sprintf(text,"%d^%c%s^",i,DT_LOCATION,Location[i].Name);
         SendServerMessage(NULL,C_NONE,C_DATA,To,text);
      }
   }
}

void HandleServerMessage(char *buf,struct PLAYER *ReallyFrom) {
/* Given a message "buf" which identifies itself as being from player */
/* "ReallyFrom" by the incoming socket, performs processing and sends */
/* suitable replies.                                                  */
   struct PLAYER *From,*To,*tmp,*pt;
   char Code,Data[BUFLEN],AICode;
   DopeEntry NewEntry;
   int i;
   price_t money;
   if (ProcessMessage(buf,&From,&AICode,&Code,&To,Data,FirstServer)==-1) {
      LOGF("Bad message\n");
      return;
   }
   if (From!=ReallyFrom && (From!=&Noone ||
       (Code!=C_NAME && Code!=C_NETMESSAGE))) {
       LOGF("Message is lying about its origin\n");
       LOGF("%s: %c: %s: %s\n",From ? From->Name : "",Code,
                                 To ? To->Name : "",Data);
       LOGF("Should be from %s\n",ReallyFrom ? ReallyFrom->Name : "NULL");
       return;
   }
   switch(Code) {
      case C_MSGTO:
         LOGF("%s->%s: %s\n",From->Name,To->Name,Data);
         SendServerMessage(From,AICode,Code,To,Data);
         break;
      case C_NETMESSAGE:
         printf("Net:%s\n",Data);
         shutdown(ReallyFrom->fd,SD_SEND);
/* Make sure they do actually disconnect, eventually! */
         if (ConnectTimeout) {
            ReallyFrom->ConnectTimeout=time(NULL)+(time_t)ConnectTimeout;
         }
         break;
      case C_NAME:
         pt=GetPlayerByName(Data,FirstServer);
         if (strlen(Data)>NAMELEN || (pt && pt!=From)) {
            if (ConnectTimeout) {
               ReallyFrom->ConnectTimeout=time(NULL)+(time_t)ConnectTimeout;
            }
            SendServerMessage(NULL,C_NONE,C_NEWNAME,ReallyFrom,NULL);
         } else if ((ReallyFrom && ReallyFrom->Name[0]==0 && Network) || 
                   (!Network && From==&Noone)) {
            if (CountPlayers(FirstServer)<=MaxClients || !Network) { 
               SendGameData(ReallyFrom);
               if (!Network) From=AddPlayer(0,&FirstServer);
               else From=ReallyFrom;
               InitialisePlayer(From);
               ClearFightTimeout(From);
               strcpy(From->Name,Data);
               for (pt=FirstServer;pt;pt=pt->Next) {
                  if (pt!=From) {
                     SendServerMessage(NULL,C_NONE,C_LIST,From,pt->Name);
                  }
               }
               SendServerMessage(NULL,C_NONE,C_ENDLIST,From,NULL);
               RegisterWithMetaServer(TRUE,FALSE);
               From->ConnectTimeout=0;

               LOGF("%s joins the game!\n",From->Name);
               BroadcastToClients(C_NONE,C_JOIN,From->Name,NULL,From);
               From->EventNum=E_ARRIVE;
               SendPlayerData(From);
               SendEvent(From);
            } else {
               From=ReallyFrom;
               printf("MaxClients (%d) exceeded - dropping connection\n",
                      MaxClients);
               sprintf(buf,"Sorry, but this server has a limit of %d \
player%s, which has been reached.^Please try connecting again later.",
                       MaxClients,MaxClients==1 ? "" : "s");
               SendServerMessage(NULL,C_NONE,C_PRINTMESSAGE,From,buf);
               shutdown(From->fd,SD_SEND);
/* Make sure they do actually disconnect, eventually! */
               if (ConnectTimeout) {
                  From->ConnectTimeout=time(NULL)+(time_t)ConnectTimeout;
               }
            }
         } else {
            LOGF("%s will now be known as %s\n",From->Name,Data);
            BroadcastToClients(C_NONE,C_RENAME,Data,From,From);
            strcpy(From->Name,Data);
         }
         break;
      case C_WANTQUIT:
         if (From->EventNum!=E_FINISH) FinishGame(From,NULL);
         break;
      case C_REQUESTJET:
         i=atoi(Data);
         if (From->EventNum==E_ATTACK || From->EventNum==E_DEFEND ||
             From->EventNum==E_WAITATTACK || From->EventNum==E_FREEFORALL) {
             BreakoffCombat(From,FALSE);
         }
         if (NumTurns>0 && From->Turn>=NumTurns && From->EventNum!=E_FINISH) {
            FinishGame(From,"Your dealing time is up...");
         } else if (i!=From->IsAt && (NumTurns==0 || From->Turn<NumTurns) && 
                    From->EventNum==E_NONE && From->Health>0) {
/*          LOGF("%s: Moving to %s\n",From->Name,Location[i].Name); */
            From->IsAt=(char)i;
            From->Turn++;
            From->Debt=(price_t)((float)From->Debt*1.1);
            From->Bank=(price_t)((float)From->Bank*1.05);
            SendPlayerData(From);
            From->EventNum=E_SUBWAY;
            SendEvent(From);
         } else {
            LOGF("%s: DENIED jet to %s\n",From->Name,Location[i].Name);
         }
         break;
      case C_REQUESTSCORE:
         SendHighScores(From,FALSE,NULL);
         break;
      case C_CONTACTSPY:
         for (tmp=FirstServer;tmp;tmp=tmp->Next) {
            if (tmp!=From && GetListEntry(&(tmp->SpyList),From)>=0) {
               SendSpyReport(From,tmp);
            }
        }
        break;
      case C_DEPOSIT:
         money=strtoprice(Data);
         if (From->Bank+money >=0 && From->Cash-money >=0) {
            From->Bank+=money; From->Cash-=money;
           SendPlayerData(From);
         }
         break;
      case C_PAYLOAN:
         money=strtoprice(Data);
         if (From->Debt-money >=0 && From->Cash-money >=0) {
            From->Debt-=money; From->Cash-=money;
            SendPlayerData(From);
         }
         break;
      case C_BUYOBJECT:
         BuyObject(From,Data);
         break;
      case C_FIGHTACT:
         if (From->EventNum==E_ATTACK || From->EventNum==E_FREEFORALL) {
            AttackPlayer(From,From->Attacked,
               TotalGunsCarried(From)>0 ? AT_SHOOT : 0);
         } else if (From->EventNum==E_DEFEND) {
            for (tmp=FirstServer;tmp;tmp=tmp->Next) {
               if ((tmp->EventNum==E_FREEFORALL || tmp->EventNum==E_WAITATTACK) 
                   && tmp->Attacked==From) {
                  AttackPlayer(From,tmp,
                               TotalGunsCarried(From)>0 ? AT_SHOOT : 0);
               }
            }
         }
         break;
      case C_ANSWER:
         HandleAnswer(From,To,Data);
         break;
      case C_DONE:
         if (From->EventNum!=E_NONE && From->EventNum<E_OUTOFSYNC) {
            From->EventNum++; SendEvent(From);
         }
         break;
      case C_SPYON:
         if (From->Cash >= Prices.Spy) {
            LOGF("%s now spying on %s\n",From->Name,To->Name);
            From->Cash -= Prices.Spy;
            LoseBitch(From,NULL,NULL);
            NewEntry.Play=From; NewEntry.Turns=-1;
            AddListEntry(&(To->SpyList),&NewEntry);
            SendPlayerData(From);
         } else {
            LOGF("%s spy on %s: DENIED\n",From->Name,To->Name);
         }
         break;
      case C_TIPOFF:
         if (From->Cash >= Prices.Tipoff) {
            LOGF("%s tipped off the cops to %s\n",From->Name,To->Name);
            From->Cash -= Prices.Tipoff;
            LoseBitch(From,NULL,NULL);
            NewEntry.Play=From; NewEntry.Turns=0;
            AddListEntry(&(To->TipList),&NewEntry);
            SendPlayerData(From);
         } else {
            LOGF("%s tipoff about %s: DENIED\n",From->Name,To->Name);
         }
         break;
      case C_SACKBITCH:
         LoseBitch(From,NULL,NULL);
         SendPlayerData(From);
         break;
      case C_MSG:
         LOGF("%s: %s\n",From->Name,Data);
         BroadcastToClients(C_NONE,C_MSG,Data,From,From);
         break;
      default:
        LOGF("%s:%c:%s:%s\n",From->Name,Code,To->Name,Data);
        break;
   }
}

void ClientLeftServer(struct PLAYER *Play) {
/* Notifies all clients that player "Play" has left the game and */
/* cleans up after them if necessary.                            */
   struct PLAYER *tmp;
   if (Play->EventNum==E_ATTACK || Play->EventNum==E_DEFEND ||
       Play->EventNum==E_WAITATTACK || Play->EventNum==E_FREEFORALL) {
      BreakoffCombat(Play,TRUE);
   }
   for (tmp=FirstServer;tmp;tmp=tmp->Next) if (tmp!=Play) {
      RemoveAllEntries(&(tmp->TipList),Play);
      RemoveAllEntries(&(tmp->SpyList),Play);
   }
   BroadcastToClients(C_NONE,C_LEAVE,Play->Name,Play,Play);
}

#if NETWORKING
int CheckServerSockets(fd_set *readfs,char WantQuit) {
/* Checks all client players for messages. If the server "wants" to  */
/* quit (i.e. it's interactive, and the user has typed "quit") then  */
/* WantQuit should be set to TRUE. If all the sockets in readfs have */
/* closed and WantQuit is TRUE, then -1 is returned to signal the    */
/* server to quit for good. Otherwise, 0 is returned.                */
   struct PLAYER *tmp;
   char buf[BUFLEN];
   time_t timenow;

   timenow=time(NULL);
   tmp=FirstServer;
   while (tmp) {
      if (FD_ISSET(tmp->fd,readfs)) {
         if (bgets(buf,BUFLEN,tmp->fd)==NULL) {
            if (!WantQuit && tmp->Name[0]) {
               LOGF("%s leaves the server!\n",tmp->Name);
               ClientLeftServer(tmp);
/* Blank the name, so that CountPlayers ignores this player */
               tmp->Name[0]=0;
/* Report the new high scores (if any) and the new number of players
   to the metaserver */
               RegisterWithMetaServer(TRUE,TRUE);
            }
            RemovePlayer(tmp,&FirstServer);
            if (!FirstServer && WantQuit) return -1;
            else return 0;
         } else {
/* OK, we have some data. Reset the idle timeout (if necessary) and
   handle the message */
            if (IdleTimeout) tmp->IdleTimeout=timenow+(time_t)IdleTimeout;
            chomp(buf);
            HandleServerMessage(buf,tmp);
         }
      }
      tmp=tmp->Next;
   }
   return 0;
}
#endif /* NETWORKING */

void CleanUpServer() {
/* Closes down the server and frees up associated handles and memory */
   if (Server) RegisterWithMetaServer(FALSE,FALSE);
   while (FirstServer) {
      RemovePlayer(FirstServer,&FirstServer);
   }
#if NETWORKING
   if (Server) CloseSocket(ListenSock);
#endif
}

void ReregisterHandle(int sig) {
/* Responds to a SIGUSR1 signal, and requests the main event loop to  */
/* reregister the server with the dopewars metaserver.                */
   ReregisterRequest=1;
}

void BreakHandle(int sig) {
/* Traps an attempt by the user to send dopewars a SIGTERM or SIGINT  */
/* (e.g. pressing Ctrl-C) and signals for a "nice" shutdown. Restores */
/* the default signal action (to terminate without cleanup) so that   */
/* the user can still close the program easily if this cleanup code   */
/* then causes problems or long delays.                               */
   struct sigaction sact;
   TerminateRequest=1;
   sact.sa_handler=SIG_DFL;
   sact.sa_flags=0;
   sigaction(SIGTERM,&sact,NULL);
   sigaction(SIGINT,&sact,NULL);
}

void PrintHelpTo(FILE *fp) {
/* Prints the server help screen to the given file pointer */
   int i;
   char Variable[80];
   fprintf(fp,"dopewars server version %s commands and settings\n\n",
          VERSION);

   fprintf(fp,"help                       Displays this help screen\n");
   fprintf(fp,"list                       Lists all players logged on\n");
   fprintf(fp,"push <player>              Politely asks the named player to \
leave\n");
   fprintf(fp,"kill <player>              Abruptly breaks the connection \
with the named player\n");
   fprintf(fp,"msg:<mesg>                 Send message to all players\n");
   fprintf(fp,"quit                       Gracefully quit, after notifying \
all players\n");
   fprintf(fp,"<variable>=<value>         Sets the named variable to the \
given value\n");
   fprintf(fp,"<variable>                 Displays the value of the named \
variable\n");
   fprintf(fp,"<list>[x].<var>=<value>    Sets the named variable in the \
given list,\n");
   fprintf(fp,"                           index x, to the given value\n");
   fprintf(fp,"<list>[x].<var>            Displays the value of the named \
list variable\n");
   fprintf(fp,"\nValid variables are listed below:-\n\n");
   for (i=0;i<NUMGLOB;i++) {
      if (Globals[i].NameStruct[0]) {
         sprintf(Variable,"%s%s.%s",Globals[i].NameStruct,
                 Globals[i].StructListPt ? "[x]" : "",Globals[i].Name);
      } else {
         strcpy(Variable,Globals[i].Name);
      }
      fprintf(fp,"%-26s %s\n",Variable,Globals[i].Help);
   }
   fprintf(fp,"\n\n");
}

void ServerHelp() {
/* Displays a simple help screen listing the server commands and options */
   int i;
   FILE *fp;
   fp=popen(Pager,"w");
   if (fp) {
      PrintHelpTo(fp);
      i=pclose(fp);
      if (i==-1 || (WIFEXITED(i) && WEXITSTATUS(i)==127)) {
         ReportError("Pager exited abnormally - using stdout instead...");
         PrintHelpTo(stdout);
      }
   }
}

#if NETWORKING
void CreatePidFile() {
/* Creates a pid file (if "PidFile" is non-NULL) and writes the process */
/* ID into it                                                           */
   FILE *fp;
   if (!PidFile) return;
   fp=fopen(PidFile,"w");
   if (fp) {
      printf("Maintaining pid file %s\n",PidFile);
      fprintf(fp,"%ld\n",(long)getpid());
      fclose(fp);
      chmod(PidFile,S_IREAD|S_IWRITE);
   } else fprintf(stderr,"Cannot create pid file %s\n",PidFile);
}

void RemovePidFile() {
/* Removes the previously-created pid file "PidFile" */
   if (PidFile) unlink(PidFile);
}

void ServerLoop() {
/* Initialises server, processes network and interactive messages, and */
/* finally cleans up the server on exit.                               */
   struct PLAYER *tmp,*tmp2;
   struct sockaddr_in ServerAddr,ClientAddr;
   fd_set readfs,errorfs;
   int cadsize,topsock,ClientSock,Index;
   char WantQuit=FALSE,buf[BUFLEN],HasEquals;
   char InputClosed=FALSE;
   char name[80],sname[80],value[80];
   struct timeval timeout;
   struct sigaction sact;
   int MinTimeout;

   CreatePidFile();

/* Make the output line-buffered, so that the log file (if used) is */
/* updated regularly                                                */
   fflush(stdout);

#ifdef SETVBUF_REVERSED /* 2nd and 3rd arguments are reversed on some systems */
   setvbuf(stdout,_IOLBF,(char *)NULL,0);
#else
   setvbuf(stdout,(char *)NULL,_IOLBF,0);
#endif

   Network=TRUE;
   FirstServer=NULL;
   ListenSock=socket(AF_INET,SOCK_STREAM,0);
   if (ListenSock==-1) {
      perror("create socket"); exit(1);
   }
   SetReuse(ListenSock);
   fcntl(ListenSock,F_SETFL,O_NONBLOCK);
  
   ServerAddr.sin_family=AF_INET;
   ServerAddr.sin_port=htons(Port);
   ServerAddr.sin_addr.s_addr=INADDR_ANY;
   memset(ServerAddr.sin_zero,0,sizeof(ServerAddr.sin_zero));
   if (bind(ListenSock,(struct sockaddr *)&ServerAddr,
       sizeof(struct sockaddr)) == -1) {
      perror("bind socket"); exit(1);
   }

   printf("dopewars server version %s ready and waiting for \
connections on port %d\n\n",VERSION,Port);
   printf("For assistance with server commands, enter the command \"help\"\n");

   if (listen(ListenSock,10)==-1) {
      perror("listen socket"); exit(1);
   }

   MetaTimeout=0;

   TerminateRequest=ReregisterRequest=0;

#if !CYGWIN
   sact.sa_handler=ReregisterHandle;
   sact.sa_flags=0;
   sigemptyset(&sact.sa_mask);
   if (sigaction(SIGUSR1,&sact,NULL)==-1) {
      printf("Cannot install SIGUSR1 interrupt handler!\n");
   }
#endif

   sact.sa_handler=BreakHandle;
   sact.sa_flags=0;
   sigemptyset(&sact.sa_mask);
   if (sigaction(SIGINT,&sact,NULL)==-1) {
      printf("Cannot install SIGINT interrupt handler!\n");
   }
   if (sigaction(SIGTERM,&sact,NULL)==-1) {
      printf("Cannot install SIGTERM interrupt handler!\n");
   }
#if !CYGWIN
   if (sigaction(SIGHUP,&sact,NULL)==-1) {
      printf("Cannot install SIGHUP interrupt handler!\n");
   }
#endif
   sact.sa_handler=SIG_IGN;
   sact.sa_flags=0;
   if (sigaction(SIGPIPE,&sact,NULL)==-1) {
      printf("Cannot install pipe handler!\n");
   }

   RegisterWithMetaServer(TRUE,TRUE);

   while (1) {
      FD_ZERO(&readfs);
      if (!InputClosed) FD_SET(0,&readfs);
      FD_SET(ListenSock,&readfs);
      FD_ZERO(&errorfs);
      FD_SET(ListenSock,&errorfs);
      topsock=ListenSock+1;
      for (tmp=FirstServer;tmp;tmp=tmp->Next) {
         if (tmp->fd>0) {
            FD_SET(tmp->fd,&readfs);
            FD_SET(tmp->fd,&errorfs);
            if (tmp->fd>=topsock) topsock=tmp->fd+1;
         }
      }
      MinTimeout=GetMinimumTimeout(FirstServer);
      if (MinTimeout!=-1) {
         timeout.tv_sec=MinTimeout;
         timeout.tv_usec=0;
      }
      if (bselect(topsock,&readfs,NULL,&errorfs,
                 MinTimeout==-1 ? NULL : &timeout)==-1) {
         if (errno==EINTR) {
            if (ReregisterRequest) {
               ReregisterRequest=0;
               RegisterWithMetaServer(TRUE,TRUE);
               continue;
            } else if (TerminateRequest) break; else continue;
         }
         perror("select"); exit(1);
      }
      HandleTimeouts(&FirstServer);
      if (FD_ISSET(0,&readfs)) {
         if (fgets(buf,sizeof(buf),stdin)==NULL) {
            buf[0]=0;
            puts("Standard input closed.");
            InputClosed=TRUE;
         }
         chomp(buf);
         if (ParseNameValue(buf,name,80,value,80,sname,80,&Index,
             &HasEquals)==0 && !HandleOption(name,sname,Index,value,
             "in command line",TRUE,CountPlayers(FirstServer) > 0,HasEquals)) {
            if (strcasecmp(name,"help")==0 || strcasecmp(name,"h")==0 ||
                strcmp(name,"?")==0) {
               ServerHelp();
            } else if (strcasecmp(name,"quit")==0) {
               if (!FirstServer) break;
               WantQuit=TRUE;
               BroadcastToClients(C_NONE,C_QUIT,NULL,NULL,NULL);
            } else if (strncasecmp(name,"msg:",4)==0) {
               BroadcastToClients(C_NONE,C_MSG,name+4,NULL,NULL);
            } else if (strcasecmp(name,"list")==0) {
               if (FirstServer) {
                  puts("Users currently logged on:-");
                  for (tmp=FirstServer;tmp;tmp=tmp->Next) puts(tmp->Name);
               } else puts("No users currently logged on!");
            } else if (strncasecmp(name,"push ",5)==0) {
               tmp=GetPlayerByName(name+5,FirstServer);
               if (tmp) {
                  printf("Pushing %s\n",tmp->Name);
                  SendServerMessage(NULL,C_NONE,C_PUSH,tmp,NULL);
               } else puts("No such user!");
            } else if (strncasecmp(name,"kill ",5)==0) {
               tmp=GetPlayerByName(name+5,FirstServer);
               if (tmp) {
                  printf("%s killed\n",tmp->Name);
                  BroadcastToClients(C_NONE,C_KILL,tmp->Name,tmp,FirstServer);
                  RemovePlayer(tmp,&FirstServer);
               } else puts("No such user!");
            } else {
               ReportError("Unknown command - try \"help\" for help...");
            }
         }
      }
      if (CheckServerSockets(&readfs,WantQuit)==-1) {
         break;
      }
      if (FD_ISSET(ListenSock,&readfs)) {
         cadsize=sizeof(struct sockaddr);
         if ((ClientSock=accept(ListenSock,(struct sockaddr *)&ClientAddr,
             &cadsize))==-1) {
            perror("accept socket"); exit(1);
         }
         LOGF("server: got connection from %s\n",
                inet_ntoa(ClientAddr.sin_addr));
         tmp=AddPlayer(ClientSock,&FirstServer);
         if (ConnectTimeout) {
            tmp->ConnectTimeout=time(NULL)+(time_t)ConnectTimeout;
         }
      }
      for (tmp=FirstServer;tmp;tmp=tmp->Next) if (FD_ISSET(tmp->fd,&errorfs)) {
         LOGF("error from client: %d\n",tmp->fd);
         CleanUpServer(); exit(1);
      }
   }
   CleanUpServer();
   RemovePidFile();
}
#endif /* NETWORKING */

void FinishGame(struct PLAYER *Play,char *Message) {
/* Tells player "Play" that the game is over; display "Message" */
   ClientLeftServer(Play);
   Play->EventNum=E_FINISH;
   SendHighScores(Play,TRUE,Message);
   shutdown(Play->fd,SD_SEND);
/* Make sure they do actually disconnect, eventually! */
   if (ConnectTimeout) {
      Play->ConnectTimeout=time(NULL)+(time_t)ConnectTimeout;
   }
}

void HighScoreTypeRead(struct HISCORE *HiScore,FILE *fp) {
/* Reads a batch of NUMHISCORE high scores into "HiScore" from "fp" */
   int i;
   char text[100];
   for (i=0;i<NUMHISCORE;i++) {
      if (read_string(fp,HiScore[i].Name,80)==EOF) break;
      read_string(fp,HiScore[i].Time,80);
      read_string(fp,text,80);
      HiScore[i].Money=strtoprice(text);
      HiScore[i].Dead=fgetc(fp);
   }
}

void HighScoreTypeWrite(struct HISCORE *HiScore,FILE *fp) {
/* Writes out a batch of NUMHISCORE high scores from "HiScore" to "fp" */
   int i;
   char text[100];
   for (i=0;i<NUMHISCORE;i++) {
      fwrite(HiScore[i].Name,strlen(HiScore[i].Name)+1,1,fp);
      fwrite(HiScore[i].Time,strlen(HiScore[i].Time)+1,1,fp);
      pricetostr(HiScore[i].Money,text,100);
      fwrite(text,strlen(text)+1,1,fp);
      fputc(HiScore[i].Dead,fp);
   }
}

int HighScoreRead(struct HISCORE *MultiScore,struct HISCORE *AntiqueScore) {
/* Reads all the high scores into MultiScore and                           */
/* AntiqueScore (antique mode scores). Returns 1 on success, 0 on failure. */
   FILE *fp;
   memset(MultiScore,0,sizeof(struct HISCORE)*NUMHISCORE);
   memset(AntiqueScore,0,sizeof(struct HISCORE)*NUMHISCORE);
   fp=fopen(HiScoreFile,"r");
   if (fp) {
      HighScoreTypeRead(AntiqueScore,fp);
      HighScoreTypeRead(MultiScore,fp);
      fclose(fp);
   } else return 0;
   return 1;
}

int HighScoreWrite(struct HISCORE *MultiScore,struct HISCORE *AntiqueScore) {
/* Writes out all the high scores from MultiScore and AntiqueScore; returns */
/* 1 on success, 0 on failure.                                              */
   FILE *fp;
   fp=fopen(HiScoreFile,"w");
   if (fp) {
      HighScoreTypeWrite(AntiqueScore,fp);
      HighScoreTypeWrite(MultiScore,fp);
      fclose(fp);
   } else return 0;
   return 1;
}

void SendHighScores(struct PLAYER *Play,char EndGame,char *Message) {
/* Adds "Play" to the high score list if necessary, and then sends the */
/* scores over the network to "Play"                                   */
/* If "EndGame" is TRUE, add the current score if it's high enough and */
/* display an explanatory message. "Message" is tacked onto the start  */
/* if it's non-NULL. The client is then informed that the game's over. */
   struct HISCORE MultiScore[NUMHISCORE],AntiqueScore[NUMHISCORE],Score;
   struct HISCORE *HiScore;
   struct tm *timep;
   time_t tim;
   char text[BUFLEN];
   int i,j,InList=-1;
   if (!HighScoreRead(MultiScore,AntiqueScore)) {
      LOGF("Error: Unable to read high score file %s\n",HiScoreFile);
   }
   text[0]=0;
   if (Message) {
      strcpy(text,Message);
      if (strlen(text)>0) strcat(text,"^");
   }
   if (WantAntique) HiScore=AntiqueScore; else HiScore=MultiScore;
   if (EndGame) {
      Score.Money=Play->Cash+Play->Bank-Play->Debt;
      strcpy(Score.Name,Play->Name);
      if (Play->Health==0) Score.Dead=1; else Score.Dead=0;
      tim=time(NULL);
      timep=gmtime(&tim);
      strftime(Score.Time,80,"%d-%m-%Y",timep);
      for (i=0;i<NUMHISCORE;i++) {
         if (InList==-1 && (Score.Money > HiScore[i].Money ||
             HiScore[i].Time[0]==0)) {
            InList=i;
            strcat(text,"Congratulations! You made the high scores!");
            SendPrintMessage(NULL,C_NONE,Play,text);
            for (j=NUMHISCORE-1;j>i;j--) {
               memcpy(&HiScore[j],&HiScore[j-1],sizeof(struct HISCORE));
            }
            memcpy(&HiScore[i],&Score,sizeof(struct HISCORE));
            break;
         }
      }
      if (InList==-1) {
         strcat(text,"You didn't even make the high score table...");
         SendPrintMessage(NULL,C_NONE,Play,text);
      }
   }
   SendServerMessage(NULL,C_NONE,C_STARTHISCORE,Play,NULL);

   j=0;
   for (i=0;i<NUMHISCORE;i++) {
      if (SendSingleHighScore(Play,&HiScore[i],j,InList==i)) j++;
   }
   if (InList==-1 && EndGame) SendSingleHighScore(Play,&Score,j,1);
   SendServerMessage(NULL,C_NONE,C_ENDHISCORE,Play,EndGame ? "end" : NULL);
   if (!EndGame) SendDrugsHere(Play,FALSE);
   if (EndGame && !HighScoreWrite(MultiScore,AntiqueScore)) {
      LOGF("Error: Unable to write high score file %s\n",HiScoreFile);
   }
}

int SendSingleHighScore(struct PLAYER *Play,struct HISCORE *Score,
                        int index,char Bold) {
/* Sends a single high score in "Score" with position "index" to player  */
/* "Play". If Bold is TRUE, instructs the client to display the score in */
/* bold text.                                                            */
   char Data[100],Price[100];
   if (Score->Time[0]==0) return 0;
   sprintf(Data,"%d^%c%c%18s  %-14s %-34s %8s%c",index,Bold ? 'B' : 'N',
                Bold ? '>' : ' ',FormatPrice(Score->Money,Price),
                Score->Time,Score->Name,Score->Dead ? "(R.I.P.)" : "",
                Bold ? '<' : ' ');
   SendServerMessage(NULL,C_NONE,C_HISCORE,Play,Data);
   return 1;
}

void SendEvent(struct PLAYER *To) {
/* In order for the server to keep track of the state of each client, each    */
/* client's state is identified by its EventNum data member. So, for example, */
/* there is a state for fighting the cops, a state for going to the bank, and */
/* so on. This function instructs client player "To" to carry out the actions */
/* expected of it in its current state. It is the client's responsibility to  */
/* ensure that it carries out the correct actions to advance itself to the    */
/* "next" state; if it fails in this duty it will hang!                       */
   price_t Money;
   int i,j;
   char text[BUFLEN],buf[BUFLEN];
   struct PLAYER *Play;
   if (!To) return;
   if (To->EventNum==E_MAX) To->EventNum=E_NONE;
   if (To->EventNum==E_NONE || To->EventNum>=E_OUTOFSYNC) return;
   Money=To->Cash+To->Bank-To->Debt;

   ClearPrices(To);

   while (To->EventNum<E_MAX) {
      switch (To->EventNum) {
         case E_SUBWAY:
            SendServerMessage(NULL,C_NONE,C_SUBWAYFLASH,To,NULL);
            break;
         case E_OFFOBJECT:
            for (i=0;i<To->TipList.Number;i++) {
               LOGF("%s: Tipoff from %s\n",To->Name,
                      To->TipList.Data[i].Play->Name);
               To->OnBehalfOf=To->TipList.Data[i].Play;
               SendCopOffer(To,FORCECOPS);
               return;
            }
            for (i=0;i<To->SpyList.Number;i++) {
/*               LOGF("%s: Checking spy from %s, turns=%d\n",To->Name,
                      To->SpyList.Data[i].Play->Name,
                      To->SpyList.Data[i].Turns);*/
               if (To->SpyList.Data[i].Turns<0) {
/*                  LOGF("%s: New spy from %s\n",To->Name,
                         To->SpyList.Data[i].Play->Name);*/
                  To->OnBehalfOf=To->SpyList.Data[i].Play;
                  SendCopOffer(To,FORCEBITCH);
                  return;
               }
               To->SpyList.Data[i].Turns++;
               if (To->SpyList.Data[i].Turns>3 && 
                   brandom(0,100)<10+To->SpyList.Data[i].Turns) {
                   if (TotalGunsCarried(To) > 0) j=brandom(0,NUMDISCOVER);
                   else j=brandom(0,NUMDISCOVER-1);
                   sprintf(text,"One of your %s was spying for %s.\
^The spy %s!",Names.Bitches,To->SpyList.Data[i].Play->Name,Discover[j]);
                   if (j!=DEFECT) LoseBitch(To,NULL,NULL);
                   SendPlayerData(To);
                   SendPrintMessage(NULL,C_NONE,To,text);
                   sprintf(text,"Your spy working with %s has been \
discovered!^The spy %s!",To->Name,Discover[j]);
                   if (j==ESCAPE) GainBitch(To->SpyList.Data[i].Play);
                   To->SpyList.Data[i].Play->Flags &= ~SPYINGON;
                   SendPlayerData(To->SpyList.Data[i].Play);
                   SendPrintMessage(NULL,C_NONE,
                                     To->SpyList.Data[i].Play,text);
                   RemoveListEntry(&(To->SpyList),i);
                   i--;
               }
            }
            if (Money>3000000) i=130;   
            else if (Money>1000000) i=115;   
            else i=100;   
            if (brandom(0,i)>75) {
               if (SendCopOffer(To,NOFORCE)) return;
            }
            break;
         case E_SAYING:
            if (!Sanitized && (brandom(0,100) < 15)) {
               if (brandom(0,100)<50) {
                  sprintf(text," The lady next to you on the subway said,^\
   \"%s\"%s",SubwaySaying[brandom(0,NUMSUBWAY)],brandom(0,100)<30 ? 
                  "^    (at least, you -think- that's what she said)" : "");
               } else {
                  sprintf(text," You hear someone playing %s",
                          Playing[brandom(0,NUMPLAYING)]);
               }
               SendPrintMessage(NULL,C_NONE,To,text);
            }
            break;
         case E_LOANSHARK:
            if (To->IsAt+1==LoanSharkLoc && To->Debt>0) {
               sprintf(text,"YN^Would you like to visit %s?",LoanSharkName);
               SendQuestion(NULL,C_ASKLOAN,To,text);
               return;
            }
            break;
         case E_BANK:
            if (To->IsAt+1==BankLoc) {
               sprintf(text,"YN^Would you like to visit %s?",BankName);
               SendQuestion(NULL,C_ASKBANK,To,text);
               return;
            }
            break;
         case E_GUNSHOP:
            if (To->IsAt+1==GunShopLoc && !Sanitized && !WantAntique) {
               sprintf(text,"YN^Would you like to visit %s?",GunShopName);
               SendQuestion(NULL,C_ASKGUNSHOP,To,text);
               return;
            }
            break;
         case E_ROUGHPUB:
            if (To->IsAt+1==RoughPubLoc && !WantAntique) {
               sprintf(text,"YN^Would you like to visit %s?",RoughPubName);
               SendQuestion(NULL,C_ASKPUB,To,text);
               return;
            }
            break;
         case E_HIREBITCH:
            if (To->IsAt+1==RoughPubLoc && !WantAntique) {
               To->Bitches.Price=brandom(Bitch.MinPrice,Bitch.MaxPrice);
               sprintf(text,"YN^^Would you like to hire %s %s for %s?",
                       StartsWithVowel(Names.Bitch) ? "an" : "a",
                       Names.Bitch,FormatPrice(To->Bitches.Price,buf));
               SendQuestion(NULL,C_ASKBITCH,To,text);
               return;
            }
            break;
         case E_ARRIVE:
            for (Play=FirstServer;Play;Play=Play->Next) {
               if (Play!=To && Play->IsAt==To->IsAt &&
                   Play->EventNum==E_NONE) {
                  text[0]='\0';
/*                  sprintf(buf,"%s is already here!^Do you T>rade, B>ribe, ",
                          Play->Name);
                  strcpy(text,"TB");*/
                  sprintf(buf,"%s is already here!^Do you ",Play->Name);
                  if (TotalGunsCarried(To)>0) {
                     strcat(buf,"A>ttack, ");
                     strcat(text,"A");
                  }
                  strcat(buf,"or E>vade?");
                  strcat(text,"E"); 
                  if (strcmp(text,"E")!=0) {
                     strcat(text,"^");
                     strcat(text,buf);
                     To->Attacked=Play;
                     SendDrugsHere(To,TRUE);
                     SendQuestion(NULL,C_MEETPLAYER,To,text);
                     return;
                  }
               }
            }
            SendDrugsHere(To,TRUE);
            break;
      }
      To->EventNum++;
   } 
   if (To->EventNum >= E_MAX) To->EventNum=E_NONE;
}

int SendCopOffer(struct PLAYER *To,char Force) {
/* In response to client player "To" being in state E_OFFOBJECT,   */
/* randomly engages the client in combat with the cops or offers   */
/* other random events. Returns 0 if the client should then be     */
/* advanced to the next state, 1 otherwise (i.e. if there are      */
/* questions pending which the client must answer first)           */
/* If Force==FORCECOPS, engage in combat with the cops for certain */
/* If Force==FORCEBITCH, offer the client a bitch for certain      */
   int i;
   i=brandom(0,100);
   if (Force==FORCECOPS) i=100;
   else if (Force==FORCEBITCH) i=0;
   else To->OnBehalfOf=NULL;
   if (i<33) { 
      return(OfferObject(To,Force==FORCEBITCH)); 
   } else if (i<50) { return(RandomOffer(To));
   } else if (Sanitized) { return 0;
   } else {
      StartOfficerHardass(To,To->EventNum+1,NULL,NULL);
   }
   return 1;
}

void StartOfficerHardass(struct PLAYER *Play,int ResyncNum,
                         char *LoneMessage,char *DeputyMessage) {
/* Starts combat between player "Play" and the cops. "ResyncNum" is */
/* the event number to be returned to after combat is complete.     */
/* "LoneMessage" and "DeputyMessage" are the format strings passed  */
/* to OfficerHardass if they are non-NULL.                          */
   price_t Money;
   if (!Play) return;
   Money=Play->Cash+Play->Bank-Play->Debt;
   if (Money>3000000) Play->Cops=brandom(11,27);
   else if (Money>1000000) Play->Cops=brandom(7,14);
   else if (Money>500000) Play->Cops=brandom(6,12);
   else if (Money>100000) Play->Cops=brandom(2,8);
   else Play->Cops=brandom(1,5);

   Play->ResyncNum=ResyncNum;
   Play->EventNum=E_COPS;
   if (Play->ResyncNum==E_MAX || Play->ResyncNum==E_NONE) {
      SendServerMessage(NULL,C_NONE,C_CHANGEDISP,Play,"N");
   }
   OfficerHardass(Play,LoneMessage,DeputyMessage);
}

void OfficerHardass(struct PLAYER *Play,char *LoneMessage,char *DeputyMessage) {
/* Send client "Play" a message announcing the attack of the cops */
/* The format string used for this purpose can be altered by      */
/* passing non-NULL "LoneMessage" (for unaccompanied Officer      */
/* Hardass) and/or "DeputyMessage" (for him with x deputies)      */
   char text[BUFLEN];
   char LoneDefault[] = { "YN^Officer %s is chasing you!" };
   char DeputyDefault[] = { 
      "YN^Officer %s and %d of his deputies are chasing you!" 
   };
   char *OfficerName;
   
   if (!Play || Play->EventNum!=E_COPS) return;
   if (Play->Cops==0) { 
      Play->EventNum=Play->ResyncNum; SendEvent(Play); return; 
   }
   OfficerName=(Play->Flags&DEADHARDASS ? Names.ReserveOfficer : 
                                          Names.Officer);
   if (Play->Cops==1) {
      sprintf(text,LoneMessage ? LoneMessage : LoneDefault,OfficerName);
   } else {
      sprintf(text,DeputyMessage ? DeputyMessage : DeputyDefault,
	      OfficerName, Play->Cops - 1);
   }
   SendPlayerData(Play);
   if (TotalGunsCarried(Play)==0) {
      strcat(text,"^Do you run?");
      SendQuestion(NULL,C_ASKRUN,Play,text);
   } else {
      strcat(text,"^Do you R>un, or F>ight?");
      text[0]='R'; text[1]='F';
      SendQuestion(NULL,C_ASKRUNFIGHT,Play,text);
   }
}

void FinishFightWithHardass(struct PLAYER *Play,char *Message) {
/* Clean up after a fight between "Play" and the cops. If the cops were */
/* tipped off by another player, inform them of the results.            */
/* If the player died, pass "Message" to the FinishGame subroutine.     */
   char text[BUFLEN],buf[BUFLEN];
   if (IsValidPlayer(Play->OnBehalfOf,FirstServer)) {
      LOGF("%s: tipoff by %s finished OK.\n",Play->Name,
             Play->OnBehalfOf->Name);
      RemoveListPlayer(&(Play->TipList),Play->OnBehalfOf);
      sprintf(text,"Following your tipoff, the cops ambushed %s, who ",
              Play->Name);
      if (Play->Health==0) strcat(text,"was shot dead.");
      else {
         sprintf(buf,"escaped with %d %s.",Play->Bitches.Carried,
		  Names.Bitches);
         strcat(text,buf);
      }
      GainBitch(Play->OnBehalfOf);
      SendPlayerData(Play->OnBehalfOf);
      SendPrintMessage(NULL,C_NONE,Play->OnBehalfOf,text);
   }
   Play->OnBehalfOf=NULL;
   if (Play->Health==0) FinishGame(Play,Message);
   else {
      Play->EventNum=Play->ResyncNum; 
      if (Play->ResyncNum==E_MAX || Play->ResyncNum==E_NONE) {
         SendServerMessage(NULL,C_NONE,C_CHANGEDISP,Play,"Y");
      }
      SendEvent(Play);
   }
}

void FireAtHardass(struct PLAYER *Play,char FireType) {
/* Have player "Play" attack the cops.                   */
/* FireType is F_STAND: Player has no gun and didn't run */
/*             F_RUN:   Player chose to run              */
/*             F_FIGHT: Player chose to fire back        */
   char text[BUFLEN],buf[BUFLEN];
   int Damage,i,j;
   char *OfficerName;

   if (!Play || Play->EventNum!=E_COPS) return;
   if (Play->Cops==0) { FinishFightWithHardass(Play,NULL); return; }
   
   strcpy(text,"^");
   OfficerName=(Play->Flags&DEADHARDASS ? Names.ReserveOfficer : 
                                          Names.Officer);
   if (FireType==F_STAND) {
      strcpy(text,"^^You stand there like an idiot.");
   } else if (FireType==F_RUN) {
      if (brandom(0,100) < Cops.EscapeProb-(Play->Cops-1)*Cops.DeputyEscape) {
         if (Play->Cops==1) strcpy(text,"^^You lose him in the alleys.");
         else strcpy(text,"^^You lose them in the alleys.");
         SendPrintMessage(NULL,C_NONE,Play,text);
         FinishFightWithHardass(Play,NULL);
         return;
      } else {
         if (Play->Cops==1) strcpy(text,"^^You can\'t shake him, man!");
         else strcpy(text,"^^You can\'t shake them, man!");
      }
   } else if (FireType==F_FIGHT) {
      Damage=100-brandom(0,Play->Cops)*Cops.Toughness;
      for (i=0;i<NumGun;i++) for (j=0;j<Play->Guns[i].Carried;j++) {
         Damage+=brandom(0,Gun[i].Damage);
      }
      if (Damage>=100) {
         if (Play->Cops==1) {
            i=brandom(1500,3000);
            sprintf(text,"^^You killed Officer %s! You find %s on his corpse!",
                    OfficerName, FormatPrice(i,buf));
            Play->Cash += i;
            Play->Flags |= DEADHARDASS;
            SendPlayerData(Play);
            SendPrintMessage(NULL,C_NONE,Play,text);
            Play->DocPrice=brandom(1000,2000-5*Play->Health);
            if (brandom(0,100)<75 && Play->DocPrice<=Play->Cash &&
                Play->Health<100) {
               Play->EventNum=E_DOCTOR;
               sprintf(text,"YN^^^^Do you pay a doctor %s to sew you",
                      FormatPrice(Play->DocPrice,buf));
	       if (Play->Bitches.Carried && !WantAntique) {
		 strcat(text, "r ");
		 strcat(text, Names.Bitches);
	       }
	       strcat(text, " up?");
               SendQuestion(NULL,C_ASKSEW,Play,text);
            } else {
               FinishFightWithHardass(Play,NULL);
            }
            return;
         } else {
            strcpy(text,"^^You got one, man!");
            Play->Cops--;
         }
      } else strcpy(text,"^^You missed!");
   }

   if (Play->Cops==1) strcat(text,"^He\'s"); else strcat(text,"^They\'re");
   strcat(text," firing on you, man! ");
   if (brandom(0,100) < Cops.HitProb+(Play->Cops-1)*Cops.DeputyHit) {
      strcat(text,"You've been hit! ");
      Damage=0;
      for (i=0;i<Play->Cops;i++) Damage+=brandom(0,Cops.Damage);
      if (Damage==0) Damage=1;
      if (Damage>Play->Health) {
         if (Play->Bitches.Carried==0 || WantAntique) {
            if (Play->Cops==1) strcat(text,"He"); else strcat(text,"They");
            strcat(text," wasted you, man! What a drag!");
            Play->Health=0;
            SendPlayerData(Play);
            FinishFightWithHardass(Play,text);
            return;
         } else {
            strcat(text,"You lost one of your "); strcat(text,Names.Bitches);
            strcat(text,"!");
            LoseBitch(Play,NULL,NULL);
            Play->Health=100;
         }
      } else {
         Play->Health-=Damage;
      }
   } else {
      if (Play->Cops==1) strcat(text,"He"); else strcat(text,"They");
      strcat(text," missed!");
   }
   SendPlayerData(Play);
   SendPrintMessage(NULL,C_NONE,Play,text);
   OfficerHardass(Play,NULL,NULL);
}

int RandomOffer(struct PLAYER *To) {
/* Inform player "To" of random offers or happenings. Returns 0 if */
/* the client can immediately be advanced to the next state, or 1  */
/* there are first questions to be answered.                       */
   int r,amount,ind;
   char text[BUFLEN],buf[80];
   r=brandom(0,100);

   if (!Sanitized && (r < 10)) {
      strcpy(text,"You were mugged in the subway!");
      To->Cash=To->Cash*(price_t)brandom(80,95)/100l;
   } else if (r<30) {
      amount=brandom(3,7);
      ind=IsCarryingRandom(To,amount);
      if (ind==-1 && amount>To->CoatSize) return 0;
      strcpy(text,"You meet a friend! ");
      if (ind==-1) {
         ind=brandom(0,NumDrug);
         sprintf(buf,"He gives you %d %s.",amount,Drug[ind].Name);
         strcat(text,buf);
         To->Drugs[ind].Carried+=amount;
         To->CoatSize-=amount;
      } else {
         sprintf(buf,"You give him %d %s.",amount,Drug[ind].Name);
         strcat(text,buf);
         To->Drugs[ind].Carried-=amount;
         To->CoatSize+=amount;
      }
      SendPlayerData(To);
      SendPrintMessage(NULL,C_NONE,To,text);
   } else if (Sanitized) {
      printf("Sanitized away a RandomOffer\n") ;
      return 0 ;
   } else if (r<50) {
      amount=brandom(3,7);
      ind=IsCarryingRandom(To,amount);
      if (ind!=-1) {
         sprintf(text,"Police dogs chase you for %d blocks! You dropped \
some %s! That's a drag, man!",brandom(3,7),Names.Drugs);
         To->Drugs[ind].Carried-=amount;
         To->CoatSize+=amount;
         SendPlayerData(To);
         SendPrintMessage(NULL,C_NONE,To,text);
      } else {
         ind=brandom(0,NumDrug);
         amount=brandom(3,7);
         if (amount>To->CoatSize) return 0;
         sprintf(text,"You find %d %s on a dead dude in the subway!",
                 amount,Drug[ind].Name);
         To->Drugs[ind].Carried+=amount;
         To->CoatSize-=amount;
         SendPlayerData(To);
         SendPrintMessage(NULL,C_NONE,To,text);
      }
   } else if (r<60 && To->Drugs[WEED].Carried+To->Drugs[HASHISH].Carried>0) {
      ind = (To->Drugs[WEED].Carried>To->Drugs[HASHISH].Carried) ?
            WEED : HASHISH;
      amount=brandom(2,6);
      if (amount>To->Drugs[ind].Carried) amount=To->Drugs[ind].Carried;
      sprintf(text,"Your mama made brownies with some of your %s! They \
were great!",Drug[ind].Name);
      To->Drugs[ind].Carried-=amount;
      To->CoatSize+=amount;
      SendPlayerData(To);
      SendPrintMessage(NULL,C_NONE,To,text);
   } else if (r<65) {
      strcpy(text,"YN^There is some weed that smells like paraquat here!");
      strcat(text,"^It looks good! Will you smoke it? ");
      To->EventNum=E_WEED;
      SendQuestion(NULL,C_NONE,To,text);
      return 1;
   } else {
      sprintf(text,"You stopped to %s.",
              StoppedTo[brandom(0,NUMSTOPPEDTO)]);
      amount=brandom(1,10);
      if (To->Cash>=amount) To->Cash-=amount;
      SendPlayerData(To);
      SendPrintMessage(NULL,C_NONE,To,text);
   }
   return 0;
}

int OfferObject(struct PLAYER *To,char ForceBitch) {
/* Offers player "To" bitches/trenchcoats or guns. If ForceBitch is  */
/* TRUE, then a bitch is definitely offered. Returns 0 if the client */
/* can advance immediately to the next state, 1 otherwise.           */
   char text[80],buf[80];
   int ObjNum;

   if (brandom(0,100)<50 || ForceBitch) {
      if (WantAntique) {
         To->Bitches.Price=brandom(MINTRENCHPRICE,MAXTRENCHPRICE);
         sprintf(text,"Would you like to buy a bigger trenchcoat for %s?",
                 FormatPrice(To->Bitches.Price,buf));
      } else {
         To->Bitches.Price=brandom(Bitch.MinPrice,Bitch.MaxPrice)/10l;
         sprintf(text,"YN^Hey dude! I'll help carry your %s for a \
mere %s. Yes or no?",Names.Drugs,FormatPrice(To->Bitches.Price,buf));
      }
      SendQuestion(NULL,C_ASKBITCH,To,text);
      return 1;
   } else if (!Sanitized && (TotalGunsCarried(To) < To->Bitches.Carried+2)) {
      ObjNum=brandom(0,NumGun);
      To->Guns[ObjNum].Price=Gun[ObjNum].Price/10;
      if (Gun[ObjNum].Space>To->CoatSize) return 0;
      sprintf(text,"YN^Would you like to buy a %s for %s?",
              Gun[ObjNum].Name,FormatPrice(To->Guns[ObjNum].Price,buf));
      SendQuestion(NULL,C_ASKGUN,To,text);
      return 1;
   }
   return 0;
}

void SendDrugsHere(struct PLAYER *To,char DisplayBusts) {
/* Sends details of drug prices to player "To". If "DisplayBusts"   */
/* is TRUE, also regenerates drug prices and sends details of       */
/* special events such as drug busts                                */
   int i;
   char msg[BUFLEN],text[BUFLEN],buf[80];
   char *Deal;
   Deal=calloc(NumDrug,1);
   if (!Deal) { fprintf(stderr,"Memory allocation error\n"); exit(1); }
   if (DisplayBusts) GenerateDrugsHere(To,Deal);
   msg[0]=0;
   if (DisplayBusts) for (i=0;i<NumDrug;i++) if (Deal[i]) {
      if (Drug[i].Expensive) {
         sprintf(text,Deal[i]==1 ? Drugs.ExpensiveStr1 : Drugs.ExpensiveStr2,
                 Drug[i].Name);
      } else if (Drug[i].Cheap) {
         strcpy(text,Drug[i].CheapStr);
      } 
      if (msg[0]!=0) strcat(msg,"^");
      strcat(msg,text);
   }
   if (msg[0]) SendPrintMessage(NULL,C_NONE,To,msg);
   text[0]=0; 
   for (i=0;i<NumDrug;i++) {
      pricetostr(To->Drugs[i].Price,buf,80);
      strcat(text,buf);
      strcat(text,"^");
   }
   SendServerMessage(NULL,C_NONE,C_DRUGHERE,To,text);
}

void GenerateDrugsHere(struct PLAYER *To,char *Deal) {
/* Generates drug prices and drug busts etc. for player "To" */
/* "Deal" is an array of chars of size NumDrug               */
   int NumEvents,NumDrugs,i;
   for (i=0;i<NumDrug;i++) {
      To->Drugs[i].Price=0;
      Deal[i]=0;
   }
   NumEvents=0;
   if (brandom(0,100)<70) NumEvents=1;
   if (brandom(0,100)<40 && NumEvents==1) NumEvents=2;
   if (brandom(0,100)<5 && NumEvents==2) NumEvents=3;
   NumDrugs=0;
   while (NumEvents>0) {
      i=brandom(0,NumDrug);
      if (Deal[i]!=0) continue;
      if (Drug[i].Expensive && (!Drug[i].Cheap || brandom(0,100)<50)) {
         Deal[i]=brandom(1,3);
         To->Drugs[i].Price=brandom(Drug[i].MinPrice,Drug[i].MaxPrice)
                            *Drugs.ExpensiveMultiply;
         NumDrugs++;
         NumEvents--;
      } else if (Drug[i].Cheap) {
         Deal[i]=1;
         To->Drugs[i].Price=brandom(Drug[i].MinPrice,Drug[i].MaxPrice)
                            /Drugs.CheapDivide;
         NumDrugs++;
         NumEvents--;
      }
   }
   NumDrugs=brandom(Location[(int)To->IsAt].MinDrug,
                    Location[(int)To->IsAt].MaxDrug)-NumDrugs;
   if (NumDrugs >= NumDrug) NumDrugs=NumDrug;
   while (NumDrugs>0) {
      i=brandom(0,NumDrug);
      if (To->Drugs[i].Price==0) {
         To->Drugs[i].Price=brandom(Drug[i].MinPrice,Drug[i].MaxPrice);
         NumDrugs--;
      }
   }
}

void HandleAnswer(struct PLAYER *From,struct PLAYER *To,char *answer) {
/* Handles the incoming message in "answer" from player "From" and */
/* intended for player "To".                                       */
   char text[BUFLEN],buf[BUFLEN];
   int i;
   if (!From || From->EventNum==E_NONE) return;
/*   LOGF("Answer=%s, EventNum=%d\n",answer,From->EventNum); */
   if (answer[0]=='Y' && From->EventNum==E_OFFOBJECT && From->Bitches.Price
       && From->Bitches.Price>From->Cash) answer[0]='N';
   if (answer[0]=='Y') switch (From->EventNum) { 
      case E_OFFOBJECT:
         text[0]=0;
         if (From->Bitches.Price) {
            sprintf(text,"bitch^0^1");
         } else {
            for (i=0;i<NumGun;i++) if (From->Guns[i].Price) {
               sprintf(text,"gun^%d^1",i);
            }
         }
         if (IsValidPlayer(From->OnBehalfOf,FirstServer)) {
            LOGF("%s: offer was on behalf of %s\n",From->Name,
                   From->OnBehalfOf->Name);
            if (From->Bitches.Price) {
               sprintf(buf,"%s has accepted your %s!",From->Name,Names.Bitch);
               strcat(buf,"^Use the G key to contact your spy.");
               From->OnBehalfOf->Flags |= SPYINGON;
               SendPlayerData(From->OnBehalfOf);
               SendPrintMessage(NULL,C_NONE,From->OnBehalfOf,buf);
               i=GetListEntry(&(From->SpyList),From->OnBehalfOf);
               if (i>=0) From->SpyList.Data[i].Turns=0;
            }
         }
         if (text[0]) BuyObject(From,text);
         From->OnBehalfOf=NULL;
         From->EventNum++; SendEvent(From);
         break;
      case E_LOANSHARK:
         SendServerMessage(NULL,C_NONE,C_LOANSHARK,From,NULL);
         break;
      case E_BANK:
         SendServerMessage(NULL,C_NONE,C_BANK,From,NULL);
         break;
      case E_GUNSHOP:
         for (i=0;i<NumGun;i++) From->Guns[i].Price=Gun[i].Price;
         SendServerMessage(NULL,C_NONE,C_GUNSHOP,From,NULL);
         break;
      case E_HIREBITCH:
         strcpy(text,"bitch^0^1");
         BuyObject(From,text);
         From->EventNum++; SendEvent(From);
         break;
      case E_ROUGHPUB:
         From->EventNum++; SendEvent(From);
         break;
      case E_WEED:
         FinishGame(From,"You hallucinated for three days on the wildest \
trip you ever imagined!^Then you died because your brain disintegrated!"); 
         break;
      case E_COPS:
         FireAtHardass(From,F_RUN);
         break;
      case E_DOCTOR:
         if (From->Cash >= From->DocPrice) {
            From->Cash -= From->DocPrice;
            From->Health=100;
            SendPlayerData(From);
         }
         FinishFightWithHardass(From,NULL);
         break;
   } else if (From->EventNum==E_COPS && 
              (answer[0]=='F' || answer[0]=='R')) {
      FireAtHardass(From,answer[0]=='F' ? F_FIGHT : F_RUN);
   } else if (From->EventNum==E_ARRIVE) {
      if ((answer[0]=='A' || answer[0]=='T') && 
          IsValidPlayer(From->Attacked,FirstServer)) {
         if (From->Attacked->IsAt==From->IsAt) {
            if (answer[0]=='A') {
               if (From->Attacked->EventNum<E_MAX) {
                  From->Attacked->ResyncNum=From->Attacked->EventNum;
               }
               From->Attacked->Flags |= FIGHTING;
               SendPlayerData(From->Attacked);
               From->Flags |= FIGHTING;
               SendPlayerData(From);
               From->Attacked->EventNum=E_DEFEND;
               From->ResyncNum=E_ARRIVE+1;
               From->EventNum=E_ATTACK;
               AttackPlayer(From,From->Attacked,AT_FIRST | AT_SHOOT);
/*          } else if (answer[0]=='T') {
               From->Flags |= TRADING;
               SendPlayerData(From);
               SendServerMessage(NULL,C_NONE,C_TRADE,From,NULL);*/
            }
         } else {
            sprintf(text,"Too late - %s has just left!",From->Attacked->Name);
            SendPrintMessage(NULL,C_NONE,From,text);
            From->EventNum++; SendEvent(From);
         }
      } else {
         From->EventNum++; SendEvent(From);
      }
   } else switch(From->EventNum) { 
      case E_ROUGHPUB:
         From->EventNum++;
         From->EventNum++; SendEvent(From);
         break;
      case E_COPS:
         FireAtHardass(From,F_STAND);
         break;
      case E_DOCTOR:
         FinishFightWithHardass(From,NULL);
         break;
      case E_HIREBITCH: case E_GUNSHOP: case E_BANK: case E_LOANSHARK:
      case E_OFFOBJECT: case E_WEED:
         if (IsValidPlayer(From->OnBehalfOf,FirstServer)) {
            LOGF("%s: offer was on behalf of %s\n",From->Name,
                   From->OnBehalfOf->Name);
            if (From->Bitches.Price && From->EventNum==E_OFFOBJECT) {
               sprintf(text,"%s has rejected your %s!",From->Name,Names.Bitch);
               GainBitch(From->OnBehalfOf);
               SendPlayerData(From->OnBehalfOf);
               SendPrintMessage(NULL,C_NONE,From->OnBehalfOf,text);
               RemoveListPlayer(&(From->SpyList),From->OnBehalfOf);
            }   
         }
         From->EventNum++; SendEvent(From);
         break;
   }
}

void BreakoffCombat(struct PLAYER *Attack,char LeftGame) {
/* Withdraws from player-player combat that player "Attack" is     */
/* currently involved in. "LeftGame" is TRUE if "Attack" has just  */
/* left the game, in which case no more messages are sent to this  */
/* player (just the other side of the fight is cleaned up)         */
   struct PLAYER *Defend;
   struct PLAYER *Play,*Victor;
   char FightDone,text[BUFLEN];
   if (!IsValidPlayer(Attack,FirstServer)) {
      LOGF("Players involved in a fight are not valid!\n");
      return;
   }
   if (Attack->EventNum!=E_DEFEND && Attack->EventNum!=E_ATTACK &&
        Attack->EventNum!=E_FREEFORALL && Attack->EventNum!=E_WAITATTACK) {
      LOGF("Players in fight are not attack/defending!\n");
      return;
   }
   Victor=NULL;

   if (Attack->EventNum==E_DEFEND) {
      sprintf(text,"%s has got away!",Attack->Name);
      for (Play=FirstServer;Play;Play=Play->Next) {
         if (Play->Attacked==Attack && (Play->EventNum==E_ATTACK ||
             Play->EventNum==E_WAITATTACK || Play->EventNum==E_FREEFORALL)) {
            ClearFightTimeout(Play);
            Play->Attacked=NULL;
            Play->Flags &= ~FIGHTING;
            SendPlayerData(Play);
            if (Attack->Health!=0) {
               SendServerMessage(NULL,C_NONE,C_FIGHTPRINT,Play,text);
            }
            Victor=Play;
            Play->EventNum=Play->ResyncNum; SendEvent(Play);
         } 
      }
   } else {
      ClearFightTimeout(Attack);
      Victor=Attack;
      Defend=Attack->Attacked;
      if (!IsValidPlayer(Defend,FirstServer)) {
         LOGF("Players involved in a fight are not valid!\n");
         return;
      }
      if (Defend->EventNum!=E_DEFEND) {
         LOGF("Players in fight are not attack/defending!\n");
         return;
      }
 
      Attack->Attacked=NULL;
      FightDone=TRUE;
      for (Play=FirstServer;Play;Play=Play->Next) {
         if (Play->Attacked==Defend && (Play->EventNum==E_ATTACK ||
             Play->EventNum==E_WAITATTACK || Play->EventNum==E_FREEFORALL)) { 
            FightDone=FALSE; 
            break; 
         }
      }
      if (Attack->Health>0) {
         sprintf(text,"%s has run off!",Attack->Name);
         SendServerMessage(NULL,C_NONE,C_FIGHTPRINT,Defend,text);
      }
      if (FightDone) {
         Defend->Flags &= ~FIGHTING;
         SendPlayerData(Defend);
         Defend->EventNum=Defend->ResyncNum; SendEvent(Defend);
      }
   }
   if (!LeftGame) { 
      if (Attack->Health>0) SendPrintMessage(NULL,C_NONE,Attack,
         "Coward! You successfully escaped from the fight.");
      Attack->Flags &= ~FIGHTING; SendPlayerData(Attack);
      Attack->EventNum=Attack->ResyncNum; SendEvent(Attack);
   }
}

void AttackPlayer(struct PLAYER *Attack,struct PLAYER *Defend,char AttackType) {
/* Processes a player-player attack from player "Attack" to player "Defend" */
/* AttackType is the type of attack, and may contain the following flags:-  */
/*  AT_FIRST: Set if this is the first attack                               */
/*  AT_SHOOT: Set if this is an 'active' attack - i.e. player "Attack" is   */
/*            actually shooting, not just "not running"                     */
   char text[BUFLEN],buf[BUFLEN],AICode,tmp[40];
   int i,j;
   Inventory *Guns,*Drugs;
   price_t Bounty;
   int Damage,MaxDamage;
 
   AICode=C_NONE;

   if (!IsValidPlayer(Attack,FirstServer) || 
       !IsValidPlayer(Defend,FirstServer)) {
      LOGF("Players involved in a fight are not valid!\n");
      return;
   }
   if (Attack->EventNum!=E_DEFEND && Attack->EventNum!=E_ATTACK &&
       Attack->EventNum!=E_FREEFORALL) {
      LOGF("Error: %s is in wrong state (%d) to attack!",
             Attack->Name,Attack->EventNum);
      return;
   }
   if ((Attack->EventNum==E_ATTACK || Attack->EventNum==E_FREEFORALL) && 
       Defend->EventNum!=E_DEFEND) {
      LOGF("Error: %s is trying to attack %s, who is in wrong state (%d)!",
           Attack->Name,Defend->Name,Defend->EventNum);
      return;
   }
   if (Attack->EventNum==E_DEFEND && Defend->EventNum!=E_WAITATTACK &&
       Defend->EventNum!=E_FREEFORALL) {
      LOGF("Error: %s is trying to defend against %s, who is in wrong \
state (%d)!",Attack->Name,Defend->Name,Defend->EventNum);
      return;
   }
   MaxDamage=0;
   Damage=0;
   for (i=0;i<NumGun;i++) {
      if (Gun[i].Damage>MaxDamage) MaxDamage=Gun[i].Damage;
      Damage+=Gun[i].Damage*Attack->Guns[i].Carried;
   }
   MaxDamage *= (Attack->Bitches.Carried+2);
   MaxDamage = Damage*100/MaxDamage;

   Guns=(Inventory *)calloc(sizeof(Inventory),NumGun);
   Drugs=(Inventory *)calloc(sizeof(Inventory),NumDrug);
   if (!Guns || !Drugs) {
      fprintf(stderr,"Error: Cannot alloc memory in AttackPlayer\n");
      exit(1);
   }
   ClearInventory(Guns,Drugs);
   if (AttackType&AT_FIRST) {
      sprintf(text,"%s arrives",Attack->Name);
      if (Attack->Bitches.Carried>0) {
         sprintf(buf," with %d %s",Attack->Bitches.Carried,Names.Bitches);
         strcat(text,buf);
      }
      if (MaxDamage<10) strcat(text,", pitifully armed");
      else if (MaxDamage<25) strcat(text,", lightly armed");
      else if (MaxDamage<60) strcat(text,", moderately well armed");
      else if (MaxDamage<80) strcat(text,", heavily armed");
      else strcat(text,", armed to the teeth");
      strcat(text,",^");
   } else strcpy(text,Attack->Name);
   strcat(text,AttackType&AT_SHOOT ? " fires and " : " stands and takes it.");
   Damage=0;
   if (AttackType&AT_SHOOT) {
      for (i=0;i<NumGun;i++) for (j=0;j<Attack->Guns[i].Carried;j++) {
         Damage+=brandom(0,Gun[i].Damage);
      }
   }
   if (Damage==0) {
      if (AttackType & AT_SHOOT) {
         strcat(text,"misses you!");
         sprintf(buf,"You failed to hit %s.",Defend->Name);
      } else {
         sprintf(buf,"You stand and take it.");
      }
      SendServerMessage(NULL,C_NONE,C_FIGHTPRINT,Attack,buf);
   } else {
      strcat(text,"hits you, man!");
      if (Damage>=Defend->Health) {
         if (Defend->Bitches.Carried==0) {
            strcat(text," You've been wasted! What a drag!");
            sprintf(buf,"You hit and killed %s",Defend->Name);
            Defend->Health=0;
            Bounty=Defend->Cash+Defend->Bank-Defend->Debt;
            if (Bounty>0) Attack->Cash+=Bounty;
            SendPlayerData(Defend);
            SendServerMessage(NULL,C_NONE,C_FIGHTPRINT,Defend,text);
            AddInventory(Guns,Defend->Guns,NumGun);
            AddInventory(Drugs,Defend->Drugs,NumDrug);
            TruncateInventoryFor(Guns,Drugs,Attack);
            if (!IsInventoryClear(Guns,Drugs)) {
               AddInventory(Attack->Guns,Guns,NumGun);
               AddInventory(Attack->Drugs,Drugs,NumDrug);
               ChangeSpaceForInventory(Guns,Drugs,Attack);
               SendPlayerData(Attack);
               strcat(buf,", and loot the body!");
            } else strcat(buf,"!");
            SendServerMessage(NULL,C_NONE,C_FIGHTPRINT,Attack,buf);
            free(Guns); free(Drugs); 
            FinishGame(Defend,text);
            return;
         } else {
            strcat(text,"^You lost a "); strcat(text,Names.Bitch);
            strcat(text,", man!");
            Bounty=(Defend->Cash+Defend->Bank-Defend->Debt)*
                   Defend->Bitches.Carried/1000L;
            if (Bounty>0) {
               sprintf(buf,"You are paid a bounty of %s in reward for killing^\
one of %s's %s",FormatPrice(Bounty,tmp),Defend->Name,Names.Bitches);
               Attack->Cash+=Bounty;
               SendPlayerData(Attack);
            } else {
               sprintf(buf,"You killed one of %s's %s (%d left)",
                           Defend->Name,Names.Bitches,
                           Defend->Bitches.Carried-1);
            }
            LoseBitch(Defend,Guns,Drugs);
            TruncateInventoryFor(Guns,Drugs,Attack);
            if (!IsInventoryClear(Guns,Drugs)) {
               AddInventory(Attack->Guns,Guns,NumGun);
               AddInventory(Attack->Drugs,Drugs,NumDrug);
               ChangeSpaceForInventory(Guns,Drugs,Attack);
               SendPlayerData(Attack);
               strcat(buf,", and loot the body!");
            } else strcat(buf,"!");
            SendServerMessage(NULL,C_NONE,C_FIGHTPRINT,Attack,buf);
            Defend->Health=100;
         }
      } else {
         Defend->Health-=Damage;
         sprintf(buf,"You fire, and hit %s!",Defend->Name);
         SendServerMessage(NULL,C_NONE,C_FIGHTPRINT,Attack,buf);
      }
   }
/*   LOGF("%s attacks %s\n",Attack->Name,Defend->Name); */
   if (Attack->EventNum==E_ATTACK || Attack->EventNum==E_FREEFORALL) {
      Attack->EventNum=E_WAITATTACK; SetFightTimeout(Attack);
   } else if (Attack->EventNum==E_DEFEND) {
      Defend->EventNum=E_ATTACK; SetFightTimeout(Defend);
   }
   SendPlayerData(Defend);
   SendServerMessage(Attack,AICode,C_FIGHTPRINT,Defend,text);
   free(Guns); free(Drugs);
}

void BuyObject(struct PLAYER *From,char *data) {
/* Processes a request stored in "data" from player "From" to buy an  */
/* object (bitch, gun, or drug)                                       */
/* Objects can be sold if the amount given in "data" is negative, and */
/* given away if their current price is zero.                         */
   char *cp,wrd[BUFLEN],type[BUFLEN],lone[200],deputy[200];
   int index,i,amount;
   cp=data;
   cp=ExtractWord(type,BUFLEN,cp);
   cp=ExtractWord(wrd,BUFLEN,cp);
   index=atoi(wrd);
   amount=atoi(cp);
   if (strcmp(type,"drug")==0) {
      if (index>=0 && index<NumDrug && From->Drugs[index].Carried+amount >= 0
          && From->CoatSize-amount >= 0 && (From->Drugs[index].Price!=0 || 
          amount<0) && From->Cash >= amount*From->Drugs[index].Price) {
         From->Drugs[index].Carried+=amount;
         From->CoatSize-=amount;
         From->Cash-=amount*From->Drugs[index].Price;
         SendPlayerData(From); 

         if (!Sanitized && (From->Drugs[index].Price==0 && brandom(0,100)<Cops.DropProb)) {
            sprintf(lone,"YN^Officer %%s spots you dropping %s, and \
chases you!",Names.Drugs);
            sprintf(deputy,"YN^Officer %%s and %%d of his deputies spot \
you dropping %s, and chase you!",Names.Drugs);
            StartOfficerHardass(From,From->EventNum,lone,deputy);
         }
      }
   } else if (strcmp(type,"gun")==0) {
      if (index>=0 && index<NumGun && TotalGunsCarried(From)+amount >= 0
          && TotalGunsCarried(From)+amount <= From->Bitches.Carried+2
          && From->Guns[index].Price!=0
          && From->CoatSize-amount*Gun[index].Space >= 0
          && From->Cash >= amount*From->Guns[index].Price) {
         From->Guns[index].Carried+=amount;
         From->CoatSize-=amount*Gun[index].Space;
         From->Cash-=amount*From->Guns[index].Price;
         SendPlayerData(From);
      }
   } else if (strcmp(type,"bitch")==0) {
      if (From->Bitches.Carried+amount >= 0 
          && From->Bitches.Price!=0 
          && amount*From->Bitches.Price <= From->Cash) {
         for (i=0;i<amount;i++) GainBitch(From);
         if (amount>0) From->Cash-=amount*From->Bitches.Price;
         SendPlayerData(From);
      }
   } 
}

void ClearPrices(struct PLAYER *Play) {
/* Clears the bitch and gun prices stored for player "Play" */
   int i;
   Play->Bitches.Price=0;
   for (i=0;i<NumGun;i++) Play->Guns[i].Price=0;
}

void GainBitch(struct PLAYER *Play) {
/* Gives player "Play" a new bitch (or larger trenchcoat) */
   Play->CoatSize+=10;
   Play->Bitches.Carried++;
}

int LoseBitch(struct PLAYER *Play,Inventory *Guns,Inventory *Drugs) {
/* Loses one bitch belonging to player "Play". If drugs or guns are */
/* lost with the bitch, 1 is returned (0 otherwise) and the lost    */
/* items are added to "Guns" and "Drugs" if non-NULL                */
   int losedrug=0,i,num,drugslost;
   int *GunIndex,tmp;
   GunIndex=(int *)malloc(sizeof(int)*NumGun);
   ClearInventory(Guns,Drugs);
   Play->CoatSize-=10;
   if (TotalGunsCarried(Play)>0) {
      if (brandom(0,100)<TotalGunsCarried(Play)*100/(Play->Bitches.Carried+2)) {
         for (i=0;i<NumGun;i++) GunIndex[i]=i;
         for (i=0;i<NumGun*5;i++) {
            num=brandom(0,NumGun-1);
            tmp=GunIndex[num+1];
            GunIndex[num+1]=GunIndex[num];
            GunIndex[num]=tmp;
         }
         for (i=0;i<NumGun;i++) {
            if (Play->Guns[GunIndex[i]].Carried > 0) {
               Play->Guns[GunIndex[i]].Carried--; losedrug=1;
               Play->CoatSize+=Gun[GunIndex[i]].Space;
               if (Guns) Guns[GunIndex[i]].Carried++;
               break;
            }
         }
      }
   }
   for (i=0;i<NumDrug;i++) if (Play->Drugs[i].Carried>0) {
      num=(int)((float)Play->Drugs[i].Carried/(Play->Bitches.Carried+2.0)+0.5);
      if (num>0) {
         Play->Drugs[i].Carried-=num;
         if (Drugs) Drugs[i].Carried+=num;
         Play->CoatSize+=num;
         losedrug=1;
      }
   }
   while (Play->CoatSize<0) {
      drugslost=0;
      for (i=0;i<NumDrug;i++) {
         if (Play->Drugs[i].Carried>0) {
            losedrug=1; drugslost=1;
            Play->Drugs[i].Carried--;
            Play->CoatSize++;
            if (Play->CoatSize>=0) break;
         }
      }
      if (!drugslost) for (i=0;i<NumGun;i++) {
         if (Play->Guns[i].Carried>0) {
            losedrug=1;
            Play->Guns[i].Carried--;
            Play->CoatSize+=Gun[i].Space;
            if (Play->CoatSize>=0) break;
         }
      }
   }
   Play->Bitches.Carried--;
   free(GunIndex);
   return losedrug;
}

void SetFightTimeout(struct PLAYER *Play) {
   if (FightTimeout) Play->FightTimeout=time(NULL)+(time_t)FightTimeout;
   else Play->FightTimeout=0;
}

void ClearFightTimeout(struct PLAYER *Play) {
   Play->FightTimeout=0;
}

int AddTimeout(time_t timeout,time_t timenow,int *mintime) {
   if (timeout==0) return 0;
   else if (timeout<=timenow) return 1;
   else {
      *mintime=timeout-timenow;
      return 0;
   }
}

int GetMinimumTimeout(struct PLAYER *First) {
   struct PLAYER *tmp;
   int mintime=-1;
   time_t timenow;

   timenow=time(NULL);
   if (AddTimeout(MetaTimeout,timenow,&mintime)) return 0;
   for (tmp=First;tmp;tmp=tmp->Next) {
      if (AddTimeout(tmp->FightTimeout,timenow,&mintime) ||
          AddTimeout(tmp->IdleTimeout,timenow,&mintime) ||
          AddTimeout(tmp->ConnectTimeout,timenow,&mintime)) return 0;
   }
   return mintime;
}

void HandleTimeouts(struct PLAYER **FirstPt) {
   struct PLAYER *tmp,*tmp2,*First;
   char text[BUFLEN];
   time_t timenow;

   First=*FirstPt;
   timenow=time(NULL);
   if (MetaTimeout!=0 && MetaTimeout<=timenow) {
      MetaTimeout=0;
      RegisterWithMetaServer(TRUE,FALSE);
   }
   tmp=First;
   while (tmp) {
      tmp2=tmp->Next;
      if (tmp->IdleTimeout!=0 && tmp->IdleTimeout<=timenow) {
         tmp->IdleTimeout=0;
         LOGF("Player removed due to idle timeout\n");
         SendPrintMessage(NULL,C_NONE,tmp,"Disconnected due to idle timeout");
         ClientLeftServer(tmp);
         shutdown(tmp->fd,SD_SEND);
/* Make sure they do actually disconnect, eventually! */
         if (ConnectTimeout) {
            tmp->ConnectTimeout=time(NULL)+(time_t)ConnectTimeout;
         }
      } else if (tmp->ConnectTimeout!=0 && tmp->ConnectTimeout<=timenow) {
         tmp->ConnectTimeout=0;
         LOGF("Player removed due to connect timeout\n");
         RemovePlayer(tmp,FirstPt);
      } else if (tmp->FightTimeout!=0 && tmp->FightTimeout<=timenow) {
         ClearFightTimeout(tmp);
         if (tmp->EventNum==E_WAITATTACK) {
            tmp->EventNum=E_FREEFORALL;
            sprintf(text,"%s fails to return fire...",tmp->Attacked->Name);
            SendServerMessage(tmp->Attacked,C_NONE,C_FIGHTPRINT,tmp,text);
         } else if (tmp->EventNum==E_ATTACK) {
            tmp->EventNum=E_FREEFORALL;
            sprintf(text,"%s fails to return fire...",tmp->Name);
            SendServerMessage(tmp,C_NONE,C_FIGHTPRINT,tmp->Attacked,text);
         }
      }
      tmp=tmp2;
   }
}
