/*
**  Copyright (c) 2005-2007 Sendmail, Inc. and its suppliers.
**    All rights reserved.
*/

#ifndef lint
static char dkim_c_id[] = "@(#)$Id: dkim.c,v 1.286 2007/05/23 01:02:50 msk Exp $";
#endif /* !lint */

/* system includes */
#include <sys/param.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <netdb.h>
#include <stdlib.h>
#include <ctype.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <unistd.h>
#include <resolv.h>
#include <regex.h>

#ifdef __STDC__
# include <stdarg.h>
#else /* __STDC__ */
# include <varargs.h>
#endif /* _STDC_ */

/* libsm includes */
#include <sm/string.h>

/* libar includes */
#if USE_ARLIB
# include <ar.h>
#endif /* USE_ARLIB */

/* OpenSSL includes */
#include <openssl/opensslv.h>
#include <openssl/pem.h>
#include <openssl/rsa.h>
#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/sha.h>

/* libdkim includes */
#include "dkim.h"
#include "dkim-types.h"
#include "dkim-tables.h"
#include "dkim-keys.h"
#include "dkim-policy.h"
#ifdef _FFR_QUERY_CACHE
# include "dkim-cache.h"
#endif /* _FFR_QUERY_CACHE */
#include "util.h"
#include "base64.h"
#ifdef _FFR_VBR
# include "vbr.h"
#endif /* _FFR_VBR */

/* prototypes */
void dkim_error __P((DKIM *, const char *, ...));
#ifdef _FFR_HASH_BUFFERING
static void dkim_canonbuffer __P((DKIM *, u_char *, size_t));
#endif /* _FFR_HASH_BUFFERING */
static void dkim_canonwrite __P((DKIM *, u_char *, size_t));
static void *dkim_malloc __P((DKIM_LIB *, void *, size_t));
static void dkim_mfree __P((DKIM_LIB *, void *, void *));

/* macros */
#define	DKIM_MALLOC(x,y)	dkim_malloc((x)->dkim_libhandle, \
				            (x)->dkim_closure, y)
#define	DKIM_FREE(x,y)		dkim_mfree((x)->dkim_libhandle, \
				           (x)->dkim_closure, y)

#define	DKIM_STATE_INIT		0
#define	DKIM_STATE_HEADER	1
#define	DKIM_STATE_EOH		2
#define	DKIM_STATE_BODY		3
#define	DKIM_STATE_EOM		4

#define	DKIM_MODE_UNKNOWN	(-1)
#define	DKIM_MODE_SIGN		0
#define	DKIM_MODE_VERIFY	1

#define	DKIM_SIGTYPE_UNKNOWN	(-1)
#define	DKIM_SIGTYPE_RSA	0

#define	DKIM_HDR_SIGNED		0x01

#define	DKIM_HASHBUFSIZE	4096

#define	BUFRSZ			1024
#define	CRLF			"\r\n"
#define	SP			" "
#define	DEFTIMEOUT		10

/* local definitions needed for DNS queries */
#ifndef MAXHOSTNAMELEN
# define MAXHOSTNAMELEN		256
#endif /* ! MAXHOSTNAMELEN */
#define MAXPACKET		8192
#if defined(__RES) && (__RES >= 19940415)
# define RES_UNC_T		char *
#else /* __RES && __RES >= 19940415 */
# define RES_UNC_T		unsigned char *
#endif /* __RES && __RES >= 19940415 */

/* list of headers which may contain the sender */
const u_char *default_senderhdrs[] =
{
	"resent-sender",
	"resent-from",
	"sender",
	"from",
	NULL
};

/* default list of headers to sign */
const u_char *default_signhdrs[] =
{
	"resent-*",
	"sender",
	"from",
	"date",
	"subject",
	"content-*",
	"message-id",
	"to",
	"cc",
	"mime-version",
	"in-reply-to",
	"references",
	"list-*",
	NULL
};

/* required list of headers to sign */
const u_char *required_signhdrs[] =
{
	"from",
	NULL
};

/* default/minimum list of headers to ignore */
const u_char *default_skiphdrs[] =
{
	"return-path",
	NULL
};

/* ========================= PRIVATE SECTION ========================= */

/*
**  DKIM_MALLOC -- allocate memory
**
**  Parameters:
**  	libhandle -- DKIM library context in which this is performed
**  	closure -- opaque closure handle for the allocation
**  	nbytes -- number of bytes desired
**
**  Return value:
**  	Pointer to allocated memory, or NULL on failure.
*/

static void *
dkim_malloc(DKIM_LIB *libhandle, void *closure, size_t nbytes)
{
	assert(libhandle != NULL);

	if (libhandle->dkiml_malloc == NULL)
		return malloc(nbytes);
	else
		return libhandle->dkiml_malloc(closure, nbytes);
}

/*
**  DKIM_MFREE -- release memory
**
**  Parameters:
**  	dkim -- DKIM context in which this is performed
**  	closure -- opaque closure handle for the allocation
**  	ptr -- pointer to memory to be freed
**
**  Return value:
**  	None.
*/

static void
dkim_mfree(DKIM_LIB *libhandle, void *closure, void *ptr)
{
	assert(libhandle != NULL);

	if (libhandle->dkiml_free == NULL)
		free(ptr);
	else
		libhandle->dkiml_free(closure, ptr);
}

/*
**  DKIM_STRDUP -- duplicate a string
**
**  Parameters:
**  	dkim -- DKIM handle
**  	str -- string to clone
**  	len -- bytes to copy (0 == copy to NULL byte)
**
**  Return value:
**  	Pointer to a new copy of "str" allocated from within the appropriate
**  	context, or NULL on failure.
*/

static unsigned char *
dkim_strdup(DKIM *dkim, const unsigned char *str, size_t len)
{
	unsigned char *new;

	assert(dkim != NULL);
	assert(str != NULL);

	if (len == 0)
		len = strlen(str);
	new = DKIM_MALLOC(dkim, len + 1);
	if (new != NULL)
		sm_strlcpy(new, str, len + 1);
	else
		dkim_error(dkim, "unable to allocate %d byte(s)", len + 1);
	return new;
}

/*
**  DKIM_SELECTHDRS -- choose headers to be included in canonicalization
**
**  Parameters:
**  	dkim -- DKIM context in which this is performed
**  	hdrlist -- string containing headers that should be marked, separated
**  	           by the ":" character
**  	ptrs -- array of header pointers (modified)
**  	nptr -- number of pointers available at "ptrs"
**
**  Return value:
**  	Count of headers added to "ptrs", or -1 on error.
**
**  Notes:
**  	Selects headers to be passed to canonicalization and the order in
**  	which this is done.  "ptrs" is populated by pointers to headers
**  	in the order in which they should be fed to canonicalization.
**
**  	If any of the returned pointers is NULL, then a header named by
**  	"hdrlist" was not found.
*/

static int
dkim_selecthdrs(DKIM *dkim, u_char *hdrlist, struct dkim_header **ptrs,
                int nptrs)
{
	u_char *bar;
	u_char *colon;
	char *ctx;
	int c;
	int n;
	int m;
	size_t len;
	size_t hlen;
	struct dkim_header *hdr;
	struct dkim_header *lhdrs[MAXHDRCNT];
	u_char *hdrs[MAXHDRCNT];

	assert(dkim != NULL);
	assert(ptrs != NULL);
	assert(nptrs != 0);

	/* if there are no headers named, use them all */
	if (hdrlist == NULL)
	{
		n = 0;

		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			if (n >= nptrs)
				break;
			ptrs[n] = hdr;
			n++;
		}

		return n;
	}

	dkim->dkim_hdrlist = dkim_strdup(dkim, hdrlist, 0);
	if (dkim->dkim_hdrlist == NULL)
		return -1;

	/* mark all headers as not used */
	for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		hdr->hdr_flags &= ~DKIM_HDR_SIGNED;

	n = 0;

	/* make a split-out copy of hdrlist */
	memset(hdrs, '\0', sizeof hdrs);
	for (bar = strtok_r(dkim->dkim_hdrlist, ":", &ctx);
	     bar != NULL;
	     bar = strtok_r(NULL, ":", &ctx))
	{
		if (n >= MAXHDRCNT)
		{
			dkim_error(dkim, "too many headers (max %d)",
			           MAXHDRCNT);
			return -1;
		}

		hdrs[n] = bar;
		n++;
	}

	/* bounds check */
	if (n >= nptrs)
	{
		dkim_error(dkim, "too many headers (max %d)", MAXHDRCNT);
		return -1;
	}

	/* for each named header, find the last unused one and use it up */
	for (c = 0; c < n; c++)
	{
		lhdrs[c] = NULL;

		len = MIN(MAXHEADER, strlen(hdrs[c]));

		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			if (hdr->hdr_flags & DKIM_HDR_SIGNED)
				continue;

			colon = strchr(hdr->hdr_text, ':');
			if (colon != NULL)
			{
				hlen = colon - hdr->hdr_text;

				if (len == hlen &&
				    strncasecmp(hdr->hdr_text, hdrs[c],
				                hlen) == 0)
					lhdrs[c] = hdr;
			}
			else
			{
				if (strcasecmp(hdr->hdr_text, hdrs[c]) == 0)
					lhdrs[c] = hdr;
			}
		}

		if (lhdrs[c] != NULL)
			lhdrs[c]->hdr_flags |= DKIM_HDR_SIGNED;
	}

	/* copy to the caller's buffers */
	m = 0;
	for (c = 0; c < n; c++)
	{
		if (lhdrs[c] != NULL)
		{
			ptrs[m] = lhdrs[c];
			m++;
		}
	}

	return m;
}

/*
**  DKIM_SET_FIRST -- return first set in a context
**
**  Parameters:
**  	dkim -- DKIM context
**  	type -- type to find, or DKIM_SETTYPE_ANY
**
**  Return value:
**  	Pointer to the first DKIM_SET in the context, or NULL if none.
*/

static DKIM_SET *
dkim_set_first(DKIM *dkim, dkim_set_t type)
{
	DKIM_SET *set;

	assert(dkim != NULL);

	if (type == DKIM_SETTYPE_ANY)
		return dkim->dkim_sethead;

	for (set = dkim->dkim_sethead; set != NULL; set = set->set_next)
	{
		if (set->set_type == type)
			return set;
	}

	return NULL;
}

/*
**  DKIM_SET_NEXT -- return next set in a context
**
**  Parameters:
**  	set -- last set reported (i.e. starting point for this search)
**  	type -- type to find, or DKIM_SETTYPE_ANY
**
**  Return value:
**  	Pointer to the next DKIM_SET in the context, or NULL if none.
*/

static DKIM_SET *
dkim_set_next(DKIM_SET *cur, dkim_set_t type)
{
	DKIM_SET *set;

	assert(cur != NULL);

	if (type == DKIM_SETTYPE_ANY)
		return cur->set_next;

	for (set = cur->set_next; set != NULL; set = set->set_next)
	{
		if (set->set_type == type)
			return set;
	}

	return NULL;
}

/*
**  DKIM_PARAM_GET -- get a parameter from a set
**
**  Parameters:
**  	set -- set to search
**  	param -- parameter to find
**
**  Return value:
**  	Pointer to the parameter requested, or NULL if it's not in the set.
*/

static u_char *
dkim_param_get(DKIM_SET *set, u_char *param)
{
	DKIM_PLIST *plist;

	assert(set != NULL);
	assert(param != NULL);

	for (plist = set->set_plist; plist != NULL; plist = plist->plist_next)
	{
		if (strcasecmp(plist->plist_param, param) == 0)
			return plist->plist_value;
	}

	return NULL;
}

/*
**  DKIM_ADD_PLIST -- add an entry to a parameter-value set
**
**  Parameters:
**  	dkim -- DKIM context in which this is performed
**  	set -- set to modify
**   	param -- parameter
**  	value -- value
**  	force -- override existing value, if any
**
**  Return value:
**  	0 on success, -1 on failure.
**
**  Notes:
**  	Data is not copied; a reference to it is stored.
*/

static int
dkim_add_plist(DKIM *dkim, DKIM_SET *set, u_char *param, u_char *value,
               bool force)
{
	DKIM_PLIST *plist;

	assert(dkim != NULL);
	assert(set != NULL);
	assert(param != NULL);
	assert(value != NULL);

	/* see if we have one already */
	for (plist = set->set_plist; plist != NULL; plist = plist->plist_next)
	{
		if (strcasecmp(plist->plist_param, param) == 0)
			break;
	}

	/* nope; make one and connect it */
	if (plist == NULL)
	{
		plist = (DKIM_PLIST *) DKIM_MALLOC(dkim, sizeof(DKIM_PLIST));
		if (plist == NULL)
		{
			dkim_error(dkim, "unable to allocate %d byte(s)",
			           sizeof(DKIM_PLIST));
			return -1;
		}
		force = TRUE;
		plist->plist_next = set->set_plist;
		set->set_plist = plist;
		plist->plist_param = param;
	}

	/* set the value if "force" was set (or this was a new entry) */
	if (force)
		plist->plist_value = value;

	return 0;
}

/*
**  DKIM_PROCESS_SET -- process a parameter set, i.e. a string of the form
**                      param=value[; param=value]*
**
**  Parameters:
**  	dkim -- DKIM context in which this is performed
**  	type -- a DKIM_SETTYPE constant
**  	str -- string to be scanned
**  	len -- number of bytes available at "str"
**
**  Return value:
**  	A DKIM_STAT constant.
*/

static DKIM_STAT
dkim_process_set(DKIM *dkim, dkim_set_t type, u_char *str, size_t len)
{
	bool spaced;
	int state;
	int status;
	u_char *p;
	u_char *param;
	u_char *value;
	u_char *hcopy;
	DKIM_SET *set;
	const char *settype;

	assert(dkim != NULL);
	assert(str != NULL);
#if _FFR_VBR
	assert(type == DKIM_SETTYPE_SIGNATURE ||
	       type == DKIM_SETTYPE_KEY ||
	       type == DKIM_SETTYPE_VBRINFO ||
	       type == DKIM_SETTYPE_POLICY);
#else /* _FFR_VBR */
	assert(type == DKIM_SETTYPE_SIGNATURE ||
	       type == DKIM_SETTYPE_KEY ||
	       type == DKIM_SETTYPE_POLICY);
#endif /* _FFR_VBR */

	param = NULL;
	value = NULL;
	state = 0;
	spaced = FALSE;

	hcopy = (u_char *) DKIM_MALLOC(dkim, len + 1);
	if (hcopy == NULL)
	{
		dkim_error(dkim, "unable to allocate %d byte(s)", len + 1);
		return DKIM_STAT_INTERNAL;
	}
	sm_strlcpy(hcopy, str, len + 1);

	set = (DKIM_SET *) DKIM_MALLOC(dkim, sizeof(DKIM_SET));
	if (set == NULL)
	{
		DKIM_FREE(dkim, hcopy);
		dkim_error(dkim, "unable to allocate %d byte(s)",
		           sizeof(DKIM_SET));
		return DKIM_STAT_INTERNAL;
	}

	set->set_type = type;
	settype = dkim_code_to_name(settypes, type);

	if (dkim->dkim_sethead == NULL)
		dkim->dkim_sethead = set;
	else
		dkim->dkim_settail->set_next = set;

	dkim->dkim_settail = set;

	set->set_next = NULL;
	set->set_plist = NULL;
	set->set_data = hcopy;
	set->set_done = FALSE;

	for (p = hcopy; *p != '\0'; p++)
	{
		if (!isascii(*p))
		{
			dkim_error(dkim,
			           "invalid character (0x%02x) in %s data",
			           *p, settype);
			return DKIM_STAT_SYNTAX;
		}

		switch (state)
		{
		  case 0:				/* before param */
			if (isspace(*p))
			{
				continue;
			}
			else if (isalnum(*p))
			{
				param = p;
				state = 1;
			}
			else
			{
				dkim_error(dkim, "syntax error in %s data",
				           settype);
				return DKIM_STAT_SYNTAX;
			}
			break;

		  case 1:				/* in param */
			if (isspace(*p))
			{
				spaced = TRUE;
			}
			else if (*p == '=')
			{
				*p = '\0';
				state = 2;
				spaced = FALSE;
			}
			else if (*p == ';' || spaced)
			{
				dkim_error(dkim, "syntax error in %s data",
				           settype);
				return DKIM_STAT_SYNTAX;
			}
			break;

		  case 2:				/* before value */
			if (isspace(*p))
			{
				continue;
			}
			else if (*p == ';')		/* empty value */
			{
				*p = '\0';
				value = p;

				/* collapse the parameter */
				dkim_collapse(param);

				/* create the DKIM_PLIST entry */
				status = dkim_add_plist(dkim, set, param,
				                        value, TRUE);
				if (status == -1)
					return DKIM_STAT_INTERNAL;

				/* reset */
				param = NULL;
				value = NULL;
				state = 0;
			}
			else
			{
				value = p;
				state = 3;
			}
			break;

		  case 3:				/* in value */
			if (*p == ';')
			{
				*p = '\0';

				/* collapse the parameter and value */
				dkim_collapse(param);
				dkim_collapse(value);

				/* create the DKIM_PLIST entry */
				status = dkim_add_plist(dkim, set, param,
				                        value, TRUE);
				if (status == -1)
					return DKIM_STAT_INTERNAL;

				/* reset */
				param = NULL;
				value = NULL;
				state = 0;
			}
			break;

		  default:				/* shouldn't happen */
			assert(0);
		}
	}

	switch (state)
	{
	  case 0:					/* before param */
	  case 3:					/* in value */
		/* parse the data found, if any */
		if (value != NULL)
		{
			/* collapse the parameter and value */
			dkim_collapse(param);
			dkim_collapse(value);

			/* create the DKIM_PLIST entry */
			status = dkim_add_plist(dkim, set, param, value, TRUE);
			if (status == -1)
				return DKIM_STAT_INTERNAL;
		}
		break;

	  case 1:					/* after param */
	  case 2:					/* before value */
		dkim_error(dkim, "syntax error in %s data", settype);
		return DKIM_STAT_SYNTAX;

	  default:					/* shouldn't happen */
		assert(0);
	}

	/* load up defaults, assert requirements */
	switch (set->set_type)
	{
	  case DKIM_SETTYPE_SIGNATURE:
		/* make sure required stuff is here */
		if (dkim_param_get(set, "s") == NULL ||
		    dkim_param_get(set, "h") == NULL ||
		    dkim_param_get(set, "d") == NULL ||
		    dkim_param_get(set, "b") == NULL ||
		    dkim_param_get(set, "a") == NULL)
		{
			dkim_error(dkim, "missing parameter(s) in %s data",
			           settype);
			return DKIM_STAT_SYNTAX;
		}

		/* default for "c" */
		status = dkim_add_plist(dkim, set, "c", "simple/simple", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		/* default for "q" */
		status = dkim_add_plist(dkim, set, "q", "dns/txt", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		/* default for "i" */
		if (dkim_param_get(set, "i") == NULL)
		{
			char addr[MAXADDRESS + 1];

			value = dkim_param_get(set, "d");
			snprintf(addr, sizeof addr, "@%s", value);
			dkim->dkim_signer = dkim_strdup(dkim, addr, 0);
			if (dkim->dkim_signer == NULL)
				return DKIM_STAT_INTERNAL;

			status = dkim_add_plist(dkim, set, "i",
			                        dkim->dkim_signer, FALSE);
			if (status == -1)
				return DKIM_STAT_INTERNAL;
		}

		break;


	  case DKIM_SETTYPE_POLICY:
		status = dkim_add_plist(dkim, set, "o", "~", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		break;

	  case DKIM_SETTYPE_KEY:
		status = dkim_add_plist(dkim, set, "g", "*", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		status = dkim_add_plist(dkim, set, "k", "rsa", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		status = dkim_add_plist(dkim, set, "s", "*", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		status = dkim_add_plist(dkim, set, "v", DKIM_VERSION_KEY,
		                        FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		break;
			
#ifdef _FFR_VBR
	  case DKIM_SETTYPE_VBRINFO:
		break;
#endif /* _FFR_VBR */

	  default:
		assert(0);
	}

	set->set_done = TRUE;

	return DKIM_STAT_OK;
}

/*
**  DKIM_OHDRS -- extract and decode original headers
**
**  Parameters:
**  	dkim -- DKIM handle
**  	ptrs -- user-provided array of pointers to header strings (updated)
**  	pcnt -- number of pointers available (updated)
**
**  Return value:
**  	A DKIM_STAT_* constant.
**
**  Notes:
**  	If the returned value of pcnt is greater that what it was originally,
**  	then there were more headers than there were pointers.
**
**  	This should act on a DKIM_SIGINFO handle, not a DKIM handle.
**  	This will probably change in the very near future when full
**  	multiple signature support is added.
*/

DKIM_STAT
dkim_ohdrs(DKIM *dkim, char **ptrs, int *pcnt)
{
	int n = 0;
	char *z;
	char *ch;
	char *p;
	char *q;
	char *last;

	assert(dkim != NULL);
	assert(ptrs != NULL);
	assert(pcnt != NULL);

	if (dkim->dkim_mode != DKIM_MODE_VERIFY)
		return DKIM_STAT_INVALID;

	/* find the tag */
	z = dkim_param_get(dkim->dkim_sigset, "z");
	if (z == NULL || *z == '\0')
	{
		*pcnt = 0;
		return DKIM_STAT_OK;
	}

	/* get memory for the decode */
	if (dkim->dkim_zdecode == NULL)
	{
		dkim->dkim_zdecode = DKIM_MALLOC(dkim, strlen(z));
		if (dkim->dkim_zdecode == NULL)
		{
			dkim_error(dkim, "unable to allocate %d byte(s)",
			           strlen(z));
			return DKIM_STAT_NORESOURCE;
		}
	}

	/* copy it */
	sm_strlcpy(dkim->dkim_zdecode, z, strlen(z));

	/* decode */
	for (ch = strtok_r(z, "|", &last);
	     ch != NULL;
	     ch = strtok_r(NULL, "|", &last))
	{
		for (p = ch, q = ch; *p != '\0'; p++)
		{
			if (*p == '=')
			{
				char c;

				if (!isxdigit(*(p + 1)) || !isxdigit(*(p + 2)))
					return DKIM_STAT_INVALID;

				c = 16 * dkim_hexchar(*(p + 1)) + dkim_hexchar(*(p + 2));

				p += 2;

				*q = c;
				q++;
			}
			else
			{
				if (q != p)
					*q = *p;
				q++;
			}
		}

		*q = '\0';

		if (n < *pcnt)
			ptrs[n] = ch;
		n++;
	}

	*pcnt = n;

	return DKIM_STAT_OK;
}

/*
**  DKIM_GET_HEADER -- find a header in a queue of headers
**
**  Parameters:
**  	dkim -- DKIM handle
**  	name -- name of the header to find
**  	inst -- instance to find (0 == first/any)
**
**  Return value:
**  	Pointer to a (struct dkim_header), or NULL if not found.
*/

struct dkim_header *
dkim_get_header(DKIM *dkim, u_char *name, int inst)
{
	u_char *colon;
	struct dkim_header *hdr;

	assert(dkim != NULL);
	assert(name != NULL);

	for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
	{
		colon = strchr(hdr->hdr_text, ':');
		if (colon == NULL && strcasecmp(hdr->hdr_text, name) == 0)
		{
			if (inst == 0)
				return hdr;
			else
				inst--;
		}
		else
		{
			size_t len;

			len = colon - hdr->hdr_text;
			if (strncasecmp(hdr->hdr_text, name, len) == 0)
			{
				if (inst == 0)
					return hdr;
				else
					inst--;
			}
		}
	}

	return NULL;
}

/*
**  DKIM_KEY_SMTP -- return TRUE iff a parameter set defines an SMTP key
**
**  Parameters:
**  	set -- set to be checked
**
**  Return value:
**  	TRUE iff "set" contains an "s" parameter whose value is either
**  	"email" or "*".
*/

static bool
dkim_key_smtp(DKIM_SET *set)
{
	u_char *val;

	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_KEY);

	val = dkim_param_get(set, "s");

	if (val != NULL &&
	    (strcmp(val, "*") == 0 || strcasecmp(val, "email") == 0))
		return TRUE;
	else
		return FALSE;
}

/*
**  DKIM_KEY_GRANOK -- return TRUE iff the granularity of the key is
**                     appropriate to the signature being evaluated
**
**  Parameters:
**  	dkim -- DKIM handle
**  	gran -- granularity string from the retrieved key
**
**  Return value:
**  	TRUE iff the value of the granularity is a match for the signer.
*/

static bool
dkim_key_granok(DKIM *dkim, u_char *gran)
{
	int status;
	DKIM_SET *set;
	char *at;
	char *end;
	char *p;
	char *q;
	char restr[MAXADDRESS + 1];
	char user[MAXADDRESS + 1];
	regex_t re;

	assert(dkim != NULL);
	assert(gran != NULL);

	/* if it's empty, it matches nothing */
	if (gran[0] == '\0')
		return FALSE;

	/* if it's just "*", it matches everything */
	if (gran[0] == '*' && gran[1] == '\0')
		return TRUE;

	/* ensure we're evaluating against a signature data set */
	set = dkim->dkim_sigset;
	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	/* get the value of the "i" parameter */
	p = dkim_param_get(set, "i");
	assert(p != NULL);
	sm_strlcpy(user, p, sizeof user);

	/* validate the "i" parameter */
	at = strchr(user, '@');
	assert(at != NULL);

	/* if it's not wildcarded, enforce an exact match */
	if (strchr(gran, '*') == NULL)
		return (strncmp(gran, user, at - user) == 0);

	/* evaluate the wildcard */
	end = restr + sizeof restr;
	memset(restr, '\0', sizeof restr);
	for (p = gran, q = restr; q < end - 1; p++)
	{
		if (isascii(*p) && ispunct(*p))
		{
			if (*p == '*')
			{
				*q++ = '.';
				*q++ = '*';
			}
			else
			{
				*q++ = '\\';
				*q++ = *p;
			}
		}
		else
		{
			*q++ = *p;
		}
	}

	if (q >= end - 1)
		return FALSE;

	status = regcomp(&re, user, 0);
	if (status != 0)
		return FALSE;

	status = regexec(&re, user, 0, NULL, 0);
	(void) regfree(&re);

	return (status == 0 ? TRUE : FALSE);
}

/*
**  DKIM_KEY_HASHOK -- return TRUE iff a particular hash is in the approved
**                     list of hashes for a given key
**
**  Parameters:
**  	dkim -- DKIM handle
**  	hashlist -- colon-separated approved hash list
**
**  Return value:
**  	TRUE iff a particular hash is in the approved list of hashes.
*/

static bool
dkim_key_hashok(DKIM *dkim, u_char *hashlist)
{
	int hashalg;
	u_char *x, *y;
	u_char tmp[BUFRSZ + 1];

	assert(dkim != NULL);

	if (hashlist == NULL)
		return TRUE;

	x = NULL;
	memset(tmp, '\0', sizeof tmp);

	y = hashlist;
	for (;;)
	{
		if (*y == ':' || *y == '\0')
		{
			if (x != NULL)
			{
				sm_strlcpy(tmp, x, sizeof tmp);
				tmp[y - x] = '\0';
				hashalg = dkim_name_to_code(hashes, tmp);
				if (hashalg == dkim->dkim_hashtype)
					return TRUE;
			}

			x = NULL;
		}
		else if (x == NULL)
		{
			x = y;
		}

		if (*y == '\0')
			return FALSE;
		y++;
	}

	/* NOTREACHED */
}

/*
**  DKIM_KEY_HASHESOK -- return TRUE iff this key supports at least one
**                       hash method we know about (or doesn't specify)
**
**  Parameters:
**  	hashlist -- colon-separated list of hashes (or NULL)
**
**  Return value:
**  	TRUE iff this key supports at least one hash method we know about
**  	(or doesn't specify)
*/

static bool
dkim_key_hashesok(u_char *hashlist)
{
	u_char *x, *y;
	u_char tmp[BUFRSZ + 1];

	if (hashlist == NULL)
		return TRUE;

	x = NULL;
	memset(tmp, '\0', sizeof tmp);

	y = hashlist;
	for (;;)
	{
		if (*y == ':' || *y == '\0')
		{
			if (x != NULL)
			{
				sm_strlcpy(tmp, x, sizeof tmp);
				tmp[y - x] = '\0';
				if (dkim_name_to_code(hashes, tmp) != -1)
					return TRUE;
			}

			x = NULL;
		}
		else if (x == NULL)
		{
			x = y;
		}

		if (*y == '\0')
			return FALSE;
		y++;
	}

	/* NOTREACHED */
}

#if 0
/*
**  DKIM_SIG_SIGNEROK -- return TRUE iff the signer is specified in a signed
**                       sender header
**
**  Parameters:
**  	dkim -- DKIM handle
**  	set -- signature set to be checked
**  	hdrs -- names of sender headers
**
**  Return value:
**  	TRUE iff the value of the "i" parameter appears in a signed sender
**  	header.
**
**  Note:
**  	This essentially detects third-party signatures.  It's not of use
**  	yet until SSP addresses this question.
*/

static bool
dkim_sig_signerok(DKIM *dkim, DKIM_SET *set, u_char **hdrs)
{
	int status;
	int c;
	int clen;
	struct dkim_header *cur;
	u_char *colon;
	char *i;
	char *user;
	char *domain;
	char buf[MAXADDRESS + 1];
	char addr[MAXADDRESS + 1];

	assert(dkim != NULL);
	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	i = dkim_param_get(set, "i");

	assert(i != NULL);

	/* for each header in the "sender header" list */
	for (c = 0; hdrs[c] != NULL; c++)
	{
		/* for each header in the message */
		for (cur = dkim->dkim_hhead; cur != NULL; cur = cur->hdr_next)
		{
			/* skip unsigned headers */
			if ((cur->hdr_flags & DKIM_HDR_SIGNED) == 0)
				continue;

			/* determine header name size */
			colon = strchr(cur->hdr_text, ':');
			if (colon == NULL)
				clen = strlen(cur->hdr_text);
			else
				clen = colon - cur->hdr_text;

			/* if this is a sender header */
			if (strncasecmp(hdrs[c], cur->hdr_text, clen) == 0)
			{
				if (colon == NULL)
					colon = cur->hdr_text;
				else
					colon += 1;

				sm_strlcpy(buf, colon, sizeof buf);

				status = rfc2822_mailbox_split(buf, &user,
				                               &domain);
				if (status != 0)
					continue;

				snprintf(addr, sizeof addr, "%s@%s",
				         user, domain);

				/* see if the sender matches "i" */
				if (dkim_addrcmp(addr, i) == 0)
					return TRUE;
			}
		}
	}

	return FALSE;
}
#endif /* 0 */

/*
**  DKIM_SIG_DOMAINOK -- return TRUE iff a signature appears to have valid
**                       domain correlation; that is, "i" must be the same
**                       domain as or a subdomain of "d"
**
**  Parameters:
**  	dkim -- DKIM handle
**  	set -- signature set to be checked
**
**  Return value:
**  	TRUE iff the "i" parameter and the "d" parameter match up.
*/

static bool
dkim_sig_domainok(DKIM *dkim, DKIM_SET *set)
{
	char *at;
	char *dot;
	char *i;
	char *d;

	assert(dkim != NULL);
	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	i = dkim_param_get(set, "i");
	d = dkim_param_get(set, "d");

	assert(i != NULL);
	assert(d != NULL);

	at = strchr(i, '@');
	if (at == NULL)
		return FALSE;

	if (strcasecmp(at + 1, d) == 0)
		return TRUE;

	for (dot = strchr(at, '.'); dot != NULL; dot = strchr(dot + 1, '.'))
	{
		if (strcasecmp(dot + 1, d) == 0)
		{
			dkim->dkim_subdomain = TRUE;
			return TRUE;
		}
	}

	return FALSE;
}

/*
**  DKIM_SIG_EXPIRED -- return TRUE iff a signature appears to have expired
**
**  Parameters:
**  	set -- signature set to be checked
**
**  Return value:
**  	TRUE iff "set" contains an "x=" parameter which indicates a time
**  	which has passed.
**
**  Notes:
**  	Syntax is not checked here.  It should be checked elsewhere.
*/

static bool
dkim_sig_expired(DKIM_SET *set)
{
	time_t expire;
	time_t now;
	u_char *val;
	char *end;

	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	val = dkim_param_get(set, "x");
	if (val == NULL)
		return FALSE;

	expire = strtoul(val, &end, 10);

	(void) time(&now);

	return (now >= expire);
}

/*
**  DKIM_SIG_TIMESTAMPSOK -- return TRUE iff a signature appears to have
**                           both a timestamp and an expiration date and they
**                           are properly ordered
**
**  Parameters:
**  	set -- signature set to be checked
**
**  Return value:
**  	TRUE: - "set" contains both a "t=" parameter and an "x=" parameter
**  	        and the latter is greater than the former
**  	      - "set" is missing either "t=" or "x=" (or both)
**  	FALSE: otherwise
**
**  Notes:
**  	Syntax is not checked here.  It should be checked elsewhere.
*/

static bool
dkim_sig_timestampsok(DKIM_SET *set)
{
	time_t signtime;
	time_t expire;
	u_char *val;
	char *end;

	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	val = dkim_param_get(set, "t");
	if (val == NULL)
		return TRUE;
	signtime = strtoul(val, &end, 10);

	val = dkim_param_get(set, "x");
	if (val == NULL)
		return TRUE;
	expire = strtoul(val, &end, 10);

	return (signtime < expire);
}

/*
**  DKIM_SIG_FUTURE -- return TRUE iff a signature appears to have been
**                     generated in the future
**
**  Parameters:
**  	set -- signature set to be checked
**
**  Return value:
**  	TRUE iff "set" contains a "t=" parameter which indicates a time
**  	in the future.
**
**  Notes:
**  	Syntax is not checked here.  It should be checked elsewhere.
*/

static bool
dkim_sig_future(DKIM_SET *set)
{
	time_t signtime;
	time_t now;
	u_char *val;
	char *end;

	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	val = dkim_param_get(set, "t");
	if (val == NULL)
		return FALSE;

	signtime = strtoul(val, &end, 10);

	(void) time(&now);

	return (now < signtime);
}

/*
**  DKIM_SIG_VERSIONOK -- return TRUE iff a signature appears to have a version
**                        we can accept
**
**  Parameters:
**  	set -- signature set to be checked
**
**  Return value:
**  	TRUE iff "set" appears to be based on a version of DKIM that is
**  	supported by this API.
*/

static bool
dkim_sig_versionok(DKIM_SET *set)
{
	char *v;

	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	v = dkim_param_get(set, "v");
	if (v == NULL ||
	    (strcmp(v, DKIM_VERSION_SIG) == 0 ||
	     strcmp(v, DKIM_VERSION_SIG_DRAFT1) == 0 ||
	     strcmp(v, DKIM_VERSION_SIG_DRAFT2) == 0))
		return TRUE;

	return FALSE;
}

/*
**  DKIM_TMPFILE -- open a temporary file
**
**  Parameters:
**  	dkim -- DKIM handle
**  	fp -- descriptor (returned)
**  	keep -- if FALSE, unlink() the file once created
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

static DKIM_STAT
dkim_tmpfile(DKIM *dkim, int *fp, bool keep)
{
	int fd;
	char path[MAXPATHLEN + 1];

	assert(dkim != NULL);
	assert(fp != NULL);

	if (dkim->dkim_id != NULL)
	{
		snprintf(path, MAXPATHLEN, "%s/dkim.%s.XXXXXX",
		         dkim->dkim_libhandle->dkiml_tmpdir, dkim->dkim_id);
	}
	else
	{
		snprintf(path, MAXPATHLEN, "%s/dkim.XXXXXX",
		         dkim->dkim_libhandle->dkiml_tmpdir);
	}

	fd = mkstemp(path);
	if (fd == -1)
	{
		dkim_error(dkim, "can't create temporary file at %s: %s",
		           path, strerror(errno));
		return DKIM_STAT_NORESOURCE;
	}

	*fp = fd;

	if (!keep)
		(void) unlink(path);

	return DKIM_STAT_OK;
}

/*
**  DKIM_CANONINIT -- initialize canonicalization
**
**  Parameters:
**  	dkim -- DKIM handle
**  	htype -- type of the hash to be performed
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

static DKIM_STAT
dkim_canoninit(DKIM *dkim, int htype)
{
	bool tmp = FALSE;
	bool keep = FALSE;

	assert(dkim != NULL);
#ifdef SHA256_DIGEST_LENGTH
	assert(htype == DKIM_HASHTYPE_SHA1 || htype == DKIM_HASHTYPE_SHA256);
#else /* SHA256_DIGEST_LENGTH */
	assert(htype == DKIM_HASHTYPE_SHA1);
#endif /* SHA256_DIGEST_LENGTH */
	assert(dkim->dkim_hash == NULL);

	tmp = (dkim->dkim_libhandle->dkiml_flags & DKIM_LIBFLAGS_TMPFILES) != 0;
	keep = (dkim->dkim_libhandle->dkiml_flags & DKIM_LIBFLAGS_KEEPFILES) != 0;

#ifdef _FFR_HASH_BUFFERING
	dkim->dkim_hashbuf = DKIM_MALLOC(dkim, DKIM_HASHBUFSIZE);
	if (dkim->dkim_hashbuf == NULL)
	{
		dkim_error(dkim, "unable to allocate %d byte(s)",
		           DKIM_HASHBUFSIZE);
		return DKIM_STAT_NORESOURCE;
	}
	dkim->dkim_hashbufsize = DKIM_HASHBUFSIZE;
	dkim->dkim_hashbuflen = 0;
#endif /* _FFR_HASH_BUFFERING */

	switch (htype)
	{
	  case DKIM_HASHTYPE_SHA1:
	  {
		struct dkim_sha1 *sha1_h;
		struct dkim_sha1 *sha1_b;

		sha1_h = (struct dkim_sha1 *) DKIM_MALLOC(dkim,
		                                          sizeof(struct dkim_sha1));
		if (sha1_h == NULL)
		{
			dkim_error(dkim, "unable to allocate %d byte(s)",
			           sizeof(struct dkim_sha1));
			return DKIM_STAT_NORESOURCE;
		}

		sha1_b = (struct dkim_sha1 *) DKIM_MALLOC(dkim,
		                                          sizeof(struct dkim_sha1));
		if (sha1_b == NULL)
		{
			DKIM_FREE(dkim, sha1_h);
			dkim_error(dkim, "unable to allocate %d byte(s)",
			           sizeof(struct dkim_sha1));
			return DKIM_STAT_NORESOURCE;
		}

		memset(sha1_h, '\0', sizeof(struct dkim_sha1));
		memset(sha1_b, '\0', sizeof(struct dkim_sha1));
		SHA1_Init(&sha1_h->sha1_ctx);
		SHA1_Init(&sha1_b->sha1_ctx);

		dkim->dkim_hash = sha1_h;
		dkim->dkim_bhash = sha1_b;

		if (tmp)
		{
			DKIM_STAT status;

			status = dkim_tmpfile(dkim, &sha1_h->sha1_tmpfd, keep);
			if (status != DKIM_STAT_OK)
			{
				DKIM_FREE(dkim, sha1_h);
				DKIM_FREE(dkim, sha1_b);
				return status;
			}

			status = dkim_tmpfile(dkim, &sha1_b->sha1_tmpfd, keep);
			if (status != DKIM_STAT_OK)
			{
				close(sha1_h->sha1_tmpfd);
				DKIM_FREE(dkim, sha1_h);
				DKIM_FREE(dkim, sha1_b);
				return status;
			}

			sha1_h->sha1_tmpbio = BIO_new_fd(sha1_h->sha1_tmpfd, 1);
			sha1_b->sha1_tmpbio = BIO_new_fd(sha1_b->sha1_tmpfd, 1);
		}

		break;
	  }

#ifdef SHA256_DIGEST_LENGTH
	  case DKIM_HASHTYPE_SHA256:
	  {
		struct dkim_sha256 *sha256_h;
		struct dkim_sha256 *sha256_b;

		sha256_h = (struct dkim_sha256 *) DKIM_MALLOC(dkim,
		                                              sizeof(struct dkim_sha256));
		if (sha256_h == NULL)
		{
			dkim_error(dkim, "unable to allocate %d byte(s)",
			           sizeof(struct dkim_sha256));
			return DKIM_STAT_NORESOURCE;
		}

		sha256_b = (struct dkim_sha256 *) DKIM_MALLOC(dkim,
		                                              sizeof(struct dkim_sha256));
		if (sha256_b == NULL)
		{
			DKIM_FREE(dkim, sha256_h);
			dkim_error(dkim, "unable to allocate %d byte(s)",
			           sizeof(struct dkim_sha256));
			return DKIM_STAT_NORESOURCE;
		}

		memset(sha256_h, '\0', sizeof(struct dkim_sha256));
		memset(sha256_b, '\0', sizeof(struct dkim_sha256));
		SHA256_Init(&sha256_h->sha256_ctx);
		SHA256_Init(&sha256_b->sha256_ctx);

		dkim->dkim_hash = sha256_h;
		dkim->dkim_bhash = sha256_b;

		if (tmp)
		{
			DKIM_STAT status;

			status = dkim_tmpfile(dkim, &sha256_h->sha256_tmpfd,
			                      keep);
			if (status != DKIM_STAT_OK)
			{
				DKIM_FREE(dkim, sha256_h);
				DKIM_FREE(dkim, sha256_b);
				return status;
			}

			status = dkim_tmpfile(dkim, &sha256_b->sha256_tmpfd,
			                      keep);
			if (status != DKIM_STAT_OK)
			{
				close(sha256_h->sha256_tmpfd);
				DKIM_FREE(dkim, sha256_h);
				DKIM_FREE(dkim, sha256_b);
				return status;
			}

			sha256_h->sha256_tmpbio = BIO_new_fd(sha256_h->sha256_tmpfd,
			                                     1);
			sha256_b->sha256_tmpbio = BIO_new_fd(sha256_b->sha256_tmpfd,
			                                     1);
		}

		break;
	  }
#endif /* SHA256_DIGEST_LENGTH */
	}

	dkim->dkim_hashtype = htype;

	return DKIM_STAT_OK;
}

#ifdef _FFR_HASH_BUFFERING
/*
**  DKIM_CANONBUFFER -- buffer for dkim_canonwrite()
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- buffer containing canonicalized data
**  	buflen -- number of bytes to consume
**
**  Return value:
**  	None.
*/

static void
dkim_canonbuffer(DKIM *dkim, u_char *buf, size_t buflen)
{
	assert(dkim != NULL);

	/* NULL buffer or 0 length means flush */
	if ((buf == NULL || buflen == 0) && dkim->dkim_hashbuflen > 0)
	{
		dkim_canonwrite(dkim, dkim->dkim_hashbuf,
		                dkim->dkim_hashbuflen);
		dkim->dkim_hashbuflen = 0;
		return;
	}

	/* not enough buffer space; write the buffer out */
	if (dkim->dkim_hashbuflen + buflen > dkim->dkim_hashbufsize)
	{
		dkim_canonwrite(dkim, dkim->dkim_hashbuf,
		                dkim->dkim_hashbuflen);
		dkim->dkim_hashbuflen = 0;
	}

	/*
	**  Now, if the input is bigger than the buffer, write it too;
	**  otherwise cache it.
	*/

	if (buflen >= dkim->dkim_hashbufsize)
	{
		dkim_canonwrite(dkim, buf, buflen);
	}
	else
	{
		memcpy(&dkim->dkim_hashbuf[dkim->dkim_hashbuflen],
		       buf, buflen);
		dkim->dkim_hashbuflen += buflen;
	}
}
#endif /* _FFR_HASH_BUFFERING */

/*
**  DKIM_CANONWRITE -- write data to canonicalization stream(s)
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- buffer containing canonicalized data
**  	buflen -- number of bytes to consume
**
**  Return value:
**  	None.
*/

static void
dkim_canonwrite(DKIM *dkim, u_char *buf, size_t buflen)
{
	u_long wlen;

	assert(dkim != NULL);

	if (buf == NULL || buflen == 0)
		return;

	assert(dkim->dkim_hash != NULL);

	if (dkim->dkim_writesep &&
	    dkim->dkim_version < DKIM_VERSION_IETF_BASE_01)
	{
		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;

			SHA1_Update(&sha1->sha1_ctx, CRLF, 2);

			if (sha1->sha1_tmpbio != NULL)
				BIO_write(sha1->sha1_tmpbio, CRLF, 2);

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;

			SHA256_Update(&sha256->sha256_ctx, CRLF, 2);

			if (sha256->sha256_tmpbio != NULL)
				BIO_write(sha256->sha256_tmpbio, CRLF, 2);

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */
		}
	}

	dkim->dkim_writesep = FALSE;

	if (dkim->dkim_state == DKIM_STATE_BODY)
	{
		wlen = (u_long) MIN((u_long) buflen,
		                    (u_long) dkim->dkim_signlen);
		if (wlen == 0)
		{
			dkim->dkim_partial = TRUE;
			return;
		}
	}
	else
	{
		wlen = (u_long) buflen;
	}

	switch (dkim->dkim_hashtype)
	{
	  case DKIM_HASHTYPE_SHA1:
	  {
		struct dkim_sha1 *sha1;

		if (dkim->dkim_state >= DKIM_STATE_BODY &&
		    !dkim->dkim_bodydone &&
		    dkim->dkim_version >= DKIM_VERSION_IETF_BASE_01)
			sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;
		else
			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;

		SHA1_Update(&sha1->sha1_ctx, buf, wlen);

		if (sha1->sha1_tmpbio != NULL)
			BIO_write(sha1->sha1_tmpbio, buf, wlen);

		break;
	  }

#ifdef SHA256_DIGEST_LENGTH
	  case DKIM_HASHTYPE_SHA256:
	  {
		struct dkim_sha256 *sha256;

		if (dkim->dkim_state >= DKIM_STATE_BODY &&
		    !dkim->dkim_bodydone &&
		    dkim->dkim_version >= DKIM_VERSION_IETF_BASE_01)
			sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;
		else
			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;

		SHA256_Update(&sha256->sha256_ctx, buf, wlen);

		if (sha256->sha256_tmpbio != NULL)
			BIO_write(sha256->sha256_tmpbio, buf, wlen);

		break;
	  }
#endif /*SHA256_DIGEST_LENGTH */
	}

	if (dkim->dkim_state == DKIM_STATE_BODY)
	{
		dkim->dkim_bodylen += buflen;
		dkim->dkim_signlen -= wlen;
		dkim->dkim_canonlen += wlen;
	}
}

/*
**  DKIM_CANONHEADER -- canonicalize a header and write it
**
**  Parameters:
**  	dkim -- DKIM handle
**  	hdr -- header handle
**  	crlf -- write a CRLF at the end?
**
**  Return value:
**  	A DKIM_STAT constant.
*/

static DKIM_STAT
dkim_canonheader(DKIM *dkim, struct dkim_header *hdr, bool crlf)
{
	bool space;
	bool firstword;
	bool colon;
	int n;
	u_char *p;
	char tmp[MAXHEADER + 1];

	assert(dkim != NULL);
	assert(hdr != NULL);

	memset(tmp, '\0', sizeof tmp);
	n = 0;

#ifdef _FFR_HASH_BUFFERING
	dkim_canonbuffer(dkim, NULL, 0);
#endif /* _FFR_HASH_BUFFERING */

	switch (dkim->dkim_hdrcanonalg)
	{
	  case DKIM_CANON_SIMPLE:
		if (sm_strlcat(tmp, hdr->hdr_text, sizeof tmp) >= sizeof tmp ||
		    (crlf && sm_strlcat(tmp, CRLF, sizeof tmp) >= sizeof tmp))
		{
			dkim_error(dkim, "header too large (max %d)",
			           sizeof tmp);
			return DKIM_STAT_INTERNAL;
		}
		break;

	  case DKIM_CANON_RELAXED:
		space = FALSE;				/* just saw a space */
		firstword = FALSE;			/* in/past first word */
		colon = FALSE;				/* past colon */
		for (p = hdr->hdr_text; *p != '\0'; p++)
		{
			/*
			**  Discard spaces before the colon or before the end
			**  of the first word.
			*/

			if (isascii(*p) && isspace(*p) &&
			    (!colon || !firstword))
				continue;

			/*
			**  Discard spaces other than the first one after
			**  the first word.
			*/

			if (isascii(*p) && isspace(*p) && space && firstword)
				continue;

			/*
			**  The first space found after a colon that gets
			**  this far marks the end of a word.
			*/

			if (isascii(*p) && isspace(*p) && colon && !space)
			{
				space = TRUE;
				continue;
			}

			/*
			**  Any non-space marks the beginning of a word.
			**  If there's a stored space, use it up.
			*/

			if ((!isascii(*p) || !isspace(*p)) &&
			    firstword && space)
			{
				tmp[n++] = ' ';
				if (n == sizeof tmp)
				{
					dkim_error(dkim,
					           "header too large (max %d)",
					           sizeof tmp);
					return DKIM_STAT_INTERNAL;
				}

				space = FALSE;
			}

			/* copy the byte */
			tmp[n++] = *p;
			if (n == sizeof tmp)
			{
				dkim_error(dkim, "header too large (max %d)",
				           sizeof tmp);
				return DKIM_STAT_INTERNAL;
			}

			/*
			**  If this was the first colon, remember that we
			**  saw it.  Otherwise, anything that got this
			**  far after the first colon means we're in or
			**  past the first word.
			*/

			if (*p == ':' && !colon)
				colon = TRUE;
			else if (colon)
				firstword = TRUE;
		}

		dkim_lowerhdr(tmp);

		if (crlf && sm_strlcat(tmp, CRLF, sizeof tmp) >= sizeof tmp)
		{
			dkim_error(dkim, "header too large (max %d)",
			           sizeof tmp);
			return DKIM_STAT_INTERNAL;
		}
		break;
	}

	dkim_canonwrite(dkim, tmp, strlen(tmp));

	return DKIM_STAT_OK;
}

/*
**  DKIM_CANONFINAL -- close canonicalization
**
**  Parameters:
**  	dkim -- DKIM handle
**  	body -- TRUE iff the one to be closed is the body
**
**  Return value:
**  	None.
*/

static void
dkim_canonfinal(DKIM *dkim, bool body)
{
	assert(dkim != NULL);
	assert(dkim->dkim_hash != NULL);

	switch (dkim->dkim_hashtype)
	{
	  case DKIM_HASHTYPE_SHA1:
	  {
		struct dkim_sha1 *sha1;

		if (dkim->dkim_bhash != NULL && body)
		{
			sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;
			SHA1_Final(sha1->sha1_out, &sha1->sha1_ctx);
		}
		else
		{
			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;
			SHA1_Final(sha1->sha1_out, &sha1->sha1_ctx);
		}

		break;
	  }

#ifdef SHA256_DIGEST_LENGTH
	  case DKIM_HASHTYPE_SHA256:
	  {
		struct dkim_sha256 *sha256;

		if (dkim->dkim_bhash != NULL && body)
		{
			sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;
			SHA256_Final(sha256->sha256_out, &sha256->sha256_ctx);
		}
		else
		{
			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;
			SHA256_Final(sha256->sha256_out, &sha256->sha256_ctx);
		}

		break;
	  }
#endif /* SHA256_DIGEST_LENGTH */

	  default:
		assert(0);
	}
}

/*
**  DKIM_FLUSH_BLANKS -- use accumulated blank lines in canonicalization
**
**  Parameters:
**  	dkim -- DKIM handle
**
**  Return value:
**  	None.
*/

static void
dkim_flush_blanks(DKIM *dkim)
{
	int c;

	assert(dkim != NULL);

	for (c = 0; c < dkim->dkim_blanks; c++)
#ifdef _FFR_HASH_BUFFERING
		dkim_canonbuffer(dkim, CRLF, 2);
#else /* _FFR_HASH_BUFFERING */
		dkim_canonwrite(dkim, CRLF, 2);
#endif /* _FFR_HASH_BUFFERING */
	dkim->dkim_blanks = 0;
}

/*
**  DKIM_GENSIGHDR -- generate a signature header
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- where to write
**  	buflen -- bytes available at "buf"
**  	delim -- delimiter
**
**  Return value:
**  	Number of bytes written to "buf".
*/

static size_t
dkim_gensighdr(DKIM *dkim, u_char *buf, size_t buflen, char *delim)
{
	bool firsthdr;
	int n;
	u_char *colon;
	struct dkim_header *hdr;
	bool always[MAXHDRCNT];
	u_char tmp[MAXHEADER + 1];
	u_char tmphdr[MAXHEADER + 1];

	assert(dkim != NULL);
	assert(buf != NULL);
	assert(delim != NULL);

	memset(tmphdr, '\0', sizeof tmphdr);

	/*
	**  We need to generate a DKIM-Signature: header template
	**  and include it in the canonicalization.
	*/

	/* basic required stuff */
	if (dkim->dkim_version < DKIM_VERSION_IETF_BASE_02)
	{
		snprintf(tmphdr, sizeof tmphdr,
		         "a=%s;%sc=%s/%s;%sd=%s;%ss=%s;%st=%lu",
		         dkim_code_to_name(algorithms, dkim->dkim_signalg),
		         delim,
		         dkim_code_to_name(canonicalizations,
			                   dkim->dkim_hdrcanonalg),
		         dkim_code_to_name(canonicalizations,
		                           dkim->dkim_bodycanonalg), delim,
		         dkim->dkim_domain, delim,
		         dkim->dkim_selector, delim,
		         dkim->dkim_timestamp);
	}
	else
	{
		char *version;

		/* XXX -- when the RFC issues, force DKIM_VERSION_SIG */

		if (dkim->dkim_version == DKIM_VERSION_IETF_BASE_02)
			version = DKIM_VERSION_SIG_DRAFT1;
		else if (dkim->dkim_version == DKIM_VERSION_RFC4871)
			version = DKIM_VERSION_SIG;
		else
			version = DKIM_VERSION_SIG_DRAFT2;

		snprintf(tmphdr, sizeof tmphdr,
		         "v=%s;%sa=%s;%sc=%s/%s;%sd=%s;%ss=%s;%st=%lu",
		         version, delim,
		         dkim_code_to_name(algorithms, dkim->dkim_signalg),
		         delim,
		         dkim_code_to_name(canonicalizations,
		                           dkim->dkim_hdrcanonalg),
		         dkim_code_to_name(canonicalizations,
		                           dkim->dkim_bodycanonalg), delim,
		         dkim->dkim_domain, delim,
		         dkim->dkim_selector, delim,
		         dkim->dkim_timestamp);
	}

	if (dkim->dkim_libhandle->dkiml_sigttl != 0)
	{
		time_t expire;

		expire = dkim->dkim_timestamp + dkim->dkim_libhandle->dkiml_sigttl;
		snprintf(tmp, sizeof tmp, ";%sx=%lu", delim, (u_long) expire);
		sm_strlcat(tmphdr, tmp, sizeof tmphdr);
	}

	if (dkim->dkim_signer != NULL)
	{
		snprintf(tmp, sizeof tmp, ";%si=%s", delim, dkim->dkim_signer);
		sm_strlcat(tmphdr, tmp, sizeof tmphdr);
	}

	if (dkim->dkim_version >= DKIM_VERSION_IETF_BASE_01 &&
	    dkim->dkim_bhash != NULL)
	{
		int status;
		size_t hashlen;
		unsigned char *hash;
		char b64hash[MAXHEADER + 1];

		memset(b64hash, '\0', sizeof b64hash);

		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;
			hash = (void *) &sha1->sha1_out;
			hashlen = SHA_DIGEST_LENGTH;

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;
			hash = (void *) &sha256->sha256_out;
			hashlen = SHA256_DIGEST_LENGTH;

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */
		}

		status = dkim_base64_encode(hash, hashlen,
		                            b64hash, sizeof b64hash);

		snprintf(tmp, sizeof tmp, ";%sbh=%s", delim,
		         b64hash);
		sm_strlcat(tmphdr, tmp, sizeof tmphdr);
	}

	/* l= */
	if (dkim->dkim_partial)
	{
		snprintf(tmp, sizeof tmp, ";%sl=%lu", delim,
		         (u_long) dkim->dkim_canonlen);
		sm_strlcat(tmphdr, tmp, sizeof tmphdr);
	}

	/* h= */
	for (n = 0; n < MAXHDRCNT; n++)
		always[n] = TRUE;

	firsthdr = TRUE;

	for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
	{
		if ((hdr->hdr_flags & DKIM_HDR_SIGNED) == 0)
			continue;

		colon = strchr(hdr->hdr_text, ':');
		if (colon == NULL)
		{
			sm_strlcpy(tmp, hdr->hdr_text, sizeof tmp);
		}
		else
		{
			size_t len;

			len = MIN(MAXHEADER, colon - hdr->hdr_text);
			memset(tmp, '\0', sizeof tmp);
			strncpy(tmp, hdr->hdr_text, len);
		}

		if (!firsthdr)
		{
			sm_strlcat(tmphdr, ":", sizeof tmphdr);
		}
		else
		{
			sm_strlcat(tmphdr, ";", sizeof tmphdr);
			sm_strlcat(tmphdr, delim, sizeof tmphdr);
			sm_strlcat(tmphdr, "h=", sizeof tmphdr);
		}

		firsthdr = FALSE;

		sm_strlcat(tmphdr, tmp, sizeof tmphdr);

		if (dkim->dkim_libhandle->dkiml_alwayshdrs != NULL)
		{
			u_char **ah = dkim->dkim_libhandle->dkiml_alwayshdrs;

			for (n = 0; ah[n] != NULL && n < MAXHDRCNT; n++)
			{
				if (strcasecmp(tmp, ah[n]) == 0)
				{
					always[n] = FALSE;
					break;
				}
			}
		}
	}

	/* apply any "always sign" list */
	if (dkim->dkim_libhandle->dkiml_alwayshdrs != NULL)
	{
		u_char **ah = dkim->dkim_libhandle->dkiml_alwayshdrs;

		for (n = 0; ah[n] != NULL && n < MAXHDRCNT; n++)
		{
			if (always[n])
			{
				if (!firsthdr)
				{
					sm_strlcat(tmphdr, ":", sizeof tmphdr);
				}
				else
				{
					sm_strlcat(tmphdr, ";", sizeof tmphdr);
					sm_strlcat(tmphdr, delim,
					           sizeof tmphdr);
					sm_strlcat(tmphdr, "h=",
					           sizeof tmphdr);
				}

				firsthdr = FALSE;

				sm_strlcat(tmphdr, ah[n], sizeof tmphdr);
			}
		}
	}

	/* if diagnostic headers were included, include 'em */
	if (dkim->dkim_libhandle->dkiml_flags & DKIM_LIBFLAGS_ZTAGS)
	{
		bool first;
		char *p;
		char *q;
		char *end;
		size_t len;

		first = TRUE;

		end = tmp + sizeof tmp - 1;

		sm_strlcat(tmphdr, ";", sizeof tmphdr);
		sm_strlcat(tmphdr, delim, sizeof tmphdr);
		sm_strlcat(tmphdr, "z=", sizeof tmphdr);

		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			memset(tmp, '\0', sizeof tmp);
			q = tmp;
			len = sizeof tmp - 1;

			if (!first)
			{
				tmp[0] = '|';
				q++;
				len--;
			}

			first = FALSE;

			for (p = hdr->hdr_text; *p != '\0'; p++)
			{
				if (q >= end)
					break;

				if ((*p >= 0x21 && *p <= 0x3a) ||
				    *p == 0x3c ||
				    (*p >= 0x3e && *p <= 0x7e))
				{
					*q = *p;
					q++;
					len--;
				}
				else
				{
					snprintf(q, len, "=%02X", *p);
					q += 3;
					len -= 3;
				}
			}

			sm_strlcat(tmphdr, tmp, sizeof tmphdr);
		}
	}

	/* and finally, an empty b= */
	sm_strlcat(tmphdr, ";", sizeof tmphdr);
	sm_strlcat(tmphdr, delim, sizeof tmphdr);
	sm_strlcat(tmphdr, "b=", sizeof tmphdr);

	sm_strlcpy(buf, tmphdr, buflen);

	return strlen(buf);
}

/*
**  DKIM_GETSENDER -- determine sender (actually just multi-search)
**
**  Parameters:
**  	dkim -- DKIM handle
**  	hdrs -- list of header names to find
**  	signedhdrs -- if TRUE, only check signed headers
**
**  Return value:
**  	Pointer to the first such header found, or NULL if none.
*/

static struct dkim_header *
dkim_getsender(DKIM *dkim, u_char **hdrs, bool signedhdrs)
{
	int c;
	size_t clen;
	u_char *colon;
	struct dkim_header *cur;

	assert(dkim != NULL);
	assert(hdrs != NULL);

	for (c = 0; hdrs[c] != NULL; c++)
	{
		for (cur = dkim->dkim_hhead; cur != NULL; cur = cur->hdr_next)
		{
			if (signedhdrs &&
			    (cur->hdr_flags & DKIM_HDR_SIGNED) == 0)
				continue;

			colon = strchr(cur->hdr_text, ':');
			if (colon == NULL)
				clen = strlen(cur->hdr_text);
			else
				clen = colon - cur->hdr_text;

			if (strncasecmp(hdrs[c], cur->hdr_text, clen) == 0)
				return cur;
		}
	}

	return NULL;
}

/*
**  DKIM_GET_POLICY -- acquire a domain's policy record
**
**  Parameters:
**  	dkim -- DKIM handle
**
**  Return value:
**  	1 -- policy retrieved, parsed OK
**  	0 -- no policy found; default applied
**  	-1 -- failure
*/

static int
dkim_get_policy(DKIM *dkim)
{
	bool defpolicy;
	int status;
	unsigned char buf[BUFRSZ + 1];

	assert(dkim != NULL);

	memset(buf, '\0', sizeof buf);

	if (dkim->dkim_libhandle->dkiml_policy_lookup != NULL)
	{
		status = dkim->dkim_libhandle->dkiml_policy_lookup(dkim, buf,
		                                                   sizeof buf);
	}
	else
	{
		switch (dkim->dkim_querymethod)
		{
		  case DKIM_QUERY_DNS:
			status = dkim_get_policy_dns(dkim, buf, sizeof buf);
			break;

		  case DKIM_QUERY_FILE:
			status = dkim_get_policy_file(dkim, buf, sizeof buf);
			break;

		  default:
			assert(0);
			/* just to silence -Wall */
			return -1;
		}
	}

	if (status == 0)
	{
		defpolicy = TRUE;
		status = dkim_process_set(dkim, DKIM_SETTYPE_POLICY,
		                          DKIM_POLICY_DEFAULTTXT,
		                          strlen(DKIM_POLICY_DEFAULTTXT));
	}
	else if (status == 1)
	{
		status = dkim_process_set(dkim, DKIM_SETTYPE_POLICY,
		                          buf, strlen(buf));
	}

#ifdef _FFR_ALLMAN_SSP_02
	/*
	**  Requery if this is a site policy and it requires user
	**  re-queries.
	*/

	if (!dkim->dkim_userpolicy)
	{
		DKIM_SET *s;

		s = dkim_set_first(dkim, DKIM_SETTYPE_POLICY);
		assert(s != NULL);

		p = dkim_param_get(s, "u");
		if (p != NULL && strcasecmp(p, "yes") == 0)
		{
			dkim->dkim_userpolicy = TRUE;
			return dkim_get_policy(dkim);
		}
	}
#endif /* _FFR_ALLMAN_SSP_02 */

	if (status == DKIM_STAT_OK)
		return (defpolicy ? 0 : 1);
	else
		return -1;
}

/*
**  DKIM_POLICY_FLAG -- see if a policy flag is set
**
**  Parameters:
**  	dkim -- DKIM handle
**  	flag -- flag to check
**
**  Return value:
**  	1 -- flag set
**  	0 -- flag not set
**  	-1 -- error
*/

static int
dkim_policy_flag(DKIM *dkim, int flag)
{
	struct dkim_set *set;
	u_char *param;

	assert(dkim != NULL);
	assert(flag != '\0');

	set = dkim_set_first(dkim, DKIM_SETTYPE_POLICY);
	if (set == NULL)
		return -1;

	param = dkim_param_get(set, "t");
	if (param != NULL)
		return (strchr(param, flag) == NULL) ? 0 : 1;
	else
		return 0;
}

/*
**  DKIM_GET_KEY -- acquire a public key used for verification
**
**  Parameters:
**  	dkim -- DKIM handle
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

static DKIM_STAT
dkim_get_key(DKIM *dkim)
{
	int status;
	struct dkim_set *set;
	unsigned char *p;
	unsigned char buf[BUFRSZ + 1];

	assert(dkim != NULL);
	assert(dkim->dkim_selector != NULL);
	assert(dkim->dkim_domain != NULL);

	memset(buf, '\0', sizeof buf);

	/* if not, try a local function if there was one defined */
	if (dkim->dkim_libhandle->dkiml_key_lookup != NULL)
	{
		status = (int) dkim->dkim_libhandle->dkiml_key_lookup(dkim,
		                                                      buf,
		                                                      sizeof buf);
		if (status != (int) DKIM_STAT_OK)
			return (DKIM_STAT) status;
	}

	/* if not cached, make the query */
	if (buf[0] == '\0')
	{
		/* use appropriate get method */
		switch (dkim->dkim_querymethod)
		{
		  case DKIM_QUERY_DNS:
			status = (int) dkim_get_key_dns(dkim, buf, sizeof buf);
			if (status != (int) DKIM_STAT_OK)
				return (DKIM_STAT) status;
			break;

		  case DKIM_QUERY_FILE:
			status = (int) dkim_get_key_file(dkim, buf,
			                                 sizeof buf);
			if (status != (int) DKIM_STAT_OK)
				return (DKIM_STAT) status;
			break;

		  default:
			assert(0);
		}
	}

	/* decode the payload */
	status = dkim_process_set(dkim, DKIM_SETTYPE_KEY, buf, strlen(buf));
	if (status != DKIM_STAT_OK)
		return status;

	/* verify key version first */
	set = dkim_set_first(dkim, DKIM_SETTYPE_KEY);
	assert(set != NULL);
	p = dkim_param_get(set, "v");
	if (p != NULL && strcmp(p, DKIM_VERSION_KEY) != 0)
	{
		dkim_error(dkim, "invalid key version `%s'", p);
		return DKIM_STAT_SYNTAX;
	}

	/* then make sure the hash type is something we can handle */
	p = dkim_param_get(set, "h");
	if (!dkim_key_hashesok(p))
	{
		dkim_error(dkim, "signature-key hash mismatch");
		return DKIM_STAT_SYNTAX;
	}

	/* then key type */
	p = dkim_param_get(set, "k");
	if (p == NULL || dkim_name_to_code(keytypes, p) == -1)
	{
		dkim_error(dkim, "invalid key type `%s'", p);
		return DKIM_STAT_SYNTAX;
	}

	/* decode the key */
	dkim->dkim_b64key = dkim_param_get(set, "p");
	if (dkim->dkim_b64key == NULL)
	{
		dkim_error(dkim, "key missing");
		return DKIM_STAT_SYNTAX;
	}
	else if (dkim->dkim_b64key[0] == '\0')
	{
		return DKIM_STAT_REVOKED;
	}
	dkim->dkim_b64keylen = strlen(dkim->dkim_b64key);

	dkim->dkim_key = DKIM_MALLOC(dkim, dkim->dkim_b64keylen);
	if (dkim->dkim_key == NULL)
	{
		dkim_error(dkim, "unable to allocate %d byte(s)",
		           dkim->dkim_b64keylen);
		return DKIM_STAT_NORESOURCE;
	}

	status = dkim_base64_decode(dkim->dkim_b64key, dkim->dkim_key,
	                            dkim->dkim_b64keylen);
	if (status < 0)
	{
		dkim_error(dkim, "key missing");
		return DKIM_STAT_SYNTAX;
	}

	dkim->dkim_keylen = status;

	return DKIM_STAT_OK;
}

/*
**  DKIM_VERROR -- log an error into a DKIM handle
**                 (varargs version)
**
**  Parameters:
**  	dkim -- DKIM context in which this is performed
**  	format -- format to apply
**  	va -- argument list
**
**  Return value:
**  	None.
*/

static void
dkim_verror(DKIM *dkim, const char *format, va_list va)
{
	int flen;
	int saverr;
	char *new;

	assert(dkim != NULL);
	assert(format != NULL);

	saverr = errno;

	if (dkim->dkim_error == NULL)
	{
		dkim->dkim_error = DKIM_MALLOC(dkim, DEFERRLEN);
		if (dkim->dkim_error == NULL)
		{
			errno = saverr;
			return;
		}
		dkim->dkim_errlen = DEFERRLEN;
	}

	for (;;)
	{
		flen = vsnprintf(dkim->dkim_error, dkim->dkim_errlen, format,
		                 va);

		/* compensate for broken vsnprintf() implementations */
		if (flen == -1)
			flen = dkim->dkim_errlen * 2;

		if (flen >= dkim->dkim_errlen)
		{
			new = DKIM_MALLOC(dkim, flen + 1);
			if (new == NULL)
			{
				errno = saverr;
				return;
			}

			DKIM_FREE(dkim, dkim->dkim_error);
			dkim->dkim_error = new;
			dkim->dkim_errlen = flen + 1;
		}
		else
		{
			break;
		}
	}

	errno = saverr;
}

/*
**  DKIM_ERROR -- log an error into a DKIM handle
**
**  Parameters:
**  	dkim -- DKIM context in which this is performed
**  	format -- format to apply
**  	... -- arguments
**
**  Return value:
**  	None.
*/

void
dkim_error(DKIM *dkim, const char *format, ...)
{
	va_list va;

	assert(dkim != NULL);
	assert(format != NULL);

	va_start(va, format);
	dkim_verror(dkim, format, va);
	va_end(va);
}

/*
**  DKIM_NEW -- allocate a new message context
**
**  Parameters:
**  	libhandle -- DKIM_LIB handle
**  	id -- transaction ID string
**  	memclosure -- memory closure
**  	hdrcanon_alg -- canonicalization algorithm to use for headers
**  	bodycanon_alg -- canonicalization algorithm to use for headers
**  	sign_alg -- signature algorithm to use
**  	statp -- status (returned)
**
**  Return value:
**  	A new DKIM handle, or NULL on failure.
*/

static DKIM *
dkim_new(DKIM_LIB *libhandle, const char *id, void *memclosure,
         dkim_canon_t hdrcanon_alg, dkim_canon_t bodycanon_alg,
         dkim_alg_t sign_alg, DKIM_STAT *statp)
{
	DKIM *new;

	assert(libhandle != NULL);

	/* allocate the handle */
	new = (DKIM *) dkim_malloc(libhandle, memclosure,
	                           sizeof(struct dkim));
	if (new == NULL)
	{
		*statp = DKIM_STAT_NORESOURCE;
		return NULL;
	}

	/* populate defaults */
	memset(new, '\0', sizeof(struct dkim));
	new->dkim_id = id;
	new->dkim_signalg = (sign_alg == -1 ? DKIM_SIGN_DEFAULT
	                                    : sign_alg);
	new->dkim_hdrcanonalg = (hdrcanon_alg == -1 ? DKIM_CANON_DEFAULT
	                                            : hdrcanon_alg);
	new->dkim_bodycanonalg = (bodycanon_alg == -1 ? DKIM_CANON_DEFAULT
	                                              : bodycanon_alg);
	new->dkim_querymethod = DKIM_QUERY_DEFAULT;
	new->dkim_mode = DKIM_MODE_UNKNOWN;
	new->dkim_state = DKIM_STATE_INIT;
	new->dkim_closure = memclosure;
	new->dkim_libhandle = libhandle;
	new->dkim_tmpdir = libhandle->dkiml_tmpdir;
	new->dkim_timeout = libhandle->dkiml_timeout;
	new->dkim_hashtype = DKIM_HASHTYPE_UNKNOWN;
	new->dkim_version = DKIM_VERSION_UNKNOWN;
	new->dkim_blankline = TRUE;

	*statp = DKIM_STAT_OK;

#ifdef _FFR_QUERY_CACHE
	if ((libhandle->dkiml_flags & DKIM_LIBFLAGS_CACHE) != 0 &&
	    libhandle->dkiml_cache == NULL)
	{
		int err = 0;

		libhandle->dkiml_cache = dkim_cache_init(&err,
		                                         libhandle->dkiml_tmpdir);
	}
#endif /* _FFR_QUERY_CACHE */

	return new;
}

/* ========================= PUBLIC SECTION ========================== */

/*
**  DKIM_INIT -- initialize a DKIM library context
**
**  Parameters:
**  	caller_mallocf -- caller-provided memory allocation function
**  	caller_freef -- caller-provided memory release function
**
**  Return value:
**  	A new DKIM_LIB handle suitable for use with other DKIM functions, or
**  	NULL on failure.
**
**  Side effects:
**  	Crop circles near Birmingham.
*/

DKIM_LIB *
dkim_init(void *(*caller_mallocf)(void *closure, size_t nbytes),
          void (*caller_freef)(void *closure, void *p))
{
	u_char *td;
	DKIM_LIB *libhandle;

	/* initialize OpenSSL algorithms */
	OpenSSL_add_all_algorithms();

	/* copy the parameters */
	libhandle = (DKIM_LIB *) malloc(sizeof(struct dkim_lib));
	if (libhandle == NULL)
		return NULL;

	td = getenv("DKIM_TMPDIR");
	if (td == NULL || td[0] == '\0')
		td = DEFTMPDIR;

	libhandle->dkiml_signre = FALSE;
	libhandle->dkiml_skipre = FALSE;
	libhandle->dkiml_malloc = caller_mallocf;
	libhandle->dkiml_free = caller_freef;
	sm_strlcpy(libhandle->dkiml_tmpdir, td, 
	           sizeof libhandle->dkiml_tmpdir);
	libhandle->dkiml_flags = DKIM_LIBFLAGS_DEFAULT;
	libhandle->dkiml_timeout = DEFTIMEOUT;
	libhandle->dkiml_senderhdrs = (u_char **) default_senderhdrs;
	libhandle->dkiml_alwayshdrs = NULL;
	libhandle->dkiml_version = DKIM_VERSION_RFC4871;
	libhandle->dkiml_querymethod = DKIM_QUERY_UNKNOWN;
	memset(libhandle->dkiml_queryinfo, '\0',
	       sizeof libhandle->dkiml_queryinfo);
#ifdef _FFR_QUERY_CACHE
	libhandle->dkiml_cache = NULL;
#endif /* _FFR_QUERY_CACHE */
	libhandle->dkiml_fixedtime = 0;
	libhandle->dkiml_sigttl = 0;

	libhandle->dkiml_key_lookup = NULL;
	libhandle->dkiml_policy_lookup = NULL;
	libhandle->dkiml_sig_handle = NULL;
	libhandle->dkiml_sig_handle_free = NULL;
	libhandle->dkiml_sig_tagvalues = NULL;
	libhandle->dkiml_prescreen = NULL;
	libhandle->dkiml_dns_callback = NULL;

	/* initialize the resolver */
#if USE_ARLIB
	libhandle->dkiml_arlib = ar_init(NULL, NULL, NULL, 0);
	if (libhandle->dkiml_arlib == NULL)
	{
		free(libhandle);
		return NULL;
	}
#else /* USE_ARLIB */
	(void) res_init();
#endif /* USE_ARLIB */

	return libhandle;
}

/*
**  DKIM_CLOSE -- shut down a DKIM library package
**
**  Parameters:
**  	lib -- library handle to shut down
**
**  Return value:
**  	None.
*/

void
dkim_close(DKIM_LIB *lib)
{
	assert(lib != NULL);

#ifdef _FFR_QUERY_CACHE
	if (lib->dkiml_cache != NULL)
		(void) dkim_cache_close(lib->dkiml_cache);
#endif /* _FFR_QUERY_CACHE */

#ifdef USE_ARLIB
	if (lib->dkiml_arlib != NULL)
		(void) ar_shutdown(lib->dkiml_arlib);
#endif /* USE_ARLIB */

	free((void *) lib);

	EVP_cleanup();
}

/*
**  DKIM_OPTIONS -- get or set a library option
**
**  Parameters:
**  	lib -- DKIM library handle
**  	op -- operation to perform
**  	opt -- option to get/set
**  	ptr -- pointer to its old/new value
**  	len -- memory available at "ptr"
**
**  Return value:
**  	A DKIM_STAT constant.
**
**  XXX -- the various pieces of the switch section in here are all very
**         similar; some common-factoring is in order
*/

DKIM_STAT
dkim_options(DKIM_LIB *lib, int op, dkim_opts_t opt, void *ptr, size_t len)
{
	assert(lib != NULL);
	assert(op == DKIM_OP_SETOPT || op == DKIM_OP_GETOPT);
	assert(len != 0);

	switch (opt)
	{
	  case DKIM_OPTS_TMPDIR:
		if (ptr == NULL)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			sm_strlcpy((u_char *) ptr,
			           lib->dkiml_tmpdir, len);
		}
		else
		{
			sm_strlcpy(lib->dkiml_tmpdir, (u_char *) ptr,
			           sizeof lib->dkiml_tmpdir);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_FIXEDTIME:
		if (ptr == NULL)
			return DKIM_STAT_INVALID;

		if (len != sizeof lib->dkiml_fixedtime)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_fixedtime, len);
		}
		else
		{
			memcpy(&lib->dkiml_fixedtime, ptr, len);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_SIGNATURETTL:
		if (ptr == NULL)
			return DKIM_STAT_INVALID;

		if (len != sizeof lib->dkiml_sigttl)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_sigttl, len);
		}
		else
		{
			memcpy(&lib->dkiml_sigttl, ptr, len);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_FLAGS:
		if (ptr == NULL)
			return DKIM_STAT_INVALID;

		if (len != sizeof lib->dkiml_flags)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_flags, len);
		}
		else
		{
			memcpy(&lib->dkiml_flags, ptr, len);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_VERSION:
		if (ptr == NULL)
			return DKIM_STAT_INVALID;

		if (len != sizeof lib->dkiml_version)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_version, len);
		}
		else
		{
			u_int test;

			memcpy(&test, ptr, sizeof test);
			if (test != DKIM_VERSION_IETF_BASE_00 &&
			    test != DKIM_VERSION_IETF_BASE_01 &&
			    test != DKIM_VERSION_IETF_BASE_02 &&
			    test != DKIM_VERSION_IETF_BASE_10 &&
			    test != DKIM_VERSION_RFC4871)
				return DKIM_STAT_INVALID;

			memcpy(&lib->dkiml_version, ptr, len);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_TIMEOUT:
		if (ptr == NULL)
			return DKIM_STAT_INVALID;

		if (len != sizeof lib->dkiml_timeout)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_timeout, len);
		}
		else
		{
			memcpy(&lib->dkiml_timeout, ptr, len);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_SENDERHDRS:
		if (len != sizeof lib->dkiml_senderhdrs)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_senderhdrs, len);
		}
		else if (ptr == NULL)
		{
			lib->dkiml_senderhdrs = (u_char **) default_senderhdrs;
		}
		else
		{
			lib->dkiml_senderhdrs = (u_char **) ptr;
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_ALWAYSHDRS:
		if (len != sizeof lib->dkiml_alwayshdrs)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_alwayshdrs, len);
		}
		else if (ptr == NULL)
		{
			lib->dkiml_alwayshdrs = NULL;
		}
		else
		{
			lib->dkiml_alwayshdrs = (u_char **) ptr;
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_SIGNHDRS:
		if (len != sizeof(char **) || op == DKIM_OP_GETOPT)
		{
			return DKIM_STAT_INVALID;
		}
		else if (ptr == NULL)
		{
			if (lib->dkiml_signre)
			{
				(void) regfree(&lib->dkiml_hdrre);
				lib->dkiml_signre = FALSE;
			}
		}
		else
		{
			int status;
			u_char **hdrs;
			char buf[BUFRSZ + 1];

			if (lib->dkiml_signre)
			{
				(void) regfree(&lib->dkiml_hdrre);
				lib->dkiml_signre = FALSE;
			}

			memset(buf, '\0', sizeof buf);

			hdrs = (u_char **) ptr;

			(void) sm_strlcpy(buf, "^(", sizeof buf);

			if (!dkim_hdrlist(buf, sizeof buf,
			                  (u_char **) required_signhdrs, TRUE))
				return DKIM_STAT_INVALID;
			if (!dkim_hdrlist(buf, sizeof buf, hdrs, FALSE))
				return DKIM_STAT_INVALID;

			if (sm_strlcat(buf, ")$", sizeof buf) >= sizeof buf)
				return DKIM_STAT_INVALID;

			status = regcomp(&lib->dkiml_hdrre, buf,
			                 (REG_EXTENDED|REG_ICASE));
			if (status != 0)
				return DKIM_STAT_INTERNAL;

			lib->dkiml_signre = TRUE;
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_SKIPHDRS:
		if (len != sizeof(char **) || op == DKIM_OP_GETOPT)
		{
			return DKIM_STAT_INVALID;
		}
		else if (ptr == NULL)
		{
			if (lib->dkiml_skipre)
			{
				(void) regfree(&lib->dkiml_skiphdrre);
				lib->dkiml_skipre = FALSE;
			}
		}
		else
		{
			int status;
			u_char **hdrs;
			char buf[BUFRSZ + 1];

			if (lib->dkiml_skipre)
			{
				(void) regfree(&lib->dkiml_skiphdrre);
				lib->dkiml_skipre = FALSE;
			}

			memset(buf, '\0', sizeof buf);

			hdrs = (u_char **) ptr;

			(void) sm_strlcpy(buf, "^(", sizeof buf);

			if (!dkim_hdrlist(buf, sizeof buf, hdrs, TRUE))
				return DKIM_STAT_INVALID;

			if (sm_strlcat(buf, ")$", sizeof buf) >= sizeof buf)
				return DKIM_STAT_INVALID;

			status = regcomp(&lib->dkiml_skiphdrre, buf,
			                 (REG_EXTENDED|REG_ICASE));
			if (status != 0)
				return DKIM_STAT_INTERNAL;

			lib->dkiml_skipre = TRUE;
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_QUERYMETHOD:
		if (ptr == NULL)
			return DKIM_STAT_INVALID;

		if (len != sizeof lib->dkiml_querymethod)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_querymethod, len);
		}
		else
		{
			memcpy(&lib->dkiml_querymethod, ptr, len);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_QUERYINFO:
		if (ptr == NULL)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			sm_strlcpy(ptr, lib->dkiml_queryinfo, len);
		}
		else
		{
			sm_strlcpy(lib->dkiml_queryinfo, ptr,
			           sizeof lib->dkiml_queryinfo);
		}
		return DKIM_STAT_OK;

	  default:
		return DKIM_STAT_INVALID;
	}

	/* to silence -Wall */
	return DKIM_STAT_INTERNAL;
}

#define	CLOBBER(x)	if ((x) != NULL) \
			{ \
				dkim_mfree(dkim->dkim_libhandle, dkim->dkim_closure, (x)); \
				(x) = NULL; \
			}

#define BIO_CLOBBER(x)	if ((x) != NULL) \
			{ \
				BIO_free((x)); \
				(x) = NULL; \
			}

#define RSA_CLOBBER(x)	if ((x) != NULL) \
			{ \
				RSA_free((x)); \
				(x) = NULL; \
			}

#define	EVP_CLOBBER(x)	if ((x) != NULL) \
			{ \
				EVP_PKEY_free((x)); \
				(x) = NULL; \
			}

/*
**  DKIM_FREE -- destroy a DKIM handle
**
**  Parameters:
**  	dkim -- DKIM handle to destroy
**
**  Return value:
**  	A DKIM_STAT constant.
*/

DKIM_STAT
dkim_free(DKIM *dkim)
{
	assert(dkim != NULL);

	if (dkim->dkim_hash != NULL)
	{
		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;

			BIO_CLOBBER(sha1->sha1_tmpbio);

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;

			BIO_CLOBBER(sha256->sha256_tmpbio);

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */

		  case DKIM_HASHTYPE_UNKNOWN:
			break;

		  default:
			assert(0);
		}

		CLOBBER(dkim->dkim_hash);
	}

	if (dkim->dkim_bhash != NULL)
	{
		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;

			BIO_CLOBBER(sha1->sha1_tmpbio);

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;

			BIO_CLOBBER(sha256->sha256_tmpbio);

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */

		  case DKIM_HASHTYPE_UNKNOWN:
			break;

		  default:
			assert(0);
		}

		CLOBBER(dkim->dkim_bhash);
	}

	if (dkim->dkim_signature != NULL)
	{
		switch (dkim->dkim_signalg)
		{
		  case DKIM_SIGN_RSASHA1:
#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_SIGN_RSASHA256:
#endif /* SHA256_DIGEST_LENGTH */
		  {
			struct dkim_rsa *rsa;

			rsa = (struct dkim_rsa *) dkim->dkim_signature;

			EVP_CLOBBER(rsa->rsa_pkey);
			RSA_CLOBBER(rsa->rsa_rsa);
			CLOBBER(rsa->rsa_rsaout);

			break;
		  }

		  default:
			assert(0);
		}

		CLOBBER(dkim->dkim_signature);
	}

	/* blast the headers */
	if (dkim->dkim_hhead != NULL)
	{
		struct dkim_header *next;
		struct dkim_header *hdr;

		for (hdr = dkim->dkim_hhead; hdr != NULL; )
		{
			next = hdr->hdr_next;

			CLOBBER(hdr->hdr_text);
			CLOBBER(hdr);

			hdr = next;
		}
	}

	/* blast the data sets */
	if (dkim->dkim_sethead != NULL)
	{
		DKIM_SET *set;
		DKIM_SET *next;

		for (set = dkim->dkim_sethead; set != NULL; )
		{
			next = set->set_next;

			if (set->set_plist != NULL)
			{
				DKIM_PLIST *plist;
				DKIM_PLIST *pnext;

				for (plist = set->set_plist;
				     plist != NULL; )
				{
					pnext = plist->plist_next;

					CLOBBER(plist);

					plist = pnext;
				}
			}

			CLOBBER(set->set_data);
			CLOBBER(set);

			set = next;
		}
	}

	if (dkim->dkim_siglist != NULL)
	{
		int c;

		for (c = 0; c < dkim->dkim_sigcount; c++)
		{
			if (dkim->dkim_siglist[c]->sig_context != NULL &&
			    dkim->dkim_libhandle->dkiml_sig_handle_free != NULL)
			{
				dkim->dkim_libhandle->dkiml_sig_handle_free(dkim->dkim_closure,
				                                            dkim->dkim_siglist[c]->sig_context);
			}

			CLOBBER(dkim->dkim_siglist[c]);
		}

		CLOBBER(dkim->dkim_siglist);
	}

	CLOBBER(dkim->dkim_b64sig);
	CLOBBER(dkim->dkim_hdrlist);
	CLOBBER(dkim->dkim_selector);
	CLOBBER(dkim->dkim_domain);
	CLOBBER(dkim->dkim_user);
	CLOBBER(dkim->dkim_key);
	CLOBBER(dkim->dkim_sender);
	CLOBBER(dkim->dkim_signer);
	CLOBBER(dkim->dkim_sig);
	CLOBBER(dkim->dkim_error);
	CLOBBER(dkim->dkim_zdecode);
#ifdef _FFR_HASH_BUFFERING
	CLOBBER(dkim->dkim_hashbuf);
#endif /* _FFR_HASH_BUFFERING */

	dkim_mfree(dkim->dkim_libhandle, dkim->dkim_closure, dkim);

	return DKIM_STAT_OK;
}

/*
**  DKIM_SIGN -- allocate a handle for use in a signature operation
**
**  Parameters:
**  	libhandle -- DKIM_LIB handle
**  	id -- identification string (e.g. job ID) for logging
**  	memclosure -- memory closure for allocations (or NULL)
**  	secretkey -- secret key (PEM format)
**  	selector -- selector to be used when generating the signature header
**  	domain -- domain for which this message is being signed
**  	hdrcanonalg -- canonicalization algorithm to use for headers
**  	bodycanonalg -- canonicalization algorithm to use for body
**  	signalg -- signing algorithm to use
**  	length -- how many bytes of the body to sign (-1 for all)
**  	statp -- status (returned)
**
**  Return value:
**  	A new signing handle, or NULL.
*/

DKIM *
dkim_sign(DKIM_LIB *libhandle, const char *id, void *memclosure,
          const dkim_sigkey_t secretkey, const char *selector,
          const char *domain, dkim_canon_t hdrcanonalg,
	  dkim_canon_t bodycanonalg, dkim_alg_t signalg,
          off_t length, DKIM_STAT *statp)
{
	DKIM *new;

	assert(libhandle != NULL);
	assert(secretkey != NULL);
	assert(selector != NULL);
	assert(domain != NULL);
	assert(hdrcanonalg == DKIM_CANON_SIMPLE ||
	       hdrcanonalg == DKIM_CANON_RELAXED);
	assert(bodycanonalg == DKIM_CANON_SIMPLE ||
	       bodycanonalg == DKIM_CANON_RELAXED);
#ifdef SHA256_DIGEST_LENGTH
	assert(signalg == DKIM_SIGN_RSASHA1 || signalg == DKIM_SIGN_RSASHA256);
#else /* SHA256_DIGEST_LENGTH */
	assert(signalg == DKIM_SIGN_RSASHA1);
#endif /* SHA256_DIGEST_LENGTH */
	assert(statp != NULL);

	new = dkim_new(libhandle, id, memclosure, hdrcanonalg, bodycanonalg,
	               signalg, statp);

	if (new != NULL)
	{
		new->dkim_mode = DKIM_MODE_SIGN;

		new->dkim_keylen = strlen((const char *) secretkey);
		new->dkim_key = (unsigned char *) DKIM_MALLOC(new,
		                                              new->dkim_keylen + 1);

		if (new->dkim_key == NULL)
		{
			*statp = DKIM_STAT_NORESOURCE;
			dkim_free(new);
			return NULL;
		}

		memcpy(new->dkim_key, (char *) secretkey,
		       new->dkim_keylen + 1);
	}

	new->dkim_selector = dkim_strdup(new, selector, 0);
	new->dkim_domain = dkim_strdup(new, domain, 0);
	if (length == (off_t) -1)
		new->dkim_signlen = ULONG_MAX;
	else
		new->dkim_signlen = length;

	return new;
}

/*
**  DKIM_VERIFY -- allocate a handle for use in a verify operation
**
**  Parameters:
**  	libhandle -- DKIM_LIB handle
**  	id -- identification string (e.g. job ID) for logging
**  	memclosure -- memory closure for allocations (or NULL)
**  	statp -- status (returned)
**
**  Return value:
**  	A new signing handle, or NULL.
*/

DKIM *
dkim_verify(DKIM_LIB *libhandle, const char *id, void *memclosure,
            DKIM_STAT *statp)
{
	DKIM *new;

	assert(libhandle != NULL);
	assert(statp != NULL);

	new = dkim_new(libhandle, id, memclosure, DKIM_CANON_UNKNOWN,
	               DKIM_CANON_UNKNOWN, DKIM_SIGN_UNKNOWN, statp);

	if (new != NULL)
		new->dkim_mode = DKIM_MODE_VERIFY;

	return new;
}

/*
**  DKIM_HEADER -- process a header
**
**  Parameters:
**  	dkim -- DKIM handle
**  	hdr -- header text
**  	len -- bytes available at "hdr"
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_header(DKIM *dkim, u_char *hdr, size_t len)
{
	size_t nlen;
	u_char *colon;
	struct dkim_header *h;

	assert(dkim != NULL);
	assert(hdr != NULL);
	assert(len != 0);

	if (dkim->dkim_state > DKIM_STATE_HEADER)
		return DKIM_STAT_INVALID;
	dkim->dkim_state = DKIM_STATE_HEADER;

	/* see if this is one we should skip */
	if (dkim->dkim_libhandle->dkiml_skipre)
	{
		int status;
		char name[MAXHEADER + 1];

		sm_strlcpy(name, hdr, sizeof name);
		colon = strchr(name, ':');
		if (colon != NULL)
			*colon = '\0';

		status = regexec(&dkim->dkim_libhandle->dkiml_skiphdrre,
		                 name, 0, NULL, 0);

		if (status == 0)
			return DKIM_STAT_OK;
		else
			assert(status == REG_NOMATCH);
	}

	h = DKIM_MALLOC(dkim, sizeof(struct dkim_header));

	if (h == NULL)
	{
		dkim_error(dkim, "unable to allocate %d byte(s)",
		           sizeof(struct dkim_header));
		return DKIM_STAT_NORESOURCE;
	}

	h->hdr_text = dkim_strdup(dkim, hdr, len);
	h->hdr_flags = 0;
	h->hdr_next = NULL;

	if (dkim->dkim_hhead == NULL)
	{
		dkim->dkim_hhead = h;
		dkim->dkim_htail = h;
	}
	else
	{
		dkim->dkim_htail->hdr_next = h;
		dkim->dkim_htail = h;
	}

	colon = strchr(hdr, ':');
	if (colon != NULL)
	{
		nlen = colon - hdr;
		if (strncasecmp(hdr, DKIM_SIGNHEADER, nlen) == 0 &&
		    hdr[nlen] == ':')
		{
			DKIM_STAT status;

			status = dkim_process_set(dkim, DKIM_SETTYPE_SIGNATURE,
			                          colon + 1, len - nlen);

			if (status != DKIM_STAT_OK)
				return status;
		}

#ifdef _FFR_VBR
		if (strncasecmp(hdr, VBR_INFOHEADER, nlen) == 0)
		{
			DKIM_STAT status;

			status = dkim_process_set(dkim, DKIM_SETTYPE_VBRINFO,
			                          colon + 1, len - nlen);

			if (status != DKIM_STAT_OK)
				return status;
		}
#endif /* _FFR_VBR */
	}

	return DKIM_STAT_OK;
}

/*
**  DKIM_EOH -- declare end-of-headers
** 
**  Parameters:
**  	dkim -- DKIM handle
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_eoh(DKIM *dkim)
{
	bool found;
	DKIM_STAT status;
	int htype = DKIM_HASHTYPE_UNKNOWN;
	int nhdrs;
	int c;
	int d;
	int len;
	struct dkim_header *hdr;
	DKIM_LIB *lib;
	DKIM_SET *set;
	u_char *hdrlist;
	struct dkim_header *hdrset[MAXHDRCNT];

	assert(dkim != NULL);

	if (dkim->dkim_state >= DKIM_STATE_EOH)
		return DKIM_STAT_INVALID;
	dkim->dkim_state = DKIM_STATE_EOH;

	lib = dkim->dkim_libhandle;
	assert(lib != NULL);

	/* select headers */
	memset(hdrset, '\0', sizeof hdrset);
	if (dkim->dkim_mode == DKIM_MODE_VERIFY)
	{
		u_char *param;
		u_char *bh;

		/* count the signatures */
		for (set = dkim_set_first(dkim, DKIM_SETTYPE_SIGNATURE);
		     set != NULL;
		     set = dkim_set_next(set, DKIM_SETTYPE_SIGNATURE))
			/* XXX -- only count "good" ones */
			dkim->dkim_sigcount++;

		/* if no signatures, return such */
		if (dkim->dkim_sigcount == 0)
		{
			dkim->dkim_skipbody = TRUE;
			return DKIM_STAT_NOSIG;
		}

		/* allocate the array */
		len = dkim->dkim_sigcount * sizeof(DKIM_SIGINFO *);
		dkim->dkim_siglist = DKIM_MALLOC(dkim, len);
		if (dkim->dkim_siglist == NULL)
		{
			dkim_error(dkim, "unable to allocate %d byte(s)", len);
			return DKIM_STAT_NORESOURCE;
		}

		/* allocate the elements */
		for (c = 0; c < dkim->dkim_sigcount; c++)
		{
			dkim->dkim_siglist[c] = DKIM_MALLOC(dkim,
			                                    sizeof(DKIM_SIGINFO));
			if (dkim->dkim_siglist[c] == NULL)
			{
				int n;
	
				dkim_error(dkim,
				           "unable to allocate %d byte(s)",
				           sizeof(DKIM_SIGINFO));
				for (n = 0; n < c; n++)
					DKIM_FREE(dkim, dkim->dkim_siglist[n]);
				return DKIM_STAT_NORESOURCE;
			}
		}

		/* generate the signature handles */
		for (c = 0; c < dkim->dkim_sigcount; c++)
		{
			if (c == 0)
			{
				set = dkim_set_first(dkim,
				                     DKIM_SETTYPE_SIGNATURE);
			}
			else
			{
				set = dkim_set_next(set,
				                    DKIM_SETTYPE_SIGNATURE);
			}

			dkim->dkim_siglist[c]->sig_bh = DKIM_SIGBH_UNTESTED;
			dkim->dkim_siglist[c]->sig_index = c;
			dkim->dkim_siglist[c]->sig_flags = 0;
			dkim->dkim_siglist[c]->sig_taglist = set;

			/* allow the user to generate its handle */
			if (lib->dkiml_sig_handle != NULL)
				dkim->dkim_siglist[c]->sig_context = lib->dkiml_sig_handle(dkim->dkim_closure);

			/* populate the user handle */
			if (lib->dkiml_sig_tagvalues != NULL)
			{
				dkim_param_t pcode;
				struct dkim_plist *plist;
				void *user;

				user = dkim->dkim_siglist[c]->sig_context;

				for (plist = set->set_plist;
				     plist != NULL;
				     plist = plist->plist_next)
				{
					pcode = dkim_name_to_code(sigparams,
					                          plist->plist_param);

					(void) lib->dkiml_sig_tagvalues(user,
					                                pcode,
					                                plist->plist_param,
					                                plist->plist_value);
				}
			}
		}

		/* call the prescreen callback, if defined */
		if (lib->dkiml_prescreen != NULL)
		{
			status = lib->dkiml_prescreen(dkim->dkim_siglist,
			                              dkim->dkim_sigcount);
			switch (status)
			{
			  case DKIM_CBSTAT_CONTINUE:
				break;

			  case DKIM_CBSTAT_REJECT:
				return DKIM_STAT_CBREJECT;

			  default:
				return DKIM_STAT_CBINVALID;
			}
		}

		/* pick first signature that looks useable */
		for (c = 0; c < dkim->dkim_sigcount; c++)
		{
			set = dkim->dkim_siglist[c]->sig_taglist;

			/* skip if marked to ignore */
			if (dkim->dkim_siglist[c]->sig_flags & DKIM_SIGFLAG_IGNORE)
			{
				set = NULL;
				continue;
			}

			/* check for signed headers */
			hdrlist = dkim_param_get(set, "h");
			nhdrs = dkim_selecthdrs(dkim, hdrlist, hdrset,
			                        MAXHDRCNT);
			if (nhdrs == -1)
				return DKIM_STAT_INTERNAL;

			/* simple syntax/validity checking */
			if (dkim_sig_versionok(set) &&
			    dkim_sig_domainok(dkim, set) &&
			    !dkim_sig_expired(set) &&
			    (dkim->dkim_version < DKIM_VERSION_IETF_BASE_10 ||
			     (dkim_sig_timestampsok(set) &&
			      !dkim_sig_future(set))))
			{
				dkim->dkim_signum = dkim->dkim_siglist[c]->sig_index;
				break;
			}

			set = NULL;
		}

		if (set == NULL)
		{
			dkim->dkim_skipbody = TRUE;
			return DKIM_STAT_NOSIG;
		}

		param = dkim_param_get(set, "l");
		if (param == NULL)
		{
			dkim->dkim_signlen = ULONG_MAX;
		}
		else
		{
			char *q;

			dkim->dkim_signlen = strtoul(param, &q, 10);
			if (*q != '\0')
			{
				dkim_error(dkim, "syntax error in `l' value");
				return DKIM_STAT_SYNTAX;
			}
		}

		param = dkim_param_get(set, "c");
		if (param != NULL)
		{
			u_char *p;
			u_char tmp[BUFRSZ + 1];

			sm_strlcpy(tmp, param, sizeof tmp);
			p = strchr(tmp, '/');
			if (p != NULL)
				*p = '\0';

			dkim->dkim_hdrcanonalg = dkim_name_to_code(canonicalizations,
			                                           tmp);
			if (dkim->dkim_hdrcanonalg == -1)
			{
				dkim_error(dkim,
				           "unknown canonicalization algorithm `%s'",
				           tmp);
				return DKIM_STAT_SYNTAX;
			}

			if (p == NULL)
			{
				dkim->dkim_bodycanonalg = DKIM_CANON_SIMPLE;
			}
			else
			{
				dkim->dkim_bodycanonalg = dkim_name_to_code(canonicalizations,
				                                            p + 1);
				if (dkim->dkim_bodycanonalg == -1)
				{
					dkim_error(dkim,
					           "unknown canonicalization algorithm `%s'",
					           p + 1);
					return DKIM_STAT_SYNTAX;
				}
			}
		}

		param = dkim_param_get(set, "a");
		if (param != NULL)
		{
			dkim->dkim_signalg = dkim_name_to_code(algorithms,
			                                       param);
			if (dkim->dkim_signalg == -1)
			{
				dkim_error(dkim,
				           "unknown signing algorithm `%s'",
				           param);
				return DKIM_STAT_SYNTAX;
			}
		}

		param = dkim_param_get(set, "q");
		if (param != NULL)
		{
			u_char *opts;
			u_char tmp[BUFRSZ + 1];

			/*
			**  XXX -- this should eventually be processed as a
			**         colon-separated list
			*/

			opts = strchr(param, '/');
			if (opts != NULL)
			{
				sm_strlcpy(tmp, param, sizeof tmp);
				opts = strchr(tmp, '/');
				*opts = '\0';
				opts++;
				param = tmp;
			}

			dkim->dkim_querymethod = dkim_name_to_code(querytypes,
			                                           param);
			if (dkim->dkim_querymethod == -1)
			{
				dkim_error(dkim,
				           "unknown key query method `%s'",
				           param);
				return DKIM_STAT_SYNTAX;
			}

			if (dkim->dkim_querymethod == DKIM_QUERY_DNS)
			{
				/* "txt" option required (also default) */
				if (opts != NULL &&
				    strcmp(opts, "txt") != 0)
				{
					dkim_error(dkim,
					           "unknown key query option `%s'",
					           opts);
					return DKIM_STAT_SYNTAX;
				}
			}
		}

		if (lib->dkiml_querymethod != DKIM_QUERY_UNKNOWN)
			dkim->dkim_querymethod = lib->dkiml_querymethod;

		param = dkim_param_get(set, "d");
		if (param == NULL)
		{
			dkim_error(dkim,
			           "signature missing required `d' parameter");
			return DKIM_STAT_SYNTAX;
		}
		else if (param[0] == '\0')
		{
			dkim_error(dkim,
			           "signature contains empty `d' parameter");
			return DKIM_STAT_SYNTAX;
		}
		dkim->dkim_domain = dkim_strdup(dkim, param, 0);
		if (dkim->dkim_domain == NULL)
			return DKIM_STAT_NORESOURCE;

		param = dkim_param_get(set, "s");
		if (param == NULL)
		{
			dkim_error(dkim,
			           "signature missing required `s' parameter");
			return DKIM_STAT_SYNTAX;
		}
		else if (param[0] == '\0')
		{
			dkim_error(dkim,
			           "signature contains empty `s' parameter");
			return DKIM_STAT_SYNTAX;
		}
		dkim->dkim_selector = dkim_strdup(dkim, param, 0);
		if (dkim->dkim_selector == NULL)
			return DKIM_STAT_NORESOURCE;

		param = dkim_param_get(set, "b");
		if (param == NULL)
		{
			dkim_error(dkim,
			           "signature missing required `b' parameter");
			return DKIM_STAT_SYNTAX;
		}
		else if (param[0] == '\0')
		{
			dkim_error(dkim,
			           "signature contains empty `b' parameter");
			return DKIM_STAT_SYNTAX;
		}
		dkim->dkim_b64sig = dkim_strdup(dkim, param, 0);
		if (dkim->dkim_b64sig == NULL)
			return DKIM_STAT_NORESOURCE;
		dkim->dkim_b64siglen = strlen(dkim->dkim_b64sig);

		dkim->dkim_sig = DKIM_MALLOC(dkim, dkim->dkim_b64siglen);
		if (dkim->dkim_sig == NULL)
		{
			dkim_error(dkim, "unable to allocate %d byte(s)",
			           dkim->dkim_b64siglen);
			return DKIM_STAT_NORESOURCE;
		}

		status = dkim_base64_decode(dkim->dkim_b64sig,
		                            dkim->dkim_sig,
		                            dkim->dkim_b64siglen);
		if (status < 0)
		{
			dkim_error(dkim, "corrupt base64 payload");
			return DKIM_STAT_INTERNAL;
		}
		else
		{
			dkim->dkim_siglen = status;
		}

		/* this is the one we will use */
		dkim->dkim_sigset = set;

		/* determine version */
		bh = dkim_param_get(set, "bh");
		if (dkim->dkim_version == DKIM_VERSION_UNKNOWN)
		{
			param = dkim_param_get(set, "v");
			if (param != NULL)
			{
				if (strcmp(param, "0.2") == 0)
					dkim->dkim_version = DKIM_VERSION_IETF_BASE_02;
				else if (strcmp(param, "0.5") == 0)
					dkim->dkim_version = DKIM_VERSION_IETF_BASE_10;
				else if (strcmp(param, "1") == 0)
					dkim->dkim_version = DKIM_VERSION_RFC4871;
			}
			else
			{
				/* missing v= with bh= implies ietf-base-01 */
				if (bh != NULL)
					dkim->dkim_version = DKIM_VERSION_IETF_BASE_01;
				else
					dkim->dkim_version = DKIM_VERSION_IETF_BASE_00;
			}
		}

		if (dkim->dkim_version == DKIM_VERSION_UNKNOWN)
			dkim->dkim_version = DKIM_VERSION_DEFAULT;
	}
	else
	{
		u_int signcnt = 0;
		size_t len = 0;
		u_char *colon;
		u_char *p;
		u_char *q;

		dkim->dkim_version = lib->dkiml_version;

		/* tag headers to be signed */
		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			if (!lib->dkiml_signre)
			{
				hdr->hdr_flags |= DKIM_HDR_SIGNED;
				signcnt++;
				continue;
			}

			colon = strchr(hdr->hdr_text, ':');

			if (colon != NULL)
				*colon = '\0';
			status = regexec(&lib->dkiml_hdrre,
			                 hdr->hdr_text, 0, NULL, 0);
			if (colon != NULL)
				*colon = ':';

			if (status == 0)
			{
				hdr->hdr_flags |= DKIM_HDR_SIGNED;
				signcnt++;
			}
			else
			{
				assert(status == REG_NOMATCH);
			}
		}

		/* figure out how much space we need */
		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			if (signcnt > 0 &&
			    (hdr->hdr_flags & DKIM_HDR_SIGNED) == 0)
				continue;

			colon = strchr(hdr->hdr_text, ':');

			if (len > 0)
				len++;				/* colon */

			if (colon == NULL)
			{
				len += strlen(hdr->hdr_text);
			}
			else
			{
				len += colon - hdr->hdr_text;
			}
		}

		/* now build a header list */
		dkim->dkim_hdrlist = DKIM_MALLOC(dkim, len + 1);
		if (dkim->dkim_hdrlist == NULL)
		{
			dkim_error(dkim, "unable to allocate %d byte(s)",
			           len + 1);
			return DKIM_STAT_INTERNAL;
		}
		memset(dkim->dkim_hdrlist, '\0', len + 1);
		hdrlist = dkim->dkim_hdrlist;
		q = hdrlist;

		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			if (signcnt > 0 &&
			    (hdr->hdr_flags & DKIM_HDR_SIGNED) == 0)
				continue;

			if (q != hdrlist)
			{
				*q = ':';
				q++;
			}

			for (p = hdr->hdr_text; *p != '\0' && *p != ':'; p++)
			{
				*q = *p;
				q++;
			}
		}

		/* and finally, select the headers based on the DKIM spec */
		nhdrs = dkim_selecthdrs(dkim, hdrlist, hdrset, MAXHDRCNT);
		if (nhdrs == -1)
			return DKIM_STAT_INTERNAL;

		if (nhdrs >= MAXHDRCNT)
		{
			dkim_error(dkim, "too many headers (max %d)",
			           MAXHDRCNT);
			return DKIM_STAT_INTERNAL;
		}
	}

	/*
	**  Verify that all the required headers are present and marked for
	**  signing/verification.
	*/

	for (c = 0; required_signhdrs[c] != NULL; c++)
	{
		found = FALSE;
		len = strlen(required_signhdrs[c]);

		for (d = 0; d < MAXHDRCNT && hdrset[d] != NULL; d++)
		{
			if (strncasecmp(hdrset[d]->hdr_text,
			                required_signhdrs[c], len) == 0 &&
			    hdrset[d]->hdr_text[len] == ':')
			{
				found = TRUE;
				break;
			}
		}

		if (!found)
		{
			dkim_error(dkim, "required header \"%s\" not found",
			           required_signhdrs[c]);
			return DKIM_STAT_SYNTAX;
		}
	}

	/* start up canonicalization */
	switch (dkim->dkim_signalg)
	{
	  case DKIM_SIGN_RSASHA1:
		htype = DKIM_HASHTYPE_SHA1;
		break;

#ifdef SHA256_DIGEST_LENGTH
	  case DKIM_SIGN_RSASHA256:
		htype = DKIM_HASHTYPE_SHA256;
		break;
#endif /* SHA256_DIGEST_LENGTH */
	}

	status = dkim_canoninit(dkim, htype);
	if (status != DKIM_STAT_OK)
		return status;

	/* canonicalize headers */
	for (c = 0; c < nhdrs; c++)
	{
		status = dkim_canonheader(dkim, hdrset[c], TRUE);
		if (status != DKIM_STAT_OK)
			return status;
	}
		
	dkim->dkim_writesep = TRUE;

	return DKIM_STAT_OK;
}

/*
**  DKIM_BODY -- pass a body chunk in for processing
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- body chunk
**  	buflen -- number of bytes at "buf"
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_body(DKIM *dkim, u_char *buf, size_t buflen)
{
	size_t wlen;
	size_t lidx;
	u_char *p;
	u_char *eob;
	u_char *wrote;
	unsigned char lbuf[BUFRSZ];

	assert(dkim != NULL);
	assert(buf != NULL);

	if (dkim->dkim_state > DKIM_STATE_BODY)
		return DKIM_STAT_INVALID;
	dkim->dkim_state = DKIM_STATE_BODY;

	if (dkim->dkim_skipbody)
		return DKIM_STAT_OK;

	wrote = buf;
	wlen = 0;
	lidx = 0;
	eob = buf + buflen - 1;

	/* canonicalize */
	switch (dkim->dkim_bodycanonalg)
	{
	  case DKIM_CANON_SIMPLE:
	  case DKIM_CANON_RELAXED:
		for (p = buf; p <= eob; p++)
		{
			if (*p == '\n')
			{
				if (dkim->dkim_lastchar == '\r')
				{
					if (dkim->dkim_blankline)
					{
						dkim->dkim_blanks++;
					}
					else if (wlen == 1)
					{
#ifdef _FFR_HASH_BUFFERING
						dkim_canonbuffer(dkim, CRLF, 2);
#else /* _FFR_HASH_BUFFERING */
						dkim_canonwrite(dkim, CRLF, 2);
#endif /* _FFR_HASH_BUFFERING */
					}
					else
					{
#ifdef _FFR_HASH_BUFFERING
						dkim_canonbuffer(dkim, wrote,
						                 wlen + 1);
#else /* _FFR_HASH_BUFFERING */
						dkim_canonwrite(dkim, wrote,
						                wlen + 1);
#endif /* _FFR_HASH_BUFFERING */
					}

					wrote = p + 1;
					wlen = 0;
					dkim->dkim_blankline = TRUE;
				}
			}
			else if (dkim->dkim_bodycanonalg == DKIM_CANON_RELAXED)
			{
				/* non-space after a space */
				if (dkim_islwsp(dkim->dkim_lastchar) &&
				    !dkim_islwsp(*p) && *p != '\r')
				{
					if (dkim->dkim_blanks > 0)
						dkim_flush_blanks(dkim);
#ifdef _FFR_HASH_BUFFERING
					dkim_canonbuffer(dkim, SP, 1);
#else /* _FFR_HASH_BUFFERING */
					dkim_canonwrite(dkim, SP, 1);
#endif /* _FFR_HASH_BUFFERING */
					wrote = p;
					wlen = 1;
					dkim->dkim_blankline = FALSE;
				}

				/* space after a non-space */
				else if (!dkim_islwsp(dkim->dkim_lastchar) &&
				         dkim_islwsp(*p))
				{
#ifdef _FFR_HASH_BUFFERING
					dkim_canonbuffer(dkim, wrote, wlen);
#else /* _FFR_HASH_BUFFERING */
					dkim_canonwrite(dkim, wrote, wlen);
#endif /* _FFR_HASH_BUFFERING */
					wlen = 0;
					wrote = p + 1;
				}

				/* space after a space */
				else if (dkim_islwsp(dkim->dkim_lastchar) &&
				         dkim_islwsp(*p))
				{
					/* XXX -- fix logic */
					;
				}

				/* non-space after a non-space */
				else
				{
					wlen++;
					if (dkim->dkim_blankline &&
					    (wlen > 1 || *p != '\r'))
					{
						if (dkim->dkim_blanks > 0)
							dkim_flush_blanks(dkim);
						dkim->dkim_blankline = FALSE;
					}
				}
			}
			else
			{
				if (*p != '\r')
				{
					if (dkim->dkim_blanks > 0)
						dkim_flush_blanks(dkim);
					dkim->dkim_blankline = FALSE;
				}
				wlen++;
			}

			dkim->dkim_lastchar = *p;
		}

#ifdef _FFR_HASH_BUFFERING
		dkim_canonbuffer(dkim, wrote, wlen);
#else /* _FFR_HASH_BUFFERING */
		dkim_canonwrite(dkim, wrote, wlen);
#endif /* _FFR_HASH_BUFFERING */

		break;

	  default:
		assert(0);
	}

	return DKIM_STAT_OK;
}

/*
**  DKIM_EOM -- declare end-of-body; conduct verification or prepare to sign
**
**  Parameters:
**  	dkim -- DKIM handle
**  	testkey -- TRUE iff the a matching key was found but is marked as a
**  	           test key (returned)
**
**  Return value:
**  	A DKIM_STAT_* constant.
**
**  XXX -- consider removing "testkey", and having the caller use
**         dkim_policy() retrieve flags like that one instead
*/

DKIM_STAT
dkim_eom(DKIM *dkim, bool *testkey)
{
	DKIM_STAT ret;

	assert(dkim != NULL);

	if (dkim->dkim_state >= DKIM_STATE_EOM)
		return DKIM_STAT_INVALID;
	dkim->dkim_state = DKIM_STATE_EOM;

	if (dkim->dkim_mode == DKIM_MODE_SIGN)
	{
		size_t n;
		DKIM_STAT status;
		u_char tmp[MAXHEADER + 1];
		struct dkim_header hdr;

		if (dkim->dkim_libhandle->dkiml_fixedtime != 0)
			dkim->dkim_timestamp = dkim->dkim_libhandle->dkiml_fixedtime;
		else
			(void) time(&dkim->dkim_timestamp);

		n = sm_strlcpy(tmp, DKIM_SIGNHEADER ": ", sizeof tmp);

		dkim_canonfinal(dkim, TRUE);
		dkim->dkim_bodydone = TRUE;

		/* sign with l= if requested */
		if ((dkim->dkim_libhandle->dkiml_flags & DKIM_LIBFLAGS_SIGNLEN) != 0)
			dkim->dkim_partial = TRUE;

		status = dkim_getsighdr(dkim, tmp + n, sizeof tmp - n,
		                        DKIM_HDRMARGIN,
		                        strlen(DKIM_SIGNHEADER) + 2);
		if (status != DKIM_STAT_OK)
			return status;

		if (strlen(tmp) == MAXHEADER)
		{
			dkim_error(dkim,
			           "generated signature header too large");
			return DKIM_STAT_NORESOURCE;
		}

		hdr.hdr_text = tmp;
		hdr.hdr_flags = 0;
		hdr.hdr_next = NULL;

		/* canonicalize */
		if (dkim->dkim_version < DKIM_VERSION_IETF_BASE_01)
			dkim_canonwrite(dkim, CRLF, 2);
		dkim_canonheader(dkim, &hdr, FALSE);
	}
	else
	{
		int n;
		u_char last = '\0';
		u_char in = '\0';
		u_char *p;
		u_char *start;
		struct dkim_header *hdr;
		struct dkim_header tmphdr;
		u_char tmp[MAXHEADER + 1];

		hdr = dkim_get_header(dkim, DKIM_SIGNHEADER,
		                      dkim->dkim_signum);
		if (hdr == NULL)
		{
			if (dkim->dkim_domain == NULL)
			{
				int status;
				char *user;
				char *domain;
				char *colon;

				hdr = dkim_get_header(dkim,
				                      DKIM_FROMHEADER, 0);
				if (hdr == NULL)
				{
					dkim_error(dkim, "no %s header found",
					           DKIM_FROMHEADER);
					return DKIM_STAT_CANTVRFY;
				}

				colon = strchr(hdr->hdr_text, ':');
				if (colon == NULL)
				{
					dkim_error(dkim, "%s header malformed",
					           DKIM_FROMHEADER);
					return DKIM_STAT_CANTVRFY;
				}

				status = rfc2822_mailbox_split(colon + 1,
				                               &user,
				                               &domain);
				if (status != 0)
				{
					dkim_error(dkim, "%s header malformed",
					           DKIM_FROMHEADER);
					return DKIM_STAT_CANTVRFY;
				}

				dkim->dkim_domain = dkim_strdup(dkim,
				                                domain, 0);
				if (dkim->dkim_domain == NULL)
					return DKIM_STAT_NORESOURCE;
			}

			/* try to retrieve policy */
			if (dkim_get_policy(dkim) == -1)
			{
				dkim_error(dkim,
				           "can't retrieve policy record");
				return DKIM_STAT_CANTVRFY;
			}

			return DKIM_STAT_NOSIG;
		}

		/*
		**  We need to copy the DKIM-Signature: header being verified,
		**  minus the contents of the "b=" part, and include it in the
		**  canonicalization.  However, skip this if no hashing was
		**  done.
		*/

		if (dkim->dkim_hash != NULL ||
		    dkim->dkim_bhash != NULL)
		{
			dkim_canonfinal(dkim, TRUE);
			dkim->dkim_bodydone = TRUE;

			start = hdr->hdr_text;
			memset(tmp, '\0', sizeof tmp);
			n = 0;
			for (p = hdr->hdr_text; *p != '\0'; p++)
			{
				if (*p == ';')
					in = '\0';

				if (in == 'b')
				{
					last = *p;
					continue;
				}

				if (in == '\0' && *p == '=')
					in = last;

				tmp[n++] = *p;
				last = *p;

				if (n == MAXHEADER)
				{
					dkim_error(dkim,
					           "header too large (max %d)",
					           MAXHEADER);
					return DKIM_STAT_NORESOURCE;
				}
			}

			/* canonicalize */
			tmphdr.hdr_text = tmp;
			tmphdr.hdr_flags = 0;
			tmphdr.hdr_next = NULL;

			if (dkim->dkim_version < DKIM_VERSION_IETF_BASE_01)
				dkim_canonwrite(dkim, CRLF, 2);
			if (dkim->dkim_hdrcanonalg == DKIM_CANON_RELAXED)
				dkim_lowerhdr(tmp);
			dkim_canonheader(dkim, &tmphdr, FALSE);
		}
	}

	/* finalize canonicalization */
	if (dkim->dkim_hash != NULL ||
	    dkim->dkim_bhash != NULL)
		dkim_canonfinal(dkim, FALSE);

	/* conduct verification, or compute signature */
	if (dkim->dkim_mode == DKIM_MODE_SIGN)
	{
		size_t diglen;
		size_t siglen;
		u_int l;
		int status;
		u_char *digest;
		u_char *signature;
		BIO *key;

		key = BIO_new_mem_buf(dkim->dkim_key, dkim->dkim_keylen);
		if (key == NULL)
		{
			dkim_error(dkim, "BIO_new_mem_buf() failed");
			return DKIM_STAT_NORESOURCE;
		}

		/* get the digest */
		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;

			diglen = SHA_DIGEST_LENGTH;
			digest = sha1->sha1_out;

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;

			diglen = SHA256_DIGEST_LENGTH;
			digest = sha256->sha256_out;

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */

		  default:
			assert(0);
		}

		/* compute and store the signature */
		switch (dkim->dkim_signalg)
		{
		  case DKIM_SIGN_RSASHA1:
#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_SIGN_RSASHA256:
#endif /* SHA256_DIGEST_LENGTH */
		  {
			int nid;
			struct dkim_rsa *rsa;

#ifdef SHA256_DIGEST_LENGTH
			assert(dkim->dkim_hashtype == DKIM_HASHTYPE_SHA1 ||
			       dkim->dkim_hashtype == DKIM_HASHTYPE_SHA256);
#else /* SHA256_DIGEST_LENGTH */
			assert(dkim->dkim_hashtype == DKIM_HASHTYPE_SHA1);
#endif /* SHA256_DIGEST_LENGTH */

			rsa = DKIM_MALLOC(dkim, sizeof(struct dkim_rsa));
			if (rsa == NULL)
			{
				dkim_error(dkim,
				           "unable to allocate %d byte(s)",
				           sizeof(struct dkim_rsa));
				return DKIM_STAT_NORESOURCE;
			}
			memset(rsa, '\0', sizeof(struct dkim_rsa));

			dkim->dkim_signature = (void *) rsa;
			dkim->dkim_sigtype = DKIM_SIGTYPE_RSA;

			rsa->rsa_pkey = PEM_read_bio_PrivateKey(key, NULL,
			                                        NULL, NULL);

			if (rsa->rsa_pkey == NULL)
			{
				dkim_error(dkim,
				           "PEM_read_bio_PrivateKey() failed");
				BIO_free(key);
				return DKIM_STAT_NORESOURCE;
			}

			rsa->rsa_rsa = EVP_PKEY_get1_RSA(rsa->rsa_pkey);
			if (rsa->rsa_rsa == NULL)
			{
				dkim_error(dkim, "EVP_PKEY_get1_RSA() failed");
				BIO_free(key);
				return DKIM_STAT_NORESOURCE;
			}

			rsa->rsa_keysize = RSA_size(rsa->rsa_rsa);
			rsa->rsa_pad = RSA_PKCS1_PADDING;
			rsa->rsa_rsaout = DKIM_MALLOC(dkim, rsa->rsa_keysize);
			if (rsa->rsa_rsaout == NULL)
			{
				dkim_error(dkim,
				           "unable to allocate %d byte(s)",
				           rsa->rsa_keysize);
				BIO_free(key);
				return DKIM_STAT_NORESOURCE;
			}

			nid = NID_sha1;

#ifdef SHA256_DIGEST_LENGTH
			if (dkim->dkim_hashtype == DKIM_HASHTYPE_SHA256)
				nid = NID_sha256;
#endif /* SHA256_DIGEST_LENGTH */

			status = RSA_sign(nid, digest, diglen,
		                          rsa->rsa_rsaout, &l, rsa->rsa_rsa);
			if (status == 0 || l == 0)
			{
				BIO_free(key);
				dkim_error(dkim,
				           "signature generation failed (status %d, length %d)",
				           status, l);
				return DKIM_STAT_INTERNAL;
			}

			rsa->rsa_rsaoutlen = l;

			signature = rsa->rsa_rsaout;
			siglen = rsa->rsa_rsaoutlen;

			break;
		  }

		  default:
			assert(0);
		}

		/* base64-encode the signature */
		dkim->dkim_b64siglen = siglen * 3 + 5;
		dkim->dkim_b64siglen += (dkim->dkim_b64siglen / 60);
		dkim->dkim_b64sig = DKIM_MALLOC(dkim, dkim->dkim_b64siglen);
		if (dkim->dkim_b64sig == NULL)
		{
			dkim_error(dkim, "unable to allocate %d byte(s)",
			           dkim->dkim_b64siglen);
			BIO_free(key);
			return DKIM_STAT_NORESOURCE;
		}
		memset(dkim->dkim_b64sig, '\0', dkim->dkim_b64siglen);

		status = dkim_base64_encode(signature, siglen,
		                            dkim->dkim_b64sig,
		                            dkim->dkim_b64siglen);

		BIO_free(key);

		if (status == -1)
		{
			dkim_error(dkim,
			           "base64 encoding error (buffer too small)");
			return DKIM_STAT_NORESOURCE;
		}
	}
	else						/* verifying */
	{
		bool msgsigned = FALSE;
		bool verified = FALSE;
		dkim_policy_t policy;
		bool test = FALSE;
		u_char *domain;
		struct dkim_set *sigset;
#ifdef _FFR_VBR
		struct dkim_set *vbrset;
#endif /* _FFR_VBR */
		struct dkim_header *sender;

		sigset = dkim->dkim_sigset;

		domain = dkim_param_get(sigset, "d");
		if (domain == NULL)			/* unsigned */
		{
			int status;
			u_char *user;
			u_char *domain;
			u_char *colon;

			sender = dkim_getsender(dkim,
			                        dkim->dkim_libhandle->dkiml_senderhdrs,
			                        FALSE);
			if (sender == NULL)
			{
				dkim_error(dkim, "no sender headers detected");
				return DKIM_STAT_SYNTAX;
			}
			dkim->dkim_senderhdr = sender;

			colon = strchr(sender->hdr_text, ':');
			if (colon == NULL)
			{
				dkim_error(dkim, "syntax error in headers");
				return DKIM_STAT_SYNTAX;
			}

			dkim->dkim_sender = dkim_strdup(dkim, colon + 1, 0);
			if (dkim->dkim_sender == NULL)
				return DKIM_STAT_NORESOURCE;

			/* determine domain */
			status = rfc2822_mailbox_split(dkim->dkim_sender,
			                               (char **) &user,
			                               (char **) &domain);
			if (status != 0)
			{
				dkim_error(dkim,
				           "can't determine sender address");
				return DKIM_STAT_SYNTAX;
			}

			if (dkim->dkim_domain == NULL)
			{
				dkim->dkim_domain = dkim_strdup(dkim,
				                                domain, 0);
				if (dkim->dkim_domain == NULL)
					return DKIM_STAT_NORESOURCE;
			}
			dkim->dkim_user = dkim_strdup(dkim, user, 0);
			if (dkim->dkim_user == NULL)
				return DKIM_STAT_NORESOURCE;
		}
		else					/* signed */
		{
			int status;
			u_int keyalg;
			size_t bhashlen = 0;
			size_t hashlen = 0;
			BIO *key = NULL;
			u_char *keytype;
			u_char *hashtypes;
			u_char *keygran;
			u_char *test;
			u_char *bhash;
			void *hashout = NULL;
			void *bhashout = NULL;
			struct dkim_set *keyset;

			msgsigned = TRUE;

			/* get key, using selector */
			status = dkim_get_key(dkim);
			if (status != DKIM_STAT_OK)
				return status;

			sender = dkim_getsender(dkim,
			                        dkim->dkim_libhandle->dkiml_senderhdrs,
			                        TRUE);
			if (sender == NULL)
			{
				dkim_error(dkim, "no sender headers detected");
				return DKIM_STAT_SYNTAX;
			}
			dkim->dkim_senderhdr = sender;

			/* confirm the key and signature match */
			for (keyset = dkim_set_first(dkim, DKIM_SETTYPE_KEY);
			     keyset != NULL;
			     keyset = dkim_set_next(keyset, DKIM_SETTYPE_KEY))
			{
				if (dkim_key_smtp(keyset))
					break;
			}
			if (keyset == NULL)
			{
				dkim_error(dkim, "no SMTP keyset found");
				return DKIM_STAT_CANTVRFY;
			}

			/* key type is OK? */
			keytype = dkim_param_get(keyset, "k");
			if (keytype == NULL)
			{
				dkim_error(dkim, "key type not defined");
				return DKIM_STAT_CANTVRFY;
			}
			keyalg = dkim_name_to_code(keytypes, keytype);
			if (keyalg == -1)
			{
				dkim_error(dkim, "key type `%s' not known",
				           keytype);
				return DKIM_STAT_CANTVRFY;
			}
#ifdef SHA256_DIGEST_LENGTH
			if (keyalg == DKIM_KEYTYPE_RSA &&
			    dkim->dkim_signalg != DKIM_SIGN_RSASHA1 &&
			    dkim->dkim_signalg != DKIM_SIGN_RSASHA256)
#else /* SHA256_DIGEST_LENGTH */
			if (keyalg == DKIM_KEYTYPE_RSA &&
			    dkim->dkim_signalg != DKIM_SIGN_RSASHA1)
#endif /* SHA256_DIGEST_LENGTH */
			{
				dkim_error(dkim, "key type `%s' not supported",
				           keytype);
				return DKIM_STAT_CANTVRFY;
			}

			/* hash type is OK? */
			hashtypes = dkim_param_get(keyset, "h");
			if (!dkim_key_hashok(dkim, hashtypes))
			{
				dkim_error(dkim,
				           "hash type `%s' not allowed with this key",
				           hashtypes);
				return DKIM_STAT_CANTVRFY;
			}

			/* key granularity is OK? */
			keygran = dkim_param_get(keyset, "g");
			if (keygran != NULL && !dkim_key_granok(dkim, keygran))
			{
				dkim_error(dkim,
				           "invalid key granularity `%s'",
				           keygran);
				return DKIM_STAT_CANTVRFY;
			}

			test = dkim_param_get(keyset, "t");
			if (testkey != NULL)
			{
				if (test != NULL && test[0] == 'y')
					*testkey = TRUE;
				else
					*testkey = FALSE;
			}

			/* load the public key */
			key = BIO_new_mem_buf(dkim->dkim_key,
			                      dkim->dkim_keylen);
			if (key == NULL)
			{
				dkim_error(dkim, "BIO_new_mem_buf() failed");
				return DKIM_STAT_NORESOURCE;
			}

			switch (dkim->dkim_signalg)
			{
			  case DKIM_SIGN_RSASHA1:
				dkim->dkim_sigtype = DKIM_SIGTYPE_RSA;
				dkim->dkim_hashtype = DKIM_HASHTYPE_SHA1;
				break;

#ifdef SHA256_DIGEST_LENGTH
			  case DKIM_SIGN_RSASHA256:
				dkim->dkim_sigtype = DKIM_SIGTYPE_RSA;
				dkim->dkim_hashtype = DKIM_HASHTYPE_SHA256;
				break;
#endif /* SHA256_DIGEST_LENGTH */

			  default:
				assert(0);
			}

			switch (dkim->dkim_hashtype)
			{
			  case DKIM_HASHTYPE_SHA1:
			  {
				struct dkim_sha1 *sha1;

				sha1 = (struct dkim_sha1 *) dkim->dkim_hash;

				hashout = sha1->sha1_out;
				hashlen = SHA_DIGEST_LENGTH;

				if (dkim->dkim_bhash != NULL)
				{
					sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;

					bhashout = sha1->sha1_out;
					bhashlen = SHA_DIGEST_LENGTH;
				}

				break;
			  }

#ifdef SHA256_DIGEST_LENGTH
			  case DKIM_HASHTYPE_SHA256:
			  {
				struct dkim_sha256 *sha256;

				sha256 = (struct dkim_sha256 *) dkim->dkim_hash;

				hashout = sha256->sha256_out;
				hashlen = SHA256_DIGEST_LENGTH;

				if (dkim->dkim_bhash != NULL)
				{
					sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;

					bhashout = sha256->sha256_out;
					bhashlen = SHA256_DIGEST_LENGTH;
				}

				break;
			  }
#endif /* SHA256_DIGEST_LENGTH */

			  default:
				assert(0);
			}

			/* body hash, if present, matches this body? */
			bhash = dkim_param_get(sigset, "bh");
			if (bhash != NULL && bhashout != NULL)
			{
				char b64hash[MAXHEADER + 1];

				memset(b64hash, '\0', sizeof b64hash);
				status = dkim_base64_encode(bhashout, bhashlen,
				                            b64hash,
				                            sizeof b64hash);

				if (strcmp(b64hash, bhash) != 0)
					return DKIM_STAT_BADSIG;
			}

			switch (dkim->dkim_sigtype)
			{
			  case DKIM_SIGTYPE_RSA:
			  {
				int nid;
				struct dkim_rsa *rsa;

				rsa = DKIM_MALLOC(dkim,
				                  sizeof(struct dkim_rsa));
				if (rsa == NULL)
				{
					dkim_error(dkim,
					           "unable to allocate %d byte(s)",
					           sizeof(struct dkim_rsa));
					return DKIM_STAT_NORESOURCE;
				}
				memset(rsa, '\0', sizeof(struct dkim_rsa));

				dkim->dkim_signature = rsa;

				rsa->rsa_pkey = d2i_PUBKEY_bio(key, NULL);
				if (rsa->rsa_pkey == NULL)
				{
					dkim_error(dkim,
					           "d2i_PUBKEY_bio() failed");
					BIO_free(key);
					return DKIM_STAT_NORESOURCE;
				}

				/* set up the RSA object */
				rsa->rsa_rsa = EVP_PKEY_get1_RSA(rsa->rsa_pkey);
				if (rsa->rsa_rsa == NULL)
				{
					dkim_error(dkim,
					           "EVP_PKEY_get1_RSA() failed");
					BIO_free(key);
					return DKIM_STAT_NORESOURCE;
				}

				rsa->rsa_keysize = RSA_size(rsa->rsa_rsa);
				rsa->rsa_pad = RSA_PKCS1_PADDING;

				rsa->rsa_rsain = dkim->dkim_sig;
				rsa->rsa_rsainlen = dkim->dkim_siglen;

				dkim->dkim_keybits = 8 * rsa->rsa_keysize;

				verified = TRUE;

				nid = NID_sha1;

#ifdef SHA256_DIGEST_LENGTH
				if (dkim->dkim_hashtype == DKIM_HASHTYPE_SHA256)
					nid = NID_sha256;
#endif /* SHA256_DIGEST_LENGTH */

				status = RSA_verify(nid, hashout, hashlen,
				                    rsa->rsa_rsain,
				                    rsa->rsa_rsainlen,
				                    rsa->rsa_rsa);
				if (status == 0)
					verified = FALSE;

				break;
			  }

			  default:
				assert(0);
			}

			BIO_free(key);
		}

#ifdef _FFR_VBR
		for (vbrset = dkim_set_first(dkim, DKIM_SETTYPE_VBRINFO);
		     vbrset != NULL;
		     vbrset = dkim_set_next(vbrset, DKIM_SETTYPE_VBRINFO))
		{
			domain = dkim_param_get(vbrset, "md");
			if (domain == NULL ||
			    strcasecmp(domain, dkim->dkim_domain) != 0)
				continue;

			dkim_vbr_settype(dkim, dkim_param_get(vbrset, "mc"));
			dkim_vbr_setcert(dkim, dkim_param_get(vbrset, "mv"));

			break;
		}
#endif /* _FFR_VBR */

		/* retrieve policy */
		if (dkim_get_policy(dkim) == -1)
		{
			dkim_error(dkim, "can't retrieve policy record");
			return DKIM_STAT_CANTVRFY;
		}

		/*
		**  Determine final result.
		*/

		ret = DKIM_STAT_OK;
		policy = dkim_policy(dkim, &test);
		if (testkey != NULL)
			*testkey = test;

		/*
		**  If the "s" policy flag is set and the signature claims
		**  the "i=" domain is a subdomain of the "d=" domain,
		**  the signature is not acceptable.
		*/

		if (dkim_policy_flag(dkim, 's') && dkim->dkim_subdomain)
		{
			dkim_error(dkim, "policy does not permit subdomains");
			return DKIM_STAT_CANTVRFY;
		}

		/* react to the published policy */
		switch (policy)
		{
#ifdef _FFR_ALLMAN_SSP_02

		  case DKIM_POLICY_UNKNOWN:
			break;

		  case DKIM_POLICY_ALL:
			if (!msgsigned)
			{
				dkim_error(dkim, "policy requires signature; no signature found");
				ret = DKIM_STAT_NOSIG;
			}
			break;

		  case DKIM_POLICY_STRICT:
			if (!verified)
				ret = DKIM_STAT_BADSIG;
			break;

#else /* _FFR_ALLMAN_SSP_02 */

		  case DKIM_POLICY_SIGNSOME:
			break;

		  case DKIM_POLICY_REQUIRED:
			if (!msgsigned)
			{
				dkim_error(dkim, "policy requires signature; no signature found");
				ret = DKIM_STAT_NOSIG;
			}
			break;

		  case DKIM_POLICY_AUTHORITY:
			/* XXX -- verify signer domain and sender domain match */
			break;

		  case DKIM_POLICY_NEVER:
			if (msgsigned)
				ret = DKIM_STAT_BADSIG;
			break;

		  case DKIM_POLICY_USER:
			ret = DKIM_STAT_NOTIMPLEMENT;
			break;

#endif /* _FFR_ALLMAN_SSP_02 */

		  default:
			dkim_error(dkim, "syntax error in policy record");
			ret = DKIM_STAT_SYNTAX;
			break;
		}

		if (msgsigned && !verified)
			ret = DKIM_STAT_BADSIG;

		return ret;
	}

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETSIGHDR -- for signing operations, retrieve the complete signature
**                    header
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- buffer for the signature
**  	buflen -- bytes available at "buf"
**  	margin -- attempt to cut at specified line width; 0 == don't bother
**  	initial -- initial line width
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getsighdr(DKIM *dkim, u_char *buf, size_t buflen, size_t margin,
               size_t initial)
{
	size_t len;
	char *ctx;
	char *pv;
	u_char hdr[MAXHEADER + 1];

	assert(dkim != NULL);
	assert(buf != NULL);
	assert(buflen != 0);

	if (dkim->dkim_state != DKIM_STATE_EOM ||
	    dkim->dkim_mode != DKIM_MODE_SIGN)
		return DKIM_STAT_INVALID;

#define	DELIMITER	"\001"

	/* compute and extract the signature header */
	memset(hdr, '\0', sizeof hdr);
	len = dkim_gensighdr(dkim, hdr, sizeof hdr, DELIMITER);
	if (len == MAXHEADER)
	{
		dkim_error(dkim,
		           "generated signature header too large (max %d)",
		           MAXHEADER);
		return DKIM_STAT_NORESOURCE;
	}

	if (dkim->dkim_b64sig != NULL)
	{
		len = sm_strlcat(hdr, dkim->dkim_b64sig, sizeof hdr);
		if (len >= sizeof hdr || len >= buflen)
		{
			dkim_error(dkim,
			           "generated signature header too large (max %d)",
			           MAXHEADER);
			return DKIM_STAT_NORESOURCE;
		}
	}

	if (margin == 0)
	{
		bool first = TRUE;

		for (pv = strtok_r(hdr, DELIMITER, &ctx);
		     pv != NULL;
		     pv = strtok_r(NULL, DELIMITER, &ctx))
		{
			if (!first)
				sm_strlcat(buf, " ", buflen);

			sm_strlcat(buf, pv, buflen);

			first = FALSE;
		}
	}
	else
	{
		bool first = TRUE;
		char *p;
		char *q;
		char *end;
		size_t len;
		char which[MAXTAGNAME + 1];

		len = initial;
		end = which + MAXTAGNAME;

		for (pv = strtok_r(hdr, DELIMITER, &ctx);
		     pv != NULL;
		     pv = strtok_r(NULL, DELIMITER, &ctx))
		{
			for (p = pv, q = which; *p != '=' && q <= end; p++, q++)
			{
				*q = *p;
				*(q + 1) = '\0';
			}

			if (len == 0 || first)
			{
				sm_strlcat(buf, pv, buflen);
				len += strlen(pv);
				first = FALSE;
			}
			else if (len + strlen(pv) > margin)
			{
				bool first = TRUE;

				if (strcmp(which, "h") == 0)
				{			/* break at colons */
					char *hdr;
					char *ctx2;

					for (hdr = strtok_r(pv, ":", &ctx2);
					     hdr != NULL;
					     hdr = strtok_r(NULL, ":", &ctx2))
					{
						if (first)
						{
							if (len + strlen(pv) + 2 > margin)
							{
								sm_strlcat(buf,
								           "\r\n\t",
								           buflen);
								len = 8;
							}
							else
							{
								sm_strlcat(buf,
								           " ",
								           buflen);
								len += 1;
							}

							sm_strlcat(buf, pv,
							           buflen);
							len += strlen(pv);
							first = FALSE;
						}
						else if (len + strlen(hdr) + 1 > margin)
						{
							sm_strlcat(buf, ":",
							           buflen);
							len += 1;
							sm_strlcat(buf,
							           "\r\n\t ",
							           buflen);
							len = 9;
							sm_strlcat(buf, hdr,
							           buflen);
							len += strlen(hdr);
						}
						else
						{
							sm_strlcat(buf, ":",
							           buflen);
							len += 1;
							sm_strlcat(buf, hdr,
							           buflen);
							len += strlen(hdr);
						}
					}

				}
				else if (strcmp(which, "b") == 0 ||
				         strcmp(which, "bh") == 0 ||
				         strcmp(which, "z") == 0)
				{			/* break at margins */
					bool more;
					int idx;
					int offset;
					char *x;

					offset = strlen(which) + 1;

					if (len + offset + 1 >= margin)
					{
						sm_strlcat(buf, "\r\n\t",
						           buflen);
						len = 8;
						idx += 3;
					}
					else
					{
						sm_strlcat(buf, " ", buflen);
						len++;
						idx++;
					}

					sm_strlcat(buf, which, buflen);
					sm_strlcat(buf, "=", buflen);

					len += offset;
					idx += offset;

					idx = strlen(buf);

					for (x = pv + offset;
					     *x != '\0' && idx < buflen - 2;
					     x++)
					{
						more = (*(x + 1) != '\0');

						buf[idx] = *x;
						idx++;
						buf[idx] = '\0';
						len++;

						if (len >= margin && more)
						{
							sm_strlcat(buf,
							           "\r\n\t",
							           buflen);
							len = 8;
							idx += 3;
						}
					}
				}
				else
				{			/* break at delimiter */
					sm_strlcat(buf, "\r\n\t", buflen);
					len = 8;
					sm_strlcat(buf, pv, buflen);
					len += strlen(pv);
				}
			}
			else
			{
				if (!first)
					sm_strlcat(buf, " ", buflen);
				first = FALSE;
				len += 1;
				sm_strlcat(buf, pv, buflen);
				len += strlen(pv);
			}
		}
	}

	return DKIM_STAT_OK;
}

/*
**  DKIM_REPORTINFO -- return the address to which failure reports are sent
**
**  Parameters:
**  	dkim -- DKIM handle
**  	fd -- descriptor for canonicalized message (returned)
**  	bfd -- descriptor for canonicalized body (returned)
**  	buf -- buffer for the address
**  	buflen -- bytes available at "buf"
**
**  Return value:
**  	A DKIM_STAT_* constant.
**
**  Notes:
**  	DKIM_STAT_CANTVRFY returned by this function means no policy record
**  	was found or "r" was not part of the policy record found.
**
**  	If the IETF "base-01" draft is being applied, the headers and
**  	body are hashed separately.  Thus, there are two descriptors of
**  	interest.  In that case, the body comes back in "bfd".  Otherwise,
**  	"bfd" will be unaltered.
*/

DKIM_STAT
dkim_reportinfo(DKIM *dkim, int *fd, int *bfd, u_char *buf, size_t buflen)
{
	DKIM_SET *set;

	assert(dkim != NULL);

	if (dkim->dkim_state != DKIM_STATE_EOM ||
	    dkim->dkim_mode != DKIM_MODE_VERIFY)
		return DKIM_STAT_INVALID;

	if (dkim->dkim_hash != NULL)
	{
		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;
			if (fd != NULL)
				*fd = sha1->sha1_tmpfd;

			if (bfd != NULL &&
			    dkim->dkim_version >= DKIM_VERSION_IETF_BASE_01)
			{
				sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;
				*bfd = sha1->sha1_tmpfd;
			}

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;
			if (fd != NULL)
				*fd = sha256->sha256_tmpfd;

			if (bfd != NULL &&
			    dkim->dkim_version >= DKIM_VERSION_IETF_BASE_01)
			{
				sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;
				*bfd = sha256->sha256_tmpfd;
			}

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */
		}
	}

	/* extract the information */
	set = dkim_set_first(dkim, DKIM_SETTYPE_POLICY);
	if (set == NULL)
	{
		dkim_error(dkim, "no policy record found");
		return DKIM_STAT_CANTVRFY;
	}

	dkim->dkim_reportaddr = dkim_param_get(set, "r");
	if (dkim->dkim_reportaddr == NULL)
	{
		dkim_error(dkim, "no reporting address found in policy record");
		return DKIM_STAT_CANTVRFY;
	}

	memset(buf, '\0', buflen);

	sm_strlcpy(buf, dkim->dkim_reportaddr, buflen);

	return DKIM_STAT_OK;
}

/*
**  DKIM_POLICY -- parse policy associated with the sender's domain
**
**  Parameters:
**  	dkim -- DKIM handle
**  	test -- test flag (returned)
**
**  Return value:
**  	A DKIM_POLICY_* value, or -1 on error.
*/

dkim_policy_t
dkim_policy(DKIM *dkim, bool *test)
{
	struct dkim_set *set;
#ifdef _FFR_ALLMAN_SSP_02
	struct dkim_set *uset;
#endif /* _FFR_ALLMAN_SSP_02 */
	u_char *param;

	assert(dkim != NULL);
	assert(test != NULL);

	set = dkim_set_first(dkim, DKIM_SETTYPE_POLICY);
	if (set == NULL)
		return -1;

#ifdef _FFR_ALLMAN_SSP_02
	uset = dkim_set_next(set, DKIM_SETTYPE_POLICY);
	if (uset != NULL)
		set = uset;
#endif /* _FFR_ALLMAN_SSP_02 */

	param = dkim_param_get(set, "t");
	if (param != NULL)
	{
		if (strchr(param, 'y') != NULL)
			*test = TRUE;
		else
			*test = FALSE;
	}
	else
	{
		*test = FALSE;
	}

#ifdef _FFR_ALLMAN_SSP_02
	param = dkim_param_get(set, "p");
	if (param == NULL)
		return -1;

	return (dkim_policy_t) dkim_name_to_code(policies, param);
#else /* _FFR_ALLMAN_SSP_02 */
	param = dkim_param_get(set, "o");
	if (param == NULL)
		return -1;

	return (dkim_policy_t) dkim_char_to_code(policies, param[0]);
#endif /* _FFR_ALLMAN_SSP_02 */
}

/*
**  DKIM_GETCANONLEN -- return canonicalized and total body lengths
**
**  Parameters:
**  	dkim -- DKIM handle
**  	msglen -- total body length (returned)
**  	canonlen -- total canonicalized length (returned)
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getcanonlen(DKIM *dkim, off_t *msglen, off_t *canonlen)
{
	assert(dkim != NULL);

	if (msglen != NULL)
		*msglen = dkim->dkim_bodylen;

	if (canonlen != NULL)
		*canonlen = dkim->dkim_canonlen;

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETKEYSIZE -- retrieve key size (in bits) when verifying
**
**  Parameters:
**  	dkim -- DKIM handle
**  	bits -- number of bits in the key (returned)
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getkeysize(DKIM *dkim, unsigned int *bits)
{
	assert(dkim != NULL);
	assert(bits != NULL);

	if (dkim->dkim_state != DKIM_STATE_EOM ||
	    dkim->dkim_mode != DKIM_MODE_VERIFY)
		return DKIM_STAT_INVALID;

	*bits = dkim->dkim_keybits;

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETSIGNALG -- retrieve signature algorithm when verifying
**
**  Parameters:
**  	dkim -- DKIM handle
**  	alg -- signature algorithm used (returned)
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getsignalg(DKIM *dkim, dkim_alg_t *alg)
{
	assert(dkim != NULL);
	assert(alg != NULL);

	if (dkim->dkim_state != DKIM_STATE_EOM ||
	    dkim->dkim_mode != DKIM_MODE_VERIFY)
		return DKIM_STAT_INVALID;

	*alg = dkim->dkim_signalg;

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETSIGNTIME -- retrieve signature timestamp
**
**  Parameters:
**  	dkim -- DKIM handle
**  	when -- signature timestamp (returned)
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getsigntime(DKIM *dkim, time_t *when)
{
	time_t tmp;
	char *end;
	char *timestr;

	assert(dkim != NULL);
	assert(when != NULL);

	if (dkim->dkim_state != DKIM_STATE_EOM ||
	    dkim->dkim_mode != DKIM_MODE_VERIFY)
		return DKIM_STAT_INVALID;

	if (dkim->dkim_sigset == NULL)
	{
		dkim_error(dkim, "no signature found");
		return DKIM_STAT_INTERNAL;
	}

	timestr = dkim_param_get(dkim->dkim_sigset, "t");
	if (timestr == NULL)
	{
		dkim_error(dkim, "signature missing required 't' value");
		return DKIM_STAT_SYNTAX;
	}

	tmp = strtoul(timestr, &end, 10);
	if (end != '\0')
	{
		dkim_error(dkim, "syntax error in `t' value");
		return DKIM_STAT_SYNTAX;
	}

	*when = tmp;

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETSELECTOR -- retrieve selector used to generate the signature
**
**  Parameters:
**  	dkim -- DKIM handle from which to retrieve selector
**
**  Return value:
**  	Selector found in the signature.
*/

const char *
dkim_getselector(DKIM *dkim)
{
	return dkim->dkim_selector;
}

/*
**  DKIM_GETSIGDOMAIN -- retrieve domain responsible for the signature
**
**  Parameters:
**  	dkim -- DKIM handle from which to retrieve domain
**
**  Return value:
**  	Domain responsible for the signature.
*/

const char *
dkim_getsigdomain(DKIM *dkim)
{
	return dkim->dkim_domain;
}

#ifdef _FFR_STATS
/*
**  DKIM_GETVERSION -- retrieve version used by a signer
**
**  Parameters:
**  	dkim -- DKIM handle from which to retrieve version
**  	ver -- pointer to an int into which version should be stored
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getversion(DKIM *dkim, int *ver)
{
	assert(dkim != NULL);
	assert(ver != NULL);

	*ver = dkim->dkim_version;

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETCANONS -- retrieve canonicalizations used when signing
**
**  Parameters:
**  	dkim -- DKIM handle from which to retrieve canonicalizations
**  	hdr -- Pointer to a dkim_canon_t where the header canonicalization
**             should be stored
**  	body -- Pointer to a dkim_canon_t where the body canonicalization
**              should be stored
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getcanons(DKIM *dkim, dkim_canon_t *hdr, dkim_canon_t *body)
{
	assert(dkim != NULL);

	if (hdr != NULL)
		*hdr = dkim->dkim_hdrcanonalg;
	if (body != NULL)
		*body = dkim->dkim_bodycanonalg;

	return DKIM_STAT_OK;
}
#endif /* _FFR_STATS */

/*
**  DKIM_SET_SIGNER -- set DKIM signature's signer
**
**  Parameters:
**  	dkim -- DKIM signing handle
**  	signer -- signer to store
**
**  Parameters:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_set_signer(DKIM *dkim, const char *signer)
{
	assert(dkim != NULL);
	assert(signer != NULL);

	if (dkim->dkim_mode != DKIM_MODE_SIGN)
		return DKIM_STAT_INVALID;

	if (dkim->dkim_signer == NULL)
	{
		dkim->dkim_signer = DKIM_MALLOC(dkim, MAXADDRESS + 1);
		if (dkim->dkim_signer == NULL)
		{
			dkim_error(dkim, "unable to allocate %d byte(s)",
			           MAXADDRESS + 1);
			return DKIM_STAT_NORESOURCE;
		}
	}

	sm_strlcpy(dkim->dkim_signer, signer, MAXADDRESS + 1);

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETERROR -- return any stored error string from within the DKIM
**                   context handle
**
**  Parameters:
**  	dkim -- DKIM handle from which to retrieve an error string
**
**  Return value:
**  	A pointer to the stored string, or NULL if none was stored.
*/

const char *
dkim_geterror(DKIM *dkim)
{
	assert(dkim != NULL);

	return dkim->dkim_error;
}

/*
**  DKIM_SET_DNS_CALLBACK -- set the DNS wait callback
**
**  Parameters:
**  	libdkim -- DKIM library handle
**  	func -- function to call; should take an opaque context pointer
**  	interval -- how often to call back
**
**  Return value:
**  	DKIM_STAT_OK -- success
**  	DKIM_STAT_INVALID -- invalid use
*/

DKIM_STAT
dkim_set_dns_callback(DKIM_LIB *libdkim, void (*func)(const void *context),
                      unsigned int interval)
{
	assert(libdkim != NULL);

#if USE_ARLIB
	if (func != NULL && interval == 0)
		return DKIM_STAT_INVALID;

	libdkim->dkiml_dns_callback = func;
	libdkim->dkiml_callback_int = interval;

	return DKIM_STAT_OK;
#else /* USE_ARLIB */
	return DKIM_STAT_INVALID;
#endif /* USE_ARLIB */
}

/*
**  DKIM_SET_USER_CONTEXT -- set user context pointer
**
**  Parameters:
**  	dkim -- DKIM handle
**  	ctx -- opaque context pointer
**
**  Return value:
**  	DKIM_STAT_OK
*/

DKIM_STAT
dkim_set_user_context(DKIM *dkim, const void *ctx)
{
	assert(dkim != NULL);

	dkim->dkim_user_context = ctx;

	return DKIM_STAT_OK;
}

/*
**  DKIM_SET_KEY_LOOKUP -- set the key lookup function
**
**  Parameters:
**  	libdkim -- DKIM library handle
**  	func -- function to call
**
**  Return value:
**  	DKIM_STAT_OK
*/

DKIM_STAT
dkim_set_key_lookup(DKIM_LIB *libdkim,
                    DKIM_STAT (*func)(DKIM *dkim, u_char *buf, size_t buflen))
{
	assert(libdkim != NULL);

	libdkim->dkiml_key_lookup = func;

	return DKIM_STAT_OK;
}

/*
**  DKIM_SET_POLICY_LOOKUP -- set the policy lookup function
**
**  Parameters:
**  	libdkim -- DKIM library handle
**  	func -- function to call
**
**  Return value:
**  	DKIM_STAT_OK
*/

DKIM_STAT
dkim_set_policy_lookup(DKIM_LIB *libdkim,
                       int (*func)(DKIM *dkim, u_char *buf, size_t buflen))
{
	assert(libdkim != NULL);

	libdkim->dkiml_policy_lookup = func;

	return DKIM_STAT_OK;
}

/*
**  DKIM_SET_SIGNATURE_HANDLE -- set the user handle allocation function
**
**  Parameters:
**  	libdkim -- DKIM library handle
**  	func -- function to call
**
**  Return value:
**  	DKIM_STAT_OK -- success
**  	DKIM_STAT_INVALID -- called against a signing handle or too late
**  	                     (i.e. after dkim_eoh() was called)
*/

DKIM_STAT
dkim_set_signature_handle(DKIM_LIB *libdkim, void * (*func)(void *closure))
{
	assert(libdkim != NULL);

	libdkim->dkiml_sig_handle = func;

	return DKIM_STAT_OK;
}

/*
**  DKIM_SET_SIGNATURE_HANDLE_FREE -- set the user handle deallocation function
**
**  Parameters:
**  	libdkim -- DKIM library handle
**  	func -- function to call
**
**  Return value:
**  	DKIM_STAT_OK
*/

DKIM_STAT
dkim_set_signature_handle_free(DKIM_LIB *libdkim,
                               void (*func)(void *closure, void *user))
{
	assert(libdkim != NULL);

	libdkim->dkiml_sig_handle_free = func;

	return DKIM_STAT_OK;
}

/*
**  DKIM_SET_SIGNATURE_TAGVALUES -- set the user handle population function
**
**  Parameters:
**  	libdkim -- DKIM library handle
**  	func -- function to call
**
**  Return value:
**  	DKIM_STAT_OK
*/

DKIM_STAT
dkim_set_signature_tagvalues(DKIM_LIB *libdkim, void (*func)(void *user,
                                                             dkim_param_t pcode,
                                                             const u_char *param,
                                                             const u_char *value))
{
	assert(libdkim != NULL);

	libdkim->dkiml_sig_tagvalues = func;

	return DKIM_STAT_OK;
}

/*
**  DKIM_SET_PRESCREEN -- set the user prescreen function
**
**  Parameters:
**  	libdkim -- DKIM library handle
**  	func -- function to call
**
**  Return value:
**  	DKIM_STAT_OK
*/

DKIM_STAT
dkim_set_prescreen(DKIM_LIB *libdkim, DKIM_CBSTAT (*func)(DKIM_SIGINFO **sigs,
                                                          int nsigs))
{
	assert(libdkim != NULL);

	libdkim->dkiml_prescreen = func;

	return DKIM_STAT_OK;
}

/*
**  DKIM_SIG_GETCONTEXT -- retrieve user-provided context from a DKIM_SIGINFO
**
**  Parameters:
**  	siginfo -- pointer to a DKIM_SIGINFO from which to extract context
**
**  Return value:
**  	Pointer to the user context provided by an earlier call to the
**  	handle allocator (see above), or NULL if none was ever set.
*/

void *
dkim_sig_getcontext(DKIM_SIGINFO *siginfo)
{
	assert(siginfo != NULL);

	return siginfo->sig_context;
}

/*
**  DKIM_SIG_IGNORE -- mark a signature referenced by a DKIM_SIGINFO with
**                     an "ignore" flag
**
**  Parameters:
**  	siginfo -- pointer to a DKIM_SIGINFO to update
**
**  Return value:
**  	None.
*/

void
dkim_sig_ignore(DKIM_SIGINFO *siginfo)
{
	assert(siginfo != NULL);

	siginfo->sig_flags |= DKIM_SIGFLAG_IGNORE;
}

#ifdef _FFR_VBR
/*
**  DKIM_TXT_DECODE -- decode a TXT reply
**
**  Parameters:
**  	ansbuf -- answer buffer
**  	anslen -- size of answer buffer
**  	buf -- output buffer
**  	buflen -- size of output buffer
**
**  Return value:
**  	TRUE iff ansbuf contains an IN TXT reply that could be deocde.
*/

static bool
dkim_txt_decode(u_char *ansbuf, size_t anslen, u_char *buf, size_t buflen)
{
	int type;
	int class;
	int qdcount;
	int ancount;
	int n;
	int c;
	u_char *cp;
	u_char *eom;
	u_char *p;
	HEADER hdr;
	char qname[MAXHOSTNAMELEN + 1];

	assert(ansbuf != NULL);
	assert(buf != NULL);

	/* set up pointers */
	memcpy(&hdr, ansbuf, sizeof hdr);
	cp = (u_char *) &ansbuf + HFIXEDSZ;
	eom = (u_char *) &ansbuf + anslen;

	/* skip over the name at the front of the answer */
	for (qdcount = ntohs((unsigned short) hdr.qdcount);
	     qdcount > 0;
	     qdcount--)
	{
		/* copy it first */
		(void) dn_expand((unsigned char *) &ansbuf, eom, cp,
		                 qname, sizeof qname);

		if ((n = dn_skipname(cp, eom)) < 0)
			return FALSE;
		cp += n;

		/* extract the type and class */
		if (cp + INT16SZ + INT16SZ > eom)
			return FALSE;

		GETSHORT(type, cp);
		GETSHORT(class, cp);
	}

	if (type != T_TXT || class != C_IN)
		return FALSE;

	if (hdr.rcode == NXDOMAIN)
		return FALSE;

	/* get the answer count */
	ancount = ntohs((unsigned short) hdr.ancount);
	if (ancount == 0)
		return FALSE;

	/* if truncated, we can't do it */
	if (hdr.tc)
		return FALSE;

	/* grab the label, even though we know what we asked... */
	if ((n = dn_expand((unsigned char *) &ansbuf, eom, cp,
	                   (RES_UNC_T) qname, sizeof qname)) < 0)
		return FALSE;
	/* ...and move past it */
	cp += n;

	/* extract the type and class */
	if (cp + INT16SZ + INT16SZ > eom)
		return FALSE;
	GETSHORT(type, cp);
	GETSHORT(class, cp);

	/* reject anything that's not valid (stupid wildcards) */
	if (type != T_TXT || class != C_IN)
		return FALSE;

	/* skip the TTL */
	cp += INT32SZ;

	/* get payload length */
	if (cp + INT16SZ > eom)
		return FALSE;
	GETSHORT(n, cp);

	/* XXX -- maybe deal with a partial reply rather than require it all */
	if (cp + n > eom)
		return FALSE;

	if (n > buflen)
		return FALSE;

	/* extract the payload */
	memset(buf, '\0', buflen);
	p = buf;
	while (n > 0)
	{
		c = *cp++;
		n--;
		while (c > 0)
		{
			*p++ = *cp++;
			c--;
			n--;
		}
	}

	return TRUE;
}

/*
**  DKIM_VBR_SETCERT -- store the VBR certifiers of this message
**
**  Parameters:
**  	dkim -- DKIM handle, created by dkim_sign() or dkim_verify()
**  	cert -- certifiers string
**
**  Return value:
**  	None (yet).
*/

void
dkim_vbr_setcert(DKIM *dkim, char *cert)
{
	assert(dkim != NULL);
	assert(cert != NULL);

	dkim->dkim_vbr_cert = dkim_strdup(dkim, cert, 0);
}

/*
**  DKIM_VBR_SETTYPE -- store the VBR type of this message
**
**  Parameters:
**  	dkim -- DKIM handle, created by dkim_sign() or dkim_verify()
**  	type -- type string
**
**  Return value:
**  	None (yet).
*/

void
dkim_vbr_settype(DKIM *dkim, char *type)
{
	assert(dkim != NULL);
	assert(type != NULL);

	dkim->dkim_vbr_type = dkim_strdup(dkim, type, 0);
}

/*
**  DKIM_VBR_TRUSTEDCERTS -- store the trusted VBR certifiers
**
**  Parameters:
**  	dkim -- DKIM handle, created by dkim_verify()
**  	certs -- NULL-terminated list of trusted certifiers
**
**  Return value:
**  	None (yet).
*/

void
dkim_vbr_trustedcerts(DKIM *dkim, char **certs)
{
	assert(dkim != NULL);
	assert(certs != NULL);

	dkim->dkim_vbr_trusted = (u_char **) certs;
}

/*
**  DKIM_VBR_QUERY -- query the vouching servers for results
**
**  Parameters:
**  	dkim -- DKIM handle, created by dkim_verify()
**  	res -- result string (one of "fail", "pass"); returned
**  	cert -- name of the certifier that returned a "pass"; returned
**  	domain -- name of the sending domain; returned
**
**  Return value:
**  	DKIM_STAT_OK -- able to determine a result
**  	DKIM_STAT_INVALID -- dkim_vbr_trustedcerts(), dkim_vbr_settype() and
**  	                     dkim_vbr_setcert() were not all called
**  	DKIM_STAT_CANTVRFY -- DNS issue prevented resolution
**
**  Notes:
**  	- "pass" is the result if ANY certifier vouched for the message.
**  	- "res" is not modified if no result could be determined
**  	- there's no attempt to validate the values found
*/

DKIM_STAT
dkim_vbr_query(DKIM *dkim, char **res, char **cert, char **domain)
{
	int n;
	char *p;
	char *last;
	DKIM_LIB *lib;
	char certs[MAXHEADER + 1];
	char query[MAXHOSTNAMELEN + 1];

	assert(dkim != NULL);
	assert(res != NULL);
	assert(cert != NULL);
	assert(domain != NULL);

	lib = dkim->dkim_libhandle;

	if (dkim->dkim_vbr_type == NULL ||
	    dkim->dkim_vbr_cert == NULL ||
	    dkim->dkim_vbr_trusted == NULL ||
	    dkim->dkim_domain == NULL)
	{
		dkim_error(dkim, "required data for VBR check missing");
		return DKIM_STAT_INVALID;
	}

	sm_strlcpy(certs, dkim->dkim_vbr_cert, sizeof certs);

	*domain = dkim->dkim_domain;

	for (p = strtok_r(certs, ":", &last);
	     p != NULL;
	     p = strtok_r(NULL, ":", &last))
	{
		for (n = 0; dkim->dkim_vbr_trusted[n] != NULL; n++)
		{
			if (strcasecmp(p, dkim->dkim_vbr_trusted[n]) == 0)
			{
				int status;
#ifdef USE_ARLIB
				int arerror;
				AR_QUERY q;
				AR_LIB ar;
#endif /* USE_ARLIB */
				char *last2;
				char *p2;
#ifdef USE_ARLIB
				struct timeval timeout;
#endif /* USE_ARLIB */
				unsigned char ansbuf[MAXPACKET];
				unsigned char buf[BUFRSZ];

				snprintf(query, sizeof query, "%s.%s.%s",
				         dkim->dkim_domain, VBR_PREFIX, p);

#ifdef USE_ARLIB
				ar = dkim->dkim_libhandle->dkiml_arlib;
				timeout.tv_sec = dkim->dkim_timeout;
				timeout.tv_usec = 0;
				q = ar_addquery(ar, query, C_IN, T_TXT,
				                MAXCNAMEDEPTH, ansbuf,
		                                sizeof ansbuf, &arerror,
		                                dkim->dkim_timeout == 0 ? NULL
				                                        : &timeout);
				if (q == NULL)
				{
					dkim_error(dkim,
					           "ar_addquery() failed");
					return DKIM_STAT_CANTVRFY;
				}

				if (lib->dkiml_dns_callback == NULL)
				{
					status = ar_waitreply(ar, q, NULL,
					                      NULL);
				}
				else
				{
					for (;;)
					{
						timeout.tv_sec = lib->dkiml_callback_int;
						timeout.tv_usec = 0;

						status = ar_waitreply(ar, q,
						                      NULL,
						                      &timeout);

						if (status != AR_STAT_NOREPLY)
							break;

						lib->dkiml_dns_callback(dkim->dkim_user_context);
					}
				}

				(void) ar_cancelquery(ar, q);
#else /* USE_ARLIB */
				status = res_query(query, C_IN, T_TXT, ansbuf,
				                   sizeof ansbuf);
#endif /* USE_ARLIB */

#if USE_ARLIB
				if (status == AR_STAT_ERROR ||
				    status == AR_STAT_EXPIRED)
				{
					dkim_error(dkim,
					           "failed to retrieve %s",
						   query);
					return DKIM_STAT_CANTVRFY;
				}
#else /* USE_ARLIB */
				if (status == -1)
				{
					switch (h_errno)
					{
					  case HOST_NOT_FOUND:
					  case NO_DATA:
						continue;

					  case TRY_AGAIN:
					  case NO_RECOVERY:
					  default:
						dkim_error(dkim,
						           "failed to retreive %s",
						           query);
						return DKIM_STAT_CANTVRFY;
					}
				}
#endif /* USE_ARLIB */

				/* try to decode the reply */
				if (!dkim_txt_decode(ansbuf, sizeof ansbuf,
				                     buf, sizeof buf))
					continue;

				/* see if there's a vouch match */
				for (p2 = strtok_r(buf, " \t", &last2);
				     p2 != NULL;
				     p2 = strtok_r(NULL, " \t", &last2))
				{
					if (strcasecmp(p2, VBR_ALL) == 0 ||
					    strcasecmp(p2,
					               dkim->dkim_vbr_type) == 0)
					{
						/* we have a winner! */
						*res = "pass";
						*cert = p;
						return DKIM_STAT_OK;
					}
				}
			}
		}
	}

	/* nobody vouched */
	*res = "fail";
	return DKIM_STAT_OK;
}

/*
**  DKIM_VBR_GETHEADER -- generate and store the VBR-Info header
**
**  Parameters:
**  	dkim -- DKIM handle, created by dkim_sign() or dkim_verify()
**  	hdr -- header buffer
**  	len -- number of bytes available at "hdr"
**
**  Return value:
**  	DKIM_STAT_OK -- success
**  	DKIM_STAT_NORESOURCE -- "hdr" was too short
**  	DKIM_STAT_INVALID -- not all VBR information was provided
*/

DKIM_STAT
dkim_vbr_getheader(DKIM *dkim, char *hdr, size_t len)
{
	size_t olen;

	assert(dkim != NULL);
	assert(hdr != NULL);

	if (dkim->dkim_vbr_cert == NULL || dkim->dkim_vbr_type == NULL)
	{
		dkim_error(dkim, "VBR certifiers or type missing");
		return DKIM_STAT_INVALID;
	}

	olen = snprintf(hdr, len, "md=%s; mc=%s; mv=%s",
	                dkim->dkim_domain, dkim->dkim_vbr_type,
	                dkim->dkim_vbr_cert);
	if (olen >= len)
	{
		dkim_error(dkim, "VBR buffer too small");
		return DKIM_STAT_NORESOURCE;
	}

	return DKIM_STAT_OK;
}
#endif /* _FFR_VBR */

/*
**  DKIM_SSL_VERSION -- return version of OpenSSL that was used to build
**                      the library
**
**  Parameters:
**  	None.
**
**  Return value:
**  	The constant OPENSSL_VERSION_NUMBER as defined by OpenSSL.
*/

unsigned long
dkim_ssl_version(void)
{
	return OPENSSL_VERSION_NUMBER;
}

#ifdef _FFR_QUERY_CACHE
/*
**  DKIM_FLUSH_CACHE -- purge expired records from the cache
**
**  Parameters:
**  	lib -- DKIM library handle, returned by dkim_init()
**
**  Return value:
**  	-1 -- caching is not in effect
**  	>= 0 -- number of purged records
*/

int
dkim_flush_cache(DKIM_LIB *lib)
{
	int err;

	assert(lib != NULL);

	if (lib->dkiml_cache == NULL)
		return -1;

	return dkim_cache_expire(lib->dkiml_cache, 0, &err);
}
#endif /* _FFR_QUERY_CACHE */
