/*
**  Copyright (c) 2005-2007 Sendmail, Inc. and its suppliers.
**    All rights reserved.
*/

#ifndef _DKIM_TYPES_H_
#define _DKIM_TYPES_H_

#ifndef lint
static char dkim_types_h_id[] = "@(#)$Id: dkim-types.h,v 1.60 2007/05/21 22:19:33 msk Exp $";
#endif /* !lint */

/* system includes */
#include <sys/types.h>
#include <regex.h>

/* libsm includes */
#include <sm/gen.h>
#include <sm/types.h>
#include <sm/cdefs.h>

/* libar includes */
#if USE_ARLIB
# include <ar.h>
#endif /* USE_ARLIB */

/* OpenSSL includes */
#include <openssl/pem.h>
#include <openssl/rsa.h>
#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/sha.h>

#ifdef _FFR_QUERY_CACHE
/* libdb includes */
# include <db.h>
#endif /* _FFR_QUERY_CACHE */

/* libdkim includes */
#include "dkim.h"

/* struct dkim_header -- an RFC2822 header of some kind */
struct dkim_header
{
	int			hdr_flags;
	u_char *		hdr_text;
	struct dkim_header *	hdr_next;
};

/* struct dkim_plist -- a parameter/value pair */
struct dkim_plist
{
	u_char *		plist_param;
	u_char *		plist_value;
	struct dkim_plist *	plist_next;
};

/* struct dkim_set -- a set of parameter/value pairs */
struct dkim_set
{
	bool			set_done;
	dkim_set_t		set_type;
	u_char *		set_data;
	struct dkim_plist *	set_plist;
	struct dkim_set *	set_next;
};

/* struct dkim_siginfo -- signature information for use by the caller */
struct dkim_siginfo
{
	u_int			sig_flags;
	u_int			sig_index;
	u_int			sig_bh;
	void *			sig_context;
	struct dkim_set *	sig_taglist;
};

/* struct dkim_sha1 -- stuff needed to do a sha1 hash */
struct dkim_sha1
{
	int			sha1_tmpfd;
	BIO *			sha1_tmpbio;
	SHA_CTX			sha1_ctx;
	u_char			sha1_out[SHA_DIGEST_LENGTH];
};

#ifdef SHA256_DIGEST_LENGTH
/* struct dkim_sha256 -- stuff needed to do a sha256 hash */
struct dkim_sha256
{
	int			sha256_tmpfd;
	BIO *			sha256_tmpbio;
	SHA256_CTX		sha256_ctx;
	u_char			sha256_out[SHA256_DIGEST_LENGTH];
};
#endif /* SHA256_DIGEST_LENGTH */

/* struct dkim_rsa -- stuff needed to do RSA sign/verify */
struct dkim_rsa
{
	u_char			rsa_pad;
	size_t			rsa_keysize;
	size_t			rsa_rsainlen;
	size_t			rsa_rsaoutlen;
	EVP_PKEY *		rsa_pkey;
	RSA *			rsa_rsa;
	u_char *		rsa_rsain;
	u_char *		rsa_rsaout;
};

/* struct dkim -- a complete DKIM transaction context */
struct dkim
{
	u_char			dkim_lastchar;
	bool			dkim_writesep;
	bool			dkim_partial;
	bool			dkim_bodydone;
	bool			dkim_blankline;
	bool			dkim_subdomain;
	bool			dkim_skipbody;
#ifdef _FFR_ALLMAN_SSP_02
	bool			dkim_userpolicy;
#endif /* _FFR_ALLMAN_SSP_02 */
	int			dkim_signum;
	int			dkim_mode;
	int			dkim_state;
	int			dkim_timeout;
	int			dkim_hashtype;
	int			dkim_sigtype;
#ifdef _FFR_QUERY_CACHE
	u_int			dkim_cache_queries;
	u_int			dkim_cache_hits;
#endif /* _FFR_QUERY_CACHE */
	u_int			dkim_blanks;
	u_int			dkim_keybits;
	u_int			dkim_version;
	u_int			dkim_sigcount;
	size_t			dkim_b64keylen;
	size_t			dkim_b64siglen;
	size_t			dkim_keylen;
	size_t			dkim_siglen;
	size_t			dkim_errlen;
	time_t			dkim_timestamp;
	dkim_query_t		dkim_querymethod;
	dkim_canon_t		dkim_hdrcanonalg;
	dkim_canon_t		dkim_bodycanonalg;
	dkim_alg_t		dkim_signalg;
	off_t			dkim_bodylen;
	off_t			dkim_signlen;
	off_t			dkim_canonlen;
#ifdef _FFR_HASH_BUFFERING
	u_int			dkim_hashbuflen;
	u_int			dkim_hashbufsize;
	u_char *		dkim_hashbuf;
#endif /* _FFR_HASH_BUFFERING */
	const char *		dkim_id;
	u_char *		dkim_domain;
	u_char *		dkim_user;
	u_char *		dkim_selector;
	u_char *		dkim_hdrlist;
	u_char *		dkim_b64key;
	u_char *		dkim_b64sig;
	u_char *		dkim_key;
	u_char *		dkim_sig;
	u_char *		dkim_reportaddr;
	u_char *		dkim_sender;
	u_char *		dkim_signer;
	u_char *		dkim_error;
	u_char *		dkim_zdecode;
#ifdef _FFR_VBR
	u_char *		dkim_vbr_type;
	u_char *		dkim_vbr_cert;
	u_char **		dkim_vbr_trusted;
#endif /* _FFR_VBR */
	char *			dkim_tmpdir;
	void *			dkim_hash;
	void *			dkim_bhash;
	void *			dkim_signature;
	void *			dkim_closure;
	const void *		dkim_user_context;
	struct dkim_siginfo **	dkim_siglist;
	struct dkim_set *	dkim_sethead;
	struct dkim_set *	dkim_settail;
	struct dkim_set *	dkim_sigset;
	struct dkim_header *	dkim_hhead;
	struct dkim_header *	dkim_htail;
	struct dkim_header *	dkim_senderhdr;
	DKIM_LIB *		dkim_libhandle;
};

/* struct dkim_lib -- a DKIM library context */
struct dkim_lib
{
	bool			dkiml_signre;
	bool			dkiml_skipre;
	u_int			dkiml_flags;
	u_int			dkiml_timeout;
	u_int			dkiml_version;
	u_int			dkiml_callback_int;
	time_t			dkiml_fixedtime;
	time_t			dkiml_sigttl;
	dkim_query_t		dkiml_querymethod;
	void *			(*dkiml_malloc) (void *closure, size_t nbytes);
	void			(*dkiml_free) (void *closure, void *p);
#if USE_ARLIB
	AR_LIB			dkiml_arlib;
#endif /* USE_ARLIB */
	u_char **		dkiml_senderhdrs;
	u_char **		dkiml_alwayshdrs;
#ifdef _FFR_QUERY_CACHE
	DB *			dkiml_cache;
#endif /* _FFR_QUERY_CACHE */
	regex_t			dkiml_hdrre;
	regex_t			dkiml_skiphdrre;
	int			(*dkiml_key_lookup) (DKIM *dkim, u_char *buf,
				                     size_t buflen);
	int			(*dkiml_policy_lookup) (DKIM *dkim,
				                        u_char *buf,
				                        size_t buflen);
	void *			(*dkiml_sig_handle) (void *closure);
	void			(*dkiml_sig_handle_free) (void *closure,
				                          void *user);
	void			(*dkiml_sig_tagvalues) (void *user,
				                        dkim_param_t pcode,
				                        const u_char *param,
				                        const u_char *value);
	DKIM_STAT		(*dkiml_prescreen) (DKIM_SIGINFO **sigs,
				                    int nsigs);
	void			(*dkiml_dns_callback) (const void *context);
	u_char			dkiml_tmpdir[MAXPATHLEN + 1];
	u_char			dkiml_queryinfo[MAXPATHLEN + 1];
};

#endif /* _DKIM_TYPES_H_ */
