/*
**  Copyright (c) 2005-2007 Sendmail, Inc. and its suppliers.
**    All rights reserved.
*/

#ifndef _DKIM_H_
#define _DKIM_H_

#ifndef lint
static char dkim_h_id[] = "@(#)$Id: dkim.h,v 1.55 2007/03/02 00:53:59 msk Exp $";
#endif /* !lint */

/* system includes */
#include <sys/types.h>
#include <sys/param.h>

/* libsm includes */
#include <sm/gen.h>
#include <sm/types.h>
#include <sm/cdefs.h>

/* openssl includes */
#include <openssl/sha.h>

/* the basics */
#ifndef NULL
# define NULL	0
#endif /* ! NULL */
#ifndef FALSE
# define FALSE	0
#endif /* ! FALSE */
#ifndef TRUE
# define TRUE	1
#endif /* ! TRUE */
#ifndef MAXPATHLEN
# define MAXPATHLEN		256
#endif /* ! MAXPATHLEN */

#ifndef MIN
# define MIN(x,y)		((x) < (y) ? (x) : (y))
#endif /* ! MIN */
#ifndef MAX
# define MAX(x,y)		((x) > (y) ? (x) : (y))
#endif /* ! MAX */

/* limits, macros, etc. */
#define MAXADDRESS		256	/* biggest user@host we accept */
#define MAXCNAMEDEPTH		3	/* max. CNAME recursion we allow */
#define MAXHDRCNT		64	/* max. headers signed */
#define MAXHEADER		4096	/* buffer for caching one header */
#define MAXHEADERS		32768	/* buffer for caching headers */
#define MAXPOLICYDEPTH		5	/* policy recursion we allow */
#define MAXTAGNAME		8	/* biggest tag name */

/* defaults */
#define	DEFERRLEN		64	/* default error string length */
#define	DEFTMPDIR		"/var/tmp"
					/* default temporary directory */

/* version */
#define	DKIM_VERSION_KEY	"DKIM1"	/* current version token for keys */
#define	DKIM_VERSION_SIG	"1"	/* current version token for sigs */
#define	DKIM_VERSION_SIG_DRAFT1	"0.2"	/* draft version token for sigs */
#define	DKIM_VERSION_SIG_DRAFT2	"0.5"	/* draft version token for sigs */

/* headers */
#define	DKIM_SIGNHEADER		"DKIM-Signature"
					/* DKIM signature header */
#define	DKIM_HDRMARGIN		75	/* "standard" header margin */

/* special DNS tokens */
#define	DKIM_DNSKEYNAME		"_domainkey"
					/* reserved DNS sub-zone */
#define	DKIM_DNSPOLICYNAME	"_policy"
					/* reserved DNS sub-zone */
#ifdef _FFR_ALLMAN_SSP_02
# define DKIM_DNSUSERPOLICYNAME	"_user"
					/* reserved DNS sub-zone */
#endif /* _FFR_ALLMAN_SSP_02 */

/*
**  DKIM_VERSION -- version; attempts to support backward compatibility
*/

#define DKIM_VERSION_UNKNOWN		0
#define DKIM_VERSION_ALLMAN_BASE_00	1
#define DKIM_VERSION_IETF_BASE_00	2
#define DKIM_VERSION_IETF_BASE_01	3
#define DKIM_VERSION_IETF_BASE_02	4
#define DKIM_VERSION_IETF_BASE_10	5
#define DKIM_VERSION_DEFAULT		DKIM_VERSION_IETF_BASE_10

/*
**  DKIM_STAT -- status code type
*/

typedef int DKIM_STAT;

#define	DKIM_STAT_OK		0	/* function completed successfully */
#define	DKIM_STAT_BADSIG	1	/* signature available but failed */
#define	DKIM_STAT_NOSIG		2	/* no signature available */
#define	DKIM_STAT_NOKEY		3	/* public key not found */
#define	DKIM_STAT_CANTVRFY	4	/* can't get domain key to verify */
#define	DKIM_STAT_SYNTAX	5	/* message is not valid syntax */
#define	DKIM_STAT_NORESOURCE	6	/* resource unavailable */
#define	DKIM_STAT_INTERNAL	7	/* internal error */
#define	DKIM_STAT_REVOKED	8	/* key found, but revoked */
#define	DKIM_STAT_INVALID	9	/* invalid function parameter */
#define	DKIM_STAT_NOTIMPLEMENT	10	/* function not implemented */
#define	DKIM_STAT_KEYFAIL	11	/* key retrieval failed */

/*
**  DKIM_CANON -- canonicalization method
*/

typedef int dkim_canon_t;

#define DKIM_CANON_UNKNOWN	-1	/* unknown method */
#define DKIM_CANON_SIMPLE	0	/* as specified in DKIM 01 draft */
#define DKIM_CANON_RELAXED	1	/* as specified in DKIM 01 draft */
#define DKIM_CANON_NOWSP	2	/* as specified in DKIM 00 draft */

#define DKIM_CANON_DEFAULT	DKIM_CANON_SIMPLE

/*
**  DKIM_SIGN -- signing method
*/

typedef int dkim_alg_t;

#define DKIM_SIGN_UNKNOWN	-1	/* unknown method */
#define DKIM_SIGN_RSASHA1	0	/* an RSA-signed SHA1 digest */
#ifdef SHA256_DIGEST_LENGTH
# define DKIM_SIGN_RSASHA256	1	/* an RSA-signed SHA256 digest */
#endif /* SHA256_DIGEST_LENGTH */

#define DKIM_SIGN_DEFAULT	DKIM_SIGN_RSASHA1

/*
**  DKIM_QUERY -- query method
*/

typedef int dkim_query_t;

#define DKIM_QUERY_UNKNOWN	-1	/* unknown method */
#define DKIM_QUERY_DNS		0	/* DNS query method (per the draft) */
#if DKIM_QUERY_FILE
# define DKIM_QUERY_FILE	1	/* text file method */
#endif /* DKIM_QUERY_FILE */

#define DKIM_QUERY_DEFAULT	DKIM_QUERY_DNS

/*
**  DKIM_PARAM -- known signature parameters
*/

typedef int dkim_param_t;

#define DKIM_PARAM_SIGNATURE	0	/* b */
#define DKIM_PARAM_SIGNALG	1	/* a */
#define DKIM_PARAM_DOMAIN	2	/* d */
#define DKIM_PARAM_CANONALG	3	/* c */
#define DKIM_PARAM_QUERYMETHOD	4	/* q */
#define DKIM_PARAM_SELECTOR	5	/* s */
#define DKIM_PARAM_HDRLIST	6	/* h */
#define DKIM_PARAM_VERSION	7	/* v */
#define DKIM_PARAM_IDENTITY	8	/* i */
#define DKIM_PARAM_TIMESTAMP	9	/* t */
#define DKIM_PARAM_EXPIRATION	10	/* x */
#define DKIM_PARAM_COPIEDHDRS	11	/* z */
#define DKIM_PARAM_BODYHASH	12	/* bh */
#define DKIM_PARAM_BODYLENGTH	13	/* l */

/*
**  DKIM_KEY -- known key parameters
*/

typedef int dkim_key_t;

#define	DKIM_KEY_VERSION	0	/* v */
#define	DKIM_KEY_GRANULARITY	1	/* g */
#define	DKIM_KEY_ALGORITHM	2	/* a */
#define	DKIM_KEY_NOTES		3	/* n */
#define	DKIM_KEY_DATA		4	/* p */
#define	DKIM_KEY_SERVICE	5	/* s */
#define	DKIM_KEY_FLAGS		6	/* t */

/*
**  DKIM_PPARAM -- policy parameters
*/

typedef int dkim_pparam_t;

#ifdef _FFR_ALLMAN_SSP_02

# define DKIM_PPARAM_POLICY	0	/* p */
# define DKIM_PPARAM_REPORTADDR	1	/* r */
# define DKIM_PPARAM_FLAGS	2	/* t */
# define DKIM_PPARAM_USERQUERY	3	/* u */

#else /* _FFR_ALLMAN_SSP_02 */

# define DKIM_PPARAM_POLICY	0	/* o */
# define DKIM_PPARAM_FLAGS	1	/* f */
# define DKIM_PPARAM_NOTES	2	/* n */
# define DKIM_PPARAM_REPORTADDR	3	/* r */
# define DKIM_PPARAM_URI	4	/* u */

#endif /* _FFR_ALLMAN_SSP_02 */

/*
**  DKIM_POLICY -- policies
*/

typedef int dkim_policy_t;

#ifdef _FFR_ALLMAN_SSP_02

# define DKIM_POLICY_UNKNOWN	0	/* unknown */
# define DKIM_POLICY_ALL	1	/* all */
# define DKIM_POLICY_STRICT	2	/* strict */

# define DKIM_POLICY_DEFAULT	DKIM_POLICY_UNKNOWN
# define DKIM_POLICY_DEFAULTTXT	"p=unknown; u=no"

#else /* _FFR_ALLMAN_SSP_02 */

# define DKIM_POLICY_SIGNSOME	0	/* ~ */
# define DKIM_POLICY_REQUIRED	1	/* - */
# define DKIM_POLICY_AUTHORITY	2	/* ! */
# define DKIM_POLICY_NEVER	3	/* . */
# define DKIM_POLICY_USER	4	/* ^ */

# define DKIM_POLICY_DEFAULT	DKIM_POLICY_SIGNSOME
# define DKIM_POLICY_DEFAULTTXT	"o=~"

#endif /* _FFR_ALLMAN_SSP_02 */

/*
**  DKIM_SETTYPE -- types of sets
*/

typedef int dkim_set_t;

#define	DKIM_SETTYPE_ANY	(-1)
#define	DKIM_SETTYPE_SIGNATURE	0
#define	DKIM_SETTYPE_POLICY	1
#define	DKIM_SETTYPE_KEY	2

/*
**  DKIM_HASHTYPE -- types of hashes
*/

#define DKIM_HASHTYPE_UNKNOWN	(-1)
#define DKIM_HASHTYPE_SHA1	0
#ifdef SHA256_DIGEST_LENGTH
# define DKIM_HASHTYPE_SHA256	1
#endif /* SHA256_DIGEST_LENGTH */

/*
**  DKIM_KEYTYPE -- types of keys
*/

#define	DKIM_KEYTYPE_UNKNOWN	(-1)
#define	DKIM_KEYTYPE_RSA	0

/*
**  DKIM_OPTS -- library-specific options
*/

typedef int dkim_opt_t;

#define DKIM_OP_GETOPT		0
#define	DKIM_OP_SETOPT		1

typedef int dkim_opts_t;

#define	DKIM_OPTS_FLAGS		0
#define	DKIM_OPTS_TMPDIR	1
#define	DKIM_OPTS_TIMEOUT	2
#define	DKIM_OPTS_SENDERHDRS	3
#define	DKIM_OPTS_SIGNHDRS	4
#define	DKIM_OPTS_VERSION	5
#define	DKIM_OPTS_QUERYMETHOD	6
#define	DKIM_OPTS_QUERYINFO	7

#define	DKIM_LIBFLAGS_NONE	0x00
#define	DKIM_LIBFLAGS_TMPFILES	0x01
#define	DKIM_LIBFLAGS_KEEPFILES	0x02
#define	DKIM_LIBFLAGS_SIGNLEN	0x04

#define	DKIM_LIBFLAGS_DEFAULT	DKIM_LIBFLAGS_NONE

/*
**  DKIM_LIB -- library handle
*/

struct dkim_lib;
typedef struct dkim_lib DKIM_LIB;

/*
**  DKIM -- DKIM context
*/

struct dkim;
typedef struct dkim DKIM;

/*
**  DKIM_SET -- a set of parameters and values
*/

struct dkim_set;
typedef struct dkim_set DKIM_SET;

/*
**  DKIM_PLIST -- a parameter/value pair, as a linked list
*/

struct dkim_plist;
typedef struct dkim_plist DKIM_PLIST;

/*
**  DKIM_SIGNATURE -- contents of a DKIM-Signature: header
*/

struct dkim_signature;
typedef struct dkim_signature DKIM_SIGNATURE;

/*
**  DKIM_SIGKEY_T -- private/public key (unencoded)
*/

typedef unsigned char * dkim_sigkey_t;

/*
**  DKIM_KEY -- contents of a published key record
*/

struct dkim_key;
typedef struct dkim_key DKIM_KEY;

/*
**  DKIM_POLICY -- contents of a published policy record
*/

struct dkim_policy;
typedef struct dkim_policy DKIM_POLICY;

/*
**  PROTOTYPES
*/

/*
**  dkim_init -- initialize the DKIM package
**
**  Parameters:
**  	None.
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_LIB *dkim_init __P((void *(*mallocf)(void *closure, size_t nbytes),
                                void (*freef)(void *closure, void *p)));

/*
**  dkim_sign -- make a new DKIM context for signing
**
**  Parameters:
**  	libhandle -- library handle, returned by dkim_init()
**  	id -- an opaque printable string for identifying this message, suitable
**  	      for use in logging or debug output; may not be NULL
**  	memclosure -- memory closure, for use by user-provided malloc/free
**  	secretkey -- pointer to secret key data to use; if NULL, it will be
**  	             obtained from disk
**  	selector -- selector being used to sign
**  	domain -- domain on behalf of which we're signing
**  	hdr_canon_alg -- canonicalization algorithm to use for headers;
**  	                 one of the DKIM_CANON_* macros, or -1 for default
**  	body_canon_alg -- canonicalization algorithm to use for body;
**  	                  one of the DKIM_CANON_* macros, or -1 for default
**  	sign_alg -- signing algorithm to use; one of the DKIM_SIGN_* macros,
**  	            or -1 for default
**  	length -- number of bytes of the body to sign (-1 == all)
**  	statp -- pointer to a DKIM_STAT which is updated by this call
**
**  Return value:
**  	A newly-allocated DKIM handle, or NULL on failure.  "statp" will be
**  	updated.
*/

extern DKIM *dkim_sign __P((DKIM_LIB *libhandle, const char *id,
                            void *memclosure, const dkim_sigkey_t secretkey,
                            const char *selector, const char *domain,
                            dkim_canon_t hdr_canon_alg,
                            dkim_canon_t body_canon_alg,
                            dkim_alg_t sign_alg,
                            off_t length, DKIM_STAT *statp));

/*
**  dkim_verify -- make a new DKIM context for verifying
**
**  Parameters:
**  	libhandle -- library handle, returned by dkim_init()
**  	id -- an opaque printable string for identifying this message, suitable
**  	      for use in logging or debug output; may not be NULL
**  	memclosure -- memory closure, for use by user-provided malloc/free
**  	statp -- pointer to a DKIM_STAT which is updated by this call
**
**  Return value:
**  	A newly-allocated DKIM handle, or NULL on failure.  "statp" will be
**  	updated.
*/

extern DKIM *dkim_verify __P((DKIM_LIB *libhandle, const char *id,
                              void *memclosure, DKIM_STAT *statp));

/*
**  dkim_header -- process a header
**
**  Parameters:
**  	dkim -- a DKIM handle previously returned by dkim_sign() or
**  	        dkim_verify()
**  	hdr -- the header to be processed, in canonical format
**  	len -- number of bytes to process starting at "hdr"
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_header __P((DKIM *dkim, u_char *hdr, size_t len));

/*
**  dkim_eoh -- identify end of headers
**
**  Parameters:
**  	dkim -- a DKIM handle previously returned by dkim_sign() or
**  	        dkim_verify()
**
**  Return value:
**  	A DKIM_STAT value.  DKIM_STAT_NOSIG will be returned if we're
**  	validating a signature but no DKIM signature was found in the headers.
*/

extern DKIM_STAT dkim_eoh __P((DKIM *dkim));

/*
**  dkim_body -- process a body chunk
**
**  Parameters:
**  	dkim -- a DKIM handle previously returned by dkim_sign() or
**  	        dkim_verify()
**  	buf -- the body chunk to be processed, in canonical format
**  	len -- number of bytes to process starting at "hdr"
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_body __P((DKIM *dkim, u_char *buf, size_t len));

/*
**  dkim_eom -- identify end of body
**
**  Parameters:
**  	dkim -- a DKIM handle previously returned by dkim_sign() or
**  	        dkim_verify()
**  	testkey -- TRUE iff the a matching key was found but is marked as a
**  	           test key (returned)
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_eom __P((DKIM *dkim, bool *testkey));

/*
**  dkim_getsighdr -- compute and return a signature header for a message
**
**  Parameters:
**  	dkim -- a DKIM handle previously returned by dkim_sign()
**  	buf -- buffer into which to write the signature
**  	len -- number of bytes available at "sig"
**  	margin -- attempt to wrap the header at the specified length
**  	initial -- width of the first line
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_getsighdr __P((DKIM *dkim, u_char *buf, size_t len,
                                     size_t margin, size_t initial));

/*
**  dkim_getcanonlen -- report number of (canonicalized) body bytes that were
**                      signed
**
**  Parameters:
**  	dkim -- a DKIM handle previously returned by dkim_sign() or
**  	        dkim_verify()
**  	msglen -- total size of the message body (returned)
**  	canonlen -- total number of canonicalized bytes (returned)
**
**  Return value:
**  	A DKIM_STAT value.
**
**  Notes:
**  	msglen or canonlen can be NULL if that information is not of interest
**  	to the caller.
*/

extern DKIM_STAT dkim_getcanonlen __P((DKIM *dkim, off_t *msglen,
                                       off_t *canonlen));

/*
**  dkim_reportinfo -- return info needed to generate failure report
**
**  Parameters:
**  	dkim -- a DKIM handle previously returned by dkim_verify()
**  	fd -- pointer to receive temporary file descriptor
**  	bfd -- pointer to receive temporary body file descriptor
**   	raddr -- buffer to receive report address
**  	rlen -- bytes available at raddr
**
**  Return vlalue:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_reportinfo __P((DKIM *dkim, int *fd, int *bfd,
                                      u_char *raddr, size_t rlen));

/*
**  DKIM_OPTIONS -- set/get options
**
**  Parameters:
**  	dklib -- DKIM library handle
**  	op -- operation (DKIM_OPT_GET or DKIM_OPT_SET)
**  	opt -- which option (a DKIM_OPTS_* constant)
**  	ptr -- value (in or out)
**  	len -- bytes available at "ptr"
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_options __P((DKIM_LIB *dklib, int op, dkim_opts_t opt,
                                   void *ptr, size_t len));

/*
**  dkim_getidentity -- retrieve signer's apparent identity
**
**  Parameters:
**  	dkim -- a DKIM handle previously returned by dkim_sign() or
**  	        dkim_verify()
**  	hname -- buffer to receive name of the header from which identity
**  	         was extracted
**  	hnamelen -- number of bytes available at hname
**  	hval -- buffer to receive decommented value of the header from which
**  	        identity was extracted
**  	hvallen -- number of bytes available at hval
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_getidentity __P((DKIM *dkim,
                                       u_char *hname, size_t hnamelen,
                                       u_char *hval, size_t hvallen));

/*
**  DKIM_POLICY -- retreive signing policy associated with the sender's domain
**
**  Parameters:
**  	dkim -- DKIM handle
**  	test -- test flag (returned)
**
**  Return value:
**  	A DKIM_POLICY_* value, or -1 on error.
*/

extern dkim_policy_t dkim_policy __P((DKIM *dkim, bool *test));

/*
**  DKIM_GETKEYSIZE -- retreive key size after verifying
**
**  Parameters:
**  	dkim -- DKIM handle
**  	bits -- size of the key in bits (returned)
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_getkeysize __P((DKIM *dkim, unsigned int *bits));

/*
**  DKIM_GETSIGNALG -- retreive signature algorithm after verifying
**
**  Parameters:
**  	dkim -- DKIM handle
**  	alg -- a DKIM_SIGN_* value (returned)
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_getsignalg __P((DKIM *dkim, dkim_alg_t *alg));

/*
**  DKIM_GETSIGNTIME -- retreive signature timestamp after verifying
**
**  Parameters:
**  	dkim -- DKIM handle
**  	when -- timestamp on the signature (returned)
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_getsigntime __P((DKIM *dkim, time_t *when));

#if _FFR_SET_DNS_CALLBACK
/*
**  DKIM_SET_DNS_CALLBACK -- set the DNS wait callback
**
**  Parameters:
**  	dkim -- DKIM handle
**  	func -- function to call; should take an opaque context pointer
**  	ctx -- opaque context handle
**  	interval -- how often to call back
**
**  Return value:
**  	None.
*/

extern DKIM_STAT dkim_set_dns_callback __P((DKIM *dkim,
                                            void (*func)(void *context),
                                            void *ctx,
                                            unsigned int interval));
#endif /* _FFR_SET_DNS_CALLBACK */

/*
**  dkim_free -- release resources associated with a DKIM handle
**
**  Parameters:
**  	dkim -- a DKIM handle previously returned by dkim_sign() or
**  	        dkim_verify()
**
**  Return value:
**  	A DKIM_STAT value.
*/

extern DKIM_STAT dkim_free __P((DKIM *dkim));

/*
**  DKIM_GETERROR -- return any stored error string from within the DKIM
**                   context handle
**
**  Parameters:
**  	dkim -- DKIM handle from which to retrieve an error string
**
**  Return value:
**  	A pointer to the stored string, or NULL if none was stored.
*/

extern const char *dkim_geterror __P((DKIM *dkim));

/*
**  rfc2822_mailbox_split -- extract the userid and host from a structured
**                           header
**
**  Parameters:
**  	addr -- the header to parse; see RFC2822 for format
**  	user -- local-part of the parsed header (returned)
**  	domain -- domain part of the parsed header (returned)
**
**  Return value:
**  	0 on success; other on error (see source)
*/

extern int rfc2822_mailbox_split __P((char *addr, char **user, char **domain));

#endif /* ! _DKIM_H_ */
