/*
**  Copyright (c) 2005-2007 Sendmail, Inc. and its suppliers.
**    All rights reserved.
*/

#ifndef lint
static char dkim_tables_c_id[] = "@(#)$Id: dkim-tables.c,v 1.17 2007/03/02 00:53:59 msk Exp $";
#endif /* !lint */

/* system includes */
#include <sys/types.h>
#include <string.h>
#include <assert.h>

/* libsm includes */
#include <sm/gen.h>
#include <sm/types.h>
#include <sm/cdefs.h>

/* libdkim includes */
#include "dkim-tables.h"
#include "dkim.h"

/* lookup tables */
static struct chartable prv_keyparams[] =	/* key parameters */
{
	{ 'a',		DKIM_KEY_ALGORITHM },
	{ 'g',		DKIM_KEY_GRANULARITY },
	{ 'n',		DKIM_KEY_NOTES },
	{ 'p',		DKIM_KEY_DATA },
	{ 's',		DKIM_KEY_SERVICE },
	{ 't',		DKIM_KEY_FLAGS },
	{ 'v',		DKIM_KEY_VERSION },
	{ '\0',		-1 }
};
struct chartable *keyparams = prv_keyparams;

#ifdef _FFR_ALLMAN_SSP_02

static struct chartable prv_policyparams[] =	/* policy parameters */
{
	{ 'p',		DKIM_PPARAM_POLICY },
	{ 'r',		DKIM_PPARAM_REPORTADDR },
	{ 't',		DKIM_PPARAM_FLAGS },
	{ 'u',		DKIM_PPARAM_USERQUERY },
	{ '\0',		-1 }
};

#else /* _FFR_ALLMAN_SSP_02 */

static struct chartable prv_policyparams[] =	/* policy parameters */
{
	{ 'f',		DKIM_PPARAM_FLAGS },
	{ 'n',		DKIM_PPARAM_NOTES },
	{ 'o',		DKIM_PPARAM_POLICY },
	{ 'r',		DKIM_PPARAM_REPORTADDR },
	{ 'u',		DKIM_PPARAM_URI },
	{ '\0',		-1 }
};

#endif /* _FFR_ALLMAN_SSP_02 */

struct chartable *policyparams = prv_policyparams;

#ifdef _FFR_ALLMAN_SSP_02

static struct nametable prv_policies[] =	/* policies */
{
	{ "unknown",	DKIM_POLICY_UNKNOWN },
	{ "all",	DKIM_POLICY_ALL },
	{ "strict",	DKIM_POLICY_STRICT },
	{ '\0',		-1 }
};
struct nametable *policies = prv_policies;

#else /* _FFR_ALLMAN_SSP_02 */

static struct chartable prv_policies[] =	/* policies */
{
	{ '~',		DKIM_POLICY_SIGNSOME },
	{ '-',		DKIM_POLICY_REQUIRED },
	{ '!',		DKIM_POLICY_AUTHORITY },
	{ '.',		DKIM_POLICY_NEVER },
	{ '^',		DKIM_POLICY_USER },
	{ '\0',		-1 }
};
struct chartable *policies = prv_policies;

#endif /* _FFR_ALLMAN_SSP_02 */

static struct nametable prv_sigparams[] =	/* signature parameters */
{
	{ "a",		DKIM_PARAM_SIGNALG },
	{ "b",		DKIM_PARAM_SIGNATURE },
	{ "bh",		DKIM_PARAM_BODYHASH },
	{ "c",		DKIM_PARAM_CANONALG },
	{ "d",		DKIM_PARAM_DOMAIN },
	{ "h",		DKIM_PARAM_HDRLIST },
	{ "i",		DKIM_PARAM_IDENTITY },
	{ "l",		DKIM_PARAM_BODYLENGTH },
	{ "q",		DKIM_PARAM_QUERYMETHOD },
	{ "s",		DKIM_PARAM_SELECTOR },
	{ "t",		DKIM_PARAM_TIMESTAMP },
	{ "v",		DKIM_PARAM_VERSION },
	{ "x",		DKIM_PARAM_EXPIRATION },
	{ "z",		DKIM_PARAM_COPIEDHDRS },
	{ NULL,		-1 }
};
struct nametable *sigparams = prv_sigparams;

static struct nametable prv_algorithms[] =	/* signing algorithms */
{
	{ "rsa-sha1",	DKIM_SIGN_RSASHA1 },
#ifdef SHA256_DIGEST_LENGTH
	{ "rsa-sha256",	DKIM_SIGN_RSASHA256 },
#endif /* SHA256_DIGEST_LENGTH */
	{ NULL,		-1 },
};
struct nametable *algorithms = prv_algorithms;

static struct nametable prv_canonicalizations[] = /* canonicalizations */
{
	{ "simple",	DKIM_CANON_SIMPLE },
	{ "relaxed",	DKIM_CANON_RELAXED },
	{ "nowsp",	DKIM_CANON_NOWSP },
	{ NULL,		-1 },
};
struct nametable *canonicalizations = prv_canonicalizations;

static struct nametable prv_hashes[] =		/* hashes */
{
	{ "sha1",	DKIM_HASHTYPE_SHA1 },
#ifdef SHA256_DIGEST_LENGTH
	{ "sha256",	DKIM_HASHTYPE_SHA256 },
#endif /* SHA256_DIGEST_LENGTH */
	{ NULL,		-1 },
};
struct nametable *hashes = prv_hashes;

static struct nametable prv_keytypes[] =	/* key types */
{
	{ "rsa",	DKIM_KEYTYPE_RSA },
	{ NULL,		-1 },
};
struct nametable *keytypes = prv_keytypes;

static struct nametable prv_querytypes[] =	/* query types */
{
	{ "dns",	DKIM_QUERY_DNS },
	{ NULL,		-1 },
};
struct nametable *querytypes = prv_querytypes;

static struct nametable prv_results[] =		/* result codes */
{
	{ "Success",			DKIM_STAT_OK },
	{ "Bad signature",		DKIM_STAT_BADSIG },
	{ "No signature",		DKIM_STAT_NOSIG },
	{ "No key",			DKIM_STAT_NOKEY },
	{ "Unable to verify",		DKIM_STAT_CANTVRFY },
	{ "Syntax error",		DKIM_STAT_SYNTAX },
	{ "Resource unavailable",	DKIM_STAT_NORESOURCE },
	{ "Internal error",		DKIM_STAT_INTERNAL },
	{ "Revoked key",		DKIM_STAT_REVOKED },
	{ NULL,				-1 },
};
struct nametable *results = prv_results;

static struct nametable prv_settypes[] =	/* set types */
{
	{ "key",	DKIM_SETTYPE_KEY },
	{ "policy",	DKIM_SETTYPE_POLICY },
	{ "signature",	DKIM_SETTYPE_SIGNATURE },
	{ NULL,		-1 },
};
struct nametable *settypes = prv_settypes;

/* ===================================================================== */

/*
**  DKIM_CODE_TO_NAME -- translate a mnemonic code to its name
**
**  Parameters:
**  	tbl -- name table
**  	code -- code to translate
**
**  Return value:
**  	Pointer to the name matching the provided code, or NULL if not found.
*/

const char *
dkim_code_to_name(struct nametable *tbl, const int code)
{
	int c;

	assert(tbl != NULL);

	for (c = 0; ; c++)
	{
		if (tbl[c].tbl_code == code)
			return tbl[c].tbl_name;

		if (tbl[c].tbl_code == -1)
			return NULL;
	}
}

/*
**  DKIM_NAME_TO_CODE -- translate a name to a mnemonic code
**
**  Parameters:
**  	tbl -- name table
**  	name -- name to translate
**
**  Return value:
**  	A mnemonic code matching the provided name, or -1 if not found.
*/

const int
dkim_name_to_code(struct nametable *tbl, const char *name)
{
	int c;

	assert(tbl != NULL);

	for (c = 0; ; c++)
	{
		if (tbl[c].tbl_name == NULL)
			return -1;
		if (strcasecmp(tbl[c].tbl_name, name) == 0)
			return tbl[c].tbl_code;
	}
}

/*
**  DKIM_CODE_TO_CHAR -- translate a mnemonic code to its matching character
**
**  Parameters:
**  	tbl -- name table
**  	code -- code to translate
**
**  Return value:
**  	Character representing the mnemonic code provided, or '\0' if
**  	not found.
*/

const int
dkim_code_to_char(struct chartable *tbl, const int code)
{
	int c;

	assert(tbl != NULL);

	for (c = 0; ; c++)
	{
		if (tbl[c].tbl_code == -1)
			return '\0';
		if (tbl[c].tbl_code == code)
			return tbl[c].tbl_char;
	}
}

/*
**  DKIM_CODE_TO_CHAR -- translate a mnemonic code to its matching character
**
**  Parameters:
**  	tbl -- name table
**  	chr -- character to translate
**
**  Return value:
**  	Mnemonic representing the character code provided, or '\0' if
**  	not found.
*/

const int
dkim_char_to_code(struct chartable *tbl, const int chr)
{
	int c;

	assert(tbl != NULL);

	for (c = 0; ; c++)
	{
		if (tbl[c].tbl_char == chr)
			return tbl[c].tbl_code;
		if (tbl[c].tbl_char == '\0')
			return -1;
	}
}
