/*
**  Copyright (c) 2005 Sendmail, Inc. and its suppliers.
**    All rights reserved.
*/

#ifndef lint
static char util_c_id[] = "@(#)$Id: util.c,v 1.9 2005/12/01 01:06:10 msk Exp $";
#endif /* !lint */

/* system includes */
#include <sys/types.h>
#include <ctype.h>
#include <assert.h>
#include <string.h>

/* libsm includes */
#include <sm/string.h>

/* libdkim includes */
#include "dkim.h"
#include "util.h"

/*
**  DKIM_COLLAPSE -- remove spaces from a string
**
**  Parameters:
**  	str -- string to process
**
**  Return value:
**  	None.
*/

void
dkim_collapse(u_char *str)
{
	u_char *q;
	u_char *r;

	assert(str != NULL);

	for (q = str, r = str; *q != '\0'; q++)
	{
		if (!isspace(*q))
		{
			if (q != r)
				*r = *q;
			r++;
		}
	}

	*r = '\0';
}

/*
**  DKIM_LOWERCASE -- convert a string to lowercase
**
**  Parameters:
**  	str -- string to process
**
**  Return value:
**  	None.
*/

void
dkim_lowercase(u_char *str)
{
	u_char *p;

	assert(str != NULL);

	for (p = str; *p != '\0'; p++)
	{
		if (isascii(*p) && isupper(*p))
			*p = tolower(*p);
	}
}

/*
**  DKIM_ISWSP -- return true iff a character is some kind of whitespace
**
**  Parameters:
**  	c -- character being examined
**
**  Return value:
**  	TRUE iff 'c' refers to something considered whitespace.
*/

bool
dkim_iswsp(u_int c)
{
	return (isascii(c) && isspace(c));
}

/*
**  DKIM_ISLWSP -- return true iff a character is some kind of linear
**                 whitespace
**
**  Parameters:
**  	c -- character being examined
**
**  Return value:
**  	TRUE iff 'c' refers to something considered linear whitespace.
*/

bool
dkim_islwsp(u_int c)
{
	return (isascii(c) && isspace(c) && c != '\n' && c != '\r');
}

/*
**  DKIM_ADDRCMP -- like strcmp(), except for e-mail addresses (i.e. local-part
**                  is case-sensitive, while the domain part is not)
**
**  Parameters:
**  	s1, s2 -- the strings
**
**  Return value:
**  	Like strcmp().
*/

int
dkim_addrcmp(u_char *s1, u_char *s2)
{
	int ret;
	u_char *at1, *at2;

	assert(s1 != NULL);
	assert(s2 != NULL);

	at1 = (u_char *) strchr(s1, '@');
	at2 = (u_char *) strchr(s2, '@');

	if (at1 == NULL || at2 == NULL)
		return strcmp(s1, s2);

	/* first check the local-parts */
	*at1 = '\0';
	*at2 = '\0';

	ret = strcmp(s1, s2);
	if (ret != 0)
	{
		*at1 = '@';
		*at2 = '@';

		return ret;
	}

	/* now check the domains */
	return strcasecmp(at1 + 1, at2 + 1);
}

/*
**  DKIM_HDRLIST -- build up a header list for use in a regexp
**
**  Parameters:
**  	buf -- where to write
**  	buflen -- bytes at "buf"
**  	hdrlist -- array of header names
**  	first -- first call
**
**  Return value:
**  	TRUE iff everything fit.
*/

bool
dkim_hdrlist(u_char *buf, size_t buflen, u_char **hdrlist, bool first)
{
	int c;
	int len;
	char *p;
	char *q;
	char *end;

	assert(buf != NULL);
	assert(hdrlist != NULL);

	for (c = 0; ; c++)
	{
		if (hdrlist[c] == NULL)
			break;

		if (!first)
		{
			len = sm_strlcat(buf, "|", buflen);
			if (len >= buflen)
				return FALSE;
		}
		else
		{
			len = strlen(buf);
		}

		first = FALSE;

		q = &buf[len];
		end = &buf[buflen - 1];

		for (p = hdrlist[c]; *p != '\0'; p++)
		{
			if (q >= end)
				return FALSE;

			switch (*p)
			{
			  case '*':
				*q = '.';
				q++;
				if (q >= end)
					return FALSE;
				*q = '*';
				q++;
				break;

			  case '.':
				*q = '\\';
				q++;
				if (q >= end)
					return FALSE;
				*q = '.';
				q++;
				break;

			  default:
				*q = *p;
				q++;
				break;
			}
		}
	}

	return TRUE;
}

/*
**  DKIM_LOWERHDR -- convert a string (presumably a header) to all lowercase,
**                   but only up to a colon
**
**  Parameters:
**  	str -- string to modify
**
**  Return value:
**  	None.
*/

void
dkim_lowerhdr(unsigned char *str)
{
	unsigned char *p;

	assert(str != NULL);

	for (p = str; *p != '\0'; p++)
	{
		if (*p == ':')
			return;

		if (isascii(*p) && isupper(*p))
			*p = tolower(*p);
	}
}
