#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/mman.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <string.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <glib.h>

#include "var.h"
#include "uaddr.h"
#include "display.h"
#include "action.h"
#include "user_manage.h"
#include "misc.h"
#include "network.h"

#define MAX_UADDR_ENTRIES 10000		/* store up to 10000 couples (nickname, download address) */
#define MAX_REGISTRATION_TIME (12*3600)		/* up to 12 hours without refresh */

/**************************/
/* Global Transfer System */
/*********************************************************************/
/* the following functions provide the capability to move a transfer */
/* from one client to another. Only queued transfers can be moved.   */
/* thus, a transfer can go from one hub to another if the user from  */
/* where we download has gone to another hub. If the .dctc dir is on */
/* a shared disc (NFS or something like that), a transfer can even   */
/* migrate from one computer to another.                             */
/* only /DL are taken into account.                                  */
/*********************************************************************/
#ifdef HAVE_MMAP

static int lock_and_map_uaddr(void);
static inline void unmap_uaddr(void);
static inline void unlock_uaddr(void);
static int increase_ping_counter(const char *addr_dl);

static int uaddr_fd=-1;
G_LOCK_DEFINE_STATIC(uaddr_fd);			/* must be locked before any UADDR operation */

static GString *uaddr_filename=NULL;

static UADDR_ENTRY *mapped_uaddr=NULL;
static int mapped_size;					/* size of UADDR file in memory (in bytes) */
static int nb_mapped_entry;			/* size of UADDR file in memory (in UADDR_ENTRY) */

static unsigned long next_id;

/**************************************************************/
/* scan the list of given address to find associated nickname */
/****************************************************************************/
/* note: string contained in addr_list are destroyed during the computation */
/****************************************************************************/
static void scan_addresses(GPtrArray *addr_list)
{
	int i;
#if 0
	int sock_fd;
#endif

	for(i=0;i<addr_list->len;i++)
	{
		char *host;

		host=g_ptr_array_index(addr_list,i);
		if(host!=NULL)
		{
			char *p;
			unsigned short port;

			p=strchr(host,':');
			if(p==NULL)
			{
				/* invalid address */
				delete_uaddr_entry_by_addr(host);
			}
			else
			{
				*p='\0';
				port=strtoul(p+1,NULL,10);
				
#if 1
				{	/* note: this methode is fastest than the following one but has one severe problem */
					/* if the remote client has "registered" a FQDN inside UADDR database, when this client */
					/* receives the pong string, it contains the remote host IP, not FQDN */
					struct sockaddr_in dest;
					dest.sin_family=AF_INET;
					dest.sin_port=htons(port);
					if(str_to_inaddr(host,&(dest.sin_addr)))
					{
						*p=':';		/* restore original string content */

						/* fail to resolv host name */
						delete_uaddr_entry_by_addr(host);
					}
					else
					{
						int counter;
						*p=':';		/* restore original string content */
						
						counter=increase_ping_counter(host);
						if((counter==-1)||(counter>MAX_UADDR_PING))
							delete_uaddr_entry_by_addr(host);
						else
						{
							const char *ping_str="$Ping |";
							counter=sendto(srch_sck,ping_str,strlen(ping_str),MSG_NOSIGNAL,(void*)&dest,sizeof(struct sockaddr_in));
							if(counter==-1)
							{
								delete_uaddr_entry_by_addr(host);
							}
						}
					}
				}
#else
				/* without the ping command, it is a quite long test */
				/* try to contact the given address */
				sock_fd=create_and_open_sock_on(host,port,0);
				*p=':';		/* restore original string content */
				if(sock_fd<0)
				{
					delete_uaddr_entry_by_addr(host);
				}
				else
				{
					fd_set rd;
					struct timeval tv;
					int nb;
					int have=0;
				
					/* wait at most 10 seconds and read 1 string without blocking */
					FD_ZERO(&rd);
					FD_SET(sock_fd,&rd);
					tv.tv_sec=10;
					tv.tv_usec=0;

					nb=select(sock_fd+1,&rd,NULL,NULL,&tv);
					if(nb>0)
					{
						/* there is something to read */
						char buf[512];

						set_non_bloquant_sock(sock_fd);
						nb=recv(sock_fd,buf,sizeof(buf)-1,MSG_NOSIGNAL);
						if(nb>0)
						{
							/* we have received something */
							buf[nb]='\0';
							if(!strncmp("$MyNick ",buf,8))
							{
								char *eonick;
								eonick=strchr(buf,'|');
								if(eonick!=NULL)
								{
									/* look like we have winner ... */
									*eonick='\0';
									add_uaddr_entry(buf+8 /* beginning of the nickname */, host);
									have=1;
								}
							}
						}
					}
					shutdown(sock_fd,2);
					close(sock_fd);

					if(have==0)
						delete_uaddr_entry_by_addr(host);
				}
#endif
			}
		}
	}
}

/************************************************************************************/
/* this thread purges oldest UADDR entries and tries to fill entry without nickname */
/************************************************************************************/
static void *uaddr_thread(void *dummy)
{
	while(1)
	{
		time_t min_time;
		GStringChunk *gsc=g_string_chunk_new(UADDR_DL);
		GPtrArray *gpa=g_ptr_array_new();

		/* expire oldest entries and get address without nickname */
		if(lock_and_map_uaddr()==0)
		{
			int i;
			min_time=time(NULL)-MAX_REGISTRATION_TIME;

			for(i=1;i<nb_mapped_entry;i++)
			{
				if(mapped_uaddr[i].slot_status==0)		/* empty slot */
					continue;

				if( (mapped_uaddr[i].register_time+mapped_uaddr[i].ping_counter*DELAY_BETWEEN_PING)<min_time)	/* slot too old ? */
				{																																/* each ping try gives a little more time */
					mapped_uaddr[i].slot_status=0;		/* free the slot */
					continue;
				}

				if(strlen(mapped_uaddr[i].nick)==0)		/* no nickname */
				{
					g_ptr_array_add(gpa,g_string_chunk_insert(gsc,mapped_uaddr[i].dl_addr));
				}
			}
		
			unmap_uaddr();		/* unmap file from memory */
			unlock_uaddr();
		}

		if(gpa->len!=0)
		{
			/* there is some addresses without nickname */
			scan_addresses(gpa);
		}

		g_ptr_array_free(gpa,TRUE);
		g_string_chunk_free(gsc);
		sleep(60);
	}
	pthread_exit(NULL);
}

/******************************************************************************/
/* to reduce duplicated code, the DCTC being the clock master is also the one */
/* performing UADDR action                                                    */
/******************************************************************************/
void create_uaddr_thread(void)
{
	pthread_attr_t thread_attr;
	pthread_t thread_id;

	pthread_attr_init (&thread_attr);
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
	if(pthread_create(&thread_id,&thread_attr, (void*)uaddr_thread,NULL))
	{
		disp_msg(ERR_MSG,"create_uaddr_thread","Fatal Error","Fail to start UADDR thread",NULL);
	}
}

/********************************/
/* perform UADDR initialisation */
/**********************************************************************/
/* this function is the only one which doesn't need to lock the UADDR */
/**********************************************************************/
void init_uaddr(void)
{
	next_id=getpid()<<16;				/* uniq ID */

	/* create uaddr filename */
	uaddr_filename=g_string_new(dctc_dir->str);
	g_string_sprintfa(uaddr_filename,"/uaddr.%d",sizeof(UADDR_ENTRY));

	uaddr_fd=open(uaddr_filename->str,O_CREAT|O_RDWR,0666);
	if(uaddr_fd==-1)
	{
		perror("init_uaddr - open fails");
		exit(0);
	}

	if(flock(uaddr_fd,LOCK_EX|LOCK_NB)!=-1)
	{
		/* the file is locked. 2 cases: */
		/* the file is empty and we must create its header */
		/* the file is not empty and is already ready to be used */
		struct stat st;
		
		if(fstat(uaddr_fd,&st)==-1)
		{
			perror("init_uaddr - fstat fails");
			exit(0);
		}

		if(st.st_size==0)
		{
			/* we have created the file, we must build its header */
			UADDR_ENTRY dummy;

			dummy.slot_status=0;
			memset(dummy.nick,'=',UADDR_NICK_SIZE);
			memset(dummy.dl_addr,'+',UADDR_DL);
			dummy.register_time=0;
			dummy.ping_counter=0;
			
			while(write(uaddr_fd,&dummy,sizeof(dummy))!=sizeof(dummy))
			{
				disp_msg(ERR_MSG,"init_uaddr","disk full, unable to initialize UADDR file. Free some space",NULL);
				sleep(1);
				lseek(uaddr_fd,0,SEEK_SET);		/* restart from the beginning */
			}
		}
		else
		{
			/* the file is already ready to be used */
		}

		flock(uaddr_fd,LOCK_UN);		/* unlock the file */
	}
	else
	{
		/* unable to lock the file. Either someone already using it or initializing it */
		/* in both case, the file is ready */
	}
}

/***********************/
/* unlock the GTS file */
/***********************/
static inline void unlock_uaddr(void)
{
	flock(uaddr_fd,LOCK_UN);
	G_UNLOCK(uaddr_fd);
}

/******************************************/
/* lock UADDR file and map it into memory */
/******************************************/
/* output: 1=error, 0=ok */
/*************************/
static int lock_and_map_uaddr(void)
{
	struct stat st;

	if(uaddr_fd==-1)
		return 1;

	G_LOCK(uaddr_fd);
	flock(uaddr_fd,LOCK_EX);

	if(fstat(uaddr_fd,&st)==-1)
	{
		unlock_uaddr();
		return 1;
	}

	mapped_size=st.st_size;
	mapped_uaddr=mmap(NULL,mapped_size,PROT_READ|PROT_WRITE,MAP_SHARED, uaddr_fd, 0);
	if(mapped_uaddr==MAP_FAILED)
	{
		unlock_uaddr();
		return 1;
	}

	nb_mapped_entry=mapped_size/sizeof(UADDR_ENTRY);

	return 0;
}

/**********************************/
/* unmap the GTS file from memory */
/**********************************/
static inline void unmap_uaddr(void)
{
	munmap(mapped_uaddr,mapped_size);
	mapped_uaddr=NULL;
}

/*************************************************************/
/* the following function removes an entry to the UADDR file */
/*************************************************************/
void delete_uaddr_entry_by_name(const char *nickname)
{
	int i;

	if(lock_and_map_uaddr())
		return;

	for(i=1;i<nb_mapped_entry;i++)
	{
		if(mapped_uaddr[i].slot_status==0)		/* empty slot */
			continue;

		if(!strcmp(nickname,mapped_uaddr[i].nick))
		{
			mapped_uaddr[i].slot_status=0;		/* the slot is now empty */
			break;
		}
	}

	unmap_uaddr();		/* unmap file from memory */
	unlock_uaddr();
}

/*************************************************************/
/* the following function removes an entry to the UADDR file */
/*************************************************************/
void delete_uaddr_entry_by_addr(const char *dl_addr)
{
	int i;

	if(lock_and_map_uaddr())
		return;

	for(i=1;i<nb_mapped_entry;i++)
	{
		if(mapped_uaddr[i].slot_status==0)		/* empty slot */
			continue;

		if	(!strcmp(dl_addr,mapped_uaddr[i].dl_addr))
		{
			mapped_uaddr[i].slot_status=0;		/* the slot is now empty */
			break;
		}
	}

	unmap_uaddr();		/* unmap file from memory */
	unlock_uaddr();
}

/**********************************************************/
/* check if the given nickname has a known remote address */
/**********************************************************/
/* output: 1=yes, 0=no */
/***********************/
int check_uaddr_entry_by_name(const char *nickname)
{
	int i;
	int ret=0;

	if(lock_and_map_uaddr())
		return ret;

	for(i=1;i<nb_mapped_entry;i++)
	{
		if(mapped_uaddr[i].slot_status==0)		/* empty slot */
			continue;

		if(!strcmp(nickname,mapped_uaddr[i].nick))
		{
			ret=1;
			break;
		}
	}

	unmap_uaddr();		/* unmap file from memory */
	unlock_uaddr();
	return ret;
}

/************************************************/
/* get the remote address of the given nickname */
/*********************************************************************************/
/* output: NULL (not found) or a GString (host:port) to free when no more useful */
/*********************************************************************************/
GString *get_uaddr_dl_addr_by_name(const char *nickname)
{
	int i;
	GString *ret=NULL;

	if(lock_and_map_uaddr())
		return ret;

	for(i=1;i<nb_mapped_entry;i++)
	{
		if(mapped_uaddr[i].slot_status==0)		/* empty slot */
			continue;

		if(!strcmp(nickname,mapped_uaddr[i].nick))
		{
			ret=g_string_new(mapped_uaddr[i].dl_addr);
			break;
		}
	}

	unmap_uaddr();		/* unmap file from memory */
	unlock_uaddr();
	return ret;
}

/**********************************************************/
/* the following function adds an entry to the UADDR file */
/**********************************************************/
/* nick is always uniq in the database */
/***************************************/
/* output: 0= ok    */
/*         1= error */
/********************/
int add_uaddr_entry(const char *nickname, const char *addr_dl)
{
	int i;
	int fnd=0;
	int ret;
	int empty_entry=-1;

	if(lock_and_map_uaddr())
		return 1;		/* error */

	/* UADDR file is in memory */
	/* we just have to scan it from the 2nd entry (the first is the header) */
	for(i=1;i<nb_mapped_entry;i++)
	{
		if(mapped_uaddr[i].slot_status==0)		/* empty slot */
		{
			if(empty_entry==-1)
				empty_entry=i;
			continue;
		}

		if((!strcmp(nickname,mapped_uaddr[i].nick)) ||
			(!strcmp(addr_dl,mapped_uaddr[i].dl_addr)) )
		{
			fnd=i;
			break;
		}
	}

	if(fnd)
	{
		/* still here, nothing to do except updating the registration time */
		mapped_uaddr[fnd].register_time=time(NULL);
		mapped_uaddr[fnd].ping_counter=0;
		strncpy_max(mapped_uaddr[fnd].dl_addr,addr_dl,UADDR_DL);
		strncpy_max(mapped_uaddr[fnd].nick,nickname,UADDR_NICK_SIZE);

		/* we must also purge all entries with the same nickname or the same addr_dl */
		for(i=1;i<nb_mapped_entry;i++)
		{
			if(i==fnd)
				continue;

			if((!strcmp(nickname,mapped_uaddr[i].nick)) ||
				(!strcmp(addr_dl,mapped_uaddr[i].dl_addr)) )
			{	/* same nick or same address ? */
				mapped_uaddr[i].slot_status=0;
			}
		}

		unmap_uaddr();
		ret=0;		/* ok */
	}
	else
	{
		/* no entry exists, we must create one */
		UADDR_ENTRY nw;

		/* the entry to add */
		nw.slot_status=1;											/* slot is busy */
		next_id++;
		strncpy_max(nw.nick,nickname,UADDR_NICK_SIZE);
		strncpy_max(nw.dl_addr,addr_dl,UADDR_DL);
		nw.register_time=time(NULL);
		nw.ping_counter=0;
		
		if(empty_entry==-1)
		{
			/* no empty entry inside UADDR file, must increase file size */
			unmap_uaddr();		/* unmap file from memory */
	
			if(nb_mapped_entry<MAX_UADDR_ENTRIES)
			{
				if(lseek(uaddr_fd,nb_mapped_entry*sizeof(UADDR_ENTRY),SEEK_SET)==(nb_mapped_entry*sizeof(UADDR_ENTRY)))
				{
					/* after moving to the right place */
					/* write the entry */
					if(write(uaddr_fd,&nw,sizeof(UADDR_ENTRY))!=sizeof(UADDR_ENTRY))
						ret=1;		/* error */
					else
						ret=0;		/* ok */
				}
				else
				{
					ret=1;	/* error */
				}
			}
			else
			{
				ret=1;	/* error */
			}
		}
		else
		{
			/* an empty entry exists, use it */
			memcpy(&mapped_uaddr[empty_entry],&nw,sizeof(UADDR_ENTRY));
			unmap_uaddr();		/* unmap file from memory */
			ret=0;		/* ok */
		}
	}

	unlock_uaddr();
	return ret;
}

/**********************************************************/
/* the following function adds an entry to the UADDR file */
/**********************************************************/
/* addr_dl is always uniq in the database */
/******************************************/
/* output: 0= ok    */
/*         1= error */
/********************/
int add_uaddr_entry_addr_dl_only(const char *addr_dl)
{
	int i;
	int fnd=0;
	int ret;
	int empty_entry=-1;

	if(lock_and_map_uaddr())
		return 1;		/* error */

	/* UADDR file is in memory */
	/* we just have to scan it from the 2nd entry (the first is the header) */
	for(i=1;i<nb_mapped_entry;i++)
	{
		if(mapped_uaddr[i].slot_status==0)		/* empty slot */
		{
			if(empty_entry==-1)
				empty_entry=i;
			continue;
		}

		if(!strcmp(addr_dl,mapped_uaddr[i].dl_addr))
		{
			fnd=i;
			break;
		}
	}

	if(fnd)
	{
		/* still here, nothing to do except updating the registration time */
		mapped_uaddr[fnd].register_time=time(NULL);
		mapped_uaddr[fnd].ping_counter=0;

		unmap_uaddr();
		ret=0;		/* ok */
	}
	else
	{
		/* no entry exists, we must create one */
		UADDR_ENTRY nw;

		/* the entry to add */
		nw.slot_status=1;											/* slot is busy */
		next_id++;
		memset(nw.nick,0,UADDR_NICK_SIZE);
		strncpy_max(nw.dl_addr,addr_dl,UADDR_DL);
		nw.register_time=time(NULL);
		nw.ping_counter=0;
		
		if(empty_entry==-1)
		{
			/* no empty entry inside UADDR file, must increase file size */

			unmap_uaddr();		/* unmap file from memory */
	
			if(lseek(uaddr_fd,nb_mapped_entry*sizeof(UADDR_ENTRY),SEEK_SET)==(nb_mapped_entry*sizeof(UADDR_ENTRY)))
			{
				/* after moving to the right place */
				/* write the entry */
				if(write(uaddr_fd,&nw,sizeof(UADDR_ENTRY))!=sizeof(UADDR_ENTRY))
					ret=1;		/* error */
				else
					ret=0;		/* ok */
			}
			else
			{
				ret=1;	/* error */
			}
		}
		else
		{
			/* an empty entry exists, use it */
			memcpy(&mapped_uaddr[empty_entry],&nw,sizeof(UADDR_ENTRY));
			unmap_uaddr();		/* unmap file from memory */
			ret=0;		/* ok */
		}
	}

	unlock_uaddr();
	return ret;
}

/*************************************************************************************************/
/* the following function increases the ping_counter of the UADDR entry having the given address */
/*************************************************************************************************/
/* addr_dl is always uniq in the database */
/******************************************/
/* output: -1= error                 */
/*         new value of ping counter */
/*************************************/
static int increase_ping_counter(const char *addr_dl)
{
	int i;
	int fnd=0;
	int ret;

	if(lock_and_map_uaddr())
		return -1;		/* error */

	/* UADDR file is in memory */
	/* we just have to scan it from the 2nd entry (the first is the header) */
	for(i=1;i<nb_mapped_entry;i++)
	{
		if(mapped_uaddr[i].slot_status==0)		/* empty slot */
			continue;

		if(!strcmp(addr_dl,mapped_uaddr[i].dl_addr))
		{
			fnd=i;
			break;
		}
	}

	if(fnd)
	{
		/* still here, nothing to do except updating the registration time */
		mapped_uaddr[fnd].register_time=time(NULL);
		mapped_uaddr[fnd].ping_counter++;

		ret=mapped_uaddr[fnd].ping_counter;		/* ok */
	}
	else
	{
		ret=-1;		/* error */
	}

	unmap_uaddr();		/* unmap file from memory */
	unlock_uaddr();
	return ret;
}


/*************/
/* end UADDR */
/*************/
void exit_uaddr(void)
{
	if(uaddr_filename!=NULL)
	{
		g_string_free(uaddr_filename,TRUE);
		uaddr_filename=NULL;
	}

	if(uaddr_fd!=-1)
	{
		close(uaddr_fd);
		uaddr_fd=-1;
	}
}

/*******************************************/
/* output UADDR content into CMD_KB format */
/*******************************************/
void list_uaddr_content(void)
{
	int i;
	char tmp[512];

	if(lock_and_map_uaddr())
		return;		/* error */
	
	for(i=1;i<nb_mapped_entry;i++)
	{
		if(mapped_uaddr[i].slot_status==0)		/* empty slot */
			continue;

		sprintf(tmp,"%lu",mapped_uaddr[i].register_time);
		disp_msg(CMD_KB,NULL,tmp,mapped_uaddr[i].dl_addr,NULL);
	}

	unmap_uaddr();
	unlock_uaddr();
}

/*****************************************************************************/
/* this function scans the UADDR files and looks if there is something to do */
/*****************************************************************************/
/* this function must be called regularly */
/******************************************/
void uaddr_action(void)
{
}

#else
/* ============================================================================ */
/* ============================================================================ */
/* ============================================================================ */
/* ============================================================================ */
/* ============================================================================ */
/* ============================================================================ */
/* ============================================================================ */
/* ============================================================================ */
/* ============================================================================ */
/* for OS without mmap, dummy functions are provided */


/******************************/
/* perform UADDR initialisation */
/******************************/
void init_uaddr(void)
{
}

/***********/
/* end UADDR */
/***********/
void exit_uaddr(void)
{
}


/********************************************************/
/* the following function adds an entry to the UADDR file */
/**********************************************************************************************************************/
/* the following cases can appear:                                                                                    */
/* no entry having the same (nick/filename/dlpath) exists and it is created.                                          */
/* an entry having the same (nick/filename/dlpath) exists, nothing is done (we don't add anything else).              */
/**********************************************************************************************************************/
/* output: 0= ok    */
/*         1= error */
/********************/
int add_uaddr_entry(const char *nickname, const char *dl_addr)
{
	return 1;
}

/**********************************************************/
/* the following function adds an entry to the UADDR file */
/**********************************************************/
/* addr_dl is always uniq in the database */
/******************************************/
/* output: 0= ok    */
/*         1= error */
/********************/
int add_uaddr_entry_addr_dl_only(const char *addr_dl)
{
	return 1;
}

/***********************************************************/
/* the following function removes an entry to the UADDR file */
/***********************************************************/
void delete_uaddr_entry_by_name(const char *nickname)
{
}

void delete_uaddr_entry_by_addr(const char *dl_addr)
{
}

/*****************************************/
/* output UADDR content into CMD_KB format */
/*****************************************/
void list_uaddr_content(void)
{
}

/******************************************************************************/
/* to reduce duplicated code, the DCTC being the clock master is also the one */
/* performing UADDR action                                                    */
/******************************************************************************/
void create_uaddr_thread(void)
{
}

/*****************************************************************************/
/* this function scans the UADDR files and looks if there is something to do */
/*****************************************************************************/
/* this function must be called regularly */
/******************************************/
void uaddr_action(void)
{
}

/**********************************************************/
/* check if the given nickname has a known remote address */
/**********************************************************/
/* output: 1=yes, 0=no */
/***********************/
int check_uaddr_entry_by_name(const char *nickname)
{
	return 0;
}

/************************************************/
/* get the remote address of the given nickname */
/*********************************************************************************/
/* output: NULL (not found) or a GString (host:port) to free when no more useful */
/*********************************************************************************/
GString *get_uaddr_dl_addr_by_name(const char *nickname)
{
	return NULL;
}

#endif

