/* darkstat 3
 * copyright (c) 2006 Emil Mikulic.
 *
 * graph_db.c: round robin database for graph data
 *
 * You may use, modify and redistribute this file under the terms of the
 * GNU General Public License version 2. (see COPYING.GPL)
 */

#include "conv.h"
#include "darkstat.h"
#include "acct.h"
#include "err.h"
#include "str.h"
#include "html.h" /* FIXME: should be pushed into a .c file? */
#include "graph_db.h"

#include <assert.h>
#include <stdlib.h>
#include <string.h> /* for memcpy() */
#include <time.h>

#define GRAPH_WIDTH "320"
#define GRAPH_HEIGHT "200"

struct graph {
   uint64_t *in, *out;
   unsigned int offset; /* i.e. seconds start at 0, days start at 1 */
   unsigned int pos, num_bars;
   const char *unit;
};

static struct graph
   graph_secs = {NULL, NULL, 0, 0, 60, "seconds"},
   graph_mins = {NULL, NULL, 0, 0, 60, "minutes"},
   graph_hrs  = {NULL, NULL, 0, 0, 24, "hours"},
   graph_days = {NULL, NULL, 1, 0, 31, "days"};

static struct graph *graph_db[] = {
   &graph_secs, &graph_mins, &graph_hrs, &graph_days
};

static unsigned int graph_db_size = sizeof(graph_db)/sizeof(*graph_db);

void
graph_init(void)
{
   unsigned int i;
   for (i=0; i<graph_db_size; i++) {
      graph_db[i]->in  = xcalloc(sizeof(uint64_t), graph_db[i]->num_bars);
      graph_db[i]->out = xcalloc(sizeof(uint64_t), graph_db[i]->num_bars);
   }
}

void
graph_free(void)
{
   unsigned int i;
   for (i=0; i<graph_db_size; i++) {
      free(graph_db[i]->in);
      free(graph_db[i]->out);
   }
}

void
graph_acct(uint64_t amount, enum graph_dir dir)
{
   unsigned int i;
   for (i=0; i<graph_db_size; i++)
   switch (dir) {
   case GRAPH_IN:
      graph_db[i]->in[ graph_db[i]->pos ] += amount;
      break;
   case GRAPH_OUT:
      graph_db[i]->out[ graph_db[i]->pos ] += amount;
      break;
   default:
      errx(1, "unknown graph_dir in graph_acct: %d", dir);
   }
}

static time_t last_time = 0;

/* Advance a graph: advance the pos, zeroing out bars as we move. */
static void
advance(struct graph *g, const unsigned int pos)
{
   if (g->pos == pos)
      return; /* didn't need to advance */
   do {
      g->pos = (g->pos + 1) % g->num_bars;
      g->in[g->pos] = g->out[g->pos] = 0;
   } while (g->pos != pos);
}

/* Rotate a graph: rotate all bars so that the bar at the current pos is moved
 * to the newly given pos.  This is non-destructive. */
static void
rotate(struct graph *g, const unsigned int pos)
{
   uint64_t *tmp;
   unsigned int i, ofs;
   size_t size;

   if (pos == g->pos)
      return; /* nothing to rotate */

   size = sizeof(*tmp) * g->num_bars;
   tmp = xmalloc(size);
   ofs = g->num_bars + pos - g->pos;

   for (i=0; i<g->num_bars; i++)
      tmp[ (i+ofs) % g->num_bars ] = g->in[i];
   memcpy(g->in, tmp, size);

   for (i=0; i<g->num_bars; i++)
      tmp[ (i+ofs) % g->num_bars ] = g->out[i];
   memcpy(g->out, tmp, size);

   free(tmp);
   assert(pos == ( (g->pos + ofs) % g->num_bars ));
   g->pos = pos;
}

static void
graph_resync(const time_t new_time)
{
   struct tm *tm;
   /*
    * If time went backwards, we assume that real time is continuous and that
    * the time adjustment should only affect display.  i.e., if we have:
    *
    * second 15: 12  bytes
    * second 16: 345 bytes
    * second 17: <-- current pos
    *
    * and time goes backwards to second 8, we will shift the graph around to
    * get:
    *
    * second 6: 12  bytes
    * second 7: 345 bytes
    * second 8: <-- current pos
    *
    * Note that we don't make any corrections for time being stepped forward.
    * We rely on graph advancement to happen at the correct real time to
    * account for, for example, bandwidth used per day.
    */
   assert(new_time < last_time);

   tm = localtime(&new_time);
   if (tm->tm_sec == 60)
      tm->tm_sec = 59; /* mis-handle leap seconds */

   rotate(&graph_secs, tm->tm_sec);
   rotate(&graph_mins, tm->tm_min);
   rotate(&graph_hrs, tm->tm_hour);
   rotate(&graph_days, tm->tm_mday - 1);

   last_time = new_time;
}

void
graph_rotate(void)
{
   time_t t;
   struct tm *tm;

   t = time(NULL);

   if (last_time == 0) {
      verbosef("first rotate");
      last_time = t;
      tm = localtime(&t);
      if (tm->tm_sec == 60)
         tm->tm_sec = 59; /* mis-handle leap seconds */

      graph_secs.pos = tm->tm_sec;
      graph_mins.pos = tm->tm_min;
      graph_hrs.pos = tm->tm_hour;
      graph_days.pos = tm->tm_mday - 1;
      return;
   }

   if (t == last_time)
      return; /* superfluous rotate */

   if (t < last_time) {
      verbosef("time went backwards! (from %u to %u, offset is %d)",
         (unsigned int)last_time, (unsigned int)t, (int)(t - last_time));
      graph_resync(t);
      return;
   }

   /* else, normal rotation */
   last_time = t;
   tm = localtime(&t);
   if (tm->tm_sec == 60)
      tm->tm_sec = 59; /* mis-handle leap seconds */
   advance(&graph_secs, tm->tm_sec);
   advance(&graph_mins, tm->tm_min);
   advance(&graph_hrs, tm->tm_hour);
   advance(&graph_days, tm->tm_mday - 1);
}

/* ---------------------------------------------------------------------------
 * Web interface: front page!
 */
struct str *
html_front_page(void)
{
   struct str *buf;
   unsigned int i;

   buf = str_make();
   str_append(buf, html_header_1);
   str_append(buf, "<title>" PACKAGE_STRING " : graphs</title>\n");
   str_append(buf, "<script src=\"graph.js\" type=\"text/javascript\">"
                   "</script>\n");
   str_append(buf, html_header_2);
   str_append(buf, "<h2 class=\"pageheader\">Graphs</h2>\n");

   str_appendf(buf,
      "<p>\n"
       "<b>Total packets:</b> <span id=\"tp\">%'qu</span><br/>\n"
       "<b>Total bytes:</b> <span id=\"tb\">%'qu</span><br/>\n"
      "</p>\n",
      total_packets, total_bytes);

   str_append(buf, "<div>\n");
   for (i=0; i<graph_db_size; i++)
      str_appendf(buf,"<div class=\"outergraph\" id=\"g%d\">%s</div>\n",
         i, (i==0)?"Graphs require JavaScript.":"");
   str_append(buf,
      "<div style=\"clear:both\"></div>\n"
      "<div id=\"graph_buttons\"></div>\n" /* this gets filled out by JS */
      "</div>\n");

   str_append(buf,
      "<script type=\"text/javascript\">\n"
      "//<![CDATA[\n"
      "var graph_width = " GRAPH_WIDTH ";\n"
      "var graph_height = " GRAPH_HEIGHT ";\n"
      "var bar_gap = 1;\n"
      "\n"
      "var graphs_uri = \"/graphs.xml\";\n"
      "var graphs = [\n"
   );

   for (i=0; i<graph_db_size; i++)
      str_appendf(buf,
         " {id:\"g%d\", name:\"%s\", title:\"last %d %s\"}%s\n",
         i, graph_db[i]->unit, graph_db[i]->num_bars, graph_db[i]->unit,
         (i < graph_db_size-1) ? "," : "");
      /* trailing comma breaks IE, it makes the array one element bigger */

   str_append(buf,
      "];\n"
      "\n"
      "window.onload = graph_loader;\n"
      "//]]>\n"
      "</script>\n"
   );

   str_append(buf, html_footer);
   return (buf);
}

/* ---------------------------------------------------------------------------
 * Web interface: graphs.xml
 */
struct str *
xml_graphs(void)
{
   unsigned int i, j;
   struct str *buf = str_make();

   str_appendf(buf, "<graphs tp=\"%'qu\" tb=\"%'qu\">\n",
      total_packets, total_bytes);

   for (i=0; i<graph_db_size; i++) {
      const struct graph *g = graph_db[i];

      str_appendf(buf, "<%s>\n", g->unit);
      j = g->pos;
      do {
         j = (j + 1) % g->num_bars;
         /* <element pos="" in="" out=""/> */
         str_appendf(buf, "<e p=\"%u\" i=\"%qu\" o=\"%qu\"/>\n",
            g->offset + j, g->in[j], g->out[j]);
      } while (j != g->pos);
      str_appendf(buf, "</%s>\n", g->unit);
   }
   str_append(buf, "</graphs>\n");
   return (buf);
}

/* vim:set ts=3 sw=3 tw=78 expandtab: */
