/* darkstat 3
 * copyright (c) 2001-2006 Emil Mikulic.
 *
 * darkstat.c: program body.
 *
 * You may use, modify and redistribute this file under the terms of the
 * GNU General Public License version 2. (see COPYING.GPL)
 */

#include "darkstat.h"
#include "acct.h"
#include "cap.h"
#include "conv.h"
#include "dns.h"
#include "graph_db.h"
#include "http.h"
#include "hosts_db.h"
#include "localip.h"
#include "ncache.h"

#include "err.h"
#include <arpa/inet.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pcap.h>

static int running = 1;

static void
sig_shutdown(int signum _unused_)
{
   running = 0;
}

static unsigned short
parse_port(const char *str)
{
   unsigned long p;
   char *ep;

   errno = 0;
   p = strtoul(str, &ep, 10);
   if (*ep != '\0')
      errx(1, "\"%s\" is not a valid number", str);
   if (errno == ERANGE)
      errx(1, "\"%s\" is out of range", str);
   return ((unsigned short)p);
}

static void
usage(const char *arg0)
{
   printf("usage: %s [-d] -i interface [-p port] [-b bindaddr] [-f filter]\n",
      arg0);
   exit(EXIT_FAILURE);
}

int
main(int argc, char **argv)
{
   /* commandline parameters */
   in_addr_t bindaddr = INADDR_ANY;
   unsigned short bindport = 667;
   int maxconn = -1;
   const char *interface = NULL;
   const char *filter = NULL;
   int ch, want_daemonize = 0;

   printf(PACKAGE_STRING " using libpcap %d.%d\n",
      PCAP_VERSION_MAJOR, PCAP_VERSION_MINOR);

   if (close(STDIN_FILENO) == -1)
      warn("close(stdin)");

   while ((ch = getopt(argc, argv, "di:p:b:f:")) != -1) {
   switch (ch) {
   case 'd':
      want_daemonize = 1;
      break;
   case 'i':
      interface = optarg;
      break;
   case 'p':
      bindport = parse_port(optarg);
      break;
   case 'b':
      bindaddr = inet_addr(optarg);
#ifndef INADDR_NONE
# define INADDR_NONE (-1) /* Solaris */
#endif
      if (bindaddr == (in_addr_t)INADDR_NONE)
         errx(1, "malformed address \"%s\"", optarg);
      break;
   case 'f':
      filter = optarg;
      break;
   default:
      usage(argv[0]);
   }}

   if (interface == NULL) {
      warnx("must specify interface");
      usage(argv[0]);
   }

   if (want_daemonize)
      daemonize_start();

   dns_init(); /* do this first as it forks - minimize memory use */

   /* Need root privs for these: */
   cap_init(interface, filter);
   http_init(bindaddr, bindport, maxconn);
   ncache_init(); /* don't need priv, but must do before chroot */

   privdrop(1 /* and chroot */);

   /* Don't need root privs for these: */
   graph_init();
   hosts_db_init();
   localip_init(interface);

   if (signal(SIGTERM, sig_shutdown) == SIG_ERR)
      errx(1, "signal() failed");
   if (signal(SIGINT, sig_shutdown) == SIG_ERR)
      errx(1, "signal() failed");

   verbosef("entering main loop");
   daemonize_finish();

   while (running) {
      int select_ret, max_fd = -1, use_timeout = 0;
      struct timeval timeout;
      fd_set rs, ws;

      FD_ZERO(&rs);
      FD_ZERO(&ws);

      cap_fd_set(&rs, &max_fd, &timeout, &use_timeout);
      http_fd_set(&rs, &ws, &max_fd, &timeout, &use_timeout);

      select_ret = select(max_fd+1, &rs, &ws, NULL,
         (use_timeout) ? &timeout : NULL);

      if ((select_ret == 0) && (!use_timeout))
            errx(1, "select() erroneously timed out");

      if (select_ret == -1) {
         if (errno == EINTR)
            continue;
         else
            err(1, "select()");
      }
      else {
         hosts_db_reduce();
         graph_rotate();
         cap_poll(&rs);
         dns_poll();
         http_poll(&rs, &ws);
      }
   }

   verbosef("shutting down");
   cap_stop();
   dns_stop();
   hosts_db_free();
   graph_free();
   ncache_free();
   verbosef("shut down");
   return (EXIT_SUCCESS);
}

/* vim:set ts=3 sw=3 tw=78 expandtab: */
