%  Copyright (C) 2003,2005 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; see the file COPYING.  If not, write to
%  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
%  Boston, MA 02110-1301, USA.

\subsection{darcs repair}
\begin{code}
module Darcs.Commands.Repair ( repair ) where
import Workaround ( getCurrentDirectory, createDirectoryIfMissing )
import System.IO
import System.Exit ( exitWith, ExitCode(..) )
import Control.Monad ( when, unless )
import Data.Maybe ( catMaybes )

import Darcs.Commands
import Darcs.Arguments ( DarcsFlag( Verbose, Quiet ),
                        working_repo_dir, umask_option,
                      )
import Darcs.Patch ( RepoPatch, patch2patchinfo )
import Darcs.Patch.Patchy ( applyAndTryToFix )
import Darcs.Patch.Info ( human_friendly )
import Darcs.Ordered ( FL(..), RL(..), lengthFL, reverseFL, reverseRL, concatRL )
import Darcs.Hopefully ( PatchInfoAnd, info )
import Darcs.Repository.Cache ( Cache, HashedDir( HashedPristineDir ) )
import Darcs.Repository ( Repository, withRepoLock, ($-), amInRepository, read_repo,
                          writePatchSet, makePatchLazy,
                          replacePristineFromSlurpy,
                          checkPristineAgainstSlurpy )
import Darcs.Repository.Format ( identifyRepoFormat, 
                                 RepoProperty ( HashedInventory ), format_has )
import Darcs.Repository.Checkpoint ( get_checkpoint_by_default )
import Darcs.Repository.InternalTypes ( extractCache )
import Darcs.Repository.HashedIO ( slurpHashedPristine, writeHashedPristine,
                                   clean_hashdir )
import Darcs.SlurpDirectory ( empty_slurpy, withSlurpy, Slurpy, SlurpMonad )
import Darcs.Repository.HashedRepo ( readHashedPristineRoot )
import Darcs.Global ( darcsdir )
import Darcs.Flags ( compression )
import Darcs.Progress ( debugMessage, beginTedious, endTedious, tediousSize, finishedOneIO )
import Darcs.Patch.Depends ( get_patches_beyond_tag )
import Darcs.Lock( rm_recursive )
import Darcs.Commands.Check ( check_uniqueness )
import Darcs.Sealed ( Sealed(..), unsafeUnflippedseal )
import Darcs.Utils ( catchall, withCurrentDirectory )
import Printer ( putDocLn, text )
\end{code}

\options{repair}
\begin{code}
repair_description :: String
repair_description = "Repair the corrupted repository."
\end{code}
\haskell{repair_help}

\begin{code}
repair_help :: String
repair_help =
 "Repair attempts to fix corruption that may have entered your\n"++
 "repository.\n"
\end{code}

\begin{code}
repair :: DarcsCommand
repair = DarcsCommand {command_name = "repair",
                       command_help = repair_help,
                       command_description = repair_description,
                       command_extra_args = 0,
                       command_extra_arg_help = [],
                       command_command = repair_cmd,
                       command_prereq = amInRepository,
                       command_get_arg_possibilities = return [],
                       command_argdefaults = nodefaults,
                       command_advanced_options = [umask_option],
                       command_basic_options = [working_repo_dir]}
\end{code}

Repair currently will only repair damage to the pristine tree.
Fortunately this is just the sort of corruption that is most
likely to happen.

\begin{code}

run_slurpy :: Slurpy -> SlurpMonad a -> IO (Slurpy, a)
run_slurpy s f =
    case withSlurpy s f of
      Left err -> fail err
      Right x -> return x

update_slurpy :: Repository p -> Cache -> [DarcsFlag] -> Slurpy -> IO Slurpy
update_slurpy r c opts s = do
  current <- readHashedPristineRoot r
  h <- writeHashedPristine c (compression opts) s
  s' <- slurpHashedPristine c (compression opts) h
  clean_hashdir c HashedPristineDir $ catMaybes [Just h, current]
  return s'

repair_cmd :: [DarcsFlag] -> [String] -> IO ()
repair_cmd opts _ = withRepoLock opts $- \repository -> do
  let putVerbose s = when (Verbose `elem` opts) $ putDocLn s
      putInfo s = when (not $ Quiet `elem` opts) $ putDocLn s
  check_uniqueness putVerbose putInfo repository
  maybe_chk <- get_checkpoint_by_default repository
  formerdir <- getCurrentDirectory
  let c = extractCache repository
  rf_or_e <- identifyRepoFormat "."
  rf <- case rf_or_e of Left e -> fail e
                        Right x -> return x
  createDirectoryIfMissing False $ darcsdir ++ "/pristine.hashed"
  rooth <- writeHashedPristine c (compression opts) empty_slurpy
  s <- slurpHashedPristine c (compression opts) rooth
  putVerbose $ text "Applying patches..."
  s' <- case maybe_chk of
    Just (Sealed chk) ->
        do let chtg = patch2patchinfo chk
           putVerbose $ text "I am repairing from a checkpoint."
           patches <- read_repo repository
           (s'', _) <- run_slurpy s $ applyAndTryToFix chk
           (_, s_) <- applyAndFix c opts s'' repository
                      (reverseRL $ concatRL $ unsafeUnflippedseal $ get_patches_beyond_tag chtg patches)
           return s_
    Nothing -> do debugMessage "Fixing any broken patches..."
                  rawpatches <- read_repo repository
                  let psin = reverseRL $ concatRL rawpatches
                  (ps, s_) <- applyAndFix c opts s repository psin
                  withCurrentDirectory formerdir $
                                       writePatchSet (reverseFL ps :<: NilRL) opts
                  debugMessage "Done fixing broken patches..."
                  return s_
  debugMessage "Checking pristine agains slurpy"
  is_same <- checkPristineAgainstSlurpy repository s' `catchall` return False
  if is_same
      then putStrLn "The repository is already consistent, no changes made."
      else do putStrLn "Fixing pristine tree..."
              replacePristineFromSlurpy repository s'
  unless (format_has HashedInventory rf) $
         rm_recursive $ darcsdir ++ "/pristine.hashed" 
  exitWith ExitSuccess

applyAndFix :: RepoPatch p => Cache -> [DarcsFlag] -> Slurpy -> Repository p -> FL (PatchInfoAnd p) -> IO (FL (PatchInfoAnd p), Slurpy)
applyAndFix _ _ s _ NilFL = return (NilFL, s)
applyAndFix c opts s_ r psin =
    do beginTedious k
       tediousSize k $ lengthFL psin
       ps <- aaf 0 s_ psin
       endTedious k
       return ps
    where k = "Repairing patch"
          aaf _ s NilFL = return (NilFL, s)
          aaf i s (p:>:ps) = do
            (s', mp') <- run_slurpy s $ applyAndTryToFix p
            finishedOneIO k $ show $ human_friendly $ info p
            p' <- case mp' of
                    Nothing -> return p
                    Just (e,pp) -> do putStrLn e
                                      return pp
            p'' <- makePatchLazy r p'
            let j = if ((i::Int) + 1 < 100) then i + 1 else 0
            (ps', s'') <- aaf j s' ps
            s''' <- if j == 0 then update_slurpy r c opts s''
                      else return s''
            return ((p'':>:ps'), s''')

\end{code}
