\begin{code}
module RemoteApply ( remote_apply, apply_as ) where

import DarcsFlags ( DarcsFlag( ApplyAs ) )
import DarcsURL ( is_url, is_ssh )
import External
import Printer
\end{code}

This module is used by the push and put commands to apply the a bundle to a
remote repository. By remote I do not necessarily mean a repository on another
machine, it is just not the repository we're located in.

\begin{code}
remote_apply :: [DarcsFlag] -> String -> Doc -> IO Doc
remote_apply opts repodir bundle 
    = case apply_as opts of
        Nothing -> if is_ssh repodir
                   then apply_via_ssh repodir bundle
                   else if is_url repodir
                        then apply_via_url repodir bundle
                        else apply_via_local repodir bundle
        Just un -> if is_ssh repodir
                   then apply_via_ssh_and_sudo repodir un bundle
                   else apply_via_sudo un repodir bundle

apply_as :: [DarcsFlag] -> Maybe String
apply_as (ApplyAs user:_) = Just user
apply_as (_:fs) = apply_as fs
apply_as [] = Nothing
apply_via_sudo :: String -> String -> Doc -> IO Doc
apply_via_sudo user repo bundle =
    execPipeIgnoreError "sudo"
        ["-u",user,"darcs","apply","--all","--repodir",repo] bundle
apply_via_local :: String -> Doc -> IO Doc
apply_via_local repo bundle =
    execPipeIgnoreError "darcs" ["apply","--all","--repodir",repo] bundle

apply_via_url :: String -> Doc -> IO Doc
apply_via_url repo bundle =
    do maybeapply <- maybeURLCmd "APPLY" repo
       case maybeapply of
         Nothing -> apply_via_local repo bundle
         Just apply ->
           do let cmd = head $ words apply
                  args = tail $ words apply
              execPipeIgnoreError cmd (args ++ [repo]) bundle

apply_via_ssh :: String -> Doc -> IO Doc
apply_via_ssh repo bundle =
    pipeDoc_SSH_IgnoreError [addr,"cd '"++path++"' && darcs apply --all"] bundle
        where (addr,':':path) = break (==':') repo

apply_via_ssh_and_sudo :: String -> String -> Doc -> IO Doc
apply_via_ssh_and_sudo repo username bundle =
    pipeDoc_SSH_IgnoreError [addr,"sudo -u "++username++
                         " darcs apply --all --repodir '"++path++"'"] bundle
        where (addr,':':path) = break (==':') repo
\end{code}
