%  Copyright (C) 2002-2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\subsection{darcs push}
\begin{code}
module Push ( push ) where
import System ( exitWith, ExitCode( ExitSuccess ) )
import Monad ( when )
import Char ( toUpper )
import DarcsCommands ( DarcsCommand(..) )
import DarcsArguments ( DarcsFlag( DryRun, Verbose, Quiet ),
                        working_repo_dir, summary,
                        print_dry_run_message_and_exit,
                        applyas, match_several, fix_filepath,
                        all_gui_interactive, dry_run,
                        any_verbosity, set_default,
                      )
import Repository ( withRepoLock, slurp_recorded )
import DarcsRepo ( read_repo, am_in_repo, absolute_dir )
import PatchInfo ( human_friendly )
import RepoPrefs ( defaultrepo, set_defaultrepo, get_preflist )
import External ( maybeURLCmd )
import DarcsURL ( is_url )
import SelectChanges ( with_selected_changes )
import DarcsUtils ( formatPath )
import DarcsURL ( is_relative )
import Depends ( get_common_and_uncommon )
import PatchBundle ( make_bundle )
import Printer ( vcat, text, ($$), (<+>), putDocLn, errorDoc )
import RemoteApply ( remote_apply, apply_as )
#include "impossible.h"
\end{code}
\begin{code}
push_description :: String
push_description =
 "Copy and apply patches from this repository to another one."
\end{code}

\options{push}
\haskell{push_help}
\begin{code}
push_help :: String
push_help =
 "Push is the opposite of pull.  Push allows you to copy changes from the\n"++
 "current repository into another repository.\n"
\end{code}
\begin{code}
push :: DarcsCommand
push = DarcsCommand {command_name = "push",
                     command_help = push_help,
                     command_description = push_description,
                     command_extra_args = 1,
                     command_extra_arg_help = ["[REPOSITORY]"],
                     command_command = push_cmd,
                     command_prereq = am_in_repo,
                     command_get_arg_possibilities = get_preflist "repos",
                     command_argdefaults = defaultrepo,
                     command_darcsoptions = [any_verbosity,
                                             match_several,
                                             all_gui_interactive,
                                             applyas, dry_run, summary,
                                             working_repo_dir,
                                             set_default]}
\end{code}
\begin{code}
push_cmd :: [DarcsFlag] -> [String] -> IO ()
push_cmd opts [unfixedrepodir] =
  let am_verbose = Verbose `elem` opts
      am_quiet = Quiet `elem` opts
      putVerbose s = when am_verbose $ putDocLn s
      putInfo s = when (not am_quiet) $ putDocLn s
      repodir = if is_relative unfixedrepodir
                then fix_filepath opts unfixedrepodir
                else unfixedrepodir
  in
 do
 -- Test to make sure we aren't trying to push to the current repo
 cur_absolute_repo_dir <- absolute_dir "."
 req_absolute_repo_dir <- absolute_dir repodir
 when (cur_absolute_repo_dir == req_absolute_repo_dir) $
       fail "Can't push to current repo!"
 bundle <- withRepoLock $ \repository -> do
  when (is_url repodir) $ do
       when (apply_as opts /= Nothing) $
           let msg = text "Cannot --apply-as when pushing to URLs" in
             if DryRun `elem` opts
             then putInfo $ text "NOTE: " <+> msg
                         $$ text ""
             else errorDoc msg
       maybeapply <- maybeURLCmd "APPLY" repodir
       when (maybeapply == Nothing) $
         let lprot = takeWhile (/= ':') repodir
             prot = map toUpper lprot
             msg = text ("Pushing to "++lprot++" URLs is not supported.\n"++
                         "You may be able to hack this to work"++
                         " using DARCS_APPLY_"++prot) in
           if DryRun `elem` opts
           then putInfo $ text "NOTE:" <+> msg
                       $$ text ""
           else errorDoc msg
  them <- read_repo repodir
  old_default <- defaultrepo "" []
  set_defaultrepo repodir opts
  when (old_default == [repodir]) $
       putInfo $ text $ "Pushing to "++formatPath repodir++"..."
  us <- read_repo "."
  case get_common_and_uncommon (us, them) of
    (common, us', _) -> do
     putVerbose $ text "We have the following patches to push:"
               $$ (vcat $ map (human_friendly.fst) $ head us')
     case us' of
         [[]] -> do putInfo $ text "No recorded local changes to push!"
                    exitWith ExitSuccess
         _ -> return ()
     s <- slurp_recorded repository
     let ps = map (fromJust.snd) $ reverse $ head us'
     with_selected_changes "push" opts s ps (Just $ length ps) $
      \ (_,to_be_pushed) -> do
      print_dry_run_message_and_exit "push" opts to_be_pushed
      when (null to_be_pushed) $ do
          putInfo $
            text "You don't want to push any patches, and that's fine with me!"
          exitWith ExitSuccess
      return $ make_bundle []
                 (bug "using slurpy in make_bundle called from Push")
                 common to_be_pushed
 out <- remote_apply opts repodir bundle 
 putDocLn out
push_cmd _ _ = impossible
\end{code}

For obvious reasons, you can only push to repositories to which you have
write access.  In addition, you can only push to repos that you access
either on the local file system or with ssh.  In order to apply with ssh,
darcs must also be installed on the remote computer.  The command invoked
to run ssh may be configured by the \verb!DARCS_SSH! environment variable
(see subsection~\ref{darcsssh}).  The command invoked via ssh is always
\verb!darcs!, i.e., the darcs executable must be in the default path on
the remote machine.

Push works by creating a patch bundle, and then running darcs apply in the
target repository using that patch bundle.  This means that the default
options for \emph{apply} in the \emph{target} repository (such as, for
example, \verb!--test!) will affect the behavior of push.  This also means
that push is somewhat less efficient than pull.

When you receive an error message such as
\begin{verbatim}
bash: darcs: command not found
\end{verbatim}
then this means that the darcs on the remote machine could
not be started.  Make sure that the darcs executable is called
\verb!darcs! and is found in the default path.  The default path can
be different in interactive and in non-interactive shells.  Say
\begin{verbatim}
ssh login@remote.machine darcs
\end{verbatim}
to try whether the remote darcs can be found, or
\begin{verbatim}
ssh login@remote.machine 'echo $PATH'
\end{verbatim}
(note the single quotes) to check the default path.

\begin{options}
--apply-as
\end{options}

If you give the \verb!--apply-as! flag, darcs will use sudo to apply the
changes as a different user.  This can be useful if you want to set up a
system where several users can modify the same repository, but you don't
want to allow them full write access.  This isn't secure against skilled
malicious attackers, but at least can protect your repository from clumsy,
inept or lazy users.

\begin{options}
--matches, --patches, --tags
\end{options}

The \verb!--patches!, \verb!--matches!, and \verb!--tags! options can be
used to select which patches to push, as described in
subsection~\ref{selecting}.  darcs will silently push along any other patches
upon which the selected patches depend.

When there are conflicts, the behavior of push is determined by the default
flags to \verb!apply! in the \emph{target} repository.  Most commonly, for
pushed-to repositories, you'd like to have \verb!--dont-allow-conflicts! as
a default option to apply (by default, it is already the default\ldots).  If
this is the case, when there are conflicts on push, darcs will fail with an
error message.  You can then resolve by pulling the conflicting patch,
recording a resolution and then pushing the resolution together with the
conflicting patch.

Darcs does not have an explicit way to tell you which patch conflicted, only the
file name. You may want to pull all the patches from the remote repo just
to be sure. If you don't want to do this in your working directory,
you can create another darcs working directory for this purpose.

If you want, you could set the target repo to use \verb!--allow-conflicts!.
In this case conflicting patches will be applied, but the conflicts will
not be marked in the working directory.

If, on the other hand, you have \verb!--mark-conflicts! specified as a
default flag for apply in the target repository, when there is a conflict,
it will be marked in the working directory of the target repository.  In
this case, you should resolve the conflict in the target repository itself.
