%  Copyright (C) 2002-2004 David Roundy
%  Copyright (C) 2004 Juliusz Chroboczek
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

\begin{code}
module Pristine ( Pristine, flagsToPristine,
                 createPristine, removePristine, identifyPristine,
                 checkPristine, slurpPristine,
                 writePristine, write_dirty_Pristine,
                 syncPristine, replacePristine, getPristinePop,
                 pristineDirectory
               ) where

import Maybe ( isJust )
import Monad ( when, liftM )
import Directory ( createDirectory, doesDirectoryExist, doesFileExist,
                   renameDirectory, removeFile )
import Lock ( rm_recursive, writeBinFile )
import Diff ( cmp, sync )
import Workaround ( getCurrentDirectory )
import SlurpDirectory ( Slurpy,  mmap_slurp, slurp_write, slurp_write_dirty)
import DarcsUtils ( withCurrentDirectory, catchall )

import PatchInfo ( PatchInfo )
import PopulationData ( Population, getPopFrom )
import DarcsFlags ( DarcsFlag( PristinePlain, PristineNone ) )


data Pristine = 
    NoPristine !String | PlainPristine !String | HashedPristine !String

davidNeverMakesAWrongDecision :: Bool
davidNeverMakesAWrongDecision = False

pristineName :: String
pristineName = if davidNeverMakesAWrongDecision then "pristine" else "current"

identifyPristine :: IO (Pristine)
identifyPristine = do mp <- reallyIdentifyPristine
                      case mp of
                          Nothing -> fail "Pristine tree doesn't exist."
                          Just pristine -> return pristine

reallyIdentifyPristine :: IO (Maybe Pristine)
reallyIdentifyPristine = 
    do dir <- findpristine doesDirectoryExist ""
       none <- findpristine doesFileExist ".none"
       hash <- findpristine doesFileExist ".hash"
       case (dir, none, hash) of
           (Nothing, Nothing, Nothing) -> return Nothing
           (Just n, Nothing, Nothing) ->
               return (Just (PlainPristine n))
           (Nothing, Just n, Nothing) ->
               return (Just (NoPristine n))
           (Nothing, Nothing, Just n) ->
               return (Just (HashedPristine n))
           _ -> fail "Multiple pristine trees."
    where findpristine fn ext =
              do e1 <- fn n1
                 e2 <- fn n2
                 case (e1, e2) of
                     (False, False) -> return Nothing
                     (True, False) -> return (Just n1)
                     (False, True) -> return (Just n2)
                     (True, True) -> fail "Multiple pristine trees."
              where  n1 = "_darcs/pristine" ++ ext
                     n2 = "_darcs/current" ++ ext

flagsToPristine :: [DarcsFlag] -> Pristine
flagsToPristine (PristinePlain : _) =
    PlainPristine ("_darcs/" ++ pristineName)
flagsToPristine (PristineNone : _) =
    NoPristine ("_darcs/" ++ pristineName ++ ".none")
flagsToPristine (_ : t) = flagsToPristine t
flagsToPristine [] = flagsToPristine [PristinePlain]

createPristine :: Pristine -> IO Pristine
createPristine mp = 
    do oldpristine <- reallyIdentifyPristine
       when (isJust oldpristine) $ fail "Pristine tree already exists."
       case mp of
           NoPristine n -> do writeBinFile n "Do not delete this file.\n"
                              return (NoPristine n)
           PlainPristine n -> do createDirectory n
                                 return (PlainPristine n)
           HashedPristine _ -> fail "HashedPristine is not implemented yet."

removePristine :: Pristine -> IO ()
removePristine (NoPristine n) = removeFile n
removePristine (PlainPristine n) = rm_recursive n
removePristine (HashedPristine _) = 
    fail "HashedPristine is not implemented yet."

checkPristine :: FilePath -> Pristine -> IO Bool
checkPristine _ (NoPristine _) = return True
checkPristine path (PlainPristine n) = do cwd <- getCurrentDirectory
                                          cmp (cwd ++ "/" ++ n) path
checkPristine _ (HashedPristine _) = 
    fail "HashedPristine is not implemented yet."

slurpPristine :: Pristine -> IO (Maybe Slurpy)
slurpPristine (PlainPristine n) = do cwd <- getCurrentDirectory
                                     slurpy <- mmap_slurp (cwd ++ "/" ++ n)
                                     return (Just slurpy)
slurpPristine _ = return Nothing

writePristine :: Slurpy -> Pristine -> IO ()
writePristine _ (NoPristine _) = return ()
writePristine cur (PlainPristine n) =
    withCurrentDirectory n $ slurp_write [] cur
writePristine _ (HashedPristine _) =
    fail "HashedPristine is not implemented yet."

write_dirty_Pristine :: Slurpy -> Pristine -> IO ()
write_dirty_Pristine _ (NoPristine _) = return ()
write_dirty_Pristine cur (PlainPristine n) =
    withCurrentDirectory n $ slurp_write_dirty [] cur
write_dirty_Pristine _ (HashedPristine _) =
    fail "HashedPristine is not implemented yet."


syncPristine :: Slurpy -> Slurpy -> Pristine -> IO ()
syncPristine _ _ (NoPristine _) = return ()
syncPristine ocur owork (PlainPristine n) = sync n ocur owork
syncPristine _ _ (HashedPristine _) = 
    fail "HashedPristine is not implemented yet."

replacePristine :: FilePath -> Pristine -> IO ()
replacePristine _ (NoPristine _) = return ()
replacePristine newcur (PlainPristine n) =
    do rm_recursive nold
           `catchall` return ()
       renameDirectory n nold
       renameDirectory newcur n
       return ()
           where nold = "_darcs/" ++ pristineName ++ "-old"
replacePristine _ (HashedPristine _) = 
    fail "HashedPristine is not implemented yet."

getPristinePop :: PatchInfo -> Pristine -> IO (Maybe Population)
getPristinePop pinfo (PlainPristine n) =
    Just `liftM` getPopFrom n pinfo
getPristinePop _ _ = return Nothing

pristineDirectory :: Pristine -> Maybe String
pristineDirectory (PlainPristine n) = Just n
pristineDirectory _ = Nothing
\end{code}
