%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\begin{code}
module Test ( run_test, test_slurpy ) where
import Directory
import IO
import System
import Monad

import SlurpDirectory ( Slurpy, slurp_hasfile, slurp_write )
import RepoPrefs ( get_prefval )
import Lock ( withTempDir )
\end{code}

If you like, you can configure your repository to be able to run a test
suite of some sort.  There are two ways you can do this.  One is to use
``setpref'' to set the ``test'' value to be a command to run.  e.g.
\begin{verbatim}
% darcs setpref test "sh configure && make && make test"
\end{verbatim}

Alternatively, you can add to your repo a file called \verb!darcs_test!.
Darcs will recognize this, and make it executable and run it as a test
script.  Typically you would make \verb!darcs_test! a simple script that
compiles your project and possibly runs a test suite.  You should make sure
that your \verb!darcs_test! returns an error code indicating either success
or failure so darcs will know whether or not the code worked.

\begin{code}
run_test :: FilePath -> IO ExitCode
run_test testdir = do
 testline <- get_prefval "test"
 setCurrentDirectory testdir
 hascheck <- doesFileExist "darcs_test"
 if hascheck || testline /= Nothing
   then do
     putStr "Starting running darcs_test...\n"
     testcode <- case testline of
                 Nothing -> return "./darcs_test"
                 Just t -> return t
     when hascheck $ do
        p <- getPermissions "darcs_test"
        setPermissions "darcs_test" (p {executable = True})
     ec <- system testcode
     if ec == ExitSuccess
       then putStr "Tests ran successfully.\n"
       else putStr "Tests failed!\n"
     return ec
   else return ExitSuccess

\end{code}

\begin{code}
test_slurpy :: Slurpy -> IO ExitCode
test_slurpy s = do
   testline <- get_prefval "test"
   formerdir <- getCurrentDirectory
   withTempDir "_darcs/testing" $ \td ->
     if testline /= Nothing || slurp_hasfile "./darcs_test" s
     then do
       slurp_write s
       setCurrentDirectory formerdir
       ec <- run_test td
       if ec == ExitSuccess
          then putStr "Looks like a good patch.\n"
          else putStr "Looks like you have a bad patch!\n"
       return ec
     else do putStr "There is no test.\n"
             return $ ExitSuccess
\end{code}
