%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs setpref}
\begin{code}
module SetPref ( setpref ) where

import IO
import System
import Monad (when)

import DarcsCommands
import Repository
import Patch
import RepoPrefs
\end{code}

\begin{code}
setpref_description =
 "Set the value for a preferences thingy."
\end{code}

\options{setpref}
Usage example:
\begin{verbatim}
% darcs setpref test "echo I am not really testing anything."
\end{verbatim}

\haskell{setpref_help} If you just want to set the pref value in your
repository only, you can just edit ``\verb!_darcs/prefs/prefs!''.  Changes
you make in that file will be preserved.

The ``\verb!_darcs/prefs/prefs!'' holds the only preferences information
that can propogate between repositories via pushes and pulls, and the only
way this happens is when the setprefs command is used.  Note that although
prefs settings are included in patches, they are \emph{not} fully version
controlled.  In particular, depending on the order in which a series of
merges is perform, you may end up with a different final prefs
configuration.  In practice I don't expect this to be a problem, as the
prefs usually won't be changed very often.

The following values are valid preferences options which can be configured
using setpref:
\begin{itemize}
\item ``test'' --- the command to run as a test script.  Default value is
  effectively ``\verb!sh ./darcs_test!''.
\item ``predist'' --- a command to run prior to tarring up a distribution
  tarball.  Typically this would consist of autoconf and/or automake.
\item ``boringfile'' --- the name of a file to read instead of the
  ``boring'' prefs file.
\end{itemize}

\begin{code}
setpref_help =
 "Setpref allows you to set a preferences value in a way
that will propogate to other repositories.
"
\end{code}

\begin{code}
validprefs = ["test"]
setpref = DarcsCommand {command_name = "setpref",
                        command_help = setpref_help,
                        command_description = setpref_description,
                        command_extra_args = 2,
                        command_command = setpref_cmd,
                        command_prereq = am_in_repo,
                        command_get_arg_possibilities = return validprefs,
                        command_argdefaults = nodefaults,
                        command_darcsoptions = []}
\end{code}

\begin{code}
setpref_cmd _ [pref,val] = do
  when (' ' `elem` pref) $ do
    putStr $ "'"++pref++"' is not a valid preference name!\n"
    exitWith $ ExitFailure 1
  oval <- get_prefval pref
  old <- case oval of Just v -> return v
                      Nothing -> return ""
  change_prefval pref old val
  putStr $ "Changing value of "++pref++" from '"++old++"' to '"++val++"'\n"
  pend <- read_pending
  case pend of
      Nothing -> write_pending $ join_patches [changepref pref old val]
      Just op -> write_pending $ join_patches $ flatten op ++
                                 [changepref pref old val]
\end{code}
