%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs get}
\begin{code}
module Get ( get ) where
import Maybe (fromJust)

import DarcsCommands
import DarcsArguments
import Directory
import IO
import System
import Repository
import Patch
import PatchInfo
import SlurpDirectory
import RegexString
import Monad
import Depends ( get_common_and_uncommon )
import RepoPrefs ( set_lastrepo, write_default_prefs )
\end{code}
\begin{code}
get_description =
 "Get a repository."
\end{code}

\options{get}

\haskell{get_help} I recommend using the \verb!--verbose! flag to get, as
this command can take a while, and with no feedback, that can be rather
boring.

If the remote repo and the current directory are in the same filesystem and
that filesystem supports hard links, get will create hard links for the
patch files, which means that the additional storage space needed will be
minimal.  This shouldn't cause any problems as darcs usually never modifies
a patch file, and if it does it deletes the old file and creates a new
file, which should mean that it won't affect the other repo which shares
the same patch file.

\begin{code}
get_help =
 "Get is used to get a local copy of a repository.
"
\end{code}
\begin{code}
get = DarcsCommand {command_name = "get",
                    command_help = get_help,
                    command_description = get_description,
                    command_extra_args = 1,
                    command_command = get_cmd,
                    command_prereq = liftM not $ am_in_repo,
                    command_get_arg_possibilities = return [],
                    command_argdefaults = nodefaults,
                    command_darcsoptions = [reponame,tagname,
                                       patchname_option,verbose]}
\end{code}
\begin{code}
get_cmd opts [inrepodir] = do
  am_verbose <- return $ Verbose `elem` opts
  former_dir <- getCurrentDirectory
  repodir <- absolute_dir inrepodir
  repovalid <- is_repo repodir
  unless repovalid $ do putStr $ "Bad repo directory: "++repodir++"\n"
                        exitWith $ ExitFailure 1
  patches_raw <- read_repo repodir
  patches <- pick_patches opts patches_raw
  myname <- make_repo_name opts repodir
  createDirectory $ myname
  setCurrentDirectory $ former_dir++"/"++myname
  createDirectory "_darcs"
  createDirectory "_darcs/patches"
  createDirectory "_darcs/current"
  createDirectory "_darcs/prefs"
  write_default_prefs
  set_lastrepo repodir
  when am_verbose $ putStr "Getting the inventory...\n"
  write_inventory "." patches
  copy_repo_patches am_verbose repodir "."
  get_rid_of_unwanted_patches opts
  local_patches <- read_repo "."
  repo_is_local <- doesDirectoryExist repodir
  if repo_is_local && not (chose_specific opts)
     then do
       s <- slurp_recorded repodir
       slurp_write s
       s' <- slurp_recorded repodir
       setCurrentDirectory "_darcs/current"
       slurp_write s'
     else do
       setCurrentDirectory "_darcs/current"
       when am_verbose $ putStr "Applying the patches to the \"current\" dir...\n"
       sequence_ $ map (apply_patch_here am_verbose) $
                 reverse $ concat local_patches
       s <- slurp "."
       setCurrentDirectory $ former_dir++"/"++myname
       slurp_write s
  putStr $ "Finished getting.\n"
\end{code}

\begin{code}
apply_patch_here :: Bool -> (PatchInfo, Maybe Patch) -> IO ()
apply_patch_here verb (pi,mp) = do
  s <- slurp "."
  when verb $ putStr $ "\nApplying patch:\n"++show pi
  case mp of
    Nothing -> do putStr "There seems to be an error in the repo!\n"
                  putStr "I'm having trouble reading a patch.  :(\n"
                  exitWith $ ExitFailure 1
    Just p ->
        case apply_to_slurpy p s of
        Just s' -> slurp_write_dirty s'
        Nothing -> do putStr "There seems to be an error in the repo!\n"
                      putStr "Error in patch:\n"
                      putStr $ show p
                      exitWith $ ExitFailure 1
\end{code}

\begin{code}
make_repo_name :: [DarcsFlag] -> FilePath -> IO String
make_repo_name (RepoName n:_) _ = make_repo_name [] n
make_repo_name (_:as) d = make_repo_name as d
make_repo_name [] d =
    let base = case matchRegex (mkRegex "(.*/)?([^/]+)/?") d of
                 Just (_:n:_) -> n
                 _ -> "anonymous_repo" in
        do
        exists <- doesDirectoryExist base
        if not exists
           then return base
           else modify_repo_name base 0
modify_repo_name n i = do
    exists <- doesDirectoryExist $ n++"_"++show i
    if not exists
       then return $ n++"_"++show i
       else modify_repo_name n $ i+1
\end{code}

If you want to get a specific version of a repository, you have a couple of
options.  The first is to use the \verb!--tag-name! option, which allows
you to specify a specific version of the repository by its tag, and get
precisely that version.  The argument is a regular expression, and get will
grab the most recent tag which matches that regular expression.  The other
option is the \verb!--patch-name! option, which will allow you to get that
patch (i.e. the most recent one matching the specified regular expression)
and all older patches.  Note that when specifying a \verb!--patch-name!,
you may get a version of your code that has never before been seen, if the
patches have gotten themselves reordered.  If you ever want to be able to
precisely reproduce a given version, you need to tag it.

\begin{code}
chose_specific :: [DarcsFlag] -> Bool
chose_specific (TagName _:_) = True
chose_specific (PatchName _:_) = True
chose_specific (_:as) = chose_specific as
chose_specific [] = False

pick_patches :: [DarcsFlag] -> PatchSet -> IO PatchSet
pick_patches (TagName t:_) = pictag (mymatch t)
pick_patches (PatchName pn:_) = picpatch (mymatch pn)
pick_patches (_:fs) = pick_patches fs
pick_patches [] = \ ps -> return ps

pictag :: (String -> Bool) -> PatchSet -> IO PatchSet
pictag t (((pi,mp):ps):r)
    | take 4 (just_name pi) == "TAG " && t (drop 4 (just_name pi))
        = do putStrLn $ "Getting tagged version:\n"++show pi
             return $ ((pi,mp):ps):r
    | otherwise = pictag t (ps:r)
pictag t ([]:r) = pictag t r
pictag t [] = do putStr "Couldn't find matching tag!\n"
                 return []

picpatch :: (String -> Bool) -> PatchSet -> IO PatchSet
picpatch t (((pi,mp):ps):r) =
    if t (just_name pi) then do putStr "Getting up to patch:\n"
                                putStrLn $ show pi
                                return $ ((pi,mp):ps):r
                        else picpatch t (ps:r)
picpatch t ([]:r) = picpatch t r
picpatch t [] = do putStr "Couldn't find matching patch!\n"
                   return []

mymatch ('^':r) s = matchRegex (mkRegex ('^':r)) s /= Nothing
mymatch r s = matchRegex (mkRegex (".*"++r)) s /= Nothing
\end{code}

\begin{code}
get_rid_of_unwanted_patches :: [DarcsFlag] -> IO ()
get_rid_of_unwanted_patches (TagName _:_) = get_rid_of_all_but_first
get_rid_of_unwanted_patches (_:as) = get_rid_of_unwanted_patches as
get_rid_of_unwanted_patches [] = return ()

get_rid_of_all_but_first = do
  patches <- read_repo "."
  (_,us',_) <- return $
               get_common_and_uncommon (patches, [[head $ head patches]])
  when (us' /= []) $ putStr $ "Debug message (not error)-getting rid of:\n"
  when (us' /= []) $ putStr $ unlines $ map (show.fst) $ reverse $ concat us'
  sequence_ $ map (eliminate_patch.fst) $ reverse $ concat us'
eliminate_patch :: PatchInfo -> IO ()
eliminate_patch pi = do
  patches <- read_repo "."
  efp $ dropWhile ((/=pi).fst) $ reverse $ concat patches
efp ((pia,mpa):(pib,mpb):ps) =
    case commute (fromJust mpb, fromJust mpa) of
    Just (pa',pb') ->
        do writePatch ("_darcs/patches/"++make_filename pib) pb'
           efp ((pia,Just pa'):ps)
efp [(pi,_)] = do patches <- read_repo "."
                  write_inventory "." [just_drop_pi pi $ concat patches]
efp [] = return ()
just_drop_pi :: PatchInfo -> [(PatchInfo,Maybe Patch)]
              -> [(PatchInfo,Maybe Patch)]
just_drop_pi pidrop ((pi,mp):ps)
    | pi == pidrop = just_drop_pi pidrop ps
    | otherwise = (pi,mp): just_drop_pi pidrop ps
just_drop_pi _ [] = []
\end{code}
