-----------------------------------------------------------------------------
-- |
-- Module      :  Coded.ASN1.ASN1
-- Copyright   :  (c) Dominic Steinitz 2003
-- License     :  BSD-style (see the file libraries/base/LICENSE)
-- 
-- Maintainer  :  dominic.steinitz@blueyonder.co.uk
-- Stability   :  experimental
-- Portability :  non-portable
--
-- Provide Haskell types associated with some of the ASN.1 base types.
-- The ASN.1, BER and other standards have been developed by the
-- International Telecomunication Union (ITU)
-- <http://www.itu.int/>
-- They can be downloaded for free from 
-- <http://asn1.elibel.tm.fr/en/standards/>.
-- See also a layman's guide to a subset of ASN.1, BER and DER available
-- at <ftp://ftp.rsa.com/pub/pkcs/ascii/layman.asc>.
--
-----------------------------------------------------------------------------

module Codec.ASN1.ASN1 (
-- * Type declarations
   VisibleString(..), PrintableString(..), NULL(NULL), 
   OctetString(..), BitString(..),
   intToTwosBinComp, twosBinCompToInt,
   Octet(..), OID(..), ASNEnum(..),
   Encodable(encode, decode),
   ASN(..),
   ASNable(toASN, fromASN),
   TagType(..), TagCons(..), TagOption(..), Tag
-- * Example
   
{- | This example program takes a modified version of the example in Annex A 
   of X.690 (ISO 8825-1) and demonstrates how to produce Haskell types 
   and encoding and decoding functions for each type. 

   Here's the ASN.1.

>PersonnelRecord ::= [APPLICATION 0] IMPLICIT SEQUENCE {
>   name         Name,
>   title        [0] VisibleString,
>   number       EmployeeNumber,
>   dateOfHire   [1] Date,
>   nameOfSpouse [2] Name,
>   children     [3] IMPLICIT
>      SEQUENCE OF ChildInformation DEFAULT {} }

>ChildInformation ::= SEQUENCE
>    { name        Name,
>      dateOfBirth [0] Date}

>Name ::= [APPLICATION 1] IMPLICIT SEQUENCE
>   {givenName  VisibleString,
>    initial    VisibleString,
>    familyName VisibleString}

>EmployeeNumber ::= [APPLICATION 2] IMPLICIT INTEGER

>Date ::= [APPLICATION 3] IMPLICIT VisibleString -- YYYYMMDD

   And here is the corresponding Haskell.

>module Main(main) where

>import Char
>import IO
>import Control.Monad.State
>import List

>import Codec.ASN1.ASN1

>newtype Date = MkDate VisibleString
>   deriving Show

>instance ASNable Date where
>   toASN NoTag (MkDate d) = toASN (Implicit (Application,3)) d
>   toASN tag   (MkDate d) = toASN tag d 
>   fromASN t x =
>      let (u,e1,e2) = 
>            case t of
>               NoTag ->
>	          ((Application,3),
>	           "fromASN: invalid application primitive tag for Date",
>	           "fromASN: invalid constructed tag for Date")
>               (Implicit v) ->
>	          (v,
>	           "fromASN: invalid implicit primitive tag for Date",
>	           "fromASN: invalid implicit constructed tag for Date") in
>         f u x e1 e2        
>      where
>         f t x errMsg1 errMsg2 =
>            case x of
>	       Primitive' t' v ->
>	          if t == t'
>		     -- VisibleString normally has the Universal tag 26.
>		     -- Decode it as an OctetString, encode it with the
>		     -- expected tag and then properly decode it.
>	             then let (y::OctetString) = fromASN (Implicit t) x 
>	                      z = map (chr . fromIntegral) $
>	                          encode $ toASN (Implicit (Universal,26)) y 
>	                      ((_,(u::ASN)),_)  = runState (decode stdin) z
>	                      (v::VisibleString) = fromASN NoTag u in 
>	                         MkDate v
>	             else error (errMsg1 ++ "\n" ++ "Expected " ++ show t ++ " actual " ++ show t')
>	       otherwise ->
>	          error errMsg2

>newtype EmployeeNumber = MkEmployeeNumber Integer
>   deriving Show

>instance ASNable EmployeeNumber where
>   toASN NoTag (MkEmployeeNumber n) =
>      toASN (Implicit (Application,2)) n
>   toASN tag (MkEmployeeNumber n) =
>      toASN tag n
>   fromASN t x =
>      let (u,e1,e2) = 
>            case t of
>               NoTag ->
>	          ((Application,2),
>	           "fromASN: invalid application primitive tag for EmployeeNumber",
>	           "fromASN: invalid constructed tag for EmployeeNumber")
>               (Implicit v) ->
>	          (v,
>	           "fromASN: invalid implicit primitive tag for EmployeeNumber",
>	           "fromASN: invalid implicit constructed tag for EmployeeNumber") in
>         f u x e1 e2        
>      where
>         f t x errMsg1 errMsg2 =
>            case x of
>	       Primitive' t' v ->
>	          if t == t'
>		     -- Integer normally has the Universal tag 2.
>		     -- Decode it as an OctetString, encode it with the
>		     -- expected tag and then properly decode it.
>	             then let (y::OctetString) = fromASN (Implicit t) x 
>	                      z = map (chr . fromIntegral) $
>	                          encode $ toASN (Implicit (Universal,2)) y 
>	                      ((_,(u::ASN)),_)  = runState (decode stdin) z
>	                      (v::Integer) = fromASN NoTag u in 
>	                         MkEmployeeNumber v
>	             else error (errMsg1 ++ "\n" ++ "Expected " ++ show t ++ " actual " ++ show t')
>	       otherwise ->
>	          error errMsg2

>data Name =
>   MkName {
>      givenName  :: VisibleString,
>      initial    :: VisibleString,
>      familyName :: VisibleString }
>   deriving Show

>instance ASNable Name where
>   toASN NoTag n =
>      toASN (Implicit (Application,1)) [toASN NoTag (givenName n),
>                                        toASN NoTag (initial n),
>                                        toASN NoTag (familyName n)]
>   toASN tag n =
>      toASN tag [toASN NoTag $ givenName n,
>                 toASN NoTag $ initial n,
>                 toASN NoTag $ familyName n]
>   fromASN t x =
>      let (u,e1,e2,e3) = 
>            case t of
>               NoTag ->
>	          ((Application,1),
>	           "fromASN: invalid application constructed tag for Name",
>	           "fromASN: invalid primitive tag for Name",
>                   "fromASN: invalid number of components for Name") 
>               (Implicit v) ->
>	          (v,
>	           "fromASN: invalid implicit constructed tag for Name",
>	           "fromASN: invalid implicit primitive tag for Name",
>                   "fromASN: invalid number of components for Name") in 
>         f u x e1 e2 e3       
>      where
>         f t x errMsg1 errMsg2 errMsg3 =
>            case x of
>	       Constructed' t' v ->
>	          if t == t'
>	             then case v of
>		             [b1,b2,b3] ->
>			        let gn = fromASN NoTag b1
>				    i  = fromASN NoTag b2
>				    fn = fromASN NoTag b3 in
>				       MkName {givenName = gn, initial = i, familyName = fn}
>			     otherwise ->
>			        error errMsg3
>	             else error (errMsg1 ++ "\n" ++ "Expected " ++ show t ++ " actual " ++ show t')
>	       otherwise ->
>	          error errMsg2

>data ChildInformation =
>   MkChildInformation {
>      name :: Name,
>      dateOfBirth ::Date }
>   deriving Show
>
>instance ASNable ChildInformation where
>   toASN t c =
>      case t of
>         NoTag ->
>            Constructed' (Universal,16) bs
>         Implicit tag ->
>            Constructed' tag bs
>      where
>          bs = [toASN NoTag (name c), 
>                toASN (Implicit (Context,1)) (dateOfBirth c)]
>   fromASN t x =
>      let (u,e1,e2,e3) = 
>            case t of
>               NoTag ->
>	          ((Universal,16),
>	           "fromASN: invalid universal constructed tag for ChildInformation",
>	           "fromASN: invalid primitive tag for ChildInformation",
>                   "fromASN: invalid number of components for ChildInformation") 
>               (Implicit v) ->
>	          (v,
>	           "fromASN: invalid implicit constructed tag for ChildInformation",
>	           "fromASN: invalid implicit primitive tag for ChildInformation",
>                   "fromASN: invalid number of components for ChildInformation") in 
>         f u x e1 e2 e3       
>      where
>         f t x errMsg1 errMsg2 errMsg3 =
>            case x of
>	       Constructed' t' v ->
>	          if t == t'
>	             then case v of
>		             [b1,b2] ->
>			        let dob = fromASN (Implicit (Context,1)) b2
>				    nm  = fromASN NoTag b1 in
>				       MkChildInformation {dateOfBirth = dob, name = nm}
>			     otherwise ->
>			        error errMsg3
>	             else error (errMsg1 ++ "\n" ++ "Expected " ++ show t ++ " actual " ++ show t')
>	       otherwise ->
>	          error errMsg2
>
>data PersonnelRecord = 
>   MkPersonnelRecord {
>      name_1       :: Name,
>      title        :: VisibleString,
>      number       :: EmployeeNumber,
>      dateOfHire   :: Date,
>      nameOfSpouse :: Name,
>      children     :: [ChildInformation] }
>   deriving Show
>
>instance ASNable PersonnelRecord where
>   toASN t p =
>      case t of
>         NoTag ->
>            Constructed' (Application,0) bs
>         Implicit tag ->
>            Constructed' tag bs
>      where
>          bs = [toASN NoTag (name_1 p), 
>                toASN (Implicit (Context,0)) (title p),
>		toASN NoTag (number p), 
>		toASN (Implicit (Context,1)) (dateOfHire p),
>		toASN (Implicit (Context,2)) (nameOfSpouse p),
>		toASN (Implicit (Context,3)) (map (toASN NoTag) (children p))]
>   fromASN t x =
>      let (u,e1,e2,e3) = 
>            case t of
>               NoTag ->
>	          ((Application,0),
>	           "fromASN: invalid application constructed tag for PersonnelRecord",
>	           "fromASN: invalid primitive tag for PersonnelRecord",
>                   "fromASN: invalid number of components for PersonnelRecord") 
>               (Implicit v) ->
>	          (v,
>	           "fromASN: invalid implicit constructed tag for PersonnelRecord",
>	           "fromASN: invalid implicit primitive tag for PersonnelRecord",
>                   "fromASN: invalid number of components for PersonnelRecord") in 
>         f u x e1 e2 e3       
>      where
>         f t x errMsg1 errMsg2 errMsg3 =
>            case x of
>	       Constructed' t' v ->
>	          if t == t'
>	             then case v of
>		             [b1,b2,b3,b4,b5,b6] ->
>			        let nm  = fromASN NoTag b1
>		                   -- VisibleString normally has the Universal tag 26.
>		                   -- Decode it as an OctetString and then encode it with the
>		                   -- expected tag and then properly decode it.
>				    tio :: OctetString
>				    tio = fromASN (Implicit (Context,0)) b2
>	                            tie = map (chr . fromIntegral) $
>	                                     encode $ toASN (Implicit (Universal,26)) tio 
>	                            ((_,(tia::ASN)),_)  = runState (decode stdin) tie
>	                            ti  = fromASN NoTag tia 
>				    en  = fromASN NoTag b3
>				    doh = fromASN (Implicit (Context,1)) b4
>				    nos = fromASN (Implicit (Context,2)) b5
>				    as  = fromASN (Implicit (Context,3)) b6 
>				    cs  = map (fromASN NoTag) as in
>				       MkPersonnelRecord { name_1       = nm, 
>				                           title        = ti,
>							   number       = en,
>							   dateOfHire   = doh,
>							   nameOfSpouse = nos,
>							   children     = cs }
>			     otherwise ->
>			        error errMsg3
>	             else error (errMsg1 ++ "\n" ++ "Expected " ++ show t ++ " actual " ++ show t')
>	       otherwise ->
>	          error errMsg2
>
>name1 = MkName { givenName  = MkVisibleString "John",
>                initial    = MkVisibleString "P",
>                familyName = MkVisibleString "Smith" }
>
>name2 = MkName { givenName  = MkVisibleString "Mary",
>                initial    = MkVisibleString "T",
>                familyName = MkVisibleString "Smith" }
>
>name3 = MkName { givenName = MkVisibleString "Ralph",
>                 initial   = MkVisibleString "T",
>                 familyName = MkVisibleString "Smith" }
>
>name4 = MkName { givenName = MkVisibleString "Susan",
>                 initial   = MkVisibleString "B",
>                 familyName = MkVisibleString "Jones" }
>
>date1 = MkDate (MkVisibleString "19710917")
>date2 = MkDate (MkVisibleString "19571111")
>date3 = MkDate (MkVisibleString "19590717")
>
>employeeNumber1 = MkEmployeeNumber 51
>
>child1 = MkChildInformation { name = name3,
>                              dateOfBirth = date2 }
>
>child2 = MkChildInformation { name = name4,
>                              dateOfBirth = date3 }
>
>personnelRecord1 = MkPersonnelRecord { name_1 = name1,
>                                       title  = MkVisibleString "Director",
>				       number = employeeNumber1,
>				       dateOfHire = date1,
>				       nameOfSpouse = name2,
>				       children = [child1,child2] }
>
>encodedPR = map (chr . fromIntegral) $ encode $ toASN NoTag personnelRecord1
>
>-- Decoding can either be done using a state monad or the IO monad - see below.
>-- stdin is a dummy file handle so that the overloaded function decode can be used
>-- with either monad.
>
>unASNedAndDecodedPR :: (PersonnelRecord,String)
>unASNedAndDecodedPR = runState (do (m,y) <- decode stdin; return $ fromASN NoTag y) encodedPR 
>
>main = 
>   do ofh <- openFile "tst.txt" WriteMode 
>      hPutStr ofh encodedPR
>      hClose ofh
>      ifh <- openFile "tst.txt" ReadMode
>      (m,y) <- decode ifh
>      putStrLn (show ((fromASN NoTag y)::PersonnelRecord))
-}
   ) where

import IO
import Bits
import Word
import Char
import List
import Control.Monad.State

newtype VisibleString = MkVisibleString String
   deriving Show

newtype PrintableString = MkPrintableString String
   deriving Show

data NULL = NULL
   deriving(Eq, Show)

newtype OctetString = MkOctetString [Octet]
   deriving Eq

instance Show OctetString where
	show (MkOctetString lo) = (concat . map show) lo

newtype Octet = MkOctet Word8
   deriving Eq

instance Show Octet where
	show (MkOctet w) = (flip (:) [] . chr . fromIntegral) w 

newtype BitString = MkBitString [Octet]
   deriving Eq

instance Show BitString where
	show (MkBitString lo) = (concat . map show) lo

newtype OID = MkOID [Int]
   deriving (Eq, Show)

newtype ASNEnum = MkASNEnum Int
   deriving (Eq, Show)

class CF a where
   cast :: CT b => a -> b

class CT b where
   doNULL            :: NULL -> b
   doInt             :: Int -> b
   doInteger         :: Integer -> b
   doVisibleString   :: VisibleString -> b
   doPrintableString :: PrintableString -> b
   doBool            :: Bool -> b
   doOctetString     :: OctetString -> b
   doBitString       :: BitString -> b
   doOID             :: OID -> b
   doASNEnum         :: ASNEnum -> b

instance CF VisibleString where
   cast = doVisibleString

instance CT VisibleString where
   doNULL x            = error "Can't cast NULL to VS"
   doInt x             = error "Can't cast Int to VS" 
   doInteger x         = error "Can't cast Integer to VS" 
   doVisibleString x   = x
   doPrintableString x = error "Can't cast PS to VS"
   doBool x            = error "Can't cast Bool to VS"
   doOctetString x     = error "Can't cast OS to VS"
   doBitString x       = error "Can't cast BS to VS"
   doOID x             = error "Can't cast OID to VS"
   doASNEnum x         = error "Can't cast ASNEnum to VS"

instance CF PrintableString where
   cast = doPrintableString

instance CT PrintableString where
   doNULL x            = error "Can't cast NULL to PS"
   doInt x             = error "Can't cast Int to PS" 
   doInteger x         = error "Can't cast Integer to PS" 
   doVisibleString x   = error "Can't cast VS to PS"
   doPrintableString x = x
   doBool x            = error "Can't cast Bool to PS"
   doOctetString x     = error "Can't cast OS to PS"
   doBitString x       = error "Can't cast BS to PS"
   doOID x             = error "Can't cast OID to PS"
   doASNEnum x         = error "Can't cast ASNEnum to PS"

instance CF Int where
   cast = doInt

instance CT Int where
   doNULL x            = error "Can't cast NULL to Int"
   doInt x             = x 
   doInteger x         = error "Can't cast Integer to Int" 
   doVisibleString x   = error "Can't cast VS to Int"
   doPrintableString x = error "Can't cast PS to Int"
   doBool x            = error "Can't cast Bool to Int"
   doOctetString x     = error "Can't cast OS to Int"
   doBitString x       = error "Can't cast BS to Int"
   doOID x             = error "Can't cast OID to Int"
   doASNEnum x         = error "Can't cast ASNEnum to Int"

instance CF Integer where
   cast = doInteger

instance CT Integer where
   doNULL x            = error "Can't cast NULL to Integer"
   doInt x             = error "Can't cast Int to Integer" 
   doInteger x         = x 
   doVisibleString x   = error "Can't cast VS to Integer"
   doPrintableString x = error "Can't cast PS to Integer"
   doBool x            = error "Can't cast Bool to Integer"
   doOctetString x     = error "Can't cast OS to Integer"
   doBitString x       = error "Can't cast BS to Integer"
   doOID x             = error "Can't cast OID to Integer"
   doASNEnum x         = error "Can't cast ASNEnum to Integer"

instance CF NULL where
   cast = doNULL

instance CT NULL where
   doNULL x            = x
   doInt  x            = error "Can't cast Int to NULL"
   doInteger  x        = error "Can't cast Integer to NULL"
   doVisibleString x   = error "Can't cast VS to NULL"
   doPrintableString x = error "Can't cast PS to NULL"
   doBool x            = error "Can't cast Bool to NULL"
   doOctetString x     = error "Can't cast OS to NULL"
   doBitString x       = error "Can't cast BS to BS"
   doOID x             = error "Can't cast OID to NULL"
   doASNEnum x         = error "Can't cast ASNEnum to NULL"

instance CF Bool where
   cast = doBool

instance CT Bool where
   doNULL x            = error "Can't cast NULL to Bool"
   doInt x             = error "Can't cast Int to Bool"
   doInteger x         = error "Can't cast Integer to Bool"
   doVisibleString x   = error "Can't cast VS to Bool"
   doPrintableString x = error "Can't cast PS to Bool"
   doBool x            = x
   doOctetString x     = error "Can't cast OS to Bool"
   doBitString x       = error "Can't cast BS to Bool"
   doOID x             = error "Can't cast OID to Bool"
   doASNEnum x         = error "Can't cast ASNEnum to Bool"

instance CF OctetString where
   cast = doOctetString

instance CT OctetString where
   doNULL x            = error "Can't cast NULL to OS"
   doInt x             = error "Can't cast Int to OS" 
   doInteger x         = error "Can't cast Integer to OS" 
   doVisibleString x   = error "Can't cast VS to OS"
   doPrintableString x = error "Can't cast PS to OS"
   doBool x            = error "Can't cast Bool to OS"
   doOctetString x     = x
   doBitString x       = error "Can't cast BS to OS"
   doOID x             = error "Can't cast OID to OS"
   doASNEnum x         = error "Can't cast ASNEnum to OS"

instance CF OID where
   cast = doOID

instance CT OID where
   doNULL x            = error "Can't cast NULL to OID"
   doInt x             = error "Can't cast Int to OID" 
   doInteger x         = error "Can't cast Integer to OID" 
   doVisibleString x   = error "Can't cast VS to OID"
   doPrintableString x = error "Can't cast PS to OID"
   doBool x            = error "Can't cast Bool to OID"
   doOctetString x     = error "Can't cast OS to OID"
   doBitString x       = error "Can't cast BS to OID"
   doOID x             = x
   doASNEnum x         = error "Can't cast ASNEnum to OID"

instance CF ASNEnum where
   cast = doASNEnum

instance CT ASNEnum where
   doNULL x            = error "Can't cast NULL to ASNEnum"
   doInt x             = error "Can't cast Int to ASNEnum" 
   doInteger x         = error "Can't cast Integer to ASNEnum" 
   doVisibleString x   = error "Can't cast VS to ASNEnum"
   doPrintableString x = error "Can't cast PS to ASNEnum"
   doBool x            = error "Can't cast Bool to ASNEnum"
   doOctetString x     = error "Can't cast OS to ASNEnum"
   doBitString x       = error "Can't cast BS to ASNEnum"
   doOID x             = error "Can't cast OID to ASNEnum"
   doASNEnum x         = x

instance CF BitString where
   cast = doBitString

instance CT BitString where
   doNULL x            = error "Can't cast NULL to BS"
   doInt x             = error "Can't cast Int to BS" 
   doInteger x         = error "Can't cast Integer to BS" 
   doVisibleString x   = error "Can't cast VS to BS"
   doPrintableString x = error "Can't cast PS to BS"
   doBool x            = error "Can't cast Bool to BS"
   doOctetString x     = error "Can't cast OS to BS"
   doBitString x       = x
   doOID x             = error "Can't cast OID to BS"
   doASNEnum x         = error "Can't cast ASNEnum to BS"

type ConcreteOctet = Word8
type OctetStream = [ConcreteOctet]

class Monad m => WrapMonad m where
   get' :: Handle -> m Char

class Encodable a where
	encode :: a -> OctetStream
	decode :: WrapMonad m => Handle -> m (Int,a)
	decode' :: WrapMonad m => Handle -> Int -> m a

data TagType = Universal | Application | Context | Private
   deriving (Eq, Enum, Show, Read, Ord)
 
type Tag = (TagType,Int)
 
data TagCons = Primitive | Constructed
   deriving (Eq, Enum, Show)
 
data TagOption = Implicit Tag
               | NoTag

data ASN = forall a . (CF a, Encodable a, Show a) =>
              Primitive' Tag a |
              Constructed' Tag [ASN]

instance Show ASN where
   show (Primitive' t a) = 
      "Primitive' " ++ show t ++ " " ++ show a
   show (Constructed' t bs) = 
      "Constructed' " ++ show t ++ " [" ++ concat (map show bs) ++ "] "

class ASNable a where
   toASN :: TagOption -> a -> ASN
   fromASN :: TagOption -> ASN -> a

data Length = Indefinite | Definite Integer
   deriving (Eq, Show)

-- The most significant bit is bit 7. The least significant bit is bit 0!

msb :: Int
msb = 7

toTag :: TagCons -> Tag -> OctetStream
toTag tagCons (tagType,value)
   | value < 0     = error ("BER toTag negative argument " ++ show value)
   | value < 31    = shortform
   | otherwise     = longform
   where
      shortform  = [t .|. c .|. sValue]
      sValue     = fromIntegral value
      longform   = [firstOctet] ++ intermeds ++ [lastOctet]
      lValue     = toBase 128 value
      firstOctet = (t .|. c .|. 0x1f)
      intermeds  = init (map (0x80 .|.) lValue)
      lastOctet  = last lValue 
      t          = shift (fromIntegral (fromEnum tagType)) 6
      c          = shift (fromIntegral (fromEnum tagCons)) 5

fromTag :: WrapMonad m => Handle -> m (Int,TagCons,Tag)
fromTag h
   = do y <- get' h
        let x :: ConcreteOctet
            x         = fromIntegral (ord y)
            shortform = fromIntegral (x .&. 0x1f)
            isShort   = shortform /= 0x1f 
            t         = toEnum . fromIntegral $ (shiftR (x .&. 0xc0) 6)
            c         = toEnum . fromIntegral $ (shiftR (x .&. 0x20) 5) in
               if isShort
                  then return (1,c,(t,shortform))
                  else do xs <- getTagOctets h
                          let longform = fromIntegral (octetStreamToInteger 128 xs) in
                                 return (length xs + 1,c,(t,longform))

getTagOctets :: WrapMonad m => Handle -> m OctetStream
getTagOctets h = 
   do x <- get' h
      let y = fromIntegral (ord x) in
         if not (testBit y msb)
            then return [y]
            else do ys <- getTagOctets h
                    return ((clearBit y msb):ys)

instance WrapMonad IO where
   get' = hGetChar 

instance WrapMonad (State String) where
   get' = \_ -> do (x:xs) <- get; put xs; return x

toLength :: Int -> OctetStream
toLength x
   | x < 0     = error ("BER toLength negative argument " ++ show x)
   | x < 128   = shortform x
   | otherwise = longform x
   where
      shortform x = [fromIntegral x]
      longform x  = (setBit (fromIntegral(length(y))) msb) : y
      y           = toBase 256 x

fromLength :: WrapMonad m => Handle -> m (Int,Length)
fromLength h =
   do y <- get' h
      let x :: ConcreteOctet
          x         = fromIntegral (ord y)
          isShort   = not (testBit x msb)
          shortform = Definite (fromIntegral x)
          length    = fromIntegral (clearBit x msb) in 
         if x == 0x80
            then return (1,Indefinite)
            else if isShort
               then return (1,shortform)
               else do xs <- getLengthOctets h length
                       let longform =
                              Definite (octetStreamToInteger 256 xs) in
                          return (length+1,longform)

getLengthOctets :: WrapMonad m => Handle -> Int -> m OctetStream
getLengthOctets h l = 
   if l <= 0 
      then return []
      else do x  <- get' h
              xs <- getLengthOctets h (l-1)
              return ((fromIntegral (ord x)):xs)
            
intToTwosBinComp :: Integral a => a -> [Word8]
intToTwosBinComp x
   | x < 0     = reverse . plusOne . reverse . (map complement) $ u
   | x == 0    = [0x00]
   | otherwise = u
   where z@(y:ys) = toBase 256 (abs x)
         u        = if testBit y msb
                       then 0x00:z
                       else z

twosBinCompToInt :: Integral a => [Word8] -> a
twosBinCompToInt x =  conv x
   where conv []       = 0 
         conv w@(x:xs) = if (testBit x msb) 
                            then neg w
                            else pos w
         neg w@(x:xs)  = let z=(clearBit x msb):xs in
                            fromIntegral((octetStreamToInteger 256 z)-
                                         (128*(256^((length w)-1))))
         pos w         = fromIntegral(octetStreamToInteger 256 w)

plusOne :: [Word8] -> [Word8]
plusOne [] = [1]
plusOne (x:xs) = 
   if x == 0xff
      then 0x00:(plusOne xs)
      else (x+1):xs

toBase x = 
   map fromIntegral .
   reverse .
   map (flip mod x) .
   takeWhile (/=0) .
   iterate (flip div x)

powersOf :: Int -> [Integer]
powersOf n = 1 : map ((fromIntegral n)*) (powersOf n)

octetStreamToInteger :: Int -> OctetStream -> Integer
octetStreamToInteger n xs = 
   sum (zipWith (*) (reverse (map fromIntegral xs)) (powersOf n))

instance Encodable VisibleString where
   encode (MkVisibleString xs) = map (fromIntegral . ord) xs
   decode = error "decode: not supported for VisibleString"
   decode' h l =
      if (l <= 0)
         then return (MkVisibleString [])
         else do x <- get' h
                 (MkVisibleString xs) <- decode' h (l-1)
                 return (MkVisibleString (x:xs))


instance Encodable PrintableString where
   encode (MkPrintableString xs) = map (fromIntegral . ord) xs
   decode = error "decode: not supported for PrintableString"
   decode' = error "decode': not supported for PrintableString"

instance Encodable OctetString where
   encode (MkOctetString xs) = map (\(MkOctet x) -> x) xs
   decode = error "decode: not supported for OctetString"
   decode' h l =
      if (l <= 0)
         then return (MkOctetString [])
         else do x <- get' h
                 (MkOctetString xs) <- decode' h (l-1)
                 return (MkOctetString ((MkOctet . fromIntegral . ord $ x):xs))


instance Encodable BitString where
   encode (MkBitString xs) = 0x00 : map (\(MkOctet x) -> x) xs
   decode = error "decode: not supported for BitString"
   decode' = error "decode': not supported for BitString"

instance Encodable NULL where
   encode NULL = []
   decode = error "decode: not supported for Bool"
   decode' = error "decode': not supported for Bool"

instance Encodable Bool where
   encode True  = [0x00]
   encode False = [0xff]
   decode = error "decode: not supported for Bool"
   decode' h l =
      do if l /= 1
            then error "decodeBool: incorrect length"
            else do x <- get' h
                    return ((ord x) == 0)

instance Encodable OID where
   encode (MkOID (x:y:zs)) =
      concat $ map encodeSubId ((x*40 + y):zs)
      where
         encodeSubId x = 
            case a of
               [] -> []
               a:as -> reverse $ a : (map (flip setBit bitsPerOctet) as)
	    where
               a = reverse $ toBase (2^bitsPerOctet) x
               bitsPerOctet = 7
   decode = error "decode: not support for ASNEnum"
   decode' h l =
      do xs <- getOctets h l
         return (decodeOIDAux xs)

decodeOIDAux (x:xs) = 
   MkOID $ ((fromIntegral x) `div` 40):((fromIntegral x) `mod` 40):ys
      where
         ys = map fromIntegral $
	      map (octetStreamToInteger (2^oidBitsPerOctet)) $
	      (map . map) (flip clearBit oidBitsPerOctet) (subIds xs)
         subIds :: OctetStream -> [OctetStream]
         subIds = unfoldr getSubId
         getSubId :: OctetStream -> Maybe (OctetStream, OctetStream)
         getSubId [] = Nothing
         getSubId xs = Just $ span' endOfSubId xs
         endOfSubId = not . (flip testBit oidBitsPerOctet)

oidBitsPerOctet = 7 :: Int

span' :: (a -> Bool) -> [a] -> ([a],[a])
span' p []
   = ([],[])
span' p xs@(x:xs') 
   | p x       = ([x],xs') 
   | otherwise = (x:ys,zs)
      where (ys,zs) = span' p xs'

instance Encodable Int where
   encode = intToTwosBinComp
   decode = error "decode: not supported for Int"
   decode' h l =
      do xs <- getOctets h l
         return (twosBinCompToInt xs)

instance Encodable Integer where
   encode = intToTwosBinComp
   decode = error "decode: not supported for Integer"
   decode' h l =
      do xs <- getOctets h l
         return (twosBinCompToInt xs)

instance Encodable ASNEnum where
   encode (MkASNEnum x) = intToTwosBinComp x
   decode = error "decode: not support for ASNEnum"
   decode' h l =
      do x <- decode' h l
         return $ MkASNEnum x

instance ASNable VisibleString where
   toASN NoTag          = Primitive' (Universal,26)
   toASN (Implicit tag) = Primitive' tag 
   fromASN NoTag x =
      case x of
         Primitive' t' v ->
	    case t' of
	       (Universal,26) ->
	          cast v
	       otherwise ->
	          error "fromASN: invalide universal primitive tag for VisibleString"
	 otherwise ->
	    error "fromASN: invalid constructed tag for VisibleString"
   fromASN (Implicit t) x =
      case x of
	 Primitive' t' v ->
	    if t == t' 
	       then cast v
	       else error "fromASN: invalid implicit primitive tag for VisibleString"
	 otherwise ->
	    error "fromASN: invalid implicit constructed tag for VisibleString"

instance ASNable PrintableString where
   toASN NoTag          = Primitive' (Universal,19)
   toASN (Implicit tag) = Primitive' tag 
   fromASN NoTag x =
      case x of
         Primitive' t' v ->
	    case t' of
	       (Universal,19) ->
	          cast v
	       otherwise ->
	          error "fromASN: invalid universal primitive tag for PrintableString"
	 otherwise ->
	    error "fromASN: invalid constructed tag for PrintableString"
   fromASN (Implicit t) x =
      case x of
	 Primitive' t' v ->
	    if t == t' 
	       then cast v
	       else error "fromASN: invalid implicit primitive tag for PrintableString"
	 otherwise ->
	    error "fromASN: invalid implicit constructed tag for PrintableString"

instance ASNable OctetString where
   toASN NoTag          = Primitive' (Universal,4)
   toASN (Implicit tag) = Primitive' tag
   fromASN NoTag x = 
      case x of
	 Primitive' t' v ->
	    case t' of
	       (Universal,4) ->  
	          cast v
	       otherwise -> 
	          error "fromASN: invalid universal primitive tag for OctetString"
	 otherwise ->
	    error "fromASN: invalid constructed tag for OctetString"
   fromASN (Implicit t) x =
      case x of
	 Primitive' t' v ->
	    if t == t' 
	       then cast v
	       else error "fromASN: invalid implicit primitive tag for OctetString"
	 otherwise ->
	    error "fromASN: invalid constructed tag for OctetString"

instance ASNable BitString where
   toASN NoTag          = Primitive' (Universal,3)
   toASN (Implicit tag) = Primitive' tag
   fromASN NoTag x = 
      case x of
	 Primitive' t' v ->
	    case t' of
	       (Universal,3) ->  
	          cast v
	       otherwise -> 
	          error "fromASN: invalid universal primitive tag for BitString"
	 otherwise ->
	    error "fromASN: invalid constructed tag for BitString"
   fromASN (Implicit t) x =
      case x of
	 Primitive' t' v ->
	    if t == t' 
	       then cast v
	       else error "fromASN: invalid implicit primitive tag for BitString"
	 otherwise ->
	    error "fromASN: invalid constructed tag for BitString"

instance ASNable NULL where
   toASN NoTag = Primitive' (Universal,5)
   toASN (Implicit tag) = Primitive' tag
   fromASN NoTag x = 
      case x of
	 Primitive' t' v ->
	    case t' of
	       (Universal,5) ->  
	          cast v
	       otherwise -> 
	          error "fromASN: invalid universal primitive tag for NULL"
	 otherwise ->
	    error "fromASN: invalid constructed tag for NULL"
   fromASN (Implicit t) x =
      case x of
	 Primitive' t' v ->
	    if t == t' 
	       then cast v
	       else error "fromASN: invalid implicit primitive tag for NULL"
	 otherwise ->
	    error "fromASN: invalid constructed tag for NULL"

instance ASNable Bool where
   toASN NoTag          = Primitive' (Universal,1)
   toASN (Implicit tag) = Primitive' tag
   fromASN NoTag x = 
      case x of
	 Primitive' t' v ->
	    case t' of
	       (Universal,1) ->  
	          cast v
	       otherwise -> 
	          error "fromASN: invalid universal primitive tag for Bool"
	 otherwise ->
	    error "fromASN: invalid constructed tag for Bool"
   fromASN (Implicit t) x =
      case x of
	 Primitive' t' v ->
	    if t == t' 
	       then cast v
	       else error "fromASN: invalid implicit primitive tag for Bool"
	 otherwise ->
	    error "fromASN: invalid implicit constructed tag for Bool"

instance ASNable OID where
   toASN NoTag          = Primitive' (Universal,6)
   toASN (Implicit tag) = Primitive' tag
   fromASN NoTag x = 
      case x of
	 Primitive' t' v ->
	    case t' of
	       (Universal,6) ->  
	          cast v
	       otherwise -> 
	          error "fromASN: invalid universal primitive tag for OID"
	 otherwise ->
	    error "fromASN: invalid constructed tag for OID"
   fromASN (Implicit t) x =
      case x of
	 Primitive' t' v ->
	    if t == t' 
	       then cast v
	       else error "fromASN: invalid implicit primitive tag for OID"
	 otherwise ->
	    error "fromASN: invalid implicit constructed tag for OID"

instance ASNable Int where
   toASN NoTag          = Primitive' (Universal,2)
   toASN (Implicit tag) = Primitive' tag
   fromASN NoTag x = 
      case x of
	 Primitive' t' v ->
	    case t' of
	       (Universal,2) ->  
	          cast v
	       otherwise -> 
	          error "fromASN: invalid universal primitive tag for Int"
	 otherwise ->
	    error "fromASN: invalid constructed tag for Int"
   fromASN (Implicit t) x =
      case x of
	 Primitive' t' v ->
	    if t == t' 
	       then cast v
	       else error "fromASN: invalid implicit primitive tag for Int"
	 otherwise ->
	    error "fromASN: invalid constructed tag for Int"

instance ASNable ASNEnum where
   toASN NoTag          = Primitive' (Universal,10)
   toASN (Implicit tag) = Primitive' tag
   fromASN NoTag x = 
      case x of
	 Primitive' t' v ->
	    case t' of
	       (Universal,10) ->  
	          cast v
	       otherwise -> 
	          error "fromASN: invalid universal primitive tag for ASNEnum"
	 otherwise ->
	    error "fromASN: invalid constructed tag for ASNEnum"
   fromASN (Implicit t) x =
      case x of
	 Primitive' t' v ->
	    if t == t' 
	       then cast v
	       else error "fromASN: invalid implicit primitive tag for ASNEnum"
	 otherwise ->
	    error "fromASN: invalid implicit constructed tag for ASNEnum"

instance ASNable Integer where
   toASN NoTag          = Primitive' (Universal,2)
   toASN (Implicit tag) = Primitive' tag
   fromASN NoTag x = 
      case x of
	 Primitive' t' v ->
	    case t' of
	       (Universal,2) ->  
	          cast v
	       otherwise -> 
	          error "fromASN: invalid universal primitive tag for Integer"
	 otherwise ->
	    error "fromASN: invalid constructed tag for Integer"
   fromASN (Implicit t) x =
      case x of
	 Primitive' t' v ->
	    if t == t' 
	       then cast v
	       else error "fromASN: invalid implicit primitive tag for Integer"
	 otherwise ->
	    error "fromASN: invalid constructed tag for Integer"

instance ASNable [ASN] where
   toASN NoTag rs =
      Constructed' (Universal,16) rs
   toASN (Implicit tag) rs =
      Constructed' tag rs
   fromASN NoTag as =
      case as of
         Constructed' (Universal,16) bs ->
	    bs
	 otherwise ->
	    error "fromASN: invalid constructed tag for [ASN]"
   fromASN (Implicit t) as =
      case as of
         Constructed' t' bs ->
	    if t == t' 
	       then bs
	       else error "fromASN: invalid implicit constructed tag for [ASN}"
	 otherwise ->
	    error "fromASN: invalid primitive tag for [ASN]"

getOctets :: WrapMonad m => Handle -> Int -> m OctetStream
getOctets h l =
   if (l <= 0)
      then return []
      else do x <- get' h
              xs <- getOctets h (l-1)
              return ((fromIntegral (ord x)):xs)

instance Encodable ASN where
 encode b@(Primitive' t x) = 
    (toTag Primitive t)++(toLength (length e))++e where e = (encode x)
 encode b@(Constructed' t bs) = 
    (toTag Constructed t)++(toLength (length e))++e 
                      where e =
                              (concat (map encode bs))
 decode h = 
   do (lt,tc,t) <- fromTag h
      (ll,j)    <- fromLength h
      case j of
         Indefinite -> error "decodeASN: indefinite length"
         Definite l ->
            if l >= fromIntegral (maxBound::Int)
               then error "decodeASN: length greater than maxBound::Int"
               else let k = fromIntegral l
                        m = lt + ll + k in 
                       case tc of 
                          Primitive ->
                             case t of
				(Universal,1) ->
                                   do (x::Bool) <- decode' h k
                                      return (m,(Primitive' t x))
                                (Universal,2) ->
                                   do (x::Integer) <- decode' h k
                                      return (m,(Primitive' t x))
                                (Universal,4) -> 
                                   do (s::OctetString) <- decode' h k
                                      return (m,(Primitive' t s))	
                                (Universal,5) -> 
                                      return (m,(Primitive' t NULL))
                                (Universal,6) -> 
                                   do (s::OID) <- decode' h k
                                      return (m,(Primitive' t s))	
				(Universal,10) ->
                                   do (x::ASNEnum) <- decode' h k
                                      return (m,(Primitive' t x))
                                (Universal,26) ->
                                   do (s::VisibleString) <- decode' h k
                                      return (m,(Primitive' t s))
                                otherwise -> 
                                   do (x::OctetString) <- decode' h k
                                      return (m,(Primitive' t x))
                          Constructed ->
                             do bs <- decodeASNs h k
                                return (m,(Constructed' t bs))
 decode' = error "decode': not supported for ASN"

decodeASNs :: WrapMonad m => Handle -> Int -> m [ASN]
decodeASNs h curLen
   | curLen < 0  = error "decodeASNs: trying to decode a negative number of octets"
   | curLen == 0 = return []
   | otherwise   = do (l,x)  <- decode h
                      ys     <- decodeASNs h (curLen-l)
                      return (x:ys)
