/* cfed -- a level editor for Crimson Fields
   Copyright (C) 2000-2004 Jens Granseuer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* cfed.cpp */

#include <iostream>
#include <fstream>
#include <string>
using namespace std;

#include <stdlib.h>
#include <string.h>

#include "SDL.h"

#include "mission.h"
#include "gamedefs.h"
#include "strutil.h"
#include "globals.h"

#define MAX_MAP_WIDTH	250
#define MAX_MAP_HEIGHT	250

#define MODE_NONE	0
#define MODE_MAP	1
#define MODE_UNIT	2
#define MODE_GAME	3
#define MODE_BUILDING	4
#define MODE_EVENT	5
#define MODE_PLAYER	6

#define HAVE_NONE	0
#define HAVE_MAP	(1<<MODE_MAP)
#define HAVE_UNIT	(1<<MODE_UNIT)
#define HAVE_GAME	(1<<MODE_GAME)
#define HAVE_BUILDING	(1<<MODE_BUILDING)
#define HAVE_EVENT	(1<<MODE_EVENT)
#define HAVE_PLAYER	(1<<MODE_PLAYER)

class MissionParser : public Mission {
public:
  MissionParser( void );

  int parse( ifstream &in );

  int load_unit_set( const char *set );
  int load_tile_set( const char *set );

private:
  int read_map( ifstream &in, unsigned short w, unsigned short h );
  int read_map_raw( ifstream &in, unsigned short w, unsigned short h );
  int read_messages( ifstream &in, const string &localeid, unsigned long *line );
  int parse_color(const string &buffer, Color &col) const;

  int check_game( void ) const;
  int check_units( void ) const;
  int check_shops( void ) const;
  int check_events( void ) const;
  int check_player( const Player &p ) const;

  List langs;

  template <typename T>
  bool find_dups( const List &l, const T *o1 ) const {
    int cnt = 0;
    for ( T *o2 = static_cast<T *>(l.Head());
          o2; o2 = static_cast<T *>(o2->Next()) ) {
      if ( o1->ID() == o2->ID() ) ++cnt;
    }
    return cnt > 1;
  }

  class LangNode : public Node {
  public:
    LangNode( Language &l ) : langid(l.ID()) {}
    const string &GetLang( void ) const { return langid; }

  private:
    const string langid;
  };
};

static void rem_whitespace( string &str );
static int satoi( const string &str );

/* start of program functions */
int main( int argc, char **argv ) {
  short show_help = 0;
  int rc, i;
  char *uset = NULL, *tset = NULL;
  string outname;

  if ( SDL_Init( 0 ) < 0 ) {
    cerr << "Error: Could not init SDL. " << SDL_GetError() << endl;
    return 1;
  }

  if ( argc < 2 ) show_help = 1;
  else {
    for ( i = argc - 1; i > 1; --i ) {
      if (strcmp(argv[i], "--help") == 0) show_help = 1;
      else if (strcmp(argv[i], "--version") == 0) {
        cout << "cfed "VERSION << endl;
        return 0;
      }
      else if (strcmp(argv[i-1], "--units") == 0) uset = argv[i];
      else if (strcmp(argv[i-1], "--tiles") == 0) tset = argv[i];
      else if (strcmp(argv[i-1], "-o") == 0) outname = argv[i];
    }
  }

  if ( !uset || !tset ) show_help = 1;

  if ( show_help ) {
    cout << "Usage: " << argv[0] << " file --tiles <tileset> --units <unitset> [-o <outfile>]" << endl
              << "  --help     display this help and exit" << endl
              << "  --version  output version information and exit" << endl;
    return 0;
  }

  MissionParser parser;
  if ( parser.load_unit_set( uset ) != 0 ) {
    cerr << "Unable to open unit set " << uset << endl;
    return 1;
  }
  if ( parser.load_tile_set( tset ) != 0 ) {
    cerr << "Unable to open tile set " << tset << endl;
    return 1;
  }

  ifstream f( argv[1] );
  if ( !f.is_open() ) {
    cerr << "Unable to open file " << argv[1] << endl;
    return 1;
  }

  rc = parser.parse( f );
  f.close();

  if ( !rc ) {
    if ( outname.empty() ) {
      outname = argv[1];
      size_t pos = outname.rfind( '.' );
      if ( pos != string::npos ) outname.erase( pos );
      outname.append( ".lev" );
    }

    rc = parser.Save( outname.c_str() );

    if ( rc )
      cerr << "Error opening file '" << outname << "' for writing" << endl;
  }

  return rc;
}

/* parse the file and create a binary mission file from it */
int MissionParser::parse( ifstream &in ) {
  Unit *cunit = 0;
  Building *cbuild = 0;
  Event *cevent = 0;
  Player *cplayer = 0;

  Point mapsize( 0, 0 );
  string buf, val, tileset("default"), unitset("default");
  unsigned short mode = MODE_NONE, status = HAVE_NONE;
  unsigned long line = 0;
  int err = 0, i;
  size_t pos;

  while ( !in.eof() ) {
    getline(in, buf);
    ++line;
    rem_whitespace( buf );
    if ((buf.size() > 0) && (buf[0] != '#')) {			/* comment */
      if ( buf[0] == '[' ) {					/* starts a new entity */
        pos = buf.find( ']' );
        if ( pos == string::npos ) {
          cerr << "Syntax error in line " << line << ": ']' missing" << endl;
          err = 1;
          break;
        }

        if ( buf == "[map]" ) {				/* map definitions */
          err = read_map( in, mapsize.x, mapsize.y );
          if ( err != 0 ) break;

          status |= HAVE_MAP;
          line += map.Height();

        } else if ( buf == "[map-raw]" ) {		/* raw map definitions */
          err = read_map_raw( in, mapsize.x, mapsize.y );
          if ( err != 0 ) break;

          status |= HAVE_MAP;
          line += mapsize.y;

        } else if ( buf == "[mission]" ) {		/* global mission definitions */
          mode = MODE_GAME;
          status |= HAVE_GAME;

        } else if ( buf == "[event]" ) {		/* events definitions */
          mode = MODE_EVENT;
          status |= HAVE_EVENT;
          cevent = new Event();
          cevent->SetID( 0 );
          cevent->SetType( 99 );
          cevent->SetPlayer( PLAYER_NONE );
          cevent->SetTrigger( 99 );
          cevent->SetDependency( -1 );
          for ( i = 0; i < 3; ++i ) {
            cevent->SetData( i, -9999 );
            cevent->SetTData( i, -9999 );
          }
          cevent->SetTitle( -1 );
          cevent->SetMessage( -1 );
          cevent->SetFlags( 0 );
          events.AddTail( cevent );

        } else if ( buf == "[unit]" ) {		/* unit definitions */
          mode = MODE_UNIT;
          status |= HAVE_UNIT;
          cunit = new Unit();
          cunit->SetPosition( Point(-1,-1) );
          cunit->SetID( 0 );
          cunit->UnsetFlags( ~0 );
          cunit->SetOwner( PLAYER_NONE );
          cunit->SetType( unit_set->GetUnitInfo(0) );
          cunit->SetCrystals( 0 );
          cunit->SetDirection( 99 );
          cunit->SetGroupSize( MAX_GROUP_SIZE );
          cunit->SetXP( 0 );
          units.AddTail( cunit );

        } else if ( buf == "[building]" ) {		/* building definitions */
          mode = MODE_BUILDING;
          status |= HAVE_BUILDING;
          cbuild = new Building();
          cbuild->SetPosition( Point(0,0) );
          cbuild->SetID( 0 );
          cbuild->SetOwner( PLAYER_NONE );
          cbuild->SetCrystalProduction( 0 );
          cbuild->SetCrystals( 0 );
          cbuild->SetMaxCrystals( 1000 );
          cbuild->SetUnitProduction( 0 );
          cbuild->UnsetFlags( ~0 );
          cbuild->SetMinWeight( 0 );
          cbuild->SetMaxWeight( 99 );
          cbuild->SetNameID( -1 );
          buildings.AddTail( cbuild );

        } else if ( buf.substr(0,9)  == "[messages" ) {		/* messages */
          pos = buf.find( '(' );
          size_t pos2 = buf.find( ')' );

          if ( (pos == string::npos) || (pos2 == string::npos) || (pos2 - pos != 3) ||
               (read_messages( in, buf.substr(pos+1,2), &line ) == -1) ) {
            cerr << "Error in line " << line << ": Malformed [messages] section" << endl;
            err = 1;
          }
        } else if ( buf == "[player]" ) {		/* player defaults */
          mode = MODE_PLAYER;
          status |= HAVE_PLAYER;

          if ( cplayer == 0 ) cplayer = &GetPlayer( PLAYER_ONE );
          else if ( cplayer == &GetPlayer(PLAYER_ONE) ) cplayer = &GetPlayer( PLAYER_TWO );
          else {
            cerr << "Error: More than two [Player] sections detected" << endl;
            err = 1;
            break;
          }

        } else {
          cerr << "Error: Unknown entity '" << buf << "' in line " << line << endl;
          err = 1;
          break;
        }

      } else {								/* data line for current mode */
        pos = buf.find( '=' );
        if ( pos == string::npos ) {
          cerr << "Syntax error in line " << line << ": '=' missing" << endl;
          err = 1;
          break;
        }
        val = buf.substr( pos+1 ); 					/* separate key from data */
        buf.erase( pos );
        rem_whitespace( buf );
        rem_whitespace( val );

        switch( mode ) {
          case MODE_GAME:
            if ( buf == "mapwidth" ) mapsize.x = satoi(val);		/* map width */
            else if ( buf == "mapheight" ) mapsize.y = satoi(val);	/* map height */
            else if ( buf == "name" ) SetName(satoi(val));		/* map name */
            else if ( buf == "info" ) SetLevelInfoMsg(satoi(val));	/* level info */
            else if ( buf == "campaignname" ) SetCampaignName(satoi(val)); /* campaign name */
            else if ( buf == "campaigninfo" ) SetCampaignInfo(satoi(val)); /* campaign info */
            else if ( buf == "players" ) SetNumPlayers(satoi(val));	/* 1 or 2 player map */
            else if ( buf == "unitset" ) unitset = val;			/* unit set to use */
            else if ( buf == "tileset" ) tileset = val;			/* tile set to use */
            else if ( buf == "nextmap" ) SetSequel(val.c_str()); 	/* next map */
            else if ( buf == "music" ) SetMusic(val.c_str()); 		/* soundtrack */
            else if ( buf == "campaign" ) {
              if (satoi(val) != 0) flags |= GI_CAMPAIGN;		/* campaign map */
            } else {
              err = 1;
              cerr << "Error in line " << line << ": Illegal keyword '" << buf << "'" << endl;
            }
            break;

          case MODE_UNIT:
            if ( buf == "xpos" ) cunit->SetPosition(Point(satoi(val),cunit->Position().y));	/* horizontal position */
            else if ( buf == "ypos" ) cunit->SetPosition(Point(cunit->Position().x,satoi(val)));	/* vertical position */
            else if ( buf == "id" ) cunit->SetID(satoi(val));			/* unit id */
            else if ( buf == "crystals" ) cunit->SetCrystals(satoi(val));	/* crystals */
            else if ( buf == "face" ) cunit->SetDirection(satoi(val));		/* direction */
            else if ( buf == "size" ) cunit->SetGroupSize(satoi(val));		/* group size */
            else if ( buf == "xp" ) cunit->SetXP(satoi(val) * XP_PER_LEVEL);	/* experience */
            else if ( buf == "type" ) {						/* unit type */
              err = 1;
              for ( i = 0; i < unit_set->NumUT(); ++i ) {
                if ( val == unit_set->UnitName(i) ) {
                  const UnitType *type = unit_set->GetUnitInfo(i);
                  cunit->SetType( type );
                  cunit->SetMoves( type->Speed() );
                  cunit->SetFlags( cunit->Flags() | type->Flags() );
                  err = 0;
                  break;
                }
              }
              if ( err ) cerr << "Error: Unknown unit type '" << val << "' in line " << line << endl;

            } else if ( buf == "player" ) {				/* owning player */
              /* this is different from internal use in the game! here 0 means no owner! */
              i = satoi(val);
              if ( i == 1 ) cunit->SetOwner( PLAYER_ONE );
              else if ( i == 2 ) cunit->SetOwner( PLAYER_TWO );
              else if ( i == 0 ) cunit->SetOwner( PLAYER_NONE );
              else {
                err = 1;
                cerr << "Error in line " << line << ": Illegal owner of unit" << endl;
              }
            } else {
              err = 1;
              cerr << "Error in line " << line << ": Illegal keyword '" << buf << "' in this context" << endl;
            }
            break;

          case MODE_BUILDING:
            if ( buf == "xpos" ) cbuild->SetPosition(Point(satoi(val),cbuild->Position().y));	/* horizontal position */
            else if ( buf == "ypos" ) cbuild->SetPosition(Point(cbuild->Position().x,satoi(val)));	/* vertical position */
            else if ( buf == "id" ) cbuild->SetID(satoi(val));		/* ID */
            else if ( buf == "name" ) cbuild->SetNameID(satoi(val));	/* name */
            else if ( buf == "player" ) {				/* owning player */
              /* this is different from internal use in the game! here 0 means no owner! */
              i = satoi(val);
              if ( i == 0 ) cbuild->SetOwner( PLAYER_NONE );
              else if ( i == 1 ) cbuild->SetOwner( PLAYER_ONE );
              else if ( i == 2 ) cbuild->SetOwner( PLAYER_TWO );
              else {
                err = 1;
                cerr << "Error in line " << line << ": Illegal owner of building" << endl;
              }

            } else if ( buf == "mining" ) cbuild->SetCrystalProduction(satoi(val));	/* mining */
            else if ( buf == "crystals" ) cbuild->SetCrystals(satoi(val));	/* resources */
            else if ( buf == "capacity" ) cbuild->SetMaxCrystals(satoi(val));	/* max. resources */
            else if ( buf == "minweight" ) cbuild->SetMinWeight(satoi(val));	/* smallest unit allowed */
            else if ( buf == "maxweight" ) cbuild->SetMaxWeight(satoi(val));	/* biggest unit allowed */
            else if ( buf == "type" ) {						/* type */
              if ( val == "workshop" ) cbuild->SetFlags(BLD_WORKSHOP);
              else if ( val == "mine" ) cerr << "Warning: Deprecated shop type 'mine' found. Ignored." << endl;
              else if ( val == "factory" ) cbuild->SetFlags(BLD_FACTORY);
              else {
                err = 1;
                cerr << "Error in line " << line << ": Unknown type '" << val << endl;
              }

            } else if ( buf == "factory" ) {				/* can produce which units */
              err = 1;
              cbuild->SetFlags(BLD_FACTORY);
              for ( i = 0; i < unit_set->NumUT(); ++i ) {
                if ( val == unit_set->UnitName(i) ) {
                  cbuild->SetUnitProduction(cbuild->UnitProduction()|(1<<i));
                  err = 0;
                  break;
                }
              }
              if ( err ) cerr << "Error: Unknown unit type '" << val << "' in line " << line << endl;
            } else {
              err = 1;
              cerr << "Error in line " << line << ": Illegal keyword '" << buf << "' in this context" << endl;
            }
            break;

          case MODE_EVENT:
            if ( buf == "type" ) {						/* type */
              if ( val == "message" ) cevent->SetType(EVENT_MESSAGE);
              else if ( val == "research" ) cevent->SetType(EVENT_RESEARCH);
              else if ( val == "mining" ) cevent->SetType(EVENT_MINING);
              else if ( val == "score" ) cevent->SetType(EVENT_SCORE);
              else if ( val == "createunit" ) cevent->SetType(EVENT_CREATE_UNIT);
              else if ( val == "nextmap" ) cevent->SetType(EVENT_NEXT_MAP);
              else if ( val == "manipulateevent" ) cevent->SetType(EVENT_MANIPULATE_EVENT);
              else {
                err = 1;
                cerr << "Error in line " << line << ": Unknown type '" << val << "'" << endl;
              }

            } else if ( buf == "trigger" ) {				/* trigger */
              if ( val == "turn" ) cevent->SetTrigger(ETRIGGER_TURN);
              else if ( val == "unitdestroyed" ) cevent->SetTrigger(ETRIGGER_UNIT_DESTROYED);
              else if ( val == "unitposition" ) cevent->SetTrigger(ETRIGGER_UNIT_POSITION);
              else if ( val == "handicap" ) cevent->SetTrigger(ETRIGGER_HANDICAP);
              else if ( val == "havebuilding" ) {
                cevent->SetTrigger(ETRIGGER_HAVE_BUILDING);
                cevent->SetTData(2, -1);
              } else if ( val == "haveunit" ) {
                cevent->SetTrigger(ETRIGGER_HAVE_UNIT);
                cevent->SetTData(2, -1);
              } else {
                err = 1;
                cerr << "Error in line " << line << ": Unknown trigger type '" << val << "'" << endl;
              }

            } else if ( buf == "player" ) {				/* player */
              /* this is different from internal use in the game! here 0 means noone! */
              i = satoi(val);
              if ( i == 1 ) cevent->SetPlayer( PLAYER_ONE );
              else if ( i == 2 ) cevent->SetPlayer( PLAYER_TWO );
              else {
                err = 1;
                cerr << "Error in line " << line << ": Illegal player" << cerr;
              }
            } else if ( buf == "title" ) cevent->SetTitle(satoi(val));
            else if ( buf == "message" ) cevent->SetMessage(satoi(val));
            else if ( buf == "id" ) cevent->SetID(satoi(val));
            else if ( buf == "depend" ) cevent->SetDependency(satoi(val));
            else {
              if ( (cevent->Trigger() == ETRIGGER_TURN) &&
                   (buf == "tturn") ) cevent->SetTData(0, satoi(val));		/* turn */
              else if ( ((cevent->Trigger() == ETRIGGER_UNIT_DESTROYED) ||
                         (cevent->Trigger() == ETRIGGER_HAVE_UNIT) ||
                         (cevent->Trigger() == ETRIGGER_UNIT_POSITION)) &&
                   (buf == "tunit") ) cevent->SetTData(0, satoi(val));		/* unit */
              else if ( (cevent->Trigger() == ETRIGGER_HAVE_BUILDING) &&
                   (buf == "tbuilding") ) cevent->SetTData(0, satoi(val));	/* building id */
              else if ( ((cevent->Trigger() == ETRIGGER_HAVE_BUILDING) ||
                         (cevent->Trigger() == ETRIGGER_HAVE_UNIT) ||
                         (cevent->Trigger() == ETRIGGER_UNIT_DESTROYED)) &&
                   (buf == "towner") ) cevent->SetTData(1, satoi(val)-1);	/* owner of building/unit */
              else if ( ((cevent->Trigger() == ETRIGGER_HAVE_BUILDING) ||
                         (cevent->Trigger() == ETRIGGER_HAVE_UNIT)) &&
                   (buf == "tturn") ) cevent->SetTData(2, satoi(val));		/* turn on which to check */
              else if ( (cevent->Trigger() == ETRIGGER_UNIT_POSITION) &&
                   (buf == "txpos") ) cevent->SetTData(1, satoi(val));		/* x of unit position */
              else if ( (cevent->Trigger() == ETRIGGER_UNIT_POSITION) &&
                   (buf == "typos") ) cevent->SetTData(2, satoi(val));		/* y of unit position */
              else if ( (cevent->Trigger() == ETRIGGER_HANDICAP) &&
                   (buf == "thandicap") ) cevent->SetTData(0, satoi(val));	/* handicap */

              else switch ( cevent->Type() ) {
                case EVENT_RESEARCH:
                  if ( buf == "building" ) cevent->SetData(0, satoi(val));	/* id of building */
                  else if ( buf == "unit" ) {					/* unit type */
                    err = 1;
                    for ( i = 0; i < unit_set->NumUT(); ++i ) {
                      if ( val == unit_set->UnitName(i) ) {
                        cevent->SetData(1, i);
                        err = 0;
                        break;
                      }
                    }
                    if ( err ) cerr << "Error: Unknown unit type '" << val << "' in line " << line << endl;
                  } else {
                    err = 1;
                    cerr << "Error in line " << line << ": Illegal keyword '" << buf << "'" << endl;
                  }
                  break;

                case EVENT_MINING:
                  if ( buf == "building" ) cevent->SetData(0, satoi(val));	/* id of building */
                  else if ( buf == "crystals" ) cevent->SetData(1, satoi(val));	/* crystal production */
                  else if ( buf == "action" ) cevent->SetData(2, satoi(val)); 	/* modification action */
                  else {
                    err = 1;
                    cerr << "Error in line " << line << ": Illegal keyword '" << buf << "'" << endl;
                  }
                  break;

                case EVENT_SCORE:
                  if ( buf == "score" ) cerr << "Warning: Deprecated key 'score' found for score event. Ignored." << endl;
                  else if ( buf == "success" ) cevent->SetData(0, satoi(val));	/* success rate increase */
                  else if ( buf == "othermsg" ) cevent->SetData(1, satoi(val)); /* message for other player */
                  else if ( buf == "othertitle" ) cevent->SetData(2, satoi(val)); /* message title for other player */
                  else {
                    err = 1;
                    cerr << "Error in line " << line << ": Illegal keyword '" << buf << "'" << endl;
                  }
                  break;

                case EVENT_CREATE_UNIT:
                  if ( buf == "xpos" ) cevent->SetData(1, satoi(val));		/* where to create x */
                  else if ( buf == "ypos" ) cevent->SetData(2, satoi(val));	/* where to create y */
                  else if ( buf == "unit" ) {					/* unit type */
                    err = 1;
                    for ( i = 0; i < unit_set->NumUT(); ++i ) {
                      if ( val == unit_set->UnitName(i) ) {
                        cevent->SetData(0, i);
                        err = 0;
                        break;
                      }
                    }
                    if ( err ) cerr << "Error: Unknown unit type '" << val << "' in line " << line << endl;
                  } else {
                    err = 1;
                    cerr << "Error in line " << line << ": Illegal keyword '" << buf << "'" << endl;
                  }
                  break;

                case EVENT_NEXT_MAP:
                  if ( buf == "map" ) cevent->SetData(0, satoi(val));	/* ID of map message */
                  else {
                    err = 1;
                    cerr << "Error in line " << line << ": Illegal keyword '" << buf << "'" << endl;
                  }
                 break;

                case EVENT_MANIPULATE_EVENT:
                  if ( buf == "event" ) cevent->SetData(0, satoi(val));	  /* ID of target event */
                  else if ( buf == "flags" ) cevent->SetData(1, satoi(val));  /* flags to modify */
                  else if ( buf == "action" ) cevent->SetData(2, satoi(val)); /* set/clear/toggle */
                  else {
                    err = 1;
                    cerr << "Error in line " << line << ": Illegal keyword '" << buf << "'" << endl;
                  }
                 break;

               case EVENT_MESSAGE:
                  err = 1;
                  cerr << "Error in line " << line << ": Illegal keyword '" << buf << "'" << endl;
                  break;
              }
            }
            break;

          case MODE_PLAYER:
            if ( buf == "name" ) cplayer->SetNameID(satoi(val));			/* name */
            else if ( buf == "briefing" ) cplayer->SetBriefing(satoi(val));
            else if ( (buf == "fcolor") || (buf == "bcolor") ) {
              Color col;
              if ( !parse_color(val, col) ) {
                if ( buf[0] == 'f' ) cplayer->SetLightColor(col);
                else cplayer->SetDarkColor(col);
              } else {
                err = 1;
                cerr << "Error in line " << line << ": Could not parse color '" << val << "'" << endl;
              }
            } else {
              err = 1;
              cerr << "Error in line " << line << ": Illegal keyword '" << buf << "' in this context" << endl;
            }
            break;
          default:
            break;
        }
        if ( err ) break;
      }
    }
  }

  if ( !err ) {
    if ( (status & HAVE_MAP) == 0 ) {
      cerr << "Error: No map" << endl;
      err = 1;
    } else if ( (status & HAVE_GAME) == 0 ) {
      cerr << "Error: Basic definitions missing" << endl;
      err = 1;
    }
  }

  if ( !err && !messages.SetDefaultLanguage( CF_LANG_DEFAULT ) ) {
    cerr << "Error: no english language data found" << endl;
    err = 1;
  }
  if ( !err ) err = check_game();
  if ( !err ) err = check_shops();
  if ( !err ) err = check_units();
  if ( !err ) err = check_events();
  if ( !err ) err = check_player( p1 );
  if ( !err ) err = check_player( p2 );

  return err;
}

/* read the map */
int MissionParser::read_map( ifstream &in, unsigned short w, unsigned short h ) {
  int i, j;
  unsigned short terrain = 0;
  string buf;

  if ( (w <= 0) || (w > MAX_MAP_WIDTH) || (h <= 0) || (h > MAX_MAP_HEIGHT) ) {
    cerr << "Error: Illegal map size" << endl;
    return -1;
  }

  map.SetSize( Point(w, h) );

  for ( i = 0; i < h; ++i ) {
    getline(in, buf);
    if ( in.eof() ) {
      cerr << "Error: unexpected end of map data" << endl;
      return -1;
    }

    rem_whitespace( buf );
    if ( buf.size() != w ) {
      cerr << "Error: map row " << i << " does not comply with width parameter" << endl;
      return -1;
    }

    for ( j = 0; j < w; ++j ) {
      switch ( buf[j] ) {
        case '.': terrain = 30; break;					/* plains */
        case '*': terrain = 81; break;					/* forest */
        case '%': terrain = 117; break;					/* mountains */
        case '=': terrain = 360; break;					/* shallow water */
        case '~': terrain = 361; break;					/* water */
        case '-': terrain = 362; break;					/* deep water */
        case '1': terrain = 6; break;					/* yellow hq entrance, east */
        case '2': terrain = 7; break;					/* blue hq entrance, east */
        case '0': terrain = 8; break;					/* neutral hq entrance, east */
        case '<': terrain = 22; break;					/* hq, left part */
        case '>': terrain = 23; break;					/* hq, right part */
        case '^': terrain = 21; break;					/* hq, upper part */
        case 'v': terrain = 24; break;					/* hq, lower part */
        case '#': terrain = 307; break;					/* swamp */
        case '\\': terrain = 177; break;				/* road, se-nw */
        case '|': terrain = 176; break;					/* road, s-n */
        case '/': terrain = 178; break;					/* road, sw-ne */
        case 'y': terrain = 193; break;					/* road, sw-n-ne */
        case 'Y': terrain = 194; break;					/* road, se-n-nw */
        case '(': terrain = 180; break;					/* road, n-se */
        case ')': terrain = 179; break;					/* road, n-sw */
        case ']': terrain = 181; break;					/* road, nw-s */
        case '[': terrain = 182; break;					/* road, ne-s */
        case 'n': terrain = 184; break;					/* road, sw-se */
        case 'u': terrain = 183; break;					/* road, nw-ne */
        case 'o': terrain = 201; break;					/* road, sw-nw-ne-se */
        case 'k': terrain = 192; break;					/* road, sw-s-ne */
        case 'K': terrain = 191; break;					/* road, s-se-nw */
        case 'T': terrain = 188; break;					/* road, n-s-se */
        case 'U': terrain = 187; break;					/* road, n-s-sw */
        case 'V': terrain = 198; break;					/* road, n-s-ne */
        case 'W': terrain = 189; break;					/* road, n-s-nw */
        case 'X': terrain = 199; break;					/* road, s-se-nw-n */
        case 'x': terrain = 200; break;					/* road, s-sw-ne-n */
        case '!': terrain = 208; break;					/* bridge, n-s */
        case '`': terrain = 209; break;					/* bridge, se-nw */
        case '\'': terrain = 210; break;				/* bridge, sw-ne */
        case '4': terrain = 9; break;					/* yellow depot entrance */
        case '5': terrain = 10; break;					/* blue depot entrance */
        case '3': terrain = 11; break;					/* neutral depot entrance */
        case '7': terrain = 12; break;					/* yellow factory entrance, north */
        case '8': terrain = 13; break;					/* blue factory entrance, north */
        case '6': terrain = 14; break;					/* neutral factory entrance, north */
        case 'a': terrain = 144; break;					/* wire fence, se-nw end */
        case 'b': terrain = 147; break;					/* wire fence, nw-se end */
        case 'c': terrain = 146; break;					/* wire fence, ne-sw end */
        case 'd': terrain = 145; break;					/* wire fence, sw-ne end */
        case 'e': terrain = 133; break;					/* wire fence, n-s */
        case 'f': terrain = 135; break;					/* wire fence, sw-ne */
        case 'g': terrain = 134; break;					/* wire fence, nw-se */
        case 'h': terrain = 138; break;					/* wire fence, nw-s */
        case 'i': terrain = 139; break;					/* wire fence, ne-s */
        case 'j': terrain = 136; break;					/* wire fence, sw-n */
        case 'l': terrain = 137; break;					/* wire fence, se-n */
        case 'm': terrain = 140; break;					/* wire fence, nw-ne */
        case 'p': terrain = 141; break;					/* wire fence, sw-se */
        case '"': terrain = 363; break;					/* cliff 1 */
        case 'A': terrain = 18; break;					/* yellow city */
        case 'B': terrain = 19; break;					/* blue city */
        case 'C': terrain = 20; break;					/* neutral city */
        case 'D': terrain = 3; break;					/* yellow hq entrance, west */
        case 'E': terrain = 4; break;					/* blue hq entrance, west */
        case 'F': terrain = 5; break;					/* neutral hq entrance, west */
        case 'G': terrain = 0; break;					/* yellow hq entrance, north */
        case 'H': terrain = 1; break;					/* blue hq entrance, north */
        case 'I': terrain = 2; break;					/* neutral hq entrance, north */
        case '9': terrain = 15; break;					/* yellow factory entrance, east */
        case 'J': terrain = 16; break;					/* blue factory entrance, east */
        case 'L': terrain = 17; break;					/* neutral factory entrance, east */
        default:
          cerr << "Error: Illegal character '" << buf[j] << "' in map" << endl;
          return -1;
      }

      map.SetHexType( j, i, terrain );
    }
  }

  return 0;
}

/* read a raw map; a hex is represented by its ID and hexes are separated by comma;
   this way, much more hex types can be accessed than by using plain ASCII code */
int MissionParser::read_map_raw( ifstream &in, unsigned short w, unsigned short h ) {
  int i, j;
  unsigned short terrain = 0;
  string buf;
  size_t pos;
  const char *start;

  if ( (w <= 0) || (w > MAX_MAP_WIDTH) || (h <= 0) || (h > MAX_MAP_HEIGHT) ) {
    cerr << "Error: Illegal map size" << endl;
    return -1;
  }

  map.SetSize( Point(w, h) );

  for ( i = 0; i < h; ++i ) {
    getline(in, buf);
    if ( in.eof() ) {
      cerr << "Error: unexpected end of map data" << endl;
      return -1;
    }

    rem_whitespace( buf );
    pos = 0;
    for ( j = 0; j < w; ++j ) {
      start = &buf.c_str()[pos];
      while ( (buf[pos] >= '0') && (buf[pos] <= '9') ) ++pos;

      if ( (buf[pos] == ',') || (buf[pos] == '\0') ) {
        buf[pos++] = '\0';

        terrain = atoi( start );
        if ( terrain >= terrain_set->NumTT() ) {
          cerr << "Error: Illegal hex id '" << terrain << "'" << endl;
          return -1;
        } else map.SetHexType( j, i, terrain );
      } else {
        cerr << "Error: Illegal character '" << buf[pos] << "' in map" << endl;
        return -1;
      }
    }
  }

  return 0;
}

/* check game data for validity */
int MissionParser::check_game( void ) const {

  if ( (map.Width() == 0) || (map.Height() == 0) ) {
    cerr << "Error: map width or height is 0" << endl;
    return 1;
  }

  if ( GetName() == -1 ) {
    cerr << "Error: Level has not been given a name" << endl;
    return 1;
  } else if ( !GetMessage(GetName()) ) {
    cerr << "Error: Illegal mission name " << (short)GetName() << endl;
    return 1;
  } else {

    short num_msgs = -1, cnt;

    for ( LangNode *n = static_cast<LangNode *>(langs.Head());
          n; n = static_cast<LangNode *>(n->Next()) ) {
      const Language *lang = messages.GetLanguage( n->GetLang() );

      // count messages
      for ( cnt = 0; lang->GetMsg(cnt); ++cnt ); // empty loop
      if ( num_msgs == -1 ) num_msgs = cnt;
      else if ( num_msgs != cnt ) {
        cerr << "Error: Language data for '" << lang->ID() << "' contains " << cnt << " messages." << endl
             << "       Previous languages had " << num_msgs << endl;
        return 1;
      }

      if ( strlen(lang->GetMsg(GetName())) > 30 ) {
        cerr << "Error (" << lang->ID() << "): Mission name must not exceed 30 characters" << endl;
        return 1;
      }
    }
  }

  if ( (GetLevelInfoMsg() != -1) && !GetMessage(GetLevelInfoMsg()) ) {
    cerr << "Error: Illegal level info message " << (short)GetLevelInfoMsg() << endl;
    return 1;
  }
  if ( (GetCampaignName() != -1) && !GetMessage(GetCampaignName()) ) {
    cerr << "Error: Illegal campaign name " << (short)GetCampaignName() << endl;
    return 1;
  }
  if ( (GetCampaignInfo() != -1) && !GetMessage(GetCampaignInfo()) ) {
    cerr << "Error: Illegal campaign info message " << (short)GetCampaignInfo() << endl;
    return 1;
  }
  if ( !IsCampaign() &&
     (GetSequel() || (GetCampaignName() != -1) || (GetCampaignInfo() != -1)) ) {
    cerr << "Error: Campaign settings found but map is not marked as a campaign map" << endl;
    return 1;
  }

  return 0;
}

/* check players */
int MissionParser::check_player( const Player &p ) const {

  short msg = p.NameID();
  if ( msg == -1 ) {
    cerr << "Error: No name set for player" << endl;
    return 1;
  } else {
    if ( !GetMessage(msg) ) {
      cerr << "Error: Illegal name " << msg << " for player" << endl;
      return 1;
    } else {
      for ( LangNode *n = static_cast<LangNode *>(langs.Head());
            n; n = static_cast<LangNode *>(n->Next()) ) {
        const Language *lang = messages.GetLanguage( n->GetLang() );

        if ( strlen(lang->GetMsg(msg)) > 20 ) {
          cerr << "Error(" << lang->ID() << "): Player names must not be longer than 20 characters" << endl;
          return 1;
        }
      }
    }
  }

  msg = p.Briefing();
  if ( (msg != -1) && !GetMessage(msg) ) {
    cerr << "Error: Illegal briefing " << msg << " for player" << endl;
    return 1;
  }

  return 0;
}

/* check units */
int MissionParser::check_units( void ) const {
  Unit *u, *walk;

  /* if there is a transport at a unit's location put it in if possible */
  /* this only checks for transports appearing before the actual unit in the file */
  for ( u = static_cast<Unit *>(units.Head()); u; u = static_cast<Unit *>(u->Next()) ) {

    if ( (u->Position().x >= map.Width()) || (u->Position().y >= map.Height()) ||
         (u->Position().x < 0) || (u->Position().y < 0) ) {
      cerr << "Error: unit (ID: " << u->ID() << ") out of map" << endl;
      return 1;
    }

    if ( u->IsTransport() && !u->IsSheltered() ) {

      if ( StorageLeft( *u ) < 0 ) {
        cerr << "Error: Unit " << u->Name() << " at "
                  << u->Position().x << '/' << u->Position().y
                  << " carries too many units or crystals" << endl;
        return 1;
      }

      for ( walk = static_cast<Unit *>(u->Next()); walk; walk = static_cast<Unit *>(walk->Next()) ) {

        if ( walk->Position() == u->Position() ) {
          if ( walk->Weight() > u->MaxWeight() ) {
            cerr << "Error: Unit " << walk->Name() << " too heavy for transport at "
                      << u->Position().x << '/' << u->Position().y << endl;
            return 1;
          } else if ( walk->Weight() < u->MinWeight() ) {
            cerr << "Error: Unit " << walk->Name() << " too light for transport at "
                      << u->Position().x << '/' << u->Position().y << endl;
            return 1;
          } else {
            walk->SetFlags( U_SHELTERED );

            /* mines inside mine layers need to get MCOST_MIN MP */
            if ( walk->IsMine() && u->IsMinesweeper() ) walk->SetMoves( MCOST_MIN );
          }
        }
      }
    }

    if ( u->Crystals() && !u->IsTransport() )
      cerr << "Error: non-transport unit (ID: " << u->ID() << ") cannot carry crystals" << cerr;

    for ( walk = static_cast<Unit *>(u->Next()); walk; walk = static_cast<Unit *>(walk->Next()) ) {
      if ( u->ID() == walk->ID() ) {
        cerr << "Error: Two or more units sharing one ID (" << u->ID() << ')' << endl;
        return 1;
      }

      if ( (u->Position() == walk->Position()) &&
           (!(u->IsSheltered() || u->IsTransport()) || !walk->IsSheltered()) ) {
        cerr << "Error: Two or more units sharing one hex (" << u->Position().x << '/'
                  << u->Position().y << ')' << endl;

        if ( walk->IsTransport() )
          cerr << "       This might be caused by a transport being declared after a unit it carries." << endl;
        return 1;
      }
    }

    if ( (u->Owner() == PLAYER_ONE) || (u->Owner() == PLAYER_TWO) ) {
      if ( u->GetDirection() == 99 ) u->SetDirection( u->Owner() * 3 );    /* direction not set, use defaults */
      else if ( u->GetDirection() > NORTHWEST ) {
        cerr << "Error: Illegal direction " << (short)u->GetDirection() << " for unit (ID: "
                  << u->ID() << ')' << endl;
        return 1;
      }
    } else {
      u->SetDirection( 0 );
      if ( !(u->Owner() == PLAYER_NONE) || !u->IsSheltered() ) {
        cerr << "Error: unit with ID " << u->ID() << " has no legal owner" << endl;
        return 1;
      }
    }

    if ( (u->GroupSize() > MAX_GROUP_SIZE) || (u->GroupSize() == 0) ) {
      cerr << "Error: unit with ID " << u->ID() << " has size " << (short)u->GroupSize() << endl;
      return 1;
    }

    if ( u->XPLevel() > XP_MAX_LEVEL ) {
      cerr << "Error: unit with ID " << u->ID() << " has been promoted to XP level " << (short)u->XPLevel()
           << ", maximum is " << XP_MAX_LEVEL << endl;
      return 1;
    }
  }
  return 0;
}

/* check buildings */
int MissionParser::check_shops( void ) const {
  Building *b, *walk;
  Unit *u;

  for ( b = static_cast<Building *>(buildings.Head()); b; b = static_cast<Building *>(b->Next()) ) {

    if ( (b->Position().x >= map.Width()) || (b->Position().y >= map.Height()) ) {
      cerr << "Error: Shop (" << b->Position().x << '/' << b->Position().y
                << ") out of map" << endl;
      return 1;
    }

    if ( !(map.TerrainTypes( b->Position() ) & TT_ENTRANCE) ) {
      cerr << "Error: Map does not have a shop entrance at " << b->Position().x
                << '/' << b->Position().y << endl;
      return 1;
    }

    if ( b->NameID() == -1 ) {
      cerr << "Error: No name set for shop " << b->ID() <<  endl;
      return 1;
    } else if ( !GetMessage(b->NameID()) ) {
      cerr << "Error: Invalid name " << (short)b->NameID() << " for shop " << b->ID() <<  endl;
      return 1;
    } else {
      for ( LangNode *n = static_cast<LangNode *>(langs.Head());
            n; n = static_cast<LangNode *>(n->Next()) ) {
        const Language *lang = messages.GetLanguage( n->GetLang() );

        if ( strlen(lang->GetMsg(b->NameID())) > 30 ) {
          cerr << "Error(" << lang->ID() << "): Shop names must not be longer than 30 characters" << endl;
          return 1;
        }
      }
    }

    if ( b->Crystals() > b->CrystalStore() ) {
      cerr << "Error: Shop at " << b->Position().x << '/' << b->Position().y
                << " contains " << b->Crystals() << " crystals, but only "
                << b->CrystalStore() << " fit in" << endl;
      return 1;
    }

    if ( b->MinWeight() > b->MaxWeight() ) {
      cerr << "Error: Shop at " << b->Position().x << '/' << b->Position().y
                << " has minweight (" << (short)b->MinWeight() << ") > maxweight ("
                << b->MaxWeight() << ')' << endl;
      return 1;
    }


    for ( walk = static_cast<Building *>(b->Next()); walk; walk = static_cast<Building *>(walk->Next()) ) {
      if ( b->ID() == walk->ID() ) {
        cerr << "Error: Two shops sharing ID " << b->ID() << endl;
        return 1;
      }

      if ( b->Position() == walk->Position() ) {
        cerr << "Error: Two shops sharing one hex ("
                  << b->Position().x << '/' << b->Position().y << ')' << endl;
        return 1;
      }
    }

    for ( u = static_cast<Unit *>(units.Head()); u; u = static_cast<Unit *>(u->Next()) ) {
      if ( b->Position() == u->Position() ) {
        if ( (u->Owner() != b->Owner()) && (b->Owner() != PLAYER_NONE) ) {
          cerr << "Error: Hostile unit (" << u->ID() << ") in building ("
                    << b->Position().x << '/' << b->Position().y << ')' << endl;
          return 1;
        } else u->SetOwner( b->Owner() );

        if ( u->Weight() < b->MinWeight() ) {
          cerr << "Error: Unit (" << u->ID() << ") too light for building at "
	            << b->Position().x << '/' << b->Position().y << endl;
          return 1;
       } else if ( u->Weight() > b->MaxWeight() ) {
          cerr << "Error: Unit (" << u->ID() << ") too heavy for building at "
	            << b->Position().x << '/' << b->Position().y << endl;
          return 1;
        }

        u->SetFlags( U_SHELTERED );
      }
    }
  }
  return 0;
}

/* check events for consistency */
int MissionParser::check_events( void ) const {
  Event *e;
  Building *b;
  Unit *u;
  short msg;

  for ( e = static_cast<Event *>(events.Head()); e; e = static_cast<Event *>(e->Next()) ){

    if ( (e->Player() != PLAYER_ONE) && (e->Player() != PLAYER_TWO) ) {
      cerr << "Error: Event " << (short)e->ID() << " has no player assigned" << endl;
      return 1;
    }

    msg = e->Message();
    if ( (msg != -1) && !messages.GetMsg(msg) ) {
      cerr << "Error: Event " << (short)e->ID() << " has illegal message '" << msg << "'" << endl;
      return 1;
    }
    msg = e->Title();
    if ( (msg != -1) && !messages.GetMsg(msg) ) {
      cerr << "Error: Event " << (short)e->ID() << " has illegal title '" << msg << "'" << endl;
      return 1;
    }

    /* check dependencies */
    if ( e->Dependency() != -1 ) {
      if ( !GetEventByID( e->Dependency() ) ) {
        cerr << "Error: Event " << (short)e->ID() << " depends on non-existing event "
                  << (short)e->Dependency() << endl;
        return 1;
      }
    }

    /* check ID's */
    if ( find_dups( events, e ) ) {
      cerr << "Error: Two or more events with the same ID '" << (short)e->ID() << "'" << endl;
      return 1;
    }


    switch ( e->Type() ) {
      case EVENT_MESSAGE:
        if ( e->Message() == -1 ) {
          cerr << "Error: Message event " << (short)e->ID() << " has no message" << endl;
          return 1;
        }
        e->SetData(0, 0);
        e->SetData(1, 0);
        e->SetData(2, 0);
        break;
      case EVENT_CREATE_UNIT:
        if ( e->GetData(0) < 0 ) {
          cerr << "Error: No unit type specified for Create Unit event " << (short)e->ID() << cerr;
          return 1;
        }

        if ( !map.Contains( Point(e->GetData(1), e->GetData(2)) ) ) {
          cerr << "Error: Location " << e->GetData(1) << '/' << e->GetData(2) << " out of map (event "
                    << (short)e->ID() << ')' << endl;
          return 1;
        }
        break;
      case EVENT_MINING:
        if ( e->GetData(0) < 0 ) {
          cerr << "Error: Mining event " << (short)e->ID() << " with no shop" << endl;
          return 1;
        }

        if ( (e->GetData(2) < 0) || (e->GetData(2) > 3) ) {
          cerr << "Error: Mining event " << (short)e->ID() << " with illegal action "
                    << e->GetData(2) << endl;
          return 1;
        }

        if ( ((e->GetData(2) == 0) || (e->GetData(2) == 2)) && (e->GetData(1) < 0) ) {
          cerr << "Error: Trying to set negative absolute amount for mining event "
                    << (short)e->ID() << endl;
          return 1;
        }

        if ( !GetBuildingByID(e->GetData(0)) ) {
          cerr << "Error: Shop with ID " << e->GetData(0) << " does not exist (event "
                    << (short)e->ID() << ')' << endl;
          return 1;
        }
        break;
      case EVENT_RESEARCH:
        if ( e->GetData(0) < 0 ) {
          cerr << "Error: Research event " << (short)e->ID() << " with no shop" << endl;
          return 1;
        }

        if ( e->GetData(1) < 0 ) {
          cerr << "Error: Research event " << (short)e->ID() << " with no unit type" << endl;
          return 1;
        }

        if ( !GetBuildingByID(e->GetData(0)) ) {
          cerr << "Error: Shop with ID " << e->GetData(0) << " does not exist (event "
                    << (short)e->ID() << ')' << endl;
          return 1;
        }
        break;
      case EVENT_SCORE:
        if ( e->GetData(0) < 0 ) {
          cerr << "Warning: Corrected success rate for score event < 0" << endl;
          e->SetData(0, 0);
        }

        if ( e->GetData(1) < -1 ) e->SetData(1, -1);
        msg = e->GetData(1);
        if ( (msg != -1) && !messages.GetMsg(msg) ) {
          cerr << "Error: Event " << (short)e->ID() << " references illegal message '" << msg << "'" << endl;
          return 1;
        }

        if ( e->GetData(2) < -1 ) e->SetData(2, -1);
        msg = e->GetData(2);
        if ( (e->GetData(1) != -1) && (msg != -1) && !messages.GetMsg(msg) ) {
          cerr << "Error: Event " << (short)e->ID() << " references illegal message '" << msg << "'" << endl;
          return 1;
        } else if ( (e->GetData(1) == -1) && (msg != -1) ) {
          cerr << "Warning: Event " << (short)e->ID() << " set a title but no message" << endl;
          e->SetData(2, -1);
        }
        break;
      case EVENT_NEXT_MAP:
        msg = e->GetData(0);
        if ( (msg != -1) && !messages.GetMsg(msg) ) {
          cerr << "Error: Event " << (short)e->ID() << " references illegal map name '" << msg << "'" << endl;
          return 1;
        }
        e->SetData(1, 0);
        e->SetData(2, 0);
        break;
      case EVENT_MANIPULATE_EVENT:
        if ( e->GetData(0) < 0 ) {
          cerr << "Error: Event manipulation event " << (short)e->ID() << " without valid target" << endl;
          return 1;
        }

        /* for now, this event can only enable/disable other events */
        e->SetData(1, EFLAG_DISABLED);
        if ( e->GetData(1) == 0 ) {
          cerr << "Error: Event manipulation (" << (short)e->ID() << ") with illegal flags 0" << endl;
          return 1;
        }

        if ( (e->GetData(2) < 0) || (e->GetData(2) > 2) ) {
          cerr << "Error: Event manipulation (" << (short)e->ID() << ") with illegal action "
                    << e->GetData(2) << endl;
          return 1;
        }

        if ( !GetEventByID( e->GetData(0) ) ) {
          cerr << "Error: Event with ID " << e->GetData(0) << " does not exist" << endl;
          return 1;
        }
        break;
      default:
          cerr << "Error: Event with ID " << (short)e->ID() << " has illegal type" << endl;
    }

    switch ( e->Trigger() ) {
      case ETRIGGER_TURN:
        if ( e->GetTData(0) < 0 ) {
          cerr << "Error: Event trigger lacks turn (" << (short)e->ID() << ')' << endl;
          return 1;
        }
        e->SetTData(1, 0);
        e->SetTData(2, 0);
        break;
      case ETRIGGER_UNIT_DESTROYED:
        if ( e->GetTData(0) != -1 ) {
          u = GetUnitByID( e->GetTData(0) );

          /* the event must also be triggered when the unit is not
             destroyed but captured by the enemy. Therefore we need
             the original owner */
          if (u) e->SetTData(1, u->Owner());
          else {
            cerr << "Error: Event trigger targets non-existing unit with ID " << e->GetTData(0) << endl;
            return 1;
          }
        } else if ( e->GetTData(1) == -9999 ) e->SetTData(1, e->Player()^1);
        e->SetTData(2, 0);
        break;
      case ETRIGGER_HAVE_UNIT:
        if ( (e->GetTData(1) != PLAYER_ONE) && (e->GetTData(1) != PLAYER_TWO) ) {
          cerr << "Error: Event trigger wants illegal player to own a unit ("
                    << (short)e->ID() << endl;
          return 1;
        }
        if ( e->GetTData(2) < 0 ) e->SetTData(2, -1);

        u = GetUnitByID( e->GetTData(0) );

        if (u) {
          if ( (u->Owner() == e->GetTData(1)) && (e->GetTData(2) < 0) && (e->Dependency() == -1) ) {
            cerr << "Error: Event trigger: unit " << u->ID() << " is already owned by player "
                      << u->Owner()+1 << '(' << (short)e->ID() << ')' << endl;
              return 1;
            }
        } else {
          cerr << "Error: Event trigger targets non-existing unit " << e->GetTData(0)
                    << " (" << (short)e->ID() << ')' << endl;
          return 1;
        }
        break;
      case ETRIGGER_HAVE_BUILDING:
        if ( (e->GetTData(1) != PLAYER_ONE) && (e->GetTData(1) != PLAYER_TWO) ) {
          cerr << "Error: Event trigger wants illegal player to own a shop ("
                    << (short)e->ID() << endl;
          return 1;
        }
        if ( e->GetTData(2) < 0 ) e->SetTData(2, -1);

        b = GetBuildingByID( e->GetTData(0) );

        if (b) {
          if ( (b->Owner() == e->GetTData(1)) && (e->GetTData(2) < 0) && (e->Dependency() == -1) ) {
            cerr << "Error: Event trigger: shop " << b->ID() << " is already owned by player "
                      << b->Owner()+1 << '(' << (short)e->ID() << ')' << endl;
              return 1;
            }
        } else {
          cerr << "Error: Event trigger targets non-existing shop " << e->GetTData(0)
                    << " (" << (short)e->ID() << ')' << endl;
          return 1;
        }
        break;
      case ETRIGGER_UNIT_POSITION:
        if ( (e->GetTData(1) < 0) || (e->GetTData(1) >= map.Width()) ||
             (e->GetTData(2) < 0) || (e->GetTData(2) >= map.Height()) ) {
          cerr << "Error: Illegal position " << e->GetTData(1) << '/' << e->GetTData(2)
                    << " set for event trigger (" << (short)e->ID() << ')' << endl;
          return 1;
        }

        if ( e->GetTData(0) < 0 ) e->SetTData(0, -1);
        else {
          if (!GetUnitByID( e->GetTData(0) )) {
            cerr << "Error: Event trigger targets non-existing unit " << e->GetTData(0)
                      << " (" << (short)e->ID() << ')' << endl;
            return 1;
          }
        }
        break;
      case ETRIGGER_HANDICAP:
        if ( e->GetTData(0) == -9999 ) {
          cerr << "Error: Handicap event trigger without a handicap ("
                    << (short)e->ID() << ')' << endl;
          return 1;
        } else if ( e->GetTData(0) & 0xFFF8 ) {
          cerr << "Error: Invalid handicap " << e->GetTData(0) << " ("
                    << (short)e->ID() << ')' << endl;
          return 1;
        }
        e->SetTData(1, 0);
        e->SetTData(2, 0);
        break;
      default:
        cerr << "Error: Invalid event trigger type " << (short)e->Trigger() << " ("
                  << (short)e->ID() << ')' << endl;
        return 1;
    }
  }
  return 0;
}

/* get text messages from the file */
int MissionParser::read_messages( ifstream &in, const string &localeid, unsigned long *line ) {
  Language lang;
  lang.SetID( localeid.c_str() );
  string buf, msg;
  bool done = false;

  do {
    getline(in, buf);
    ++(*line);

    rem_whitespace( buf );

    if ((buf.size() > 0) && 
       ((buf[0] == '%') || (buf.substr(0,11) == "[/messages]"))) {
      /* save last message */
      if (!msg.empty()) {
        lang.AddMsg(msg);
        msg.erase();
      }

      if (buf.substr(0,11) == "[/messages]") done = true;
    } else {
      if (!msg.empty()) msg += '\n';
      msg.append(buf);
    }
  } while ( !in.eof() && !done );

  int rc = 0;

  if (in.eof()) {
    rc = -1;
    cerr << "Error: messages section unterminated" << endl;
  } else {
    messages.AddLanguage( lang );
    langs.AddTail( new LangNode(lang) );
  }

  return rc;
}

/* load a unit set */
int MissionParser::load_unit_set( const char *set ) {
  string setshort( set );

  // keep only the file part; check for both Unix and Windows path
  // separator characters or this breaks when building with MinGW
  setshort = setshort.substr( setshort.find_last_of( "/\\" ) + 1 );

  size_t pos = setshort.find( ".units" );
  if ( pos != string::npos ) setshort.erase( pos );
  File file( set );
  if ( !file.Open("rb") ) return -1;

  unit_set = new UnitSet();
  if ( unit_set->Load( file, setshort.c_str() ) == -1 ) {
    delete unit_set;
    unit_set = 0;
    return -1;
  }

  map.SetUnitSet( unit_set );
  return 0;
}

/* load a terrain set */
int MissionParser::load_tile_set( const char *set ) {
  string setshort( set );

  // keep only the file part; check for both Unix and Windows path
  // separator characters or this breaks when building with MinGW
  setshort = setshort.substr( setshort.find_last_of( "/\\" ) + 1 );

  size_t pos = setshort.find( ".tiles" );
  if ( pos != string::npos ) setshort.erase( pos );
  File file( set );
  if ( !file.Open("rb") ) return -1;

  terrain_set = new TerrainSet();
  if ( terrain_set->Load( file, setshort.c_str() ) == -1 ) {
    delete terrain_set;
    return -1;
  }

  map.SetTerrainSet( terrain_set );
  return 0;
}

/* parse a color */
int MissionParser::parse_color( const string &buffer, Color &col ) const {
  size_t pos1, pos2;
  short comp;
  string val;

  pos1 = buffer.find( ',' );
  pos2 = buffer.rfind( ',' );

  if ( pos1 == pos2 ) return -1;

  val = buffer.substr( 0, pos1 );
  rem_whitespace( val );
  if ( val.size() == 0 ) return -1;
  comp = satoi( val );
  if ( (comp < 0) || (comp > 255) ) return -1;
  col.r = comp;

  val = buffer.substr( pos1 + 1, pos2 - pos1 - 1 );
  rem_whitespace( val );
  if ( val.size() == 0 ) return -1;
  comp = satoi( val );
  if ( (comp < 0) || (comp > 255) ) return -1;
  col.g = comp;

  val = buffer.substr( pos2 + 1 );
  rem_whitespace( val );
  if ( val.size() == 0 ) return -1;
  comp = satoi( val );
  if ( (comp < 0) || (comp > 255) ) return -1;
  col.b = comp;

  return 0;
}

MissionParser::MissionParser( void ) {
  flags = 0;
  SetLevelInfoMsg( -1 );
  SetCampaignName( -1 );
  SetCampaignInfo( -1 );
  SetTitle( "Unknown" );
}

int satoi( const string &s ) {
  if ( s.size() > 0 && s[0] != '-' && !isdigit(s[0]) ) // complain
    cerr << "Warning: trying to convert non-numeric chars to number: " << s << endl;
  return atoi(s.c_str());
}

/* remove leading and trailing spaces and cr + lf from the string */
void rem_whitespace( string &str ) {
  int len = str.size(), start = 0, end;

  while ( (start < len) && (str[start] == ' ') ) ++start;
  for ( end = len - 1;
        ((str[end] == ' ') || (str[end] == '\n') ||
         (str[end] == '\r')) && (end >= start);
        --end );  /* empty loop */

  str.erase( end + 1 );
  str.erase( 0, start );
}

