// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2003 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// lset.cpp -- level set handling
////////////////////////////////////////////////////////////////////////

#include <string>
using namespace std;

#include "SDL_endian.h"
#include "lset.h"
#include "fileio.h"
#include "misc.h"

////////////////////////////////////////////////////////////////////////
// NAME       : UnitSet::UnitSet
// DESCRIPTION: Initialize a new unit set.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

UnitSet::UnitSet( void ) {
  num_ut = num_sfx = 0;
  ut = NULL;
  sfx = NULL;
  portraits = NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitSet::~UnitSet
// DESCRIPTION: Destroy the currently loaded unit set.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

UnitSet::~UnitSet( void ) {
  delete [] ut;
  if ( sfx ) {
    for ( int i = 0; i < num_sfx; ++i ) delete sfx[i];
    delete [] sfx;
  }
  delete [] portraits;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitSet::GetUnitInfo
// DESCRIPTION: Retrieve information about a unit type by its ID.
// PARAMETERS : utid - unit type ID
// RETURNS    : pointer to unit info or NULL if ID is invalid
////////////////////////////////////////////////////////////////////////

const UnitType *UnitSet::GetUnitInfo( unsigned short utid ) const {
  if ( utid < num_ut ) return &ut[utid];
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitSet::GetSound
// DESCRIPTION: Get a sound effect by its ID.
// PARAMETERS : sfxid - effect ID
// RETURNS    : pointer to effect or NULL if ID is invalid
////////////////////////////////////////////////////////////////////////

SoundEffect *UnitSet::GetSound( unsigned short sfxid ) const {
  if ( sfxid < num_sfx ) return sfx[sfxid];
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitSet::Load
// DESCRIPTION: Load a unit set from a file.
// PARAMETERS : setname - name of the requested unit set. The actual
//                        filename is computed by prefixing it with the
//                        data directory path and adding a .units suffix.
// RETURNS    : 0 on success, non-zero on error
////////////////////////////////////////////////////////////////////////

int UnitSet::Load( const char *setname ) {
  int rc = -1;
  string fname( get_data_dir() );
  fname.append( setname );
  fname.append( ".units" );

  SDL_RWops *file = SDL_RWFromFile( fname.c_str(), "rb" );
  if ( file ) {
    if ( SDL_ReadLE32( file ) == FID_UNITSET ) {
      unsigned char sounds;

      num_ut = SDL_ReadLE16( file );
      SDL_RWread( file, &sounds, 1, 1 );
      num_sfx = sounds;

      if ( !LoadSfx( file ) && !LoadUnitTypes( file ) &&
           !tiles.LoadImageData( file, true ) ) {
        unsigned char num_portraits;
        SDL_RWread( file, &num_portraits, 1, 1 );
        if ( num_portraits ) {
          portraits = new Surface [num_portraits];

          for ( int i = 0; i < num_portraits; ++i ) {
            if ( portraits[i].LoadImageData( file ) ) return -1;
            portraits[i].DisplayFormat();
          }
        }

        tiles.DisplayFormat();
        name.assign( setname );
        rc = 0;
      }
    }
    SDL_RWclose( file );
  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitSet::LoadUnitTypes
// DESCRIPTION: Load unit type definitions from a unit set file.
// PARAMETERS : file - SDL_RWops descriptor of the unit set file
// RETURNS    : 0 on success, non-zero on error
////////////////////////////////////////////////////////////////////////

int UnitSet::LoadUnitTypes( SDL_RWops *file ) {
  ut = new UnitType [num_ut];
  if ( !ut ) return -1;

  for ( int i = 0; i < num_ut; ++i ) {
    if ( ut[i].Load( file, this ) ) {
      delete [] ut;
      ut = NULL;
      num_ut = 0;
      return -1;
    }
  }
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitSet::LoadSfx
// DESCRIPTION: Load unit sound effects from a unit set file.
// PARAMETERS : file - SDL_RWops descriptor of the unit set file
// RETURNS    : 0 on success, non-zero on error
////////////////////////////////////////////////////////////////////////

int UnitSet::LoadSfx( SDL_RWops *file ) {
  if ( num_sfx ) {
    string sfxd( get_sounds_dir() );
    char sfxname[256], len;

    sfx = new SoundEffect * [num_sfx];
    if ( !sfx ) return -1;

    for ( int i = 0; i < num_sfx; ++i ) {
      SDL_RWread( file, &len, 1, 1 );
      SDL_RWread( file, sfxname, 1, len );
      sfxname[len] = '\0';
      sfx[i] = new SoundEffect( (sfxd + sfxname).c_str() );
    }
  }

  return 0;
}


////////////////////////////////////////////////////////////////////////
// NAME       : TerrainType::Load
// DESCRIPTION: Load a terrain type definition from a file.
// PARAMETERS : file - SDL_RWops file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int TerrainType::Load( SDL_RWops *file ) {
  tt_type = SDL_ReadLE16( file );
  tt_image = SDL_ReadLE16( file );
  SDL_RWread( file, &tt_att_mod, 1, 3 );
  tt_color = SDL_ReadLE32( file );
  return 0;
}


////////////////////////////////////////////////////////////////////////
// NAME       : TerrainSet::~TerrainSet
// DESCRIPTION: Destroy the currently loaded terrain set.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

TerrainSet::~TerrainSet( void ) {
  delete [] tt;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TerrainSet::Load
// DESCRIPTION: Load a terrain set from a file.
// PARAMETERS : setname - name of the requested terrain set. The actual
//                        filename is computed by prefixing it with the
//                        data directory path and adding a .tiles suffix.
// RETURNS    : 0 on success, non-zero on error
////////////////////////////////////////////////////////////////////////

int TerrainSet::Load( const char *setname ) {
  int rc = -1;
  string fname( get_data_dir() );
  fname.append( setname );
  fname.append( ".tiles" );

  SDL_RWops *file = SDL_RWFromFile( fname.c_str(), "rb" );
  if ( file ) {
    if ( SDL_ReadLE32( file ) == FID_TERRAINSET ) {
      num_tt = SDL_ReadLE16( file );

      for ( int i = 0; i < 10; ++i )
        classid[i] = SDL_ReadLE16( file );

      if ( !LoadTerrainTypes( file ) && !tiles.LoadImageData( file, true ) ) {
        tiles.DisplayFormat();
        name.assign( setname );
        rc = 0;
      }
    }
    SDL_RWclose( file );
  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TerrainSet::GetTerrainInfo
// DESCRIPTION: Retrieve information about a terrain type by its ID.
// PARAMETERS : ttid - terrain type ID
// RETURNS    : pointer to terrain info or NULL if ID is invalid
////////////////////////////////////////////////////////////////////////

const TerrainType *TerrainSet::GetTerrainInfo( unsigned short ttid ) const {
  if ( ttid < num_tt ) return &tt[ttid];
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TerrainSet::LoadTerrainTypes
// DESCRIPTION: Load terrain type definitions from a terrain set file.
// PARAMETERS : file - SDL_RWops descriptor of the terrain set file
// RETURNS    : 0 on success, non-zero on error
////////////////////////////////////////////////////////////////////////

int TerrainSet::LoadTerrainTypes( SDL_RWops *file ) {
  tt = new TerrainType [num_tt];
  if ( !tt ) return -1;

  for ( int i = 0; i < num_tt; ++i ) {
    if ( tt[i].Load( file ) ) {
      delete [] tt;
      tt = NULL;
      num_tt = 0;
      return -1;
    }
  }
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitType::Load
// DESCRIPTION: Load a unit type definition from a file.
// PARAMETERS : file - SDL_RWops data source descriptor
//              set  - level set
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int UnitType::Load( SDL_RWops *file, const UnitSet *set ) {
  ut_snd_move = ut_snd_fire = NULL;
  ut_terrain = SDL_ReadLE16( file );
  ut_image = SDL_ReadLE16( file );
  ut_flags = SDL_ReadLE16( file );
  SDL_RWread( file, &ut_moves, 1, 13 );

  // set sound effects
  unsigned char sfx[2];
  SDL_RWread( file, sfx, 1, 2 );
  if ( sfx[0] != UT_NO_SOUND ) ut_snd_move = set->GetSound( sfx[0] );
  if ( sfx[1] != UT_NO_SOUND ) ut_snd_fire = set->GetSound( sfx[1] );

  SDL_RWread( file, &ut_trans_slots, 1, 5 );
  SDL_RWread( file, ut_name, 1, 20 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitType::Firepower
// DESCRIPTION: Get firepower of this unit type against a specific
//              target unit class.
// PARAMETERS : target_type - target class (U_*)
// RETURNS    : firepower
////////////////////////////////////////////////////////////////////////

unsigned char UnitType::Firepower( unsigned long target_type ) const {
  unsigned char fp = 0;

  if ( target_type == U_GROUND ) fp = ut_pow_ground;
  else if ( target_type == U_SHIP ) fp = ut_pow_ship;
  else if ( target_type == U_AIR ) fp = ut_pow_air;
  return fp;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitType::IsInFOF
// DESCRIPTION: Check whether a unit of a certain class at a given
//              distance is still in the field of fire.
// PARAMETERS : dist        - distance to target
//              target_type - target unit class (U_*)
// RETURNS    : TRUE if unit is in FOF, FALSE otherwise
////////////////////////////////////////////////////////////////////////

bool UnitType::IsInFOF( unsigned short dist, unsigned long target_type ) const {
  if ( target_type == U_GROUND )
    return ( (ut_min_range_ground <= dist) && (ut_max_range_ground >= dist) );

  else if ( target_type == U_SHIP )
    return ( (ut_min_range_ship <= dist) && (ut_max_range_ship >= dist) );

  else if ( target_type == U_AIR )
    return ( (ut_min_range_air <= dist) && (ut_max_range_air >= dist) );

  return false;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitType::MaxFOF
// DESCRIPTION: Get maximum weapon range against a specific unit class.
// PARAMETERS : target_type - target class (U_*)
// RETURNS    : maximum weapon range
////////////////////////////////////////////////////////////////////////

unsigned char UnitType::MaxFOF( unsigned long target_type ) const {
  unsigned char range = 0;

  if ( target_type == U_GROUND ) range = ut_max_range_ground;
  else if ( target_type == U_SHIP ) range = ut_max_range_ship;
  else if ( target_type == U_AIR ) range = ut_max_range_air;
  return range;
}

////////////////////////////////////////////////////////////////////////
// NAME       : UnitType::MinFOF
// DESCRIPTION: Get minimum weapon range against a specific unit class.
// PARAMETERS : target_type - target class (U_*)
// RETURNS    : minimum weapon range
////////////////////////////////////////////////////////////////////////

unsigned char UnitType::MinFOF( unsigned long target_type ) const {
  unsigned char range = 1;

  if ( target_type == U_GROUND ) range = ut_min_range_ground;
  else if ( target_type == U_SHIP ) range = ut_min_range_ship;
  else if ( target_type == U_AIR ) range = ut_min_range_air;
  return range;
}

