// CoMET - The Crimson Fields Map Editing Tool
// Copyright (C) 2002-2003 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// building.cpp
////////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "building.h"

////////////////////////////////////////////////////////////////////////
// NAME       : Building::Building
// DESCRIPTION: Create a new building.
// PARAMETERS : pos  - position on map
//              id   - building indentifier
//              pid  - identifier of controlling player
//              name - building name
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Building::Building( const Point &pos, unsigned short id,
                    unsigned char pid, const char *name ) :
          MapObject( MO_BUILDING, name ) {
  b_id = id;
  b_pos = pos;
  b_flags = BLD_WORKSHOP;
  b_crystals = 0;
  b_maxcrystals = 1000;
  b_uprod = 0;
  b_cprod = 0;
  b_pid = pid;
  b_minweight = 0;
  b_maxweight = 99;
  strcpy( b_name, name );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Building::Load
// DESCRIPTION: Load a building definition from a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Building::Load( SDL_RWops *file ) {
  b_id = SDL_ReadLE16( file );
  b_pos.x = SDL_ReadLE16( file );
  b_pos.y = SDL_ReadLE16( file );
  b_flags = SDL_ReadLE16( file );
  b_crystals = SDL_ReadLE16( file );
  b_maxcrystals = SDL_ReadLE16( file );
  b_uprod = SDL_ReadLE32( file );

  SDL_RWread( file, &b_cprod, 1, 1 );
  SDL_RWread( file, &b_pid, 1, 1 );
  SDL_RWread( file, &b_minweight, 1, 1 );
  SDL_RWread( file, &b_maxweight, 1, 1 );
  SDL_RWread( file, b_name, 1, 20 );

  SetName( b_name );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Building::Save
// DESCRIPTION: Save the building to a file.
// PARAMETERS : file - save file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Building::Save( SDL_RWops *file ) const {
  SDL_WriteLE16( file, b_id );
  SDL_WriteLE16( file, b_pos.x );
  SDL_WriteLE16( file, b_pos.y );
  SDL_WriteLE16( file, b_flags );
  SDL_WriteLE16( file, b_crystals );
  SDL_WriteLE16( file, b_maxcrystals );
  SDL_WriteLE32( file, b_uprod );

  SDL_RWwrite( file, &b_cprod, 1, 1 );
  SDL_RWwrite( file, &b_pid, 1, 1 );
  SDL_RWwrite( file, &b_minweight, 1, 1 );
  SDL_RWwrite( file, &b_maxweight, 1, 1 );
  SDL_RWwrite( file, b_name, 1, 20 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Building::Export
// DESCRIPTION: Save the building to a plain text file.
// PARAMETERS : file - save file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Building::Export( ofstream &file, const UnitSet *uset ) const {
  file << "[building]\n";
  file << "name = " << b_name << '\n';
  file << "player = " << (b_pid+1)%3 << '\n';
  file << "id = " << b_id << '\n';
  file << "xpos = " << b_pos.x << '\n';
  file << "ypos = " << b_pos.y << '\n';

  if ( (b_flags & BLD_WORKSHOP) != 0 ) file << "type = workshop\n";
  if ( (b_flags & BLD_FACTORY) != 0 ) {
    file << "type = factory\n";

    for ( int i = 0; i < uset->NumUT(); ++i ) {
      if ( ((1<<i) & UnitProduction()) != 0 )
        file << "factory = " << uset->GetUnitInfo(i)->Name() << '\n';
    }
  }
  if ( (b_flags & BLD_MINE) != 0 ) {
    file << "type = mine\n";
    file << "mining = " << (short)CrystalProduction() << '\n';
  }

  file << "crystals = " << b_crystals << '\n';
  file << "capacity = " << b_maxcrystals << '\n';
  file << "minweight = " << (short)b_minweight << '\n';
  file << "maxweight = " << (short)b_maxweight << '\n' << '\n';

  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Building::SetName
// DESCRIPTION: Set the building name.
// PARAMETERS : name - building descriptor
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Building::SetName( const char *name ) {
  if ( name ) {
    strncpy( b_name, name, 19 );
    b_name[19] = '\0';
  } else b_name[0] = '\0';
}

