// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2003 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// initwindow.cpp
////////////////////////////////////////////////////////////////////////

#include <string.h>
#include "SDL_endian.h"

#include "initwindow.h"
#include "fileio.h"
#include "misc.h"
#include "game.h"
#include "options.h"

#define ICON_QUIT_X	185
#define ICON_QUIT_Y	57
#define ICON_QUIT_W	7
#define ICON_QUIT_H	7

#define IW_WIDGET_LIST  	0
#define IW_WIDGET_START_NEW	1
#define IW_WIDGET_START_SAVED	2
#define IW_WIDGET_LOAD_SAVED	3
#define IW_WIDGET_LOAD_NEW	4
#define IW_WIDGET_PBEM		5
#define IW_WIDGET_AI		6
#define IW_WIDGET_VIDEO_OPTIONS 7

extern Options CFOptions;

struct initLevInfo {    // used as user_data for the LevListWidget
  bool password;
  bool singleplayer;
  string *full_path;
};

const short GenericOptionsWindow::B_ID_OK      = 1;
const short SoundOptionsWindow::B_ID_SFX       = 10;
const short SoundOptionsWindow::B_ID_MUSIC     = 11;
const short SoundOptionsWindow::S_ID_VOL_SFX   = 12;
const short SoundOptionsWindow::S_ID_VOL_MUSIC = 13;

////////////////////////////////////////////////////////////////////////
// NAME       : LevListWidget::PrintItem
// DESCRIPTION: Print a level name. If the map is protected by a
//              password print it in red colour, otherwise in green.
//              Put the default number of players behind the title in
//              brackets.
// PARAMETERS : item - node to print
//              dest - destination surface
//              x    - left edge of printing area
//              y    - top edge of printing area
//              clip - clipping rectangle
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void LevListWidget::PrintItem( const TLWNode *item, Surface *dest,
                               short x, short y, const Rect &clip ) const {
  Color col( CF_COLOR_WHITE );
  string name( item->Name() );
  name.erase( name.length() - 4 );   // remove .lev or .sav suffix

  if ( item->UserData() ) {
    initLevInfo *info = (initLevInfo *)item->UserData();
    if ( info->password ) col = surface->GetFGPen();
    name += ' ';
    name += '(';
    name += (info->singleplayer ? '1' : '2');
    name += ')';
  }

  Rect oldclip;
  dest->GetClipRect( oldclip );
  dest->SetClipRect( clip );
  font->Write( name.c_str(), dest, x, y, col );
  dest->SetClipRect( oldclip );
}


////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::InitWindow
// DESCRIPTION: Create the main menu window. The player(s) can select
//              the level they want to play and set other options.
// PARAMETERS : view  - view to attach the window to
//              title - pointer to title window; must be closed when this
//                      window is closed
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

InitWindow::InitWindow( View *view, Window *title ) : Window( WIN_CENTER, view ) {
  this->title = title;

  // calculate window dimensions
  SetSize( MIN(380, view->Width()), sfont->Height() * 15 + 10 );
  view->SetFGPen( Color( 0x00d87c00 ) );
  view->SetBGPen( Color( CF_COLOR_BLACK ) );

  const string home_lev( get_home_levels_dir() );
  if ( !home_lev.empty() ) create_files_list( home_lev.c_str(), ".lev", levels );
  create_files_list( get_levels_dir().c_str(), ".lev", levels );
  CompleteFilesList( levels );
  create_files_list( get_save_dir().c_str(), ".sav", saves );
  CompleteFilesList( saves );

  // create widgets
  levwidget = new LevListWidget( IW_WIDGET_LIST, 5, 5,
                    w/2 - 10, sfont->Height() * 15, &levels,
                    -1, WIDGET_HSCROLLKEY|WIDGET_VSCROLLKEY,
                    NULL, this );

  startwidget = new ButtonWidget( IW_WIDGET_START_NEW, w/2 + 5,
                levwidget->TopEdge() + levwidget->Height() - sfont->Height() - 8,
                w/2 - 10, sfont->Height() + 8, 's',
                WIDGET_DEFAULT, "Start", this );
  startwidget->SetHook( this );

  videowidget = new ButtonWidget( IW_WIDGET_VIDEO_OPTIONS,
                startwidget->LeftEdge(), levwidget->TopEdge(),
                startwidget->Width() - ICON_QUIT_W - 13, startwidget->Height(),
                'v', 0, "Video Options", this );
  videowidget->SetHook( this );

  listctrlwidget = new ButtonWidget( IW_WIDGET_LOAD_SAVED, startwidget->LeftEdge(),
                   videowidget->TopEdge() + videowidget->Height() + 5,
                   startwidget->Width(), videowidget->Height(), 'l', 0,
                   "Load saved game", this );
  listctrlwidget->SetHook( this );

  pbemwidget = new CheckboxWidget( IW_WIDGET_PBEM, listctrlwidget->LeftEdge() + 5,
                   listctrlwidget->TopEdge() + listctrlwidget->Height() + 10,
                   DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, CFOptions.GetPBEM(), 'e',
                   WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT|
                   (CFOptions.GetComputerPlayer() ? WIDGET_DISABLED : 0),
                   "Play By e-Mail", this );
  pbemwidget->SetHook( this );

  aiwidget = new CheckboxWidget( IW_WIDGET_AI, pbemwidget->LeftEdge(),
                   pbemwidget->TopEdge() + pbemwidget->Height() + 3,
                   DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, CFOptions.GetComputerPlayer(),
                   'c', WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT|
                   (CFOptions.GetPBEM() ? WIDGET_DISABLED : 0),
                   "Play vs. Computer", this );
  aiwidget->SetHook( this );

  // the quit button
  ButtonWidget *btn = new ButtonWidget( GUI_QUIT, w - ICON_QUIT_W - 9,
                   videowidget->TopEdge(), ICON_QUIT_W + 4, ICON_QUIT_H + 4,
                   SDLK_ESCAPE, WIDGET_STYLE_GFX, NULL, this );
  btn->SetImage( view->GetSystemIcons(),
                   Rect( ICON_QUIT_X, ICON_QUIT_Y, ICON_QUIT_W, ICON_QUIT_H ),
                   Rect( ICON_QUIT_X, ICON_QUIT_Y, ICON_QUIT_W, ICON_QUIT_H ) );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::~InitWindow
// DESCRIPTION: Destroy window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

InitWindow::~InitWindow( void ) {
  TLWNode *n;
  initLevInfo *li;

  while ( !levels.IsEmpty() ) {
    n = static_cast<TLWNode *>( levels.RemHead() );
    li = (initLevInfo *)n->UserData();
    if ( li ) {
      delete li->full_path;
      delete li;
    }
    delete n;
  }

  while ( !saves.IsEmpty() ) {
    n = static_cast<TLWNode *>( saves.RemHead() );
    li = (initLevInfo *)n->UserData();
    if ( li ) {
      delete li->full_path;
      delete li;
    }
    delete n;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::Draw
// DESCRIPTION: Draw the main window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void InitWindow::Draw( void ) {
  Window::Draw();
  DrawBox( Rect( listctrlwidget->LeftEdge(),
                 pbemwidget->TopEdge() - 5,
                 listctrlwidget->Width(),
                 startwidget->TopEdge() - pbemwidget->TopEdge() - 5),
           BOX_CARVED );
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::WidgetActivated
// DESCRIPTION: Handle activation of widgets in the window.
// PARAMETERS : button - calling widget
//              win    - pointer to active window
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status InitWindow::WidgetActivated( Widget *button, Window *win ) {

  switch ( button->ID() ) {
  case IW_WIDGET_START_NEW:
  case IW_WIDGET_START_SAVED: {
    TLWNode *n = static_cast<TLWNode *>( levwidget->Selected() );
    if ( !n ) new NoteWindow( "Error!", "No level selected!", WIN_CLOSE_ESC, view );
    else {
      string *lname = ((initLevInfo *)n->UserData())->full_path;

      Gam = new Game( view );
      int val = Gam->Load( lname->c_str() );
      if ( val  ) {
        NoteWindow *nw = new NoteWindow( "Error!", "Failed to load mission!", 0, view );
        nw->SetButtonID( 0, GUI_RESTART );
      } else if ( !Gam->GetLevelPassword() ||
                  Gam->CheckPassword( "Access restricted", "Enter access code",
                                      Gam->GetLevelPassword(), false ) ) {
        view->CloseWindow( this );
        view->Refresh();
        Gam->StartTurn();         // let the games begin...
      } else {
        delete Gam;
        Gam = NULL;
        new NoteWindow( "Access denied!", "Invalid access code for this level.",
                        WIN_CLOSE_ESC, view );
      }
    }
    break; }
  case IW_WIDGET_LOAD_NEW:
    startwidget->SetID( IW_WIDGET_START_NEW );
    listctrlwidget->SetID( IW_WIDGET_LOAD_SAVED );
    listctrlwidget->SetTitle( "Load saved game" );
    listctrlwidget->Draw();
    levwidget->SwitchList( &levels, -1 );
    pbemwidget->Enable();
    pbemwidget->Draw();
    aiwidget->Enable();
    aiwidget->Draw();
    Show();
    break;
  case IW_WIDGET_LOAD_SAVED:
    startwidget->SetID( IW_WIDGET_START_SAVED );
    listctrlwidget->SetID( IW_WIDGET_LOAD_NEW );
    listctrlwidget->SetTitle( "Launch new game" );
    listctrlwidget->Draw();
    levwidget->SwitchList( &saves, -1 );
    pbemwidget->Disable();
    pbemwidget->Draw();
    aiwidget->Disable();
    aiwidget->Draw();
    Show();
    break;
  case IW_WIDGET_PBEM:
    CFOptions.SetPBEM( pbemwidget->Clicked() );
    if ( pbemwidget->Clicked() ) {
      CFOptions.SetComputerPlayer( false );
      aiwidget->Disable();
    } else aiwidget->Enable();
    aiwidget->Draw();
    Show();
    break;
  case IW_WIDGET_AI:
    CFOptions.SetComputerPlayer( aiwidget->Clicked() );
    if ( aiwidget->Clicked() ) {
      CFOptions.SetPBEM( false );
      pbemwidget->Disable();
    } else pbemwidget->Enable();
    pbemwidget->Draw();
    Show();
    break;
  case IW_WIDGET_VIDEO_OPTIONS:
    new VideoOptionsWindow( view );
    break;
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::CompleteFilesList
// DESCRIPTION: For the levels list display we need some information
//              about the maps (password protection, 1 or 2 players).
//              This information is attached to the list items.
// PARAMETERS : list - list of files created using create_files_list()
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

void InitWindow::CompleteFilesList( TLWList &list ) const {
  initLevInfo *info;

  for ( TLWNode *n = static_cast<TLWNode *>(list.Head()), *next; n; n = next ) {
    next = static_cast<TLWNode *>(n->Next());
    string *fname = (string *)n->UserData();
    bool ok = false;

    SDL_RWops *file = SDL_RWFromFile( fname->c_str(), "rb" );
    if ( file ) {
      unsigned long id;
      unsigned short flags;
      unsigned char ver;

      id = SDL_ReadLE32( file );
      if ( id == FID_MISSION ) {
        SDL_RWread( file, &ver, 1, 1 );
        if ( ver == FILE_VERSION ) {
          flags = SDL_ReadLE16( file );
          info = new initLevInfo;
          info->password = ((flags & GI_PASSWORD) != 0);
          info->singleplayer = ((flags & GI_AI) != 0);
          info->full_path = fname;
          n->SetUserData( info );
          ok = true;
        }
      }
      SDL_RWclose( file );
    }

    if ( !ok ) {
      n->Remove();
      delete (string *)n->UserData();
      delete n;
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : TitleWindow::TitleWindow
// DESCRIPTION: Create a mostly invisible window containing the title
//              screen image.
// PARAMETERS : view - view to attach the window to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

TitleWindow::TitleWindow( View *view ) : Window( WIN_CENTER, view ) {
  string tname = get_data_dir();
  append_path_delim( tname );
  tname.append( CF_TITLE_SCREEN );

  if ( LoadBMP( tname.c_str() ) ) SetSize( 0, 0, 0, 0 );
  else {
    DisplayFormat();
    Center( *view ); 

    Show();
    EventLoop();
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : TitleWindow::HandleEvent
// DESCRIPTION: Close (or rather, return control to main) when a key or
//              a mouse button is pressed.
// PARAMETERS : event - event received by the event handler
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status TitleWindow::HandleEvent( const SDL_Event &event ) {
  GUI_Status rc = GUI_OK;
  if ( (event.type == SDL_KEYDOWN) || (event.type == SDL_MOUSEBUTTONDOWN) )
    rc = GUI_CLOSE;
  return rc;
}


////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::VideoOptionsWindow
// DESCRIPTION: Show the video options menu.
// PARAMETERS : view - view to attach the window to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

VideoOptionsWindow::VideoOptionsWindow( View *view ) :
     GenericOptionsWindow( view->LargeFont()->Width() * 13 + 30,
                           view->SmallFont()->Height() * 11 + 30,
                           "Video Options", view ) {
  SDL_Rect std_sizes[] = { { 0, 0, 1280, 1024 }, { 0, 0, 1024, 768 },
                           { 0, 0, 800, 600}, { 0, 0, 640, 480 } };
  SDL_Rect *std_modes[5];
  short current = -1;

  // create modes list
  SDL_Rect **sdl_modes = SDL_ListModes( NULL, SDL_FULLSCREEN );

  // if any mode is ok offer some standard ones
  if ( sdl_modes == (SDL_Rect **)-1 ) {
    sdl_modes = std_modes;
    std_modes[0] = &std_sizes[0];
    std_modes[1] = &std_sizes[1];
    std_modes[2] = &std_sizes[2];
    std_modes[3] = &std_sizes[3];
    std_modes[4] = NULL;
  }
 
  if ( sdl_modes != NULL ) {
    for ( int i = 0; sdl_modes[i]; ++i ) AddMode( sdl_modes[i] );

    // add current mode
    SDL_Rect screen = { 0, 0, view->Width(), view->Height() };
    current = AddMode( &screen );
  }

  // create widgets
  const Rect &b = GetBounds();

  modewidget = new TextListWidget( 0, b.x + 5, b.y + 5,
                    b.w - 10, sfont->Height() * 10, &modes, current,
                    WIDGET_HSCROLLKEY|WIDGET_VSCROLLKEY|WIDGET_ALIGN_CENTER,
                    NULL, this );

  fswidget = new CheckboxWidget( 0, modewidget->LeftEdge() + 10,
                   modewidget->TopEdge() + modewidget->Height() + 10,
                   DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, view->IsFullScreen(), 'f',
                   WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
                   "Fullscreen", this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::~VideoOptionsWindow
// DESCRIPTION: Destroy the video options window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

VideoOptionsWindow::~VideoOptionsWindow( void ) {
  TLWNode *n2;

  for ( TLWNode *n = static_cast<TLWNode *>(modes.Head()); n; n = n2 ) {
    n2 = static_cast<TLWNode *>(n->Next());
    n->Remove();

    SDL_Rect *rect = (SDL_Rect *)n->UserData();
    delete rect;
    delete n;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::AddMode
// DESCRIPTION: Add another resultion to the list. Duplicates and
//              invalid sizes will be rejected.
// PARAMETERS : res - resolution to add
// RETURNS    : position at which the mode has been added (or existed
//              before); -1 if mode was rejected
////////////////////////////////////////////////////////////////////////

short VideoOptionsWindow::AddMode( SDL_Rect *res ) {
  short rc = -1;

  if ( (res->w >= MIN_XRES) && (res->h >= MIN_YRES) ) {
    TLWNode *walk, *prev = NULL;
    bool add = true;
    rc = 0;

    for ( walk = static_cast<TLWNode *>(modes.Head());
          walk; prev = walk, walk = static_cast<TLWNode *>(walk->Next()) ) {
      SDL_Rect *nres = (SDL_Rect *)walk->UserData();
      if ( nres->w <= res->w ) {
        if ( (nres->w == res->w) && (nres->h == res->h) ) add = false;
        break;
      }
      ++rc;
    }

    if ( add ) {
      char buf[16];
      sprintf( buf, "%d x %d", res->w, res->h );

      SDL_Rect *mode = new SDL_Rect;
      mode->w = res->w;
      mode->h = res->h;
      TLWNode *n = new TLWNode( buf, mode );
      modes.InsertNode( n, prev );
    }
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::WidgetActivated
// DESCRIPTION: When the user pushes the 'OK' button, switch to the
//              selected video mode.
// PARAMETERS : button - calling widget
//              win    - pointer to active window
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status VideoOptionsWindow::WidgetActivated( Widget *button, Window *win ) {
  view->CloseWindow( win );

  TLWNode *mode = static_cast<TLWNode *>( modewidget->Selected() );

  if ( mode ) {
    bool fs = fswidget->Clicked();
    unsigned long mode_flags = SDL_HWSURFACE|(fs ? SDL_FULLSCREEN : 0);

    // if selected mode is the same as current mode only check for fullscreen
    // dimensions of the selected mode are available in the user_data field
    SDL_Rect *dim = (SDL_Rect *)mode->UserData();
    if ( (dim->w == view->Width()) && (dim->h == view->Height()) ) {
      if ( fs != view->IsFullScreen() ) view->ToggleFullScreen();
    } else {
      view->SetVideoMode( dim->w, dim->h, DISPLAY_BPP, mode_flags );
    }
  }
  return GUI_OK;
}


////////////////////////////////////////////////////////////////////////
// NAME       : GenericOptionsWindow::GenericOptionsWindow
// DESCRIPTION: Show an options menu.
// PARAMETERS : w     - width (only the part needed by the subclass)
//              h     - height (only the part needed by the subclass)
//              title - window title
//              view  - view to attach the window to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

GenericOptionsWindow::GenericOptionsWindow( unsigned short w,
       unsigned short h, const char *title, View *view ) :
       Window( 0, 0, w + 10,
               h + view->LargeFont()->Height() + view->SmallFont()->Height() +
               25, WIN_CENTER, view ),
       clientarea(5, 13 + lfont->Height(), Width() - 10,
                  Height() - lfont->Height() - sfont->Height() - 25),
       title(title) {

  Widget *wd = new ButtonWidget( B_ID_OK, 1, Height() - sfont->Height() - 9,
                   Width() - 2, sfont->Height() + 8, 'o', WIDGET_DEFAULT, "OK", this );
  wd->SetHook( this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : GenericOptionsWindow::Draw
// DESCRIPTION: Draw the options window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void GenericOptionsWindow::Draw( void ) {
  Window::Draw();

  DrawBox( clientarea, BOX_RECESSED );

  lfont->Write( title, this, (w - lfont->TextWidth(title)) / 2 + 3,
                             8, view->GetBGPen() );
  lfont->Write( title, this, (w - lfont->TextWidth(title)) / 2,
                             5, view->GetFGPen() );
}


////////////////////////////////////////////////////////////////////////
// NAME       : GeneralOptionsWindow::GeneralOptionsWindow
// DESCRIPTION: Show the general options menu.
// PARAMETERS : mv   - current map view
//              view - view to attach the window to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

GeneralOptionsWindow::GeneralOptionsWindow( MapView &mv, View *view ) :
     GenericOptionsWindow( view->LargeFont()->Width() * 15 + 30,
                           view->SmallFont()->Height() * 2 + 35,
                           "General Options", view ), mv(mv) {
  // create widgets
  const Rect &b = GetBounds();

  // damage indicator
  diwidget = new CheckboxWidget( 0, b.x + 5, b.y + 5,
               DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, mv.UnitStatsEnabled(),
               'd', WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
               "Show damage indicators", this );

  repwidget = new CheckboxWidget( 0, b.x + 5, b.y + + diwidget->Height() + 10,
              DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, CFOptions.GetTurnReplay(),
              't', WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
              "Show turn replays", this );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : GeneralOptionsWindow::WidgetActivated
// DESCRIPTION: When the user pushes the 'OK' button, propagate the
//              selected options.
// PARAMETERS : widget - calling widget
//              win    - pointer to active window
// RETURNS    : GUI_OK
////////////////////////////////////////////////////////////////////////

GUI_Status GeneralOptionsWindow::WidgetActivated( Widget *widget, Window *win ) {
  view->CloseWindow( win );

  if ( mv.UnitStatsEnabled() != diwidget->Clicked() ) {
    CFOptions.SetDamageIndicator( diwidget->Clicked() );

    if ( diwidget->Clicked() ) mv.EnableUnitStats();
    else mv.DisableUnitStats();

    // now redraw the map display
    mv.Draw();
    view->Refresh( mv );
  }

  CFOptions.SetTurnReplay( repwidget->Clicked() );

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : SoundOptionsWindow::SoundOptionsWindow
// DESCRIPTION: Show the sound options menu.
// PARAMETERS : view - view to attach the window to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

#ifndef DISABLE_SOUND

SoundOptionsWindow::SoundOptionsWindow( View *view ) :
     GenericOptionsWindow( view->LargeFont()->Width() * 13 + 30,
                           view->SmallFont()->Height() * 2 + 20,
                           "Sound Options", view ),
     volgfx( view->GetSystemIcons(), 145, 46, 12 ,12 ) {
  // create widgets
  const Rect &b = GetBounds();
  Widget *wd;

  // sfx
  wd = new CheckboxWidget( B_ID_SFX, b.x + 5, b.y + 5,
           DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, Audio::GetSfxState(),
           's', WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
           "SFX", this );
  wd->SetHook( this );

  // sfx volume
  short xoff = wd->LeftEdge() + wd->Width() + sfont->TextWidth("Music") +
               volgfx.Width() + 15;
  sfxvol = new SliderWidget( S_ID_VOL_SFX,
           xoff, wd->TopEdge() + (wd->Height() - DEFAULT_SLIDER_SIZE)/2,
           b.w - xoff - 5, DEFAULT_SLIDER_SIZE, 0, MIX_MAX_VOLUME,
           Audio::GetSfxVolume(), 20, 'f',
           WIDGET_HSCROLL|WIDGET_HSCROLLKEY|
             (Audio::GetSfxState() ? 0 : WIDGET_DISABLED),
           NULL, this );
  sfxvol->SetHook( this );

  // music
  wd = new CheckboxWidget( B_ID_MUSIC,
           wd->LeftEdge(), wd->TopEdge() + wd->Height() + 5,
           DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, Audio::GetMusicState(), 'm',
           WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT|WIDGET_DISABLED,
           "Music", this );
  wd->SetHook( this );

  // music volume
  musicvol = new SliderWidget( S_ID_VOL_MUSIC,
           xoff, wd->TopEdge() + (wd->Height() - DEFAULT_SLIDER_SIZE)/2,
           b.w - xoff - 5, DEFAULT_SLIDER_SIZE, 0, MIX_MAX_VOLUME,
           0, 20, 'u', WIDGET_HSCROLL|WIDGET_VSCROLLKEY|WIDGET_DISABLED,
           NULL, this );
  musicvol->SetHook( this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : SoundOptionsWindow::WidgetActivated
// DESCRIPTION: When the user activates a widget, propagate the changes
//              to the sound layer.
// PARAMETERS : widget - calling widget
//              win    - pointer to active window
// RETURNS    : GUI_OK
////////////////////////////////////////////////////////////////////////

GUI_Status SoundOptionsWindow::WidgetActivated( Widget *widget, Window *win ) {

  switch ( widget->ID() ) {
  case B_ID_SFX:
    Audio::ToggleSfxState();

    if ( Audio::GetSfxState() ) sfxvol->Enable();
    else sfxvol->Disable();
    break;

  case S_ID_VOL_SFX:
    Audio::SetSfxVolume( sfxvol->Level() );
    break;

  case B_ID_MUSIC:
    Audio::ToggleMusicState();

    if ( Audio::GetMusicState() ) musicvol->Enable();
    else musicvol->Disable();
    break;

  case S_ID_VOL_MUSIC:
    Audio::SetMusicVolume( musicvol->Level() );
    break;

  case B_ID_OK:
    view->CloseWindow( this );
    break;

  default:
    break;
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : SoundOptionsWindow::Draw
// DESCRIPTION: Draw the sound options window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void SoundOptionsWindow::Draw( void ) {
  GenericOptionsWindow::Draw();

  short xoff = sfxvol->LeftEdge() - volgfx.Width() - 5;

  volgfx.Draw( this, xoff,
         sfxvol->TopEdge() + (sfxvol->Height() - volgfx.Height())/2 );
  volgfx.Draw( this, xoff,
         musicvol->TopEdge() + (musicvol->Height() - volgfx.Height())/2 );
}
#endif

