/*
 * Copyright 2001 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/queue.h>
#include <sys/socket.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>

#include "config.h"

#define DNS_MAXLRUSIZE	1024

struct dns_entry {
	TAILQ_ENTRY(dns_entry) next;

	int positive;
	char *name;
	struct addrinfo *ai;
};

struct dns_entry storage[DNS_MAXLRUSIZE];
int entries;

struct stats {
	int hits;
	int replacements;
	int iterations;
	int calls;
} dns_stats;


TAILQ_HEAD(dns_list, dns_entry) dnsqueue;

void
dns_init(void)
{
	TAILQ_INIT(&dnsqueue);

	entries = 0;
	memset(storage, 0, sizeof(storage));
	memset(&dns_stats, 0, sizeof(dns_stats));
}

void
dns_print_stats(void)
{
	fprintf(stdout, "DNS queries: %d\n", dns_stats.calls);
	fprintf(stdout, "DNS cache hits: %d, replacements: %d\n",
		dns_stats.hits, dns_stats.replacements);
	fprintf(stdout, "DNS average list search: %f\n",
		(float)dns_stats.iterations/dns_stats.calls);
}

void
dns_free(struct dns_entry *dns)
{
	if (dns->name)
		free(dns->name);
	if (dns->ai)
		freeaddrinfo(dns->ai);
	memset(dns, 0, sizeof (struct dns_entry));
}

struct addrinfo *
dns_roundrobin(struct addrinfo **pai)
{
	struct addrinfo *first = *pai;
	struct addrinfo *tmp;

	if (first == NULL || first->ai_next == NULL)
		return (first);

	tmp = first->ai_next;
	*pai = tmp;

	/* Detach at the beginning and put at the end */
	first->ai_next = NULL;
	while (tmp->ai_next != NULL)
		tmp = tmp->ai_next;

	tmp->ai_next = first;

	return (first);
}

int
dns_resolve(char *ip, u_short port, struct addrinfo **pai)
{
	char strport[6];
	char *name;
	struct addrinfo hints, *ai;
	struct dns_entry *dns;
	int res, positive;

	dns_stats.calls++;

	TAILQ_FOREACH(dns, &dnsqueue, next) {
		dns_stats.iterations++;
		if (!strcasecmp(ip, dns->name))
			break;
	}

	if (dns != NULL) {
		dns_stats.hits++;

		TAILQ_REMOVE(&dnsqueue, dns, next);
		TAILQ_INSERT_HEAD(&dnsqueue, dns, next);

		if (dns->positive != -1)
			*pai = dns_roundrobin(&dns->ai);
		return (dns->positive);
	}

	name = strdup(ip);
	if (name == NULL) {
		warn(__FUNCTION__": strdup");
		return (-1);
	}

	/* Only use IP addresses with this interface, otherwise we
	 * will block on DNS.
	 */
	positive = 0;
        memset(&hints, 0, sizeof(hints));
        hints.ai_family = AF_INET;
        hints.ai_socktype = SOCK_STREAM;
	sprintf(strport, "%d", port);
	res = getaddrinfo(ip, strport, &hints, &ai);
        if (res != 0) {
                fprintf(stderr, __FUNCTION__": getaddrinfo(%s): %s\n",
			ip, gai_strerror(res));
		if (res != EAI_NODATA) {
			/* Temporary error, don't cache */
			free(name);
			return (-1);
		}

		/* Negative caching */
		positive = -1;
		ai = NULL;
        }


	if (entries >= DNS_MAXLRUSIZE) {
		dns_stats.replacements++;

		/* Recycle an old entry */
		dns = TAILQ_LAST(&dnsqueue, dns_list);
		TAILQ_REMOVE(&dnsqueue, dns, next);
		dns_free(dns);
	} else
		dns = &storage[entries++];

	dns->name = name;
	dns->ai = ai;
	dns->positive = positive;
	TAILQ_INSERT_HEAD(&dnsqueue, dns, next);

	if (positive != -1)
		*pai = ai;

	return (positive);
}
