/* steganalysis.word-gaps.c - Letter gap steganalysis
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank-interface.h"

/* Global data */

#define MODE_LETTER_IN_WORD   1 /* nth letter in a word */
#define MODE_LETTER_IN_STRING 2 /* nth letter in a string */
#define MODE_WORD_IN_TEXT     3 /* nth word in the text */

static int nth_letter = 1;
static int mode = MODE_LETTER_IN_WORD;

/* Function declares */
void cb_spinbox_changed (GtkEditable *editable, gpointer user_data);
void cb_change_mode(GtkToggleButton *togglebutton, gpointer user_data);

/* Plugin Interface */
/* ---------------- */

const char name[] = "steganalysis.word-gaps";
const char version[] = VERSION;
const char author[] = "Matthew Russell";
const char description[] = "Displays letters and words selected at intervals, either within words or within strings, to break hidden messages.";
const int interface_version = 1;
const int plugin_type = PLUGIN_MODE;
const char menu_string[] = "/Text/Word Gaps";

int boot(void) {
    return PLUGIN_BOOT_OK;
}

void init_plugin(void) {;}

void deactivate_plugin(void) {;}

GtkWidget *make_widget(char *text) {
    GtkWidget *hbox;
    GtkObject *adjustment;
    GtkWidget *spinbox;
    GtkWidget *label;
    GSList *group;
    GtkWidget *button_letter_word, *button_letter_string, *button_word_text;

    hbox = gtk_hbox_new(FALSE, 0);
    label = gtk_label_new("Interval:");
    gtk_widget_show(label);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    adjustment = gtk_adjustment_new((float) nth_letter, 1.0, 40.0, 1.0, 1.0, 0.0);
    spinbox = gtk_spin_button_new(GTK_ADJUSTMENT(adjustment), 0.0, 0);
    gtk_signal_connect(GTK_OBJECT(spinbox), "changed", GTK_SIGNAL_FUNC(cb_spinbox_changed), NULL);
    gtk_widget_show(spinbox);
    gtk_box_pack_start(GTK_BOX(hbox), spinbox, FALSE, FALSE, 0);
    
    button_letter_word = gtk_radio_button_new_with_label(NULL, "Letter position within each word");
    group = gtk_radio_button_group(GTK_RADIO_BUTTON(button_letter_word));
    gtk_signal_connect(GTK_OBJECT(button_letter_word), "toggled", GTK_SIGNAL_FUNC(cb_change_mode), GINT_TO_POINTER(MODE_LETTER_IN_WORD));
    gtk_widget_show(button_letter_word);
    gtk_box_pack_start(GTK_BOX(hbox), button_letter_word, FALSE, FALSE, 0);

    button_letter_string = gtk_radio_button_new_with_label(group, "Letter position within entire text");
    gtk_signal_connect(GTK_OBJECT(button_letter_string), "toggled", GTK_SIGNAL_FUNC(cb_change_mode), GINT_TO_POINTER(MODE_LETTER_IN_STRING));
    gtk_widget_show(button_letter_string);
    gtk_box_pack_start(GTK_BOX(hbox), button_letter_string, FALSE, FALSE, 0);

    group = gtk_radio_button_group(GTK_RADIO_BUTTON(button_letter_string));
    button_word_text = gtk_radio_button_new_with_label(group, "Word position within entire text");
    gtk_signal_connect(GTK_OBJECT(button_word_text), "toggled", GTK_SIGNAL_FUNC(cb_change_mode), GINT_TO_POINTER(MODE_WORD_IN_TEXT));
    gtk_widget_show(button_word_text);
    gtk_box_pack_start(GTK_BOX(hbox), button_word_text, FALSE, FALSE, 0);
    
    if (mode == MODE_LETTER_IN_WORD) 
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button_letter_word), TRUE);
    else if (mode == MODE_LETTER_IN_STRING)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button_letter_string), TRUE);
    else if (mode == MODE_WORD_IN_TEXT)
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button_word_text), TRUE);
    
    return hbox;
}

char *transform(char *text) {
    char *newtext, c;
    int word_pos = 0, in_word = FALSE, i, position = 0, length = strlen(text);
    newtext = malloc((sizeof(char) * (length + 1)));

    for (i = 0; i < length; i++) {
	c = text[i];
	if (mode == MODE_LETTER_IN_WORD) {
	    if (!isspace(c)) {
		word_pos++;
	    } else {
		word_pos = 0;
	    }
	    if (word_pos == nth_letter) {
		newtext[position++] = c;
	    }
	} else if (mode == MODE_LETTER_IN_STRING) {
	    if (nth_letter && !(i % nth_letter)) {
		newtext[position++] = c;
	    }
	} else if (mode == MODE_WORD_IN_TEXT) {
	    if (isspace(c)) {
		newtext[position++] = c;
		in_word = FALSE;
	    } else {
		/* Check to see if this is the start of a new word */
		if (!in_word) {
		    word_pos++;
		}
		if (nth_letter && !((word_pos - 1) % nth_letter))
		    newtext[position++] = c;
		in_word = TRUE;
	    }
	}
    }
    newtext[position] = '\0';
    return newtext;
}

void reset(void) {
    ;
}

/* Plugin implementation */
/* --------------------- */

/* Callbacks */
/* --------- */

void cb_spinbox_changed (GtkEditable *editable, gpointer user_data) {
    nth_letter = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(editable));
    display_text();
}

void cb_change_mode(GtkToggleButton *togglebutton, gpointer user_data) {
    if (gtk_toggle_button_get_active(togglebutton)) {
	mode = GPOINTER_TO_INT(user_data);
	display_text();
    }
}
