/*
** Copyright 1998 - 1999 Double Precision, Inc.
** See COPYING for distribution information.
*/

#if	HAVE_CONFIG_H
#include	"config.h"
#endif
#include	"imaptoken.h"
#include	"imapwrite.h"
#include	"imapscanclient.h"
#include	"fetchinfo.h"
#include	"rfc822/rfc822.h"
#include	"rfc2045/rfc2045.h"
#include	"maildir/maildirgetquota.h"
#include	"maildir/maildirquota.h"

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	<errno.h>
#if	HAVE_UNISTD_H
#include	<unistd.h>
#endif
#include	<sys/types.h>
#include	<sys/stat.h>

static const char rcsid[]="$Id: fetch.c,v 1.7 2000/04/30 15:19:22 mrsam Exp $";

extern int current_mailbox_ro;
extern struct imapscaninfo current_mailbox_info;
extern char *current_mailbox;
extern char *rfc2045id(struct rfc2045 *);

extern void msgenvelope(void (*)(const char *, size_t),
                FILE *, struct rfc2045 *);
extern void msgbodystructure( void (*)(const char *, size_t), int,
        FILE *, struct rfc2045 *);

extern int is_trash(const char *);
extern void get_message_flags(struct imapscanmessageinfo *,
	char *, struct imapflags *);
extern void append_flags(char *, struct imapflags *);

static int fetchitem(FILE *, struct fetchinfo *,
	struct imapscaninfo *,  unsigned,
	struct rfc2045 **);

static void bodystructure(FILE *, struct fetchinfo *,
	struct imapscaninfo *,  unsigned,
	struct rfc2045 *);

static void body(FILE *, struct fetchinfo *,
	struct imapscaninfo *,  unsigned,
	struct rfc2045 *);

static void fetchmsgbody(FILE *, struct fetchinfo *,
	struct imapscaninfo *,  unsigned,
	struct rfc2045 *);

static void dofetchmsgbody(FILE *, struct fetchinfo *,
	struct imapscaninfo *,  unsigned,
	struct rfc2045 *);

static void envelope(FILE *, struct fetchinfo *,
	struct imapscaninfo *,  unsigned,
	struct rfc2045 *);

static void doflags(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned, struct rfc2045 *);

static void internaldate(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned, struct rfc2045 *);

static void uid(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned, struct rfc2045 *);

static void all(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned, struct rfc2045 *);

static void fast(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned, struct rfc2045 *);

static void full(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned, struct rfc2045 *);

static void rfc822size(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned, struct rfc2045 *);

#if 0
static void do_envelope(FILE *, struct fetchinfo *,
	struct imapscanmessageinfo *, struct rfc2045 *);
#endif

static void dofetchheadersbuf(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned,
	struct rfc2045 *,
	int (*)(struct fetchinfo *fi, const char *));
static void dofetchheadersfile(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned,
	struct rfc2045 *,
	int (*)(struct fetchinfo *fi, const char *));

static void print_bodysection_partial(struct fetchinfo *,
		void (*)(const char *));
static void print_bodysection_output(const char *);

static int dofetchheaderfields(struct fetchinfo *, const char *);
static int dofetchheadernotfields(struct fetchinfo *, const char *);
static int dofetchheadermime(struct fetchinfo *, const char *);

static void rfc822(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned,
	struct rfc2045 *);

static void rfc822header(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned,
	struct rfc2045 *);

static void rfc822text(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned,
	struct rfc2045 *);

void fetchflags(unsigned long);

static void fetcherrorprt(const char *p)
{
	fprintf(stderr, "%s", p);
}

static void fetcherror(const char *errmsg,
		struct fetchinfo *fi,
		struct imapscaninfo *info, unsigned j)
{
struct imapscanmessageinfo *mi=info->msgs+j;

	fprintf(stderr, "IMAP FETCH ERROR: %s, uid=%u, filename=%s: %s",
		errmsg, (unsigned)getuid(), mi->filename, fi->name);
	if (fi->bodysection)
		print_bodysection_partial(fi, &fetcherrorprt);
	fprintf(stderr, "\n");
}

int reflag_filename(struct imapscanmessageinfo *mi, struct imapflags *flags,
	int fd)
{
char    *p, *q, *r;
int	rc=0;
struct	imapflags old_flags;
struct	stat	stat_buf;
char	quotabuf[QUOTABUFSIZE];

	get_message_flags(mi, 0, &old_flags);
	p=malloc(strlen(mi->filename)+20);
	if (!p)	write_error_exit(0);
	strcpy(p, mi->filename);
	if ((q=strrchr(p, ':')) != 0)	*q=0;
	strcat(p, ":2,");
	append_flags(p, flags);

	q=malloc(strlen(current_mailbox)+strlen(mi->filename)+sizeof("/cur/"));
	r=malloc(strlen(current_mailbox)+strlen(p)+sizeof("/cur/"));
	if (!q || !r)	write_error_exit(0);
	strcat(strcat(strcpy(q, current_mailbox), "/cur/"), mi->filename);
	strcat(strcat(strcpy(r, current_mailbox), "/cur/"), p);
	if (strcmp(q, r))
	{
		if (!is_trash(current_mailbox)	/* Changes in TRASH take no
						** effect.
						*/
			&& old_flags.deleted != flags->deleted
			&& maildir_getquota(current_mailbox, quotabuf) == 0
			&& fstat(fd, &stat_buf) == 0)
		{
		long	nbytes;
		unsigned long unbytes;
		int	nmsgs=1;
		int	quotafd;

			if (maildir_parsequota(mi->filename, &unbytes) == 0)
				nbytes=unbytes;
			else
				nbytes=stat_buf.st_size;
			if ( flags->deleted )
			{
				nbytes= -nbytes;
				nmsgs= -nmsgs;
			}
			if ( maildir_checkquota(current_mailbox, &quotafd,
				quotabuf, nbytes, nmsgs) && errno != EAGAIN &&
				nbytes >= 0)
			{
				if (quotafd >= 0)	close(quotafd);
				rc= -1;
			}
			else
				maildir_addquota(current_mailbox, quotafd,
					quotabuf, nbytes, nmsgs);
		}

		if (rc == 0)
			rename(q, r);
	}
	free(q);
	free(r);
	free(mi->filename);
	mi->filename=p;

#if 0
	if (strncmp(current_mailbox, SHAREDSUBDIR "/",
		sizeof(SHAREDSUBDIR "/")) == 0)
		maildir_shared_updateflags(current_mailbox, p);
#endif

	return (rc);
}

int do_fetch(unsigned long n, int byuid, void *p)
{
struct fetchinfo *fi=(struct fetchinfo *)p;
int	fd;
FILE	*fp;
struct	rfc2045 *rfc2045p;
int	seen;

	fd=imapscan_openfile(current_mailbox, &current_mailbox_info, n-1);
	if (fd < 0 || (fp=fdopen(fd, "r")) == 0)
	{
		if (fd >= 0)	close(fd);
		writes("* NO Cannot open message ");
		writen(n);
		writes("\r\n");
		return (0);
	}
	writes("* ");
	writen(n);
	writes(" FETCH (");

	if (byuid)
	{
	struct fetchinfo *fip;

		for (fip=fi; fip; fip=fip->next)
			if (strcmp(fip->name, "UID") == 0)
				break;

		if (fip == 0)
		{
			writes("UID ");
			writen(current_mailbox_info.msgs[n-1].uid);
			writes(" ");
		}
	}
	seen=0;
	rfc2045p=0;
	while (fi)
	{
		if (fetchitem(fp, fi, &current_mailbox_info, n-1,
			&rfc2045p))	seen=1;
		if ((fi=fi->next) != 0)	writes(" ");
	}
	writes(")\r\n");
	fclose(fp);
	if (rfc2045p)	rfc2045_free(rfc2045p);
	if (seen && !current_mailbox_ro)
	{
	struct	imapflags	flags;

		get_message_flags(current_mailbox_info.msgs+(n-1),
				0, &flags);
		if (!flags.seen)
		{
			flags.seen=1;
			reflag_filename(&current_mailbox_info.msgs[n-1],&flags,
				fd);
			current_mailbox_info.msgs[n-1].changedflags=1;
		}
	}
	close(fd);

	if (current_mailbox_info.msgs[n-1].changedflags)
		fetchflags(n-1);
	return (0);
}

static int fetchitem(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 **mimep)
{
void (*fetchfunc)(FILE *, struct fetchinfo *,
	struct imapscaninfo *, unsigned,
	struct rfc2045 *);
int	parsemime=0;
int	rc=0;

	if (strcmp(fi->name, "ALL") == 0)
	{
		parsemime=1;
		fetchfunc= &all;
	}
	else if (strcmp(fi->name, "BODYSTRUCTURE") == 0)
	{
		parsemime=1;
		fetchfunc= &bodystructure;
	}
	else if (strcmp(fi->name, "BODY") == 0)
	{
		parsemime=1;
		fetchfunc= &body;
		if (fi->bodysection)
		{
			fetchfunc= &fetchmsgbody;
			rc=1;
		}
	}
	else if (strcmp(fi->name, "BODY.PEEK") == 0)
	{
		parsemime=1;
		fetchfunc= &body;
		if (fi->bodysection)
			fetchfunc= &fetchmsgbody;
	}
	else if (strcmp(fi->name, "ENVELOPE") == 0)
	{
		parsemime=1;
		fetchfunc= &envelope;
	}
	else if (strcmp(fi->name, "FAST") == 0)
	{
		parsemime=1;
		fetchfunc= &fast;
	}
	else if (strcmp(fi->name, "FULL") == 0)
	{
		parsemime=1;
		fetchfunc= &full;
	}
	else if (strcmp(fi->name, "FLAGS") == 0)
	{
		fetchfunc= &doflags;
	}
	else if (strcmp(fi->name, "INTERNALDATE") == 0)
	{
		fetchfunc= &internaldate;
	}
	else if (strcmp(fi->name, "RFC822") == 0)
	{
		fetchfunc= &rfc822;
		rc=1;
	}
	else if (strcmp(fi->name, "RFC822.HEADER") == 0)
	{
		fetchfunc= &rfc822header;
	}
	else if (strcmp(fi->name, "RFC822.SIZE") == 0)
	{
		parsemime=1;
		fetchfunc= &rfc822size;
	}
	else if (strcmp(fi->name, "RFC822.TEXT") == 0)
	{
		parsemime=1;
		fetchfunc= &rfc822text;
	}
	else if (strcmp(fi->name, "UID") == 0)
	{
		fetchfunc= &uid;
	}
	else	return (0);

	if (parsemime && !*mimep)
	{
		if (fseek(fp, 0L, SEEK_SET) == -1)
		{
			fetcherror("fseek", fi, i, msgnum);
			return (0);
		}
		*mimep=rfc2045_fromfp(fp);
		if (!*mimep)	write_error_exit(0);
	}

	(*fetchfunc)(fp, fi, i, msgnum, *mimep);
	return (rc);
}

static void bodystructure(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
	writes("BODYSTRUCTURE ");
	msgbodystructure(writemem, 1, fp, mimep);
}

static void body(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
	writes("BODY ");
	msgbodystructure(writemem, 0, fp, mimep);
}

static void envelope(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
	writes("ENVELOPE ");
	msgenvelope( &writemem, fp, mimep);
}

void fetchflags(unsigned long n)
{
	writes("* ");
	writen(n+1);
	writes(" FETCH (");
	doflags(0, 0, &current_mailbox_info, n, 0);
	writes(")\r\n");
}

static void doflags(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
char	buf[256];

	writes("FLAGS ");
	get_message_flags(i->msgs+msgnum, buf, 0);
	if (buf[0] == '\0')
		writes("()");
	else
	{
		writes("(");
		writes(buf);
		writes(")");
	}
	i->msgs[msgnum].changedflags=0;
}

static void internaldate(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
struct	stat	stat_buf;
char	buf[256];
char	*p, *q;

	writes("INTERNALDATE ");
	if (fstat(fileno(fp), &stat_buf) == 0)
	{
		rfc822_mkdate_buf(stat_buf.st_mtime, buf);

		/* Convert RFC822 date to imap date */

		p=strchr(buf, ',');
		if (p)	++p;
		else	p=buf;
		while (*p == ' ')	++p;
		if ((q=strchr(p, ' ')) != 0)	*q++='-';
		if ((q=strchr(p, ' ')) != 0)	*q++='-';
		writes("\"");
		writes(p);
		writes("\"");
	}
	else
		writes("NIL");
}

static void uid(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
	writes("UID ");
	writen(i->msgs[msgnum].uid);
}

static void rfc822size(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
off_t start_pos, end_pos, start_body;
off_t nlines, nbodylines;

	writes("RFC822.SIZE ");

	rfc2045_mimepos(mimep, &start_pos, &end_pos, &start_body,
		&nlines, &nbodylines);

	writen(end_pos - start_pos + nlines);
}

static void all(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
	doflags(fp, fi, i, msgnum, mimep);
	writes(" ");
	internaldate(fp, fi, i, msgnum, mimep);
	writes(" ");
	rfc822size(fp, fi, i, msgnum, mimep);
	writes(" ");
	envelope(fp, fi, i, msgnum, mimep);
}

static void fast(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
	doflags(fp, fi, i, msgnum, mimep);
	writes(" ");
	internaldate(fp, fi, i, msgnum, mimep);
	writes(" ");
	rfc822size(fp, fi, i, msgnum, mimep);
}

static void full(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
	doflags(fp, fi, i, msgnum, mimep);
	writes(" ");
	internaldate(fp, fi, i, msgnum, mimep);
	writes(" ");
	rfc822size(fp, fi, i, msgnum, mimep);
	writes(" ");
	envelope(fp, fi, i, msgnum, mimep);
	writes(" ");
	body(fp, fi, i, msgnum, mimep);
}

static void fetchmsgbody(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
	writes("BODY");
	print_bodysection_partial(fi, &print_bodysection_output);
	writes(" ");
	dofetchmsgbody(fp, fi, i, msgnum, mimep);
}

static void print_bodysection_output(const char *p)
{
	writes(p);
}

static void print_bodysection_partial(struct fetchinfo *fi,
	void (*func)(const char *))
{
	(*func)("[");
	if (fi->bodysection)
	{
	struct fetchinfo *subl;

		(*func)(fi->bodysection);
		if (fi->bodysublist)
		{
		char	*p=" (";

			for (subl=fi->bodysublist; subl; subl=subl->next)
			{
				(*func)(p);
				p=" ";
				(*func)("\"");
				(*func)(subl->name);
				(*func)("\"");
			}
			(*func)(")");
		}
	}
	(*func)("]");
	if (fi->ispartial)
	{
	char	buf[80];

		sprintf(buf, "<%lu>", (unsigned long)fi->partialstart);
		(*func)(buf);
	}
}

static void dofetchmsgbody(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *i, unsigned msgnum,
	struct rfc2045 *mimep)
{
const char *p=fi->bodysection;
off_t start_pos, end_pos, start_body;
off_t nlines, nbodylines;
unsigned long cnt;
char	buf[BUFSIZ];
unsigned bufptr;
unsigned long skipping;
int	ismsgrfc822=1;

	while (p && isdigit((int)(unsigned char)*p))
	{
	unsigned n=0;

		do
		{
			n=n*10 + (*p++ - '0');
		} while (isdigit((int)(unsigned char)*p));

		if (mimep)
		{
			if (ismsgrfc822)
			{
				if (mimep->firstpart == 0)
				{
					/* Not a multipart, n must be 1 */
					if (n != 1)
						mimep=0;
					if (*p == '.')
						++p;
					continue;
				}
				ismsgrfc822=0;
			}

			mimep=mimep->firstpart;
			while (mimep)
			{
				if (!mimep->isdummy && --n == 0)
					break;
				mimep=mimep->next;
			}
			if (mimep && mimep->firstpart &&
				!mimep->firstpart->isdummy)
				/* This is a message/rfc822 part */
			{
				mimep=mimep->firstpart;
				ismsgrfc822=1;
			}
		}
		if (*p == '.')
			++p;
	}
	if (mimep == 0)
	{
		writes("{0}\r\n");
		return;
	}

	rfc2045_mimepos(mimep, &start_pos, &end_pos, &start_body,
		&nlines, &nbodylines);


	if (p && strcmp(p, "TEXT") == 0)
	{
		if (fseek(fp, start_body, SEEK_SET) == -1)
		{
			writes("{0}\r\n");
			fetcherror("fseek", fi, i, msgnum);
			return;
		}

		cnt=end_pos - start_body + nbodylines;
	}
	else if (p && strcmp(p, "HEADER") == 0)
	{
		if (fseek(fp, start_pos, SEEK_SET) == -1)
		{
			writes("{0}\r\n");
			fetcherror("fseek", fi, i, msgnum);
			return;
		}
		cnt= start_body - start_pos + (nlines - nbodylines);
	}
	else if (p && strcmp(p, "HEADER.FIELDS") == 0)
	{
		if (start_body - start_pos <= BUFSIZ)
			dofetchheadersbuf(fp, fi, i, msgnum, mimep,
				&dofetchheaderfields);
		else
			dofetchheadersfile(fp, fi, i, msgnum, mimep,
				&dofetchheaderfields);
		return;
	}
	else if (p && strcmp(p, "HEADER.FIELDS.NOT") == 0)
	{
		if (start_body - start_pos <= BUFSIZ)
			dofetchheadersbuf(fp, fi, i, msgnum, mimep,
				&dofetchheadernotfields);
		else
			dofetchheadersfile(fp, fi, i, msgnum, mimep,
				&dofetchheadernotfields);
		return;
	}
	else if (p && strcmp(p, "MIME") == 0)
	{
		if (start_body - start_pos <= BUFSIZ)
			dofetchheadersbuf(fp, fi, i, msgnum, mimep,
				&dofetchheadermime);
		else
			dofetchheadersfile(fp, fi, i, msgnum, mimep,
				&dofetchheadermime);
		return;
	}
	else if (*fi->bodysection == 0)
	{
		if (fseek(fp, start_pos, SEEK_SET) == -1)
		{
			writes("{0}\r\n");
			fetcherror("fseek", fi, i, msgnum);
			return;
		}
		cnt= end_pos - start_pos + nlines;
	}
	else	/* Last possibility: entire body */
	{
		if (fseek(fp, start_body, SEEK_SET) == -1)
		{
			writes("{0}\r\n");
			fetcherror("fseek", fi, i, msgnum);
			return;
		}
		cnt= end_pos - start_body + nbodylines;
	}

	skipping=0;
	if (fi->ispartial)
	{
		skipping=fi->partialstart;
		if (skipping > cnt)	skipping=cnt;
		cnt -= skipping;
		if (fi->ispartial > 1 && cnt > fi->partialend)
			cnt=fi->partialend;
	}

	writes("{");
	writen(cnt);
	writes("}\r\n");
	bufptr=0;
	writeflush();

	while (cnt)
	{
	int	c=getc(fp);

		if (c == EOF)
		{
			fetcherror("unexpected EOF", fi, i, msgnum);
			_exit(1);
		}

		if (c == '\n')
		{
			if (skipping)
				--skipping;
			else
			{
				if (bufptr >= sizeof(buf))
				{
					writemem(buf, sizeof(buf));
					bufptr=0;
					/*writeflush();*/
				}
				buf[bufptr++]='\r';
				--cnt;

				if (cnt == 0)
					break;
			}
		}

		if (skipping)
			--skipping;
		else
		{
			if (bufptr >= sizeof(buf))
			{
				writemem(buf, sizeof(buf));
				bufptr=0;
				/*writeflush();*/
			}
			buf[bufptr++]=c;
			--cnt;
		}
	}
	writemem(buf, bufptr);
	writeflush();
}

static int dofetchheaderfields(struct fetchinfo *fi, const char *name)
{
	for (fi=fi->bodysublist; fi; fi=fi->next)
	{
	int	i, a, b;

		if (fi->name == 0)	continue;
		for (i=0; fi->name[i] && name[i]; i++)
		{
			a=(unsigned char)name[i];
			a=toupper(a);
			b=fi->name[i];
			b=toupper(b);
			if (a != b)	break;
		}
		if (fi->name[i] == 0 && name[i] == 0)	return (1);
	}

	return (0);
}

static int dofetchheadernotfields(struct fetchinfo *fi, const char *name)
{
	return (!dofetchheaderfields(fi, name));
}

static int dofetchheadermime(struct fetchinfo *fi, const char *name)
{
int	i, a;
static const char mv[]="MIME-VERSION";

	for (i=0; i<sizeof(mv)-1; i++)
	{
		a= (unsigned char)name[i];
		a=toupper(a);
		if (a != mv[i])	break;
	}
	if (mv[i] == 0 && name[i] == 0)	return (1);

	for (i=0; i<8; i++)
	{
		a= (unsigned char)name[i];
		a=toupper(a);
		if (a != "CONTENT-"[i])	return (0);
	}
	return (1);
}

static void dofetchheadersbuf(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *info, unsigned msgnum,
	struct rfc2045 *mimep,
	int (*headerfunc)(struct fetchinfo *fi, const char *))
{
off_t start_pos, end_pos, start_body;
off_t nlines, nbodylines;
size_t i,j,k,l;
int	c;
char	buf[BUFSIZ+2];
int	goodheader;
unsigned long skipping;
unsigned long cnt;
char	*p;

	rfc2045_mimepos(mimep, &start_pos, &end_pos, &start_body,
		&nlines, &nbodylines);
	if (fseek(fp, start_pos, SEEK_SET) == -1)
	{
		writes("{0}\r\n");
		fetcherror("fseek", fi, info, msgnum);
		return;
	}
	for (i=0; i<start_body-start_pos; i++)
	{
		c=getc(fp);
		if (c == EOF)
		{
			fetcherror("unexpected EOF", fi, info, msgnum);
			_exit(1);
		}
		buf[i]=c;
	}
	goodheader= (*headerfunc)(fi, "");

	l=0;
	for (j=0; j<i; )
	{
		if (buf[j] != '\n' && buf[j] != '\r' &&
			!isspace((int)(unsigned char)buf[j]))
		{
			goodheader= (*headerfunc)(fi, "");

			for (k=j; k<i; k++)
			{
				if (buf[k] == '\n' || buf[k] == ':')
					break;
			}

			if (k < i && buf[k] == ':')
			{
				buf[k]=0;
				goodheader=(*headerfunc)(fi, buf+j);
				buf[k]=':';
			}
		}
		else if (buf[j] == '\n')
			goodheader=0;

		for (k=j; k<i; k++)
			if (buf[k] == '\n')
			{
				++k;
				break;
			}

		if (goodheader)
		{
			while (j<k)
				buf[l++]=buf[j++];
		}
		j=k;
	}

	buf[l++]='\n';	/* Always append a blank line */

	cnt=l;
	for (i=0; i<l; i++)
		if (buf[i] == '\n')	++cnt;

	skipping=0;
	if (fi->ispartial)
	{
		skipping=fi->partialstart;
		if (skipping > cnt)	skipping=cnt;
		cnt -= skipping;
		if (fi->ispartial > 1 && cnt > fi->partialend)
			cnt=fi->partialend;
	}

	writes("{");
	writen(cnt);
	writes("}\r\n");
	p=buf;
	while (skipping)
	{
		if (*p == '\n')
		{
			--skipping;
			if (skipping == 0)
			{
				if (cnt)
				{
					writes("\n");
					--cnt;
				}
				break;
			}
		}
		--skipping;
		++p;
	}

	while (cnt)
	{
		if (*p == '\n')
		{
			writes("\r");
			if (--cnt == 0)	break;
			writes("\n");
			--cnt;
			++p;
			continue;
		}
		for (i=0; i<cnt; i++)
			if (p[i] == '\n')
				break;
		writemem(p, i);
		p += i;
		cnt -= i;
	}
}

struct fetchheaderinfo {
	unsigned long skipping;
	unsigned long cnt;
	} ;

static void countheader(struct fetchheaderinfo *, const char *, size_t);

static void printheader(struct fetchheaderinfo *, const char *, size_t);

static void dofetchheadersfile(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *info, unsigned msgnum,
	struct rfc2045 *mimep,
	int (*headerfunc)(struct fetchinfo *fi, const char *))
{
off_t start_pos, end_pos, start_body, left;
off_t nlines, nbodylines;
size_t i;
int	c, pass;
char	buf1[256];
int	goodheader;
struct	fetchheaderinfo finfo;

	finfo.cnt=0;
	for (pass=0; pass<2; pass++)
	{
	void (*func)(struct fetchheaderinfo *, const char *, size_t)=
			pass ? printheader:countheader;

		rfc2045_mimepos(mimep, &start_pos, &end_pos, &start_body,
			&nlines, &nbodylines);
		if (fseek(fp, start_pos, SEEK_SET) == -1)
		{
			writes("{0}\r\n");
			fetcherror("fseek", fi, info, msgnum);
			return;
		}
		if (pass)
		{
			finfo.skipping=0;
			if (fi->ispartial)
			{
				finfo.skipping=fi->partialstart;
				if (finfo.skipping > finfo.cnt)
					finfo.skipping=finfo.cnt;
				finfo.cnt -= finfo.skipping;
				if (fi->ispartial > 1 &&
					finfo.cnt > fi->partialend)
					finfo.cnt=fi->partialend;
			}

			writes("{");
			writen(finfo.cnt+2);	/* BUG */
			writes("}\r\n");
		}
		left=start_body - start_pos;

		goodheader= (*headerfunc)(fi, "");
		while (left)
		{
			for (i=0; i<sizeof(buf1)-1 && i<left; i++)
			{
				c=getc(fp);
				if (c == EOF)
				{
					fetcherror("unexpected EOF", fi, info, msgnum);
					_exit(1);
				}

				if (c == '\n' || c == ':')
				{
					ungetc(c, fp);
					break;
				}
				buf1[i]=c;
			}
			buf1[i]=0;
			left -= i;

			if (buf1[0] != '\n' && buf1[0] != '\r' &&
				!isspace((int)(unsigned char)buf1[0]))
				goodheader= (*headerfunc)(fi, buf1);
			else if (buf1[0] == '\n')
				goodheader=0;

			if (!goodheader)
			{
				while (left)
				{
					c=getc(fp);
					--left;
					if (c == EOF)
					{
						fetcherror("unexpected EOF", fi, info, msgnum);
						_exit(1);
					}
					if (c == '\n')	break;
				}
				continue;
			}

			(*func)(&finfo, buf1, i);

			i=0;
			while (left)
			{
				c=getc(fp);
				if (c == EOF)
				{
					fetcherror("unexpected EOF", fi, info, msgnum);
					_exit(1);
				}
				--left;
				if (i >= sizeof(buf1))
				{
					(*func)(&finfo, buf1, i);
					i=0;
				}
				if (c == '\n')
				{
					(*func)(&finfo, buf1, i);
					buf1[0]='\r';
					i=1;
				}
				buf1[i++]=c;
				if (c == '\n')	break;
			}
			(*func)(&finfo, buf1, i);
			if (pass && finfo.cnt == 0)	break;
		}
	}
	writes("\r\n");	/* BUG */
}

static void countheader(struct fetchheaderinfo *fi, const char *p, size_t s)
{
	fi->cnt += s;
}

static void printheader(struct fetchheaderinfo *fi, const char *p, size_t s)
{
	if (fi->skipping)
	{
		if (fi->skipping > s)
		{
			fi->skipping -= s;
			return;
		}
		p += fi->skipping;
		s -= fi->skipping;
		fi->skipping=0;
	}
	if (s > fi->cnt)	s=fi->cnt;
	writemem(p, s);
	fi->cnt -= s;
}

static void rfc822(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *info, unsigned msgnum,
	struct rfc2045 *rfcp)
{
unsigned long n=0;
int	c;
char	buf[BUFSIZ];
unsigned long i;

	writes("RFC822 ");

	if (fseek(fp, 0L, SEEK_SET) == -1)
	{
		fetcherror("fseek", fi, info, msgnum);
		writes("{0}\r\n");
		return;
	}
	while ((c=getc(fp)) != EOF)
	{
		++n;
		if (c == '\n')	++n;
	}

	if (fseek(fp, 0L, SEEK_SET) == -1)
	{
		fetcherror("fseek", fi, info, msgnum);
		writes("{0}\r\n");
		return;
	}
	writes("{");
	writen(n);
	writes("}\r\n");

	i=0;
	while (n)
	{
		c=getc(fp);
		if (c == '\n')
		{
			if (i >= sizeof(buf))
			{
				writemem(buf, i);
				i=0;
			}
			buf[i++]='\r';
			if (--n == 0)	break;
		}

		if (i >= sizeof(buf))
		{
			writemem(buf, i);
			i=0;
		}
		buf[i++]=c;
		--n;
	}
	writemem(buf, i);
}

static void rfc822header(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *info, unsigned msgnum,
	struct rfc2045 *rfcp)
{
unsigned long n=0;
int	c;
char	buf[BUFSIZ];
unsigned long i;
int	eol;

	writes("RFC822.HEADER ");

	if (fseek(fp, 0L, SEEK_SET) == -1)
	{
		fetcherror("fseek", fi, info, msgnum);
		writes("{0}\r\n");
		return;
	}

	eol=0;
	while ((c=getc(fp)) != EOF)
	{
		++n;
		if (c != '\n')
		{
			eol=0;
			continue;
		}
		++n;
		if (eol)	break;
		eol=1;
	}

	if (fseek(fp, 0L, SEEK_SET) == -1)
	{
		fetcherror("fseek", fi, info, msgnum);
		writes("{0}\r\n");
		return;
	}
	writes("{");
	writen(n);
	writes("}\r\n");

	i=0;
	while (n)
	{
		c=getc(fp);
		if (c == '\n')
		{
			if (i >= sizeof(buf))
			{
				writemem(buf, i);
				i=0;
			}
			buf[i++]='\r';
			if (--n == 0)	break;
		}

		if (i >= sizeof(buf))
		{
			writemem(buf, i);
			i=0;
		}
		buf[i++]=c;
		--n;
	}
	writemem(buf, i);
}

static void rfc822text(FILE *fp, struct fetchinfo *fi,
	struct imapscaninfo *info, unsigned msgnum,
	struct rfc2045 *rfcp)
{
off_t start_pos, end_pos, start_body;
off_t nlines, nbodylines;
unsigned long i;
int	c;
char	buf[BUFSIZ];
unsigned l;

	writes("RFC822.TEXT {");

	rfc2045_mimepos(rfcp, &start_pos, &end_pos, &start_body,
		&nlines, &nbodylines);

	if (fseek(fp, start_body, SEEK_SET) == -1)
	{
		fetcherror("fseek", fi, info, msgnum);
		writes("0}\r\n");
		return;
	}

	i=end_pos - start_body + nbodylines;

	writen(i);
	writes("}\r\n");

	l=0;
	while (i)
	{
		c=getc(fp);
		if (c == EOF)
		{
			fetcherror("unexpected EOF", fi, info, msgnum);
			_exit(1);
		}
		--i;
		if (l >= sizeof(BUFSIZ))
		{
			writemem(buf, l);
			l=0;
		}
		if (c == '\n' && i)
		{
			--i;
			buf[l++]='\r';
			if (l >= sizeof(BUFSIZ))
			{
				writemem(buf, l);
				l=0;
			}
		}
		buf[l++]=c;
	}
	writemem(buf, l);
}
