/*  $Id: CosmoSmashEngine.h,v 1.31 2008/11/15 03:56:36 sarrazip Exp $
    CosmoSmashEngine.h - A space rock shooting video game engine.

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2000-2004 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_CosmoSmashEngine
#define _H_CosmoSmashEngine

#include "Command.h"

class Controller;

#include <flatzebra/GameEngine.h>
#include <flatzebra/Sprite.h>
#include <flatzebra/SoundMixer.h>
#include <flatzebra/KeyState.h>

#include <string>
#include <vector>
#include <iostream>


class CosmoSmashEngine : public flatzebra::GameEngine
/*  A space rock shooting video game engine.
*/
{
public:
    CosmoSmashEngine(long initialScore,
			bool mirrorHyperspace, bool useSound,
			bool fullScreen, Controller *controller,
			std::ostream *gameScriptStream)
						    throw(int, std::string);
    /*  See base class.

	Throws a string or an integer code if an error occurs.

	If 'mirrorHyperspace' is true, then hyperspace pivots around
	the center of the ground.  If 'mirrorHyperspace' is false,
	then hyperspace is random.

	'controller' must be a non-null pointer to an object constructed
	with operator new.
	The destructor of this class will call delete on 'controller'.
    */

    virtual ~CosmoSmashEngine();
    /*  Destroys the Controller object given to the constructor,
	by calling operator delete.
	That object must have been constructed with operator new.
    */

    virtual void processKey(SDLKey keysym, bool pressed);
    /*  Inherited.
    */

    virtual bool tick();
    /*  Inherited.
    */

protected:

    // Inherited:
    bool animatePlayer();
    void animateAutomaticCharacters();
    void detectCollisions();
    void restoreBackground();
    void drawSprites();

private:

    typedef std::vector<flatzebra::Couple> CoupleList;

    enum PlayerBoostType
    {
	NO_BOOST, TRIPLE_BULLETS, CLOSER_BULLETS
    };


    bool paused;
    unsigned long tickCount;
    bool useSound;

    // SDL color values (see SDL_MapRGB()):
    Uint32 blackColor;
    Uint32 whiteColor;
    Uint32 greenColor;

    bool mirrorHyperspace;
    bool useGameExtensions;
    unsigned long playerBoostTicks;
    PlayerBoostType playerBoostType;

    flatzebra::PixmapArray playerPA;
    flatzebra::Sprite *playerSprite;

    flatzebra::PixmapArray baseBulletPA;
    flatzebra::SpriteList  baseBulletSprites;

    flatzebra::PixmapArray bigRockPA;
    flatzebra::PixmapArray smallRockPA;
    flatzebra::SpriteList rockSprites;  // contains big and small rock sprites

    flatzebra::PixmapArray bigSpinnerPA;
    flatzebra::PixmapArray smallSpinnerPA;
    flatzebra::SpriteList spinnerSprites;

    flatzebra::PixmapArray pulsarPA;
    flatzebra::SpriteList pulsarSprites;

    flatzebra::PixmapArray saucerPA;
    flatzebra::SpriteList saucerSprites;

    flatzebra::PixmapArray saucerBulletPA;
    flatzebra::SpriteList saucerBulletSprites;

    flatzebra::PixmapArray explosionPA;
    flatzebra::SpriteList  explosionSprites;

    flatzebra::PixmapArray questionPA;
    flatzebra::SpriteList questionSprites;

    int groundPos;     // y position of highest pixel row of ground
    int groundHeight;  // height of ground in pixels
    int mountainTopPos;  // y position of highest possible pixel for mountains

    Controller *controller;
    flatzebra::KeyState quitKS;

    long   initScore;     // score at which a game starts (determines level)

    long   theScore;      // player's score in points
    long   thePeakScore;  // not displayed
    int    theLevel;      // level number (1--6) (depends on theScore)
    bool   updateScore;   // indicates if player's score needs to be rewritten
    flatzebra::Couple scoreAreaPos;

    int    numLives;  // number of player lives left
    bool   updateNumLives;
    flatzebra::Couple numLivesAreaPos;

    unsigned long timeBeforeNextPulsar;  // in ticks
    unsigned long timeBeforeNextSaucer;
    unsigned long timeBeforeNextQuestion;

    CoupleList starPositions;
    CoupleList mountainRangePositions;


    /*  SOUND EFFECTS:
    */
    flatzebra::SoundMixer *theSoundMixer;  // see method playSoundEffect()
    enum { NUM_ROCK_HIT_SOUNDS = 2 };
    flatzebra::SoundMixer::Chunk rockHitSounds[NUM_ROCK_HIT_SOUNDS];
    flatzebra::SoundMixer::Chunk playerHitSound;
    flatzebra::SoundMixer::Chunk pulsarBeepSound;
    flatzebra::SoundMixer::Chunk saucerShootingSound;
    flatzebra::SoundMixer::Chunk hyperspaceSound;
    flatzebra::SoundMixer::Chunk cadenceSound;


    // Game script writing:
    std::ostream *gameScriptStream;  // null means not saving


    /*  Implementation functions:
    */
    void fireBullet(flatzebra::Couple pos);
    void initializeSprites() throw(flatzebra::PixmapLoadError);
    void initializeMisc(bool useSound) throw(std::string);
    void setTimeBeforeNextPulsar();
    void setTimeBeforeNextSaucer();
    void setTimeBeforeNextQuestion();
    int getSpeedUp() const;
    void readKeyboardCommands();
    flatzebra::Sprite *createExplosionSprite(flatzebra::Couple posOfCenterOfExplosion,
				    int timeToLive);
    std::pair<flatzebra::Sprite *, flatzebra::Sprite *> splitBigRock(const flatzebra::Sprite &bigRock) const;
    void makeSpriteExplode(const flatzebra::Sprite *target,
			    const flatzebra::Sprite *bullet,
			    long points);
    void addToScore(long n);
    void addToNumLives(int n);
    void killSpritesInList(flatzebra::SpriteList &sl);
    void centerPlayerSprite();
    void repositionPlayerSprite(flatzebra::Couple newpos);
    void killAllAutomaticCharacters();
    void killPlayerBase();
    bool playerIsActive() const;
    std::pair<bool, size_t> animateFallingObjects(flatzebra::SpriteList &sprites);
    void animateSaucers();
    void displayPauseMessage(bool display);
    void displayStartMessage(bool display);
    void displayMessage(int row, const char *msg);
    bool detectCollisionsWithBullet(const flatzebra::Sprite &bullet,
				    flatzebra::SpriteList &slist,
				    int scoreOnCollision);
    bool detectCollisionsWithExplosions();
    void detectCollisionsBetweenFallingObjects(flatzebra::SpriteList &slist);
    void boostPlayer();
    flatzebra::Couple accelerate(flatzebra::Couple speed) const;
    void playShortExplosionSound();
    void playSoundEffect(flatzebra::SoundMixer::Chunk &wb);

    void registerCommand(Command command);
    void registerTickEnd();


    /*	Forbidden operations:
    */
    CosmoSmashEngine(const CosmoSmashEngine &);
    CosmoSmashEngine &operator = (const CosmoSmashEngine &);
};


#endif  /* _H_CosmoSmashEngine */
