/* international.c - routines for composing international characters
   from standard ascii keypresses.
   Copyright (C) 1996-2000 Paul Sheer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.
 */

#include "global.h"

int option_latin2 = 0;

static struct compose_international {
    unsigned char extended_ascii;
    unsigned char ascii1, ascii2;
} compose_latin1[] = { \
    {0xA0, ' ', 0},		/* NO-BREAK SPACE */
    {0xA1, '^', '!'},		/* INVERTED EXCLAMATION MARK */
    {0xA2, 'c', '/'},		/* CENT SIGN */
    {0xA3, 'l', '-'},		/* POUND SIGN */
    {0xA4, 'x', 'o'},		/* CURRENCY SIGN */
    {0xA5, 'y', '='},		/* YEN SIGN */
    {0xA6, '|', '|'},		/* BROKEN BAR */
    {0xA7, 'O', 'S'},		/* SECTION SIGN */
    {0xA7, 'o', 'S'},		/* SECTION SIGN */
    {0xA8, '\'', '\''},		/* DIAERESIS */
    {0xA9, 'O', 'c'},		/* COPYRIGHT SIGN */
    {0xA9, 'o', 'c'},		/* COPYRIGHT SIGN */
    {0xAA, 'a', '_'},		/* FEMININE ORDINAL INDICATOR */
    {0xAB, '<', '<'},		/* LEFT-POINTING DOUBLE ANGLE QUOTATION MARK */
    {0xAC, '!', '!'},		/* NOT SIGN */
    {0xAD, '-', '-'},		/* SOFT HYPHEN */
    {0xAE, 'O', 'r'},		/* REGISTERED SIGN */
    {0xAE, 'o', 'r'},		/* REGISTERED SIGN */
    {0xAF, '^', '-'},		/* MACRON */
    {0xB0, '^', '0'},		/* DEGREE SIGN */
    {0xB1, '+', '-'},		/* PLUS-MINUS SIGN */
    {0xB2, '^', '2'},		/* SUPERSCRIPT TWO */
    {0xB3, '^', '3'},		/* SUPERSCRIPT THREE */
    {0xB4, '^', '\''},		/* ACUTE ACCENT */
    {0xB5, 'u', '|'},		/* MICRO SIGN */
    {0xB6, 'Q', '|'},		/* PILCROW SIGN */
    {0xB6, 'q', '|'},		/* PILCROW SIGN */
    {0xB7, '^', '.'},		/* MIDDLE DOT */
    {0xB8, '_', ','},		/* CEDILLA */
    {0xB9, '^', '1'},		/* SUPERSCRIPT ONE */
    {0xBA, 'o', '_'},		/* MASCULINE ORDINAL INDICATOR */
    {0xBB, '>', '>'},		/* RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK */
    {0xBC, '1', '4'},		/* VULGAR FRACTION ONE QUARTER */
    {0xBD, '1', '2'},		/* VULGAR FRACTION ONE HALF */
    {0xBE, '3', '4'},		/* VULGAR FRACTION THREE QUARTERS */
    {0xBF, '^', '?'},		/* INVERTED QUESTION MARK */
    {0xC0, 'A', '`'},		/* LATIN CAPITAL LETTER A WITH GRAVE ACCENT */
    {0xC1, 'A', '\''},		/* LATIN CAPITAL LETTER A WITH ACUTE ACCENT */
    {0xC2, 'A', '^'},		/* LATIN CAPITAL LETTER A WITH CIRCUMFLEX ACCENT */
    {0xC3, 'A', '~'},		/* LATIN CAPITAL LETTER A WITH TILDE */
    {0xC4, 'A', '"'},		/* LATIN CAPITAL LETTER A WITH DIAERESIS */
    {0xC5, 'A', 'o'},		/* LATIN CAPITAL LETTER A WITH RING ABOVE */
    {0xC6, 'A', 'E'},		/* LATIN CAPITAL LIGATURE AE */
    {0xC7, 'C', ','},		/* LATIN CAPITAL LETTER C WITH CEDILLA */
    {0xC8, 'E', '`'},		/* LATIN CAPITAL LETTER E WITH GRAVE ACCENT */
    {0xC9, 'E', '\''},		/* LATIN CAPITAL LETTER E WITH ACUTE ACCENT */
    {0xCA, 'E', '^'},		/* LATIN CAPITAL LETTER E WITH CIRCUMFLEX ACCENT */
    {0xCB, 'E', '"'},		/* LATIN CAPITAL LETTER E WITH DIAERESIS */
    {0xCC, 'I', '`'},		/* LATIN CAPITAL LETTER I WITH GRAVE ACCENT */
    {0xCD, 'I', '\''},		/* LATIN CAPITAL LETTER I WITH ACUTE ACCENT */
    {0xCE, 'I', '^'},		/* LATIN CAPITAL LETTER I WITH CIRCUMFLEX ACCENT */
    {0xCF, 'I', '"'},		/* LATIN CAPITAL LETTER I WITH DIAERESIS */
    {0xD0, 'D', '-'},		/* LATIN CAPITAL LETTER ETH */
    {0xD1, 'N', '~'},		/* LATIN CAPITAL LETTER N WITH TILDE */
    {0xD2, 'O', '`'},		/* LATIN CAPITAL LETTER O WITH GRAVE ACCENT */
    {0xD3, 'O', '\''},		/* LATIN CAPITAL LETTER O WITH ACUTE ACCENT */
    {0xD4, 'O', '^'},		/* LATIN CAPITAL LETTER O WITH CIRCUMFLEX ACCENT */
    {0xD5, 'O', '~'},		/* LATIN CAPITAL LETTER O WITH TILDE */
    {0xD6, 'O', '"'},		/* LATIN CAPITAL LETTER O WITH DIAERESIS */
    {0xD7, 'X', 0},		/* MULTIPLICATION SIGN */
    {0xD8, 'O', '/'},		/* LATIN CAPITAL LETTER O WITH STROKE */
    {0xD9, 'U', '`'},		/* LATIN CAPITAL LETTER U WITH GRAVE ACCENT */
    {0xDA, 'U', '\''},		/* LATIN CAPITAL LETTER U WITH ACUTE ACCENT */
    {0xDB, 'U', '^'},		/* LATIN CAPITAL LETTER U WITH CIRCUMFLEX ACCENT */
    {0xDC, 'U', '"'},		/* LATIN CAPITAL LETTER U WITH DIAERESIS */
    {0xDD, 'Y', '\''},		/* LATIN CAPITAL LETTER Y WITH ACUTE ACCENT */
    {0xDE, 'P', '|'},		/* LATIN CAPITAL LETTER THORN */
    {0xDF, 's', 0},		/* LATIN SMALL LETTER SHARP S */
    {0xE0, 'a', '`'},		/* LATIN SMALL LETTER A WITH GRAVE ACCENT */
    {0xE1, 'a', '\''},		/* LATIN SMALL LETTER A WITH ACUTE ACCENT */
    {0xE2, 'a', '^'},		/* LATIN SMALL LETTER A WITH CIRCUMFLEX ACCENT */
    {0xE3, 'a', '~'},		/* LATIN SMALL LETTER A WITH TILDE */
    {0xE4, 'a', '"'},		/* LATIN SMALL LETTER A WITH DIAERESIS */
    {0xE5, 'a', 'o'},		/* LATIN SMALL LETTER A WITH RING ABOVE */
    {0xE6, 'a', 'e'},		/* LATIN SMALL LIGATURE AE */
    {0xE7, 'c', ','},		/* LATIN SMALL LETTER C WITH CEDILLA */
    {0xE8, 'e', '`'},		/* LATIN SMALL LETTER E WITH GRAVE ACCENT */
    {0xE9, 'e', '\''},		/* LATIN SMALL LETTER E WITH ACUTE ACCENT */
    {0xEA, 'e', '^'},		/* LATIN SMALL LETTER E WITH CIRCUMFLEX ACCENT */
    {0xEB, 'e', '"'},		/* LATIN SMALL LETTER E WITH DIAERESIS */
    {0xEC, 'i', '`'},		/* LATIN SMALL LETTER I WITH GRAVE ACCENT */
    {0xED, 'i', '\''},		/* LATIN SMALL LETTER I WITH ACUTE ACCENT */
    {0xEE, 'i', '^'},		/* LATIN SMALL LETTER I WITH CIRCUMFLEX ACCENT */
    {0xEF, 'i', '"'},		/* LATIN SMALL LETTER I WITH DIAERESIS */
    {0xF0, 'd', '-'},		/* LATIN SMALL LETTER ETH */
    {0xF1, 'n', '~'},		/* LATIN SMALL LETTER N WITH TILDE */
    {0xF2, 'o', '`'},		/* LATIN SMALL LETTER O WITH GRAVE ACCENT */
    {0xF3, 'o', '\''},		/* LATIN SMALL LETTER O WITH ACUTE ACCENT */
    {0xF4, 'o', '^'},		/* LATIN SMALL LETTER O WITH CIRCUMFLEX ACCENT */
    {0xF5, 'o', '~'},		/* LATIN SMALL LETTER O WITH TILDE */
    {0xF6, 'o', '"'},		/* LATIN SMALL LETTER O WITH DIAERESIS */
    {0xF7, ':', '-'},		/* DIVISION SIGN */
    {0xF8, 'o', '/'},		/* LATIN SMALL LETTER O WITH OBLIQUE BAR */
    {0xF9, 'u', '`'},		/* LATIN SMALL LETTER U WITH GRAVE ACCENT */
    {0xFA, 'u', '\''},		/* LATIN SMALL LETTER U WITH ACUTE ACCENT */
    {0xFB, 'u', '^'},		/* LATIN SMALL LETTER U WITH CIRCUMFLEX ACCENT */
    {0xFC, 'u', '"'},		/* LATIN SMALL LETTER U WITH DIAERESIS */
    {0xFD, 'y', '\''},		/* LATIN SMALL LETTER Y WITH ACUTE ACCENT */
    {0xFE, 'p', '|'},		/* LATIN SMALL LETTER THORN */
    {0xFF, 'y', '"'},		/* LATIN SMALL LETTER Y WITH DIAERESIS */
    {0, 0, 0}
};

/*
 * Latin 2 ascii table
 * :: I am not absolutely sure with it :-(
 * :: M.K.
 */

static struct compose_international \
compose_latin2[] = { \
    {0xA0, ' ', 0},		/* NO-BREAK SPACE */
    {0xA1, 'A', ','},		/* A with tilde */
    {0xA2, ' ', '?'},		/* Caron */
    {0xA3, 'l', '-'},		/* POUND SIGN */
    {0xA4, 'x', 'o'},		/* CURRENCY SIGN */
    {0xA5, 'L', '?'},		/* L with Caron */
    {0xA6, 'S', '\''},		/* S Acute */
    {0xA7, '8', 'o'},		/* Paragraph */
    {0xA8, ' ', '"'},		/* DIAERESIS */
    {0xA9, 'S', '?'},		/* S with Caron */
    {0xAA, 'S', ','},		/* S with cedilla */
    {0xAB, 'T', '?'},		/* T with Caron */
    {0xAC, 'Z', '\''},		/* Z acute */
    {0xAD, ' ', '-'},		/* SOFT HYPHEN */
    {0xAE, 'Z', '?'},		/* Z with Caron */
    {0xAF, 'Z', 'o'},		/* Z with dot above */
    {0xB0, ' ', 'o'},		/* DEGREE SIGN */
    {0xB1, 'a', ','},		/* a with cedilla */
    {0xB2, ' ', '`'},		/* tilde !?*/
    {0xB3, 'l', '-'},		/* -l- */
    {0xB4, ' ', '\''},		/* acute */
    {0xB5, 'l', '?'},		/* l with Caron */
    {0xB6, 's', '\''},		/* s acute */
    {0xB7, ' ', '?'},		/* Caron */
    {0xB8, ' ', ','},		/* cedilla */ 
    {0xB9, 's', '?'},		/* s with Caron */
    {0xBA, 's', '\''},		/* s with tilde */
    {0xBB, 't', '?'},		/* t with Caron */
    {0xBC, 'z', '\''},		/* z acute */
    {0xBD, ' ', '~'},		/* tilde */
    {0xBE, 'z', '?'},		/* z with Caron */
    {0xBF, 'z', 'o'},		/* z with dot above */
    {0xC0, 'R', '\''},		/* R acute */
    {0xC1, 'A', '\''},		/* LATIN CAPITAL LETTER A WITH ACUTE ACCENT */
    {0xC2, 'A', '^'},		/* LATIN CAPITAL LETTER A WITH CIRCUMFLEX ACCENT */
    {0xC3, 'A', '?'},		/* A with Caron */
    {0xC4, 'A', '"'},		/* LATIN CAPITAL LETTER A WITH DIAERESIS */
    {0xC5, 'L', '\''},		/* L acute */
    {0xC6, 'C', '\''},		/* C acute */
    {0xC7, 'C', ','},		/* LATIN CAPITAL LETTER C WITH CEDILLA */
    {0xC8, 'C', '?'},		/* C with Caron */
    {0xC9, 'E', '\''},		/* LATIN CAPITAL LETTER E WITH ACUTE ACCENT */
    {0xCA, 'E', ','},		/* E with cedilla */
    {0xCB, 'E', '"'},		/* LATIN CAPITAL LETTER E WITH DIAERESIS */
    {0xCC, 'E', '?'},		/* E with Caron */
    {0xCD, 'I', '\''},		/* LATIN CAPITAL LETTER I WITH ACUTE ACCENT */
    {0xCE, 'I', '^'},		/* LATIN CAPITAL LETTER I WITH CIRCUMFLEX ACCENT */
    {0xCF, 'D', '?'},		/* D with Caron */
    {0xD0, 'D', '-'},		/* LATIN CAPITAL LETTER ETH */
    {0xD1, 'N', '\''},		/* N acute */
    {0xD2, 'N', '?'},		/* N with Caron */
    {0xD3, 'O', '\''},		/* LATIN CAPITAL LETTER O WITH ACUTE ACCENT */
    {0xD4, 'O', '^'},		/* LATIN CAPITAL LETTER O WITH CIRCUMFLEX ACCENT */
    {0xD5, 'O', '~'},		/* LATIN CAPITAL LETTER O WITH TILDE */
    {0xD6, 'O', '"'},		/* LATIN CAPITAL LETTER O WITH DIAERESIS */
    {0xD7, 'X', 'o'},		/* MULTIPLICATION SIGN */
    {0xD8, 'R', '?'},		/* R with Caron */
    {0xD9, 'U', 'o'},		/* LATIN CAPITAL LETTER U WITH GRAVE ACCENT */
    {0xDA, 'U', '\''},		/* LATIN CAPITAL LETTER U WITH ACUTE ACCENT */
    {0xDB, 'U', '~'},		/* U with cedilla */
    {0xDC, 'U', '^'},		/* U with DIAERESIS */
    {0xDD, 'Y', '\''},		/* LATIN CAPITAL LETTER Y WITH ACUTE ACCENT */
    {0xDE, 'T', ','},		/* T with cedilla */
    {0xDF, 'S', '^'},		/*  */
    {0xE0, 'r', '\''},		/* r acute */
    {0xE1, 'a', '\''},		/* LATIN SMALL LETTER A WITH ACUTE ACCENT */
    {0xE2, 'a', '^'},		/* LATIN SMALL LETTER A WITH CIRCUMFLEX ACCENT */
    {0xE3, 'a', '?'},		/* a with Caron */
    {0xE4, 'a', '"'},		/* LATIN SMALL LETTER A WITH DIAERESIS */
    {0xE5, 'l', '\''},		/* l acute */
    {0xE6, 'c', '\''},		/* c acute */
    {0xE7, 'c', ','},		/* LATIN SMALL LETTER C WITH CEDILLA */
    {0xE8, 'c', '?'},		/* c with Caret */
    {0xE9, 'e', '\''},		/* LATIN SMALL LETTER E WITH ACUTE ACCENT */
    {0xEA, 'e', ','},		/* e with cedilla */
    {0xEB, 'e', '"'},		/* LATIN SMALL LETTER E WITH DIAERESIS */
    {0xEC, 'e', '?'},		/* e with Caret */
    {0xED, 'i', '\''},		/* LATIN SMALL LETTER I WITH ACUTE ACCENT */
    {0xEE, 'i', '^'},		/* LATIN SMALL LETTER I WITH CIRCUMFLEX ACCENT */
    {0xEF, 'd', '?'},		/* d with Caret */
    {0xF0, 'd', '-'},		/* LATIN SMALL LETTER ETH */
    {0xF1, 'n', '\''},		/* n acute */
    {0xF2, 'n', '?'},		/* n with Caret */
    {0xF3, 'o', '\''},		/* LATIN SMALL LETTER O WITH ACUTE ACCENT */
    {0xF4, 'o', '^'},		/* LATIN SMALL LETTER O WITH CIRCUMFLEX ACCENT */
    {0xF5, 'o', '~'},		/* LATIN SMALL LETTER O WITH TILDE */
    {0xF6, 'o', '"'},		/* LATIN SMALL LETTER O WITH DIAERESIS */
    {0xF7, ':', '-'},		/* DIVISION SIGN */
    {0xF8, 'r', '?'},		/* r with Caret */
    {0xF9, 'u', 'o'},		/* u with ring above */
    {0xFA, 'u', '\''},		/* LATIN SMALL LETTER U WITH ACUTE ACCENT */
    {0xFB, 'u', '~'},		/* u with tilde */
    {0xFC, 'u', '"'},		/* LATIN SMALL LETTER U WITH DIAERESIS */
    {0xFD, 'y', '\''},		/* LATIN SMALL LETTER Y WITH ACUTE ACCENT */
    {0xFE, 't', ','},		/* t with cedilla */
    {0xFF, ' ', '`'},		/* dot above */
    {0, 0, 0}
};


unsigned char get_international_character (unsigned char key_press)
{
    int i;
    static int last_press = 0;
    struct compose_international *compose;
    if (!key_press) {
	last_press = 0;
	return 0;
    }
    if (option_latin2)
        compose = compose_latin2;
    else
        compose = compose_latin1;
    if (last_press) {
	for (i = 0; compose[i].ascii1; i++) {
	    if ((compose[i].ascii2 == key_press && compose[i].ascii1 == last_press)
		||
		(compose[i].ascii1 == key_press && compose[i].ascii2 == last_press)) {
		last_press = 0;
		return compose[i].extended_ascii;
	    }
	}
	last_press = 0;
	return 0;
    }
    for (i = 0; compose[i].ascii1; i++) {
	if (compose[i].ascii1 == key_press || compose[i].ascii2 == key_press) {
	    if (!compose[i].ascii2)
		return compose[i].extended_ascii;
	    else {
		last_press = key_press;
		return 1;
	    }
	}
    }
    return 0;
}
