/************************************************************************/
/*	Copyright 1987-1994 by Chuck Musciano and Harris Corporation 	*/
/*									*/
/*	Full ownership of this software, and all rights pertaining to 	*/
/*	the for-profit distribution of this software, are retained by 	*/
/*	Chuck Musciano and Harris Corporation.  You are permitted to 	*/
/*	use this software without fee.  This software is provided "as 	*/
/*	is" without express or implied warranty.  You may redistribute 	*/
/*	this software, provided that this copyright notice is retained,	*/
/*	and that the software is not distributed for profit.  If you 	*/
/*	wish to use this software in a profit-making venture, you must 	*/
/*	first license this code and its underlying technology from 	*/
/*	Harris Corporation. 						*/
/*									*/
/*	Bottom line: you can have this software, you can use it, you 	*/
/*	can give it away.  You just can't sell any or all parts of it 	*/
/*	without prior permission from Harris Corporation. 		*/
/************************************************************************/

/************************************************************************/
/*									*/
/*	contool.c	main contool driver				*/
/*									*/
/************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<fcntl.h>
#include	<string.h>
#include	<unistd.h>
#include	<time.h>

#include	<sys/ioctl.h>
#include	<sys/param.h>
#include	<sys/types.h>
#include	<sys/stat.h>

#ifdef	SVR4
#include	<sys/stream.h>
#include	<sys/stropts.h>
#include	<sys/strredir.h>
#include	<sys/termios.h>
#endif

#include	<X11/Xlib.h>
#include	<X11/Xutil.h>

#include	<xview/xview.h>
#include	<xview/icon.h>
#include	<xview/panel.h>
#include	<xview/textsw.h>
#include	<xview/notify.h>

#include	"manifest.h"
#include	"contool.h"
#include	"contool_ui.h"

#define		update_value(old, new)		((old) = ((new) > (old))? (new) : (old))

#define		INPUT_BUFFER_SIZE		4096

EXPORT	Attr_attribute	INSTANCE;

EXPORT	contool_base_objects	*contool_base;
EXPORT	char			*filter_file;

PUBLIC	Server_image		load_icon();

PRIVATE	unsigned	short	good_bits[]  = {
#include	"icons/default_good.icon"
					       };
PRIVATE	unsigned	short	bad_bits[]   = {
#include	"icons/default_bad.icon"
					       };
PRIVATE	unsigned	short	flash_bits[] = {
#include	"icons/default_flash.icon"
					       };
PRIVATE	unsigned	short	mask_bits[]  = {
#include	"icons/mask.icon"
					       };

PRIVATE	char			*ct_usage = "usage: contool [-c <file>] [-f] [-i <file>] [-l] [-L <file>] [-n]\n";

PRIVATE	Server_image		default_good_icon;
PRIVATE	Server_image		good;
PRIVATE	Server_image		good_mask;
PRIVATE	Server_image		default_bad_icon;
PRIVATE	Server_image		bad;
PRIVATE	Server_image		bad_mask;
PRIVATE	Server_image		default_flash_icon;
PRIVATE	Server_image		flash;
PRIVATE	Server_image		flash_mask;
PRIVATE	Server_image		icon_mask;

PRIVATE	int			bad_is_up;
PRIVATE	int			beep_count;
PRIVATE	int			blinking		= FALSE;
PRIVATE	FILE			*command		= NULL;
PRIVATE	Filter			*curr_filter		= NULL;
PRIVATE	int			event_in_progress	= FALSE;
PRIVATE	int			explicit_filters	= FALSE;
PRIVATE	int			fork_into_background	= FALSE;
PRIVATE	int			masking_works;
PRIVATE	int			master			= -1;
PRIVATE	int			no_console		= FALSE;
PRIVATE	int			old_time		= 0;
PRIVATE	char			*program;
PRIVATE	int			slave			= -1;
PRIVATE	int			filter_timeout		= 0;

PRIVATE	struct	itimerval	timer			= {{0, 500000}, {0, 500000}};

/************************************************************************/
PRIVATE	do_insertion(buf, len)

char	*buf;
int	len;

{	int	sw_len;
	Textsw_index	first, last;
	char	*temp;

	while ((int) xv_get(contool_base->display, TEXTSW_LENGTH) + len > defaults.max_size) {
	   sw_len = (int) xv_get(contool_base->display, TEXTSW_LENGTH);
	   first = 1;
	   last = TEXTSW_INFINITY;
	   if (textsw_find_bytes(contool_base->display, &first, &last, "\n<<<", 4, 0) == -1 || first == 0)
	      if (textsw_find_bytes(contool_base->display, &first, &last, "\n", 1, 0) == -1 || first == 0)
	         first = (defaults.delete_amount < sw_len)? defaults.delete_amount : sw_len;
	   temp = (char *) malloc((unsigned) sw_len);
	   xv_get(contool_base->display, TEXTSW_CONTENTS, 0, temp, sw_len);
	   textsw_reset(contool_base->display, 0, 0);
	   textsw_insert(contool_base->display, temp + first, sw_len - first);
	   free(temp);
	   }
	xv_set(contool_base->display, TEXTSW_INSERTION_POINT, TEXTSW_INFINITY, 0);
	textsw_insert(contool_base->display, buf, len);
}

/************************************************************************/
PRIVATE	void	time_stamp()

{	time_t	t;
	int	pos;
	char	buf[5];

	   t = time(0);
	   if (t - old_time >= defaults.stamp_resolution) {
	      xv_set(contool_base->display, TEXTSW_INSERTION_POINT, TEXTSW_INFINITY, 0);
	      pos = (int) xv_get(contool_base->display, TEXTSW_LENGTH);
	      if (pos != 0) {
	         xv_get(contool_base->display, TEXTSW_CONTENTS, pos - 1, buf, 1);
	         if (buf[0] != '\n')
	            do_insertion("\n", 1);
	         }
	      do_insertion("\n<<< ", 5);
	      do_insertion(ctime(&t), 24);
	      do_insertion(" >>>\n", 5);
	      old_time = t;
	      }
}

/************************************************************************/
PRIVATE	void	change_icon(image, image_mask)

Server_image	image;
Server_image	image_mask;

{	Icon	icon;

	icon = (Icon) xv_get(contool_base->base, FRAME_ICON);
	xv_set(icon,
		  ICON_IMAGE, image,
		  ICON_MASK_IMAGE, masking_works? image_mask : (Server_image) NULL,
		  XV_WIDTH, (int) xv_get(image, XV_WIDTH),
		  XV_HEIGHT, (int) xv_get(image, XV_HEIGHT),
	       NULL);
	xv_set(contool_base->base, FRAME_ICON, icon, NULL);
}

/************************************************************************/
EXPORT	adjust_window_limit()

{	int	sw_len;
	char	*temp;

	sw_len = (int) xv_get(contool_base->display, TEXTSW_LENGTH);
	temp = (char *) malloc((unsigned) sw_len);
	xv_get(contool_base->display, TEXTSW_CONTENTS, 0, temp, sw_len);
	xv_set(contool_base->display, TEXTSW_MEMORY_MAXIMUM, defaults.max_size + defaults.max_size / 10, NULL);
	textsw_reset(contool_base->display, 0, 0);
	if (sw_len > defaults.max_size)
	   textsw_insert(contool_base->display, temp + sw_len - defaults.max_size, defaults.max_size);
	else
	   textsw_insert(contool_base->display, temp, sw_len);
	free(temp);
	reset_archive_size();
}

/************************************************************************/
PRIVATE	Notify_value	blink_proc(me, which)

int	*me;
int	which;

{
	if (event_in_progress)
	   return(NOTIFY_DONE);
	if (beep_count > 0) {
	   window_bell(contool_base->base);
	   beep_count--;
	   }
	if (blinking) {
	   if (bad_is_up)
	      change_icon(flash, flash_mask);
	   else
	      change_icon(bad, bad_mask);
	   bad_is_up = !bad_is_up;
	   }
	if (filter_timeout > 0)
	   if (--filter_timeout <= 0)
	      filters_changed();
	if (beep_count == 0 && !blinking && filter_timeout <= 0)
	   notify_set_itimer_func(contool_base->base, blink_proc, ITIMER_REAL, NULL, NULL);
	return(NOTIFY_DONE);
}

/************************************************************************/
PRIVATE	internal_error(a, b, c, d, e, f)

char	*a, *b, *c, *d, *e, *f;

{	char	buf[1024];

	sprintf(buf, a, b, c, d, e, f);
	time_stamp();
	fprintf(stderr, "*** %s: %s\n", program, buf);
}

/************************************************************************/
PRIVATE	void	internal_message(a, b, c, d, e, f)

char	*a, *b, *c, *d, *e, *f;

{	char	buf[1024];

	sprintf(buf, a, b, c, d, e, f);
	time_stamp();
	write_log(buf);
	do_insertion(buf, strlen(buf));
}

/************************************************************************/
PRIVATE	start_command(cmd)

char	*cmd;

{
	if (command)
	   pclose(command);
	if ((command = popen(cmd, "w")) == NULL)
	   internal_error("could not execute \"%s\"", cmd);
}

/************************************************************************/
PRIVATE	continue_command(buf)

char	*buf;

{
	if (command)
	   fputs(buf, command);
}

/************************************************************************/
PRIVATE	end_command()

{
	if (command)
	   pclose(command);
}

/************************************************************************/
EXPORT	acquire_console()

{	int	console;

#ifdef	SVR4
	if (slave > 0) {
	   if ((console = open("/dev/console", O_RDONLY)) == -1)
	      abend("%s: cannot open /dev/console: %s", program, sys_errlist[errno]);
	   if (ioctl(console, SRIOCSREDIR, slave) == -1)
	      abend("%s: could not attach to /dev/console: %s", program, sys_errlist[errno]);
	   }
#else
	if (slave > 0)
	   if (ioctl(slave, TIOCCONS, NULL) == -1)
	      abend("%s: could not attach to /dev/console: %s", program, sys_errlist[errno]);
#endif
}

/************************************************************************/
EXPORT	filters_changed()

{
	curr_filter = NULL;
	xv_set(contool_base->base, FRAME_LEFT_FOOTER, "", NULL);
	end_command();
	filter_timeout = 0;
	if (!blinking && beep_count == 0)
	   notify_set_itimer_func(contool_base->base, blink_proc, ITIMER_REAL, NULL, NULL);
}

/************************************************************************/
PRIVATE	load_filters()

{	struct	stat	sb;
	static	int	load_time = 0;

	if (access(filter_file, R_OK) == -1) {
	   if (explicit_filters && load_time == 0) {
	      internal_error("filter file %s cannot be accessed", filter_file);
	      load_time = 1;
	      }
	   }
	else if (stat(filter_file, &sb) == 0 && sb.st_mtime > load_time)
	   if (lex_init(filter_file)) {
	      yyparse();
	      if (!parse_errors_occured) {
	         free_list(filters);
	         free(parsed_defaults);
	         filters = parsed_filters;
	         internal_message("*** filters loaded from %s\n", filter_file);
	         load_time = sb.st_mtime;
	         }
	      }
	   else {
	      internal_error("error accessing configuration file %s", filter_file);
	      load_time = 1;
	      }
}

/************************************************************************/
PRIVATE	parse_options(argc, argv)

int	*argc;
char	**argv;

{	char	*s, c, path[1024], *log_file = NULL, *p;
	int	log = FALSE;
	static	char	cmdline[MAXPATHLEN + 10];
	struct	stat	sb;

	if (p = getenv(CONTOOL_FILTERS))
	   filter_file = strsave(p);
	else {
	   sprintf(path, "%s/.contool", getenv("HOME"));
	   filter_file = strsave(path);
	   }

	cmdline[0] = '\0';

	while ((c = get_option(argc, argv, "c:fi:lL:n?", &s)) != EOF)
	   switch (c) {
	      case 'c' : filter_file = expand_tilde(s);
	      		 explicit_filters = TRUE;
		         strcat(cmdline, " -c ");
			 strcat(cmdline, s);
	      		 break;
	      case 'f' : fork_into_background = TRUE;
	      		 strcat(cmdline, " -f");
	      		 break;
	      case 'i' : close(master);
	      		 close(slave);
	      		 slave = -1;
	      		 if (strcmp(s, "-") == 0)
	      		    master = fileno(stdin);
	      		 else {
	      		    p = expand_tilde(s);
	      		    if (stat(p, &sb) != 0)
	      		       abend("%s: cannot stat %s: %s", program, s, sys_errlist[errno]);
	      		    else if ((sb.st_mode & S_IFMT) != S_IFCHR && (sb.st_mode & S_IFMT) != S_IFIFO)
	      		       abend("%s: %s is not a FIFO or character special device", program, s);
	      		    else if ((master = open(p, O_RDONLY | O_NDELAY)) < 0)
	      		       abend("%s: cannot open %s for reading: %s", program, s, sys_errlist[errno]);
	      		    }
	      		 strcat(cmdline, " -i ");
	      		 strcat(cmdline, s);
	      		 break;
	      case 'l' : log = TRUE;
		         strcat(cmdline, " -l");
	      		 break;
	      case 'L' : log_file = expand_tilde(s);
		         strcat(cmdline, " -L ");
			 strcat(cmdline, s);
	      		 break;
	      case 'n' : no_console = TRUE;
		         strcat(cmdline, " -n");
	      		 break;
	      case '?' : fprintf(stderr, ct_usage);
	      		 exit(0);
	      		 break;
	      default  : fprintf(stderr, ct_usage);
	                 exit(1);
	      }

	if (strlen(cmdline) > (size_t) 0)
	    xv_set(contool_base->base, WIN_CMD_LINE, cmdline, NULL);

	if (lex_init(filter_file)) {
	   yyparse();
	   if (parsed_defaults)
	      defaults = *parsed_defaults;
	   if (log_file)
	      defaults.log_file = log_file;
	   if (log)
	      enable_logging();
	   adjust_window_limit();
	   }
	else if (explicit_filters)
	   error("Could not read configuration file %s", filter_file);
}

/************************************************************************/
PRIVATE	stop_blinking()

{
	if (filter_timeout <= 0)
	   notify_set_itimer_func(contool_base->base, blink_proc, ITIMER_REAL, NULL, NULL);
	change_icon(good, good_mask);
	blinking = FALSE;
}

/************************************************************************/
EXPORT	update_icons()

{	char	msg[1024];

	if (good && good != default_good_icon)
	   xv_destroy(good);
	if (good_mask && good_mask != icon_mask)
	   xv_destroy(good_mask);
	if (bad && bad != default_bad_icon)
	   xv_destroy(bad);
	if (bad_mask && bad_mask != icon_mask)
	   xv_destroy(bad_mask);
	if (flash && flash != default_flash_icon)
	   xv_destroy(flash);
	if (flash_mask && flash_mask != icon_mask)
	   xv_destroy(flash_mask);

	if (defaults.good_icon == NULL) {
	   good = default_good_icon;
	   good_mask = icon_mask;
	   }
	else if ((good = load_icon(defaults.good_icon, msg)) == (Server_image) NULL) {
	   internal_error("Cannot load default \"All is well\" icon %s: %s", defaults.good_icon, msg);
	   good = default_good_icon;
	   good_mask = icon_mask;
	   }
	else if (defaults.good_icon_mask == NULL)
	   good_mask = (Server_image) NULL;
	else if ((good_mask = load_icon(defaults.good_icon_mask, msg)) == (Server_image) NULL)
	   internal_error("Cannot load default \"All is well\" icon mask %s: %s", defaults.good_icon_mask, msg);

	if (defaults.bad_icon == NULL) {
	   bad = default_bad_icon;
	   bad_mask = icon_mask;
	   }
	else if ((bad = load_icon(defaults.bad_icon, msg)) == (Server_image) NULL) {
	   internal_error("Cannot load default \"Check console\" icon %s: %s", defaults.bad_icon, msg);
	   bad = default_bad_icon;
	   bad_mask = icon_mask;
	   }
	else if (defaults.bad_icon_mask == NULL)
	   bad_mask = (Server_image) NULL;
	else if ((bad_mask = load_icon(defaults.bad_icon_mask, msg)) == (Server_image) NULL)
	   internal_error("Cannot load default \"Check console\" icon mask %s: %s", defaults.bad_icon_mask, msg);

	if (defaults.flash_icon == NULL) {
	   flash = default_flash_icon;
	   flash_mask = icon_mask;
	   }
	else if ((flash = load_icon(defaults.flash_icon, msg)) == (Server_image) NULL) {
	   internal_error("Cannot load default \"Flash\" icon %s: %s", defaults.flash_icon, msg);
	   flash = default_flash_icon;
	   flash_mask = icon_mask;
	   }
	else if (defaults.flash_icon_mask == NULL)
	   flash_mask = (Server_image) NULL;
	else if ((flash_mask = load_icon(defaults.flash_icon_mask, msg)) == (Server_image) NULL)
	   internal_error("Cannot load default \"Flash\" icon mask %s: %s", defaults.flash_icon_mask, msg);

	if (!blinking)
	   change_icon(good, good_mask);
}

/************************************************************************/
PRIVATE	Notify_value	close_proc(frame, event, arg, type)

Frame	frame;
Event	*event;
Notify_arg	arg;
Notify_event_type	type;

{
	event_in_progress = TRUE;
	if (event_action(event) == ACTION_OPEN) {
	   if (blinking)
	      stop_blinking();
	   create_archive();
	   }
	if (event_action(event) == ACTION_CLOSE && defaults.archive_style == ARCHIVE_ON_CLOSE)
	   archive_messages();
	event_in_progress = FALSE;
	return(notify_next_event_func(frame, (Notify_event) event, arg, type));
}

/************************************************************************/
PRIVATE	Notify_value	destroy_proc(frame, status)

Frame	frame;
Destroy_status	status;

{
	if (status == DESTROY_CHECKING) {
	   textsw_reset(contool_base->display, 0, 0);
	   return(NOTIFY_DONE);
	   }
	else
	   return(notify_next_destroy_func(frame, status));
}

/************************************************************************/
PRIVATE	Notify_value	input_proc(me, fd)

int	*me;
int	fd;

{	char	old_c, *s, *t, buf[1024];
	Filter	*f;
	int	count, do_blink = FALSE, do_open = FALSE;
	static	char	in_buf[INPUT_BUFFER_SIZE + 2];
	static	int	leftover = 0;

	while ((count = read(master, in_buf + leftover, INPUT_BUFFER_SIZE - leftover)) >= 0) {
	   if (count == 0 && slave == -1)
	      abend("%s: input source has reached EOF, exiting", program);
	   in_buf[count + leftover] = '\0';
	   leftover = 0;
	   while (s = strchr(in_buf, '\015'))
	      strcpy(s, s + 1);
	   for (t = in_buf; *t; *s = old_c, t = s) {
	      if (s = strchr(t, '\n')) {
	         old_c = *++s;
	         *s = '\0';
	         }
	      else {
	         leftover = strlen(t);
	         strcpy(in_buf, t);
	         break;
	         }
	      if (!defaults.log_after)
	         write_log(t);
	      if (curr_filter == NULL) {
	         load_filters();
	         for (f = filters; f; f = f->next)
	            if (f->start_re && match_exp(f->start_re, f->start_circf, t)) {
	               if (f->save) {
	                  update_value(do_blink, f->flash);
	                  update_value(beep_count, f->beep);
	                  update_value(do_open, f->open);
	                  if (f->stamp)
	                     time_stamp();
	                  do_insertion(t, strlen(t));
	                  if (f->command) {
	                     start_command(f->command);
	                     continue_command(t);
	                     if (f->stop == NULL)
	                        end_command();
	                     }
	                  if (defaults.log_after && f->log)
			     write_log(t);
	                  }
	               if (f->stop) {
	                  curr_filter = f;
	                  filter_timeout = f->timeout * 2;
	                  sprintf(buf, "Filtering \"%s\"...", f->start);
	                  xv_set(contool_base->base, FRAME_LEFT_FOOTER, buf, NULL);
	                  }
	               break;
	               }
	         if (f == NULL) {
	            if (defaults.stamp)
	               time_stamp();
	            if (defaults.command) {
	               start_command(defaults.command);
	               continue_command(t);
	               end_command();
	               }
		    if (defaults.log_after && defaults.log)
	               write_log(t);
	            do_insertion(t, strlen(t));
	            update_value(do_blink, defaults.flash);
	            update_value(do_open, defaults.open);
	            update_value(beep_count, defaults.beep);
	            }
	         }
	      else {
	         if (curr_filter->save) {
	            if (curr_filter->stamp)
	               time_stamp();
		    if (defaults.log_after && curr_filter->log)
	               write_log(t);
		    do_insertion(t, strlen(t));
		    if (curr_filter->command)
		       continue_command(t);
	            }
	         if (match_exp(curr_filter->stop_re, curr_filter->stop_circf, t)) {
	            xv_set(contool_base->base, FRAME_LEFT_FOOTER, "", NULL);
	            if (curr_filter->command)
	               end_command();
	            curr_filter = NULL;
	            }
	         }
	      }
	   }
	xv_set(contool_base->display, TEXTSW_UPDATE_SCROLLBAR, 0);
	if (do_open)
	   xv_set(contool_base->base, FRAME_CLOSED, FALSE, 0);
	if (do_blink)
	   if (xv_get(contool_base->base, FRAME_CLOSED) && !blinking) {
	      change_icon(bad, bad_mask);
	      xv_set(contool_base->base, WIN_SHOW, TRUE, 0);
	      blinking = TRUE;
	      bad_is_up = TRUE;
	      notify_set_itimer_func(contool_base->base, blink_proc, ITIMER_REAL, &timer, NULL);
	      }
	if (beep_count > 0 || blinking || filter_timeout > 0)
	   notify_set_itimer_func(contool_base->base, blink_proc, ITIMER_REAL, &timer, NULL);
	return(NOTIFY_DONE);
}

/************************************************************************/
PRIVATE	Notify_value	signal_proc(frame, sig, when)

Frame	frame;
int	sig;
Notify_signal_mode	when;

{
	if (sig == SIGUSR1)
	   stop_blinking();
	else if (sig == SIGHUP)
	   update_logging();
	return(NOTIFY_DONE);
}

/************************************************************************/
EXPORT	Menu_item	become_console(item, op)

Menu_item	item;
Menu_generate	op;

{
	if (op == MENU_NOTIFY)
	   acquire_console();
	return item;
}

/************************************************************************/
EXPORT	Menu_item	clear_messages(item, op)

Menu_item	item;
Menu_generate	op;

{
	if (op == MENU_NOTIFY) {
	   textsw_reset(contool_base->display, 0, 0);
	   old_time = 0;
	   }
	return item;
}

/************************************************************************/
EXPORT	Menu_item	print_messages(item, op)

Menu_item	item;
Menu_generate	op;

{	int	size;
	char	*buf;
	FILE	*cmd;

	if (op == MENU_NOTIFY) {
	   if (is_empty(defaults.print_filter))
	      error("You must specify a printer command in the Properties dialog");
	   else if ((cmd = popen(defaults.print_filter, "w")) == NULL)
	      error("Could not execute %s", defaults.print_filter);
	   else {
	      lets_get_busy(contool_base->base, TRUE, NULL);
	      size = (int) xv_get(contool_base->display, TEXTSW_LENGTH);
	      buf = (char *) malloc(size);
	      xv_get(contool_base->display, TEXTSW_CONTENTS, 0, buf, size);
	      if (fwrite(buf, 1, size, cmd) != size)
	         error("Could not write console contents to printer");
	      pclose(cmd);
	      free(buf);
	      lets_get_busy(contool_base->base, FALSE, NULL);
	      }
	   }
	return item;
}

/************************************************************************/
EXPORT	Menu_item	reset_filter(item, op)

Menu_item	item;
Menu_generate	op;

{
	if (op == MENU_NOTIFY)
	   filters_changed();
	return item;
}

/************************************************************************/
EXPORT	Menu_item	save_to_archive(item, op)


Menu_item	item;
Menu_generate	op;

{
	if (op == MENU_NOTIFY)
	   archive_messages();
	return(item);
}

/************************************************************************/
main(argc, argv)

int	argc;
char	**argv;

{	char	buf[1024], *p, *path, *open_psuedo_tty();
	int	i, console[2];
	XWindowAttributes	attr;
	XClassHint	hints;

	program = strsave(argv[0]);

#ifdef	SVR4
	if (pipe(console) != 0)
	   abend("%s: could not create a pipe: %s", program, sys_errlist[errno]);
	master = console[0];
	slave = console[1];

        if (ioctl(slave, I_PUSH, "ptem") == -1 || ioctl(slave, I_PUSH, "ldterm") == -1 || ioctl(slave, I_PUSH, "ttcompat") == -1)
           abend("%s: could not emulate terminal for console: %s", program, sys_errlist[errno]);
#else
	open_psuedo_tty(&master, &slave);
	if (master == -1 || slave == -1)
	   abend("%s: could not open a psuedo-tty: %s", program, sys_errlist[errno]);
#endif

	xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);
	INSTANCE = xv_unique_key();

	contool_base = contool_base_objects_initialize(NULL, NULL);

	if (p = getenv(CONTOOL_LABEL))
	   xv_set(contool_base->base, FRAME_LABEL, p, NULL);

	default_good_icon  = (Server_image) xv_create((Frame) NULL, SERVER_IMAGE,
							 XV_WIDTH, 64,
							 XV_HEIGHT, 64,
							 SERVER_IMAGE_BITS, good_bits,
						      NULL);
	default_bad_icon   = (Server_image) xv_create((Frame) NULL, SERVER_IMAGE,
							 XV_WIDTH, 64,
							 XV_HEIGHT, 64,
							 SERVER_IMAGE_BITS, bad_bits,
						      NULL);
	default_flash_icon = (Server_image) xv_create((Frame) NULL, SERVER_IMAGE,
							 XV_WIDTH, 64,
							 XV_HEIGHT, 64,
							 SERVER_IMAGE_BITS, flash_bits,
						      NULL);
	icon_mask          = (Server_image) xv_create((Frame) NULL, SERVER_IMAGE,
							 XV_WIDTH, 64,
							 XV_HEIGHT, 64,
							 SERVER_IMAGE_BITS, mask_bits,
						      NULL);
	disable_logging(contool_base);

	XGetWindowAttributes((Display *) xv_get(contool_base->base, XV_DISPLAY), xv_get(xv_get(contool_base->base, XV_ROOT), XV_XID), &attr);
	masking_works = (attr.depth > 1);

	hints.res_class = "Contool";
	hints.res_name = (p = strrchr(program, '/'))? p + 1 : program;
	XSetClassHint((Display *) xv_get(contool_base->base, XV_DISPLAY), xv_get(contool_base->base, XV_XID), &hints);

	parse_options(&argc, argv);

	i = fcntl(master, F_GETFL, 0);
	i |= O_NONBLOCK;
	if (fcntl(master, F_SETFL, i) == -1)
	   abend("%s: could not force %s to non-blocking i/o", program);

	load_filters();
	update_icons();
	if (!no_console)
	   acquire_console();
	if (fork_into_background)
	   if (fork() != 0)
	      exit(0);

	notify_set_input_func(contool_base->base, input_proc, master);
	notify_interpose_destroy_func(contool_base->base, destroy_proc);
	notify_interpose_event_func(contool_base->base, close_proc, NOTIFY_SAFE);
	notify_set_signal_func(contool_base->base, signal_proc, SIGUSR1, NOTIFY_SYNC);
	notify_set_signal_func(contool_base->base, signal_proc, SIGHUP, NOTIFY_SYNC);

	xv_main_loop(contool_base->base);
	exit(0);
}
