/****************************************************************************
Connection-Log
Brian Mitchell	brian@saturn.net

$Log: clog.c,v $
Revision 1.6  1997/08/08 04:54:51  brian
changed long to bpf_u_int32 in pcap_lookupnet()
masked out the flags portion of the ip_ofs field

Revision 1.5  1996/06/02 04:48:38  brian
fixed stupid error in get_date()

Revision 1.4  1996/06/02 04:24:30  brian
found out ppp is only broken in lin(s)ux, fixed.
we now toss any packets that dont have 0 as the fragmentation offset

Revision 1.3  1996/05/21 06:24:04  brian
Added SLIP, CSLIP, and (broken) PPP support. Apparently pcap does
not support filtering with PPP, and bad things happen, so we turn
off the filtering if we want to use that interface. There is a
hook for FDDI, but I doubt I will ever get that going.

Revision 1.2  1996/05/20 14:39:14  brian
libpcap does a atexit() in the pcap_open_live() function to restore
the interface to its previous state. This was causing problems with
the -b (background) option, so when I go in the background now, I
use _exit() instead of exit().

Revision 1.1  1996/05/18 11:24:54  brian
Initial revision

****************************************************************************/

static char rcsid[] = "$Id: clog.c,v 1.6 1997/08/08 04:54:51 brian Exp $";
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifdef LINUX
#include <netinet/in.h>
#include "linux-include/in_systm.h"
#include "linux-include/ip.h"
#include "linux-include/tcp.h"
#else
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#endif
#include <arpa/inet.h>
#include <pcap.h>
int fddipad;

#ifdef NO_BCOPY
#define bcopy(s, d, n) memcpy(d, s, n)
#endif

/* some global variables (ugh!) */
pcap_t *ip_socket;
FILE *logfile = stdout;
int resolve = 0;
int dlt_len = 4;
int brokedppp = 0;

/* function declarations */
void filter_packet(u_char *, struct pcap_pkthdr *, u_char *);
void log_syn(struct ip *, struct tcphdr *);
char *hostlookup(unsigned int);
void signal_handler(int);
char *get_date(void);

/* default packet filter - what could be easier :-) */
#define DEFAULT_FILTER	"tcp"

/* main function: parses arguments, sets up pcap, and goes into loop */
void main(int argc, char **argv)
{
   char *interface = NULL;
   char errbuf[PCAP_ERRBUF_SIZE];
   struct bpf_program prog;
   bpf_u_int32 network, netmask;
   int promisc = 1;
   int background = 0;
   extern int opterr;
   extern char *optarg;
   char *clog_filter = NULL;
   int c;
   
   opterr = 0;
   /* parse arguments */
   while((c = getopt(argc, argv, "hbrpbi:o:f:")) != -1)
   {
      switch(c)
      {
         /* help */
         case 'h':
            fprintf(stderr, "%s [-h] [-b] [-r] [-p] [-i interface] [-o file] [-f filter]\n", argv[0]);
            exit(0);
         /* resolve hostnames instead of printing ip address */
         case 'r':
            resolve = 1;
            break;
         /* turn promiscuous mode off */
         case 'p':
            promisc = 0;
            break;
         /* specify the interface to use */
         case 'i':
            interface = optarg;
            break;
         /* run in the background */
         case 'b':
            background = 1;
            break;
         /* specify the file to log to */
         case 'o':
            logfile = fopen(optarg, "at");
            if(logfile == NULL) logfile = stdout;
            break;
         /* specify a filter (must be in quotes if more than 1 word) */
         case 'f':        
            clog_filter = optarg;
            break;            
      }
   }
   /* if you didnt specify a filter, use the default filter */
   if(clog_filter == NULL) clog_filter = DEFAULT_FILTER;
   /* ignore SIGHUP, anything else causes a controlled shutdown */
   signal(SIGHUP, SIG_IGN);
   signal(SIGINT, signal_handler);
   signal(SIGQUIT, signal_handler);
   signal(SIGTERM, signal_handler);

   /* if we didnt specify a interface, have pcap find a suitable one */   
   if(interface == NULL)
   {
      interface = pcap_lookupdev(errbuf);
      if(interface == NULL)
      {
         fprintf(stderr, "pcap_lookupdev: %s\n", errbuf);
         exit(-1);
      }
   }   
   if(pcap_lookupnet(interface, &network, &netmask, errbuf) < 0)
   {
      fprintf(stderr, "pcap_lookupnet: %s\n", errbuf);
      exit(-1);
   }
   ip_socket = pcap_open_live(interface, 1024, promisc, 1024, errbuf);
   if(ip_socket == NULL)
   {
      fprintf(stderr, "pcap_open_live: %s\n", errbuf);
      exit(-1);
   }
   switch(pcap_datalink(ip_socket))
   {
      case DLT_EN10MB:
         dlt_len = 14;
         break;
      case DLT_SLIP:
         dlt_len = 16;
         break;
      case DLT_PPP:
#ifdef LINUX      
         brokedppp = 1;
         dlt_len = 4;
         clog_filter = NULL;
         fprintf(stderr, "PPP is broke, so no filtering is possible\n");
         fflush(stderr);
#else
         dlt_len = 4;
#endif                  
         break;
      case DLT_FDDI:
         fprintf(stderr, "FDDI is not currently a supported interface\n");
         fflush(stderr);
         signal_handler(-1);
      default:
         dlt_len = 4;
         break;
   }                        
   if(pcap_compile(ip_socket, &prog, clog_filter, 1, netmask) < 0)
   {
      fprintf(stderr, "pcap_compile: %s\n", errbuf);
      signal_handler(-1);
   }  
   if(pcap_setfilter(ip_socket, &prog) < 0)
   {
      fprintf(stderr, "pcap_setfilter: %s\n", errbuf);
      signal_handler(-1);
   }
   /* display some useful information: interface, filter, promisc status */
   fprintf(stderr, "interface: %s, filter: %s, ", interface, clog_filter);
   if(promisc == 0) fprintf(stderr, "promisc: off\n");
   else fprintf(stderr, "promisc: on\n");
   /* go in background here, if that is desired */
   if(background == 1)
   {
      fflush(stdout);
      fflush(stderr);
      c = fork();
      /* _exit() because pcap uses atexit() to restore the interface */
      if(c > 0) _exit(0);
      if(c < 0) signal_handler(-1);
      setsid();
   }
   /* go into loop processing packets */
   while(1)
      pcap_loop(ip_socket, -1, (pcap_handler)filter_packet, NULL);
}

/* this is the function that's called when pcap reads a packet */
void filter_packet(u_char *u, struct pcap_pkthdr *p, u_char *packet)
{
   /* some defines we need - this only works with ethernet devices */
   #define IP_SIZE	20
   #define TCP_SIZE	20
   
   unsigned short ip_options = 0;
   struct ip *ip;
   struct tcphdr *tcp;
   static u_char *align_buf=NULL;
      
   /* p->len should never be smaller than the smallest possible packet */
   if(p->len < (dlt_len + IP_SIZE + TCP_SIZE)) return;
   /* cast a ip pointer */
   ip = (struct ip *)(packet + dlt_len);
   /* align packet if needed */
   if(align_buf == NULL) align_buf=(u_char *)malloc(4096);
   bcopy((char *)ip, (char *)align_buf, p->len);
   packet = align_buf;
   ip = (struct ip *)align_buf;
   /* lame hack needed for PPP */
   if(ip->ip_p != IPPROTO_TCP) return;
   /* determine length of ip options (usually 0) */
   ip_options = ip->ip_hl;
   ip_options -= 5;
   ip_options *= 4;
   /* cast tcp pointer */
   tcp = (struct tcphdr *)(packet + IP_SIZE + ip_options);
   /* nuke any flags in the offset field */
   ip->ip_off &= 0xFF9F;
   /* toss packets where the fragmentation offset is not 0 */
   if(ip->ip_off != 0) return; 
   /* if a syn was recieved, but a ack was not, log the packet */
   if((tcp->th_flags & TH_SYN) && !(tcp->th_flags & TH_ACK))
      log_syn(ip, tcp);
}

/* this function logs the interesting packets */
void log_syn(struct ip *ip, struct tcphdr *tcp)
{
   fprintf(logfile, "%s|", get_date());
   fprintf(logfile, "%s|%u|", hostlookup(ip->ip_src.s_addr), ntohs(tcp->th_sport));
   fprintf(logfile, "%s|%u\n", hostlookup(ip->ip_dst.s_addr), ntohs(tcp->th_dport));
   fflush(logfile);
}

/* return either a hostname, or a ip address */
char *hostlookup(unsigned int in)
{
   static char blah[4096];
   struct in_addr i;
   struct hostent *he = NULL;
   
   i.s_addr = in;
   if(resolve == 1)
      he = gethostbyaddr((char *)&i, sizeof(struct in_addr), AF_INET);
   if(he == NULL) strncpy(blah, inet_ntoa(i), 4095);
   else strncpy(blah, he->h_name, 4095);
   return (char *)blah;
}

/* shut down in a controlled way, close log file, close socket, and exit */
void signal_handler(int s)
{
   fclose(logfile);
   pcap_close(ip_socket);
   exit(s);
}

/* print the time in a readable way, sorta like asctime() cept different */
char *get_date(void)
{
   struct tm *timeptr;
   time_t timer;
   static char buffer[40];
   char *months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
                     "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
   
   time(&timer);
   timeptr = localtime(&timer);
   sprintf(buffer, "%s %02d %02d:%02d", months[timeptr->tm_mon], timeptr->tm_mday, timeptr->tm_hour, timeptr->tm_min);
   return buffer;
}
   
            
