/*
 * Copyright (C) 2000-2001 Peter J Jones (pjones@pmade.org)
 * All Rights Reserved
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "XML.hh"

/// A macro to save typing when working with xmlChar*
#define XMLCHARCAST(x) reinterpret_cast<char*>(const_cast<xmlChar*>(x))

//# method set_filename #######################################################
/** Set the name of the XML file to parse
	
	This method allows you to set the name of the XML file that
	will get parsed.

	@param filename A string containing the filename
	@author Peter Jones
**/
//#############################################################################
void XML::set_filename(string filename) {
	m_filename = filename;
}
//# method parse ##############################################################
/** Parse the XML file and return a Generator Object
	
	This method will parse the XML config file and return a
	Generator Object that can be used to create a clo++ parser.

	@return A Generator Object
	@author Peter Jones
**/
//#############################################################################
void XML::parse(Generator &generator) {
	xmlNodePtr current_node;

	// open and parse the file, this creates a tree
	m_xml_doc = xmlParseFile(m_filename.c_str());

	// make sure the parse went ok
	if (m_xml_doc == 0)
		throw "error parsing xml file";

	current_node = xmlDocGetRootElement(m_xml_doc);

	// make sure things look good
	if (!current_node || !current_node->name || strcasecmp(XMLCHARCAST(current_node->name), "clo") != 0) {
		xmlFreeDoc(m_xml_doc);
		throw "error, root element missing, or not 'clo'";
	}

	for (current_node=current_node->xmlChildrenNode; current_node!=0; current_node=current_node->next) {
		if (current_node->type != XML_ELEMENT_NODE) continue;
		if (current_node->name == 0) continue;

		if (strcasecmp(XMLCHARCAST(current_node->name), "option") == 0) {
			Option option = parse_option(current_node);
			generator.add_option(option);
		} else if (strcasecmp(XMLCHARCAST(current_node->name), "command") == 0) {
			Command command = parse_command(current_node);
			generator.add_subcommand(command);
		} else if (strcasecmp(XMLCHARCAST(current_node->name), "autohelp") == 0) {
			generator.autohelp(true);
		} else if (strcasecmp(XMLCHARCAST(current_node->name), "autoversion") == 0) {
			generator.autoversion(true);
		} else if (strcasecmp(XMLCHARCAST(current_node->name), "mandatory") == 0) {
			generator.subcommand_mandatory(true);
		} else if (strcasecmp(XMLCHARCAST(current_node->name), "nomix") == 0) {
			generator.nomix(true);
		} else if (strcasecmp(XMLCHARCAST(current_node->name), "class") == 0) {
			generator.set_classname(get_content(current_node));
		} else if (strcasecmp(XMLCHARCAST(current_node->name), "namespace") == 0) {
			generator.set_namespace(get_content(current_node));
		} else if (strcasecmp(XMLCHARCAST(current_node->name), "file") == 0) {
			generator.set_filename(get_content(current_node));
		} else if (strcasecmp(XMLCHARCAST(current_node->name), "codefile") == 0) {
			generator.set_codefile(get_content(current_node));
		}
	}

	xmlFreeDoc(m_xml_doc);
	m_xml_doc = 0;
}
//# method parse_option #######################################################
/** Create a Option object from some XML code
	
	This method will parse some XML code and generate a Option object.

	@param option_node A xmlNodePtr pointing to the start of the option
	@return a Option
	@author Peter Jones
**/
//#############################################################################
Option XML::parse_option(xmlNodePtr option_node) {
	xmlNodePtr	node_i = option_node->xmlChildrenNode;
	Option 		option;

	for (;node_i !=0; node_i=node_i->next) {
		// some safety checks
		if (node_i->type != XML_ELEMENT_NODE) continue;
		if (node_i->name == 0) continue;

		if (strcasecmp(XMLCHARCAST(node_i->name), "type") == 0) {
			string type = get_content(node_i);
			if (strcasecmp(type.c_str(), "flag") == 0) option.set_type(Option::type_flag);
			else if (strcasecmp(type.c_str(), "count") == 0) option.set_type(Option::type_count);
			else if (strcasecmp(type.c_str(), "bool") == 0) option.set_type(Option::type_bool);
			else if (strcasecmp(type.c_str(), "int") == 0) option.set_type(Option::type_int);
			else if (strcasecmp(type.c_str(), "double") == 0) option.set_type(Option::type_double);
			else if (strcasecmp(type.c_str(), "string") == 0) option.set_type(Option::type_string);
			else { cerr << "unknown option type, setting to flag: " << type << endl; option.set_type(Option::type_flag); }
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "long") == 0) {
			option.set_name_long(get_content(node_i));
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "short") == 0) {
			option.set_name_short(get_content(node_i).at(0));
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "description") == 0) {
			option.set_description(get_content(node_i));
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "default") == 0) {
			option.set_default(get_content(node_i));
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "mandatory") == 0) {
			option.is_mandatory(true);
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "argname") == 0) {
			option.set_argname(get_content(node_i));
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "list") == 0) {
			option.is_list(true);
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "map") == 0) {
			option.is_map(true);
		}
	}
	
	return option;
}
//# method parse_command ######################################################
/** Parse <command> blocks
	
	This method will parse the <command> blocks, adding
	per-command options.

	@param command_node the xml node that contains the command
	@return a command
	@author Peter Jones
**/
//#############################################################################
Command XML::parse_command(xmlNodePtr command_node) {
	xmlNodePtr	node_i = command_node->xmlChildrenNode;
	Command		command;

	for (;node_i !=0; node_i=node_i->next) {
		// some safety checks
		if (node_i->type != XML_ELEMENT_NODE) continue;
		if (node_i->name == 0) continue;

		if (strcasecmp(XMLCHARCAST(node_i->name), "name") == 0) {
			command.set_name(get_content(node_i)); 
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "alias") == 0) {
			command.add_alias(get_content(node_i)); 
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "description") == 0) {
			command.set_description(get_content(node_i));
		} else if (strcasecmp(XMLCHARCAST(node_i->name), "option") == 0) {
			Option option = parse_option(node_i);
			command.add_option(option); //FIXME, find better way to use references
		}
	}

	return command;
}
//# method get_content ########################################################
/** Get the content of a xmlNodePtr
	
	This method returns a string containing the contents of the given
	xmlNodePtr.

	@param node The node to get the content out of
	@return A string containing the content
	@author Peter Jones
**/
//#############################################################################
string XML::get_content(xmlNodePtr node) {
	xmlChar* xml_content = xmlNodeGetContent(node);
	string   content;

	if (xml_content != 0) {
		content = XMLCHARCAST(xml_content);
		// libxml says we own this memory now
		delete [] xml_content;
	} else return "";

	if (content.size()) {
		// remove space from front
		StrUtil::rm_space_forward(content, 0);
		// remove space from end
		StrUtil::rm_space_backward(content, content.size() - 1);
		// also remove space after a newline
		StrUtil::rm_space_newline(content);
		// and and duplicate space
		StrUtil::rm_space_dup(content);
	}

	return content;
}
#undef XMLCHARCAST
