/*
 *
 * CLEX File Manager
 *
 * Copyright (C) 1997-2000 Tempest s.r.o. http://www.tempest.sk
 * Copyright (C) 2001-2004 Vlado Potisk <vlado_potisk@clex.sk>
 *
 * CLEX is free software without warranty of any kind; see the
 * GNU General Public License as set out in the "COPYING" document
 * which accompanies the CLEX File Manager package.
 *
 * CLEX can be downloaded from http://www.clex.sk
 *
 */

#include <config.h>

#include <sys/types.h>			/* clex.h */
#include <stdio.h>				/* sprintf() */
#include <stdlib.h>				/* malloc() */
#include <string.h>				/* strlen() */
#include <unistd.h>				/* read() */
#include <limits.h>				/* SSIZE_MAX */

#include "clex.h"
#include "util.h"

#include "control.h"			/* err_exit() */
#include "ustring.h"			/* us_copy() */

/* variables used in pathname_xxx() functions */
static USTRING sf_buff = { 0,0 };
static size_t sf_dirlen;

#ifndef SSIZE_MAX
# define SSIZE_MAX INT_MAX
#endif

/* get rid of directory part in 'pathname' */
const char
*base_name(const char *pathname)
{
	const char *base, *pch;
	char ch;

	for (pch = base = pathname; (ch = *pch++) != '\0'; )
		if (ch == '/')
			base = pch;
	return base;
}

static void
alloc_fail(size_t size)
{
	err_exit("Memory allocation failed, could not allocate %lu bytes",
	  (unsigned long)size);
	/* NOTREACHED */
}

/* malloc with error checking */
void *
emalloc(size_t size)
{
	void *mem;

	if (size > SSIZE_MAX)
		/*
		 * possible problems with signed/unsigned int !
		 *
		 * It is not normal to request such a huge memory
		 * block anyway (16-bit systems are not supported)
		 */
		alloc_fail(size);
	if ((mem = malloc(size)) == 0)
		alloc_fail(size);
	return mem;
}

/* realloc with error checking */
void *
erealloc(void *ptr, size_t size)
{
	void *mem;

	/* not sure if really all realloc()s can handle this case */
	if (ptr == 0)
		return emalloc(size);

	if (size > SSIZE_MAX)
		/* see emalloc() above */
		alloc_fail(size);

	if ((mem = realloc(ptr,size)) == 0)
		alloc_fail(size);
	return mem;
}

/* strdup with error checking */
char *
estrdup(const char *str)
{
	char *dup;

	if (str == 0)
		return 0;
	dup = emalloc(strlen(str) + 1);
	strcpy(dup,str);
	return dup;
}

/* set the directory name for pathname_join() */
void
pathname_set_directory(const char *dir)
{
	char *str;

	sf_dirlen = strlen(dir);
	us_resize(&sf_buff,sf_dirlen + 24);
	/* 24 extra bytes = slash + initial space for the filename */
	str = USTR(sf_buff);
	strcpy(str,dir);
	if (str[sf_dirlen - 1] != '/')
		str[sf_dirlen++] = '/';
		/* the string is not null terminated now, that's ok */
}

/*
 * join the filename 'file' with the directory set by
 * pathname_set_directory() above
 *
 * returned data is overwritten by subsequent calls
 */
char *
pathname_join(const char *file)
{
	us_resize(&sf_buff,sf_dirlen + strlen(file) + 1);
	strcpy(USTR(sf_buff) + sf_dirlen,file);
	return USTR(sf_buff);
}

char *
my_strerror(int errcode)
{
	static char buffer[32];

	sprintf(buffer,"errno = %d",errcode);
	return buffer;
}

/*
 * read data from a descriptor (file, pipe, etc.)
 * returned value - same as read()
 *
 * note: error check should be (r == -1) and not (r < 0) !
 */
ssize_t
read_fd(int fd, char *buff, size_t bytes)
{
	size_t total;
	ssize_t rd;

	for (total = 0; bytes > 0; total += rd, bytes -= rd) {
		rd = read(fd,buff + total,
		  bytes > SSIZE_MAX ? SSIZE_MAX : bytes);
		if (rd == -1)	/* error */
			return -1;
		if (rd == 0)	/* EOF */
			break;
	}
	return total;
}

/* dequote backslash quoted text: 'src' -> 'dst' */
size_t
dequote_txt(const char *src, size_t len, char *dst)
{
	char ch;
	size_t i, j;
	FLAG quote;

	for (i = j = 0, quote = 0; i < len; i++) {
		ch = src[i];
		if (TCLR(quote) || !(quote = ch == '\\'))
			dst[j++] = ch;
	}
	dst[j] = '\0';

	return j;
}
