/*
 *
 * CLEX File Manager
 *
 * Copyright (C) 1997-2000 Tempest s.r.o. http://www.tempest.sk
 * Copyright (C) 2001-2004 Vlado Potisk <vlado_potisk@clex.sk>
 *
 * CLEX is free software without warranty of any kind; see the
 * GNU General Public License as set out in the "COPYING" document
 * which accompanies the CLEX File Manager package.
 *
 * CLEX can be downloaded from http://www.clex.sk
 *
 */

#include <config.h>

#include <sys/types.h>		/* clex.h */
#include <ctype.h>			/* iscntrl() */
#ifdef HAVE_NCURSES_H
# include <ncurses.h>		/* KEY_xxx */
#else
# include <curses.h>
#endif
#include <stdarg.h>			/* va_list */
#include <stdlib.h>			/* exit() */

#include "clex.h"
#include "control.h"

#include "cfg.h"			/* config_prepare() */
#include "completion.h"		/* compl_prepare() */
#include "directory.h"		/* dir_main_prepare() */
#include "edit.h"			/* cx_edit_xxx() */
#include "filepanel.h"		/* cx_files_xxx() */
#include "help.h"			/* help_prepare() */
#include "history.h"		/* history_prepare() */
#include "inout.h"			/* win_panel() */
#include "locate.h"			/* locate_prepare() */
#include "panel.h"			/* cx_pan_xxx() */
#include "select.h"			/* select_prepare() */
#include "sort.h"			/* sort_prepare() */
#include "tty.h"			/* tty_reset() */
#include "undo.h"			/* undo_before() */
#include "xterm_title.h"	/* xterm_title_restore() */

/*
 * This is the main control section. It is table driven.
 *
 * PANEL is the main part of the screen, it shows various data
 *   depending on the panel type, the user can scroll through it.
 *
 * TEXTLINE is a line of text where the user can enter and edit
 *   his/her input.
 *
 * KEY_BINDING contains a keystroke and a corresponding function
 *   to be called every time that key is pressed. All such handler
 *   function names begin with the cx_ prefix.
 *
 * CLEX operation mode is defined by a PANEL, TEXTLINE, and a set of
 *   KEY_BINDINGs tables. The PANEL and TEXTLINE are initialized by
 *   a so-called preparation function after each mode change.
 *
 * Operation mode can be changed in one of two ways:
 *   - straightforward transition from mode A to mode B; this
 *     achieved by setting the 'next_mode' global variable
 *   - nesting of modes; this is achieved by calling another
 *     instance of 'control_loop()'. To go back the variable
 *     'next_mode' must be set to MODE_SPECIAL_RETURN.
 */

typedef struct {
	FLAG escp;			/* press escape key first */
	int key;			/* if this key was pressed ... */
	void (*fn)(void);	/* ... then this function is to be invoked */
} KEY_BINDING;

#define CXM(X,M) void cx_mode_ ## X (void) { control_loop(MODE_ ## M); }
static CXM(cfg,CFG)
static CXM(compare,COMPARE)
static CXM(dir,DIR)
static CXM(history,HIST)
static CXM(help,HELP)
static CXM(p_locate,P_LOCATE)
static CXM(s_locate,S_LOCATE)
static CXM(mainmenu,MAINMENU)
static CXM(select,SELECT)
static CXM(sort,SORT)
static CXM(deselect,DESELECT)

#define CXT(X,M) void cx_trans_ ## X (void) { next_mode = MODE_ ## M; }
static CXT(cfg_confirm,CFG_CONFIRM)
static CXT(quit,SPECIAL_QUIT)
static CXT(return,SPECIAL_RETURN)

/* defined below */
static void menu_prepare(void);
static void cx_menu_pick(void);
static void cx_menu_na(void);

/* temporarily in 3.10 for backward compatibility */
static void ctrl_G_Q(void)
{
	panel_compare.pd->curs = 0;
	win_remark("ctrl-G Q shortcut is obsolete, use <esc> = instead");
	next_mode = MODE_COMPARE;
}
static void ctrl_G_T(void)
{
	panel_compare.pd->curs = 3;
	win_remark("ctrl-G T shortcut is obsolete, use <esc> = instead");
	next_mode = MODE_COMPARE;
}

static KEY_BINDING tab_cfg[] = {
	{ 0,  CH_CTRL('C'),	cx_trans_cfg_confirm	},
	{ 0,  CH_CTRL('M'),	cx_config_edit			},
	{ 0,  'd',			cx_config_default		},
	{ 0,  'u',			cx_config_original		},
	{ 0,  0,			0						}
};

static KEY_BINDING tab_cfg_confirm[] = {
	{ 0,  CH_CTRL('M'),	cx_config_confirm	},
	{ 0,  0,			0					}
};

static KEY_BINDING tab_cfg_edit_num[] = {
	{ 0,  CH_CTRL('M'),	cx_config_num_enter	},
	{ 0,  0,			0					}
};

static KEY_BINDING tab_cfg_edit_str[] = {
	{ 0,  CH_CTRL('M'),	cx_config_str_enter	},
	{ 0,  0,			0					}
};

static KEY_BINDING tab_common[] = {
	{ 0,  CH_CTRL('C'),	cx_trans_return	},
	{ 1,  'q',			cx_trans_quit	},
	{ 1,  'v',			cx_version		},
	{ 0,  CH_CTRL('G'),	cx_menu_na		},	/* catch bogus ctrl-G */
	{ 0,  KEY_F(16),	cx_menu_na		},	/* catch bogus <MENU> */
	{ 0,  KEY_F(1),		cx_mode_help	},
	{ 0,  0,			0				}
};

static KEY_BINDING tab_compare[] = {
	{ 0,  CH_CTRL('M'),	cx_compare		},
	{ 0,  '0',			cx_compare_0	},
	{ 0,  '1',			cx_compare_1	},
	{ 0,  '2',			cx_compare_2	},
	{ 0,  '3',			cx_compare_3	},
	{ 0,  0,			0				}
};

static KEY_BINDING tab_compl[] = {
	{ 0,  CH_CTRL('I'),	cx_compl_complete	},
	{ 0,  CH_CTRL('M'),	cx_compl_complete	},
	{ 0,  0,			0					}
};

static KEY_BINDING tab_dir[] = {
	{ 0,  CH_CTRL('I'),	cx_dir_tab			},
	{ 0,  CH_CTRL('M'),	cx_dir_enter		},
	{ 0,  0,			0					}
};

static KEY_BINDING tab_edit[] = {
	{ 0,  CH_CTRL('B'),	cx_edit_w_left		},
	{ 0,  CH_CTRL('D'),	cx_edit_w_del		},
	{ 0,  CH_CTRL('H'),	cx_edit_backsp		},
	{ 0,  CH_CTRL('U'),	cx_edit_kill		},
	{ 0,  CH_CTRL('V'),	cx_edit_insert_spc	},
	{ 0,  CH_CTRL('W'),	cx_edit_w_right		},
	{ 0,  CH_CTRL('Y'),	cx_edit_delend		},
	{ 0,  CH_CTRL('Z'),	cx_undo				},
	{ 0,  '\177',		cx_edit_delchar		},
	{ 0,  KEY_DC,		cx_edit_delchar		},
	{ 0,  KEY_LEFT,		cx_edit_left		},
	{ 0,  KEY_RIGHT,	cx_edit_right		},
	{ 0,  KEY_HOME,		cx_edit_begin		},
#ifdef KEY_END
	{ 0,  KEY_END,		cx_edit_end			},
#endif
	{ 1,  KEY_UP,		cx_edit_up			},
	{ 1,  KEY_DOWN,		cx_edit_down		},
	{ 0,  0,			0					}
};

static KEY_BINDING tab_editcmd[] = {
	{ 0,  CH_CTRL('E'),	cx_edit_paste_dir	},
	{ 0,  CH_CTRL('G'),	cx_mode_mainmenu	},
	{ 0,  KEY_F(16),	cx_mode_mainmenu	},
	{ 0,  CH_CTRL('I'),	cx_files_tab		},
	{ 1,  CH_CTRL('I'),	cx_edit_fullpath	},
	{ 0,  CH_CTRL('M'),	cx_files_enter		},
	{ 1,  CH_CTRL('M'),	cx_files_cd			},
	{ 0,  CH_CTRL('K'),	cx_hist_complete	},
	{ 0,  CH_CTRL('N'),	cx_hist_next		},
	{ 0,  CH_CTRL('O'),	cx_edit_paste_link	},
	{ 0,  CH_CTRL('P'),	cx_hist_prev		},
	{ 0,  CH_CTRL('R'),	cx_files_reread		},
	{ 1,  CH_CTRL('R'),	cx_files_reread_ug	},
	{ 0,  CH_CTRL('T'),	cx_select_toggle	},
	{ 0,  CH_CTRL('X'),	cx_files_exchange	},
	{ 0,  KEY_IC,		cx_select_toggle	},
	{ 0,  KEY_IL,		cx_select_toggle	},
	{ 0,  KEY_F(2),		cx_edit_cmd_f2		},
	{ 0,  KEY_F(3),		cx_edit_cmd_f3		},
	{ 0,  KEY_F(4),		cx_edit_cmd_f4		},
	{ 0,  KEY_F(5),		cx_edit_cmd_f5		},
	{ 0,  KEY_F(6),		cx_edit_cmd_f6		},
	{ 0,  KEY_F(7),		cx_edit_cmd_f7		},
	{ 0,  KEY_F(8),		cx_edit_cmd_f8		},
	{ 0,  KEY_F(9),		cx_edit_cmd_f9		},
	{ 0,  KEY_F(10),	cx_edit_cmd_f10		},
	{ 0,  KEY_F(11),	cx_edit_cmd_f11		},
	{ 0,  KEY_F(12),	cx_edit_cmd_f12		},
	{ 0,  0,			0					}
};

static KEY_BINDING tab_help[] = {
	{ 0,  CH_CTRL('H'),	cx_help_back		},
	{ 0,  CH_CTRL('M'),	cx_help_link		},
	{ 0,  KEY_LEFT,		cx_help_back		},
	{ 0,  KEY_RIGHT,	cx_help_link		},
	{ 0,  KEY_F(1),		cx_help_contents	},	/* avoid F1 recursion */
	{ 0,  0,			0					}
};

static KEY_BINDING tab_hist[] = {
	{ 0,  CH_CTRL('I'),	cx_hist_paste	},
	{ 0,  CH_CTRL('M'),	cx_hist_enter	},
	{ 0,  CH_CTRL('N'),	cx_pan_up		},	/* redefine history next */
	{ 0,  CH_CTRL('P'),	cx_pan_down		},	/* redefine history prev */
	{ 0,  0,			0				}
};

/* pseudo-table returned by do_action() */
static KEY_BINDING tab_insertchar[] = {
	{ 0, 0,				0				}
};

static KEY_BINDING tab_locate[] = {
	{ 0,  CH_CTRL('I'),	cx_locate_tab	},
	{ 0,  CH_CTRL('M'),	cx_locate_go	},
	{ 0,  KEY_F(2),		cx_edit_cmd_f2	},
	{ 0,  KEY_F(3),		cx_edit_cmd_f3	},
	{ 0,  KEY_F(4),		cx_edit_cmd_f4	},
	{ 0,  KEY_F(5),		cx_edit_cmd_f5	},
	{ 0,  KEY_F(6),		cx_edit_cmd_f6	},
	{ 0,  KEY_F(7),		cx_edit_cmd_f7	},
	{ 0,  KEY_F(8),		cx_edit_cmd_f8	},
	{ 0,  KEY_F(9),		cx_edit_cmd_f9	},
	{ 0,  KEY_F(10),	cx_edit_cmd_f10	},
	{ 0,  KEY_F(11),	cx_edit_cmd_f11	},
	{ 0,  KEY_F(12),	cx_edit_cmd_f12	},
	{ 0,  0,			0				}
};

static KEY_BINDING tab_mainmenu[] = {
/*
 * these lines correspond with the main menu panel,
 * if you change this, you must update initialization
 * in start.c and descriptions in inout.c
 */
	{ 0,  0,			cx_mode_help		},
	/*
	 * no key for cx_mode_help, note the difference:
	 * <F1> in tab_common:    main menu -> help -> main menu
	 * help in tab_mainmenu:  main menu -> help -> file panel
	 */
	{ 1,  'd',			cx_mode_dir			},
	{ 1,  '/',			cx_files_cd_root	},
	{ 1,  '.',			cx_files_cd_parent	},
	{ 1,  '~',			cx_files_cd_home	},
	{ 1,  'h',			cx_mode_history		},
	{ 1,  's',			cx_mode_sort		},
	{ 0,  CH_CTRL('R'),	cx_files_reread		},
	{ 0,  CH_CTRL('F'),	cx_mode_s_locate	},
	{ 1,  CH_CTRL('F'),	cx_mode_p_locate	},
	{ 1,  '=',			cx_mode_compare		},
	/*
	 * No keys for next 2 functions because this table is valid
     * both for the file panel and the main menu. Keys specific
     * for the main menu are listed in the tab_mainmenu2[] below.
	 */
	{ 0,  0,			cx_select_allfiles	},		/* no key */
	{ 0,  0,			cx_select_nofiles	},		/* no key */
	{ 1,  '+',			cx_mode_select		},
	{ 1,  '-',			cx_mode_deselect	},
	{ 1,  '*',			cx_select_invert	},
	{ 1,  'c',			cx_mode_cfg			},
	{ 1,  'v',			cx_version			},
	{ 0,  0,			cx_trans_quit		},	/* no key necessary */
	{ 0,  0,			0					}
};

static KEY_BINDING tab_mainmenu2[] = {
	{ 0,  CH_CTRL('M'),	cx_menu_pick		},
	{ 0,  '+',			cx_select_allfiles	},
	{ 0,  '-',			cx_select_nofiles	},
	{ 0,  't',			ctrl_G_T			},	/* temporarily */
	{ 0,  'q',			ctrl_G_Q			},	/* temporarily */
	{ 0,  0,			0					}
};

static KEY_BINDING tab_panel[] = {
	{ 0,  KEY_UP,		cx_pan_up		},
	{ 0,  KEY_DOWN,		cx_pan_down		},
	{ 0,  KEY_PPAGE,	cx_pan_pgup		},
	{ 0,  KEY_NPAGE,	cx_pan_pgdown	},
	{ 1,  KEY_HOME,		cx_pan_home		},
#ifdef KEY_END
	{ 1,  KEY_END,		cx_pan_end		},
#endif
	{ 0,  0,			0				}
};

static KEY_BINDING tab_select[] = {
	{ 0,  CH_CTRL('M'),	cx_select_files	},
	{ 0,  CH_CTRL('N'),	cx_select_next	},
	{ 0,  CH_CTRL('P'),	cx_select_prev	},
	{ 0,  0,			0				}
};

static KEY_BINDING tab_sort[] = {
	{ 0,  ' ',			cx_sort_set		},
	{ 0,  CH_CTRL('M'),	cx_sort_set		},
	{ 0,  0,			0				}
};

typedef struct {
	CODE mode;
	void (*prepare_fn)(void);
	KEY_BINDING *table[6];	/* up to 6 tables terminated with NULL;
							   order is sometimes important, only
							   the first KEY_BINDING for a given key
							   is executed */
} MODE_DEFINITION;

static MODE_DEFINITION mode_definition[] = {
	{ MODE_CFG_CONFIRM, config_confirm_prepare,
		{ tab_cfg_confirm,tab_panel,tab_common,0 } },
	{ MODE_CFG_EDIT_NUM, config_edit_num_prepare,
		{ tab_cfg_edit_num,tab_edit,tab_common,0 } },
	{ MODE_CFG_EDIT_TXT, config_edit_str_prepare,
		{ tab_cfg_edit_str,tab_edit,tab_common,0 } },
	{ MODE_CFG, config_prepare,
		{ tab_cfg,tab_panel,tab_common,0 } },
	{ MODE_COMPARE, compare_prepare,
		{ tab_compare,tab_panel,tab_common,0 } },
	{ MODE_COMPL, compl_prepare,
		{ tab_compl,tab_panel,tab_edit,tab_common,0 } },
	{ MODE_DIR, dir_main_prepare,
		{ tab_dir,tab_panel,tab_edit,tab_common,0 } },
	{ MODE_DIR_SPLIT, dir_split_prepare,
		{ tab_dir,tab_panel,tab_edit,tab_common,0 } },
	{ MODE_FILE, files_main_prepare,
		{ tab_editcmd,tab_mainmenu,tab_panel,tab_edit,tab_common,0 } },
	{ MODE_HELP, help_prepare,
		{ tab_help,tab_panel,tab_common,0 } },
	{ MODE_HIST, hist_prepare,
		{ tab_hist,tab_panel,tab_edit,tab_common,0 } },
	{ MODE_S_LOCATE, locate_prepare,
		{ tab_locate,tab_edit,tab_panel,tab_common,0 } },
	{ MODE_P_LOCATE, locate_prepare,
		{ tab_locate,tab_edit,tab_panel,tab_common,0 } },
	{ MODE_MAINMENU, menu_prepare,
		{ tab_mainmenu,tab_mainmenu2,tab_panel,tab_common,0 } },
	{ MODE_SELECT, select_prepare,
		{ tab_select,tab_panel,tab_edit,tab_common,0 } },
	{ MODE_SORT, sort_prepare,
		{ tab_sort,tab_panel,tab_common,0 } },
	{ MODE_DESELECT, select_prepare,
		{ tab_select,tab_panel,tab_edit,tab_common,0 } },
	{ 0, 0, { 0 } }
};

/* linked list of all control loop instances */
static struct operation_mode {
	CODE mode;
	PANEL_DESC *panel;
	TEXTLINE *textline;
	struct operation_mode *previous;
} mode_init = { 0,0,0,0 }, *clex_mode = &mode_init;

int
get_current_mode(void)
{
	return clex_mode->mode;
}

int
get_previous_mode(void)
{
	return clex_mode->previous->mode;
}

static MODE_DEFINITION *
get_modedef(int mode)
{
	MODE_DEFINITION *p;

	for (p = mode_definition; p->mode; p++)
		if (p->mode == mode)
			return p;

	err_exit("BUG: requested operation mode %d is invalid",mode);

	/* NOTREACHED */
	return 0;
}

static KEY_BINDING *
do_action(int key, KEY_BINDING **tables)
{
	int lower_ch, i, noesc_idx;
	FLAG esc;
	KEY_BINDING **ptab, *tab, *noesc_tab;

	if (key == CH_ESC)
		return 0;

	esc = kbd_esc();
	lower_ch = IS_CHAR(key) ? tolower(key) : key;
	noesc_tab = 0;
	noesc_idx = 0;	/* to prevent compiler warning */
	for (ptab = tables; (tab = *ptab); ptab++) {
		for (i = 0; tab[i].fn; i++)
			if (lower_ch == tab[i].key) {
				if (esc && !tab[i].escp) {
					/*
					 * an entry with 'escp' flag has higher priority,
					 * we must continue to search the tables to see
					 * if such entry for the given key exists
					 */
					if (noesc_tab == 0) {
						/* accept the first definition only */
						noesc_tab = tab;
						noesc_idx = i;
					}
				}
				else if (esc || !tab[i].escp) {
					(*tab[i].fn)();
					return tab;
				}
			}
	}
	if (noesc_tab) {
		(*noesc_tab[noesc_idx].fn)();
		return noesc_tab;
	}

	if (textline && IS_CHAR(key) && !iscntrl(key)) {
		edit_insertchar(key);
		return tab_insertchar;
	}
#if 0
	win_remark_fmt("DEBUG: %o key (no function)",key);
#else
	win_remark("pressed key has no function "
	  "(F1 - help, ctrl-C - cancel)");
#endif
	return 0;
}

/*
 * main control loop for a selected mode 'mode'
 * control loops for different modes are nested whenever necessary
 */
void
control_loop(int mode)
{
	MODE_DEFINITION *modedef;
	KEY_BINDING *kb_tab;
	struct operation_mode current_mode;

	current_mode.previous = clex_mode;
	clex_mode = &current_mode;

	/* panel and textline inherited the from previous mode */
	clex_mode->panel = clex_mode->previous->panel;
	clex_mode->textline = clex_mode->previous->textline;

	for (next_mode = mode; /* until break */; ) {
		modedef = get_modedef(clex_mode->mode = next_mode);
		next_mode = 0;
		(*modedef->prepare_fn)();
		win_heading();
		if (panel != clex_mode->panel) {
			clex_mode->panel = panel;
			win_panel();
		}
		if (textline != clex_mode->textline) {
			if (textline)
				textline->undo_possible = 0;
			clex_mode->textline = textline;
			win_edit();
		}

		for (; /* until break */;) {
			undo_before();
			kb_tab = do_action(kbd_input(),modedef->table);
			undo_after();
			if (next_mode) {
				if (next_mode == MODE_SPECIAL_RETURN
				  && clex_mode->previous->mode == 0) {
					win_remark("to quit CLEX press <esc> Q");
					next_mode = 0;
				}
				else
					break;
			}

			/* some special handling not implemented with tables */
			switch (clex_mode->mode) {
			case MODE_MAINMENU:
				if (kb_tab == tab_mainmenu || kb_tab == tab_mainmenu2)
					next_mode = MODE_SPECIAL_RETURN;
				break;
			case MODE_COMPL:
				if (kb_tab == tab_edit || kb_tab == tab_insertchar)
					next_mode = MODE_SPECIAL_RETURN;
				break;
			case MODE_DIR:
			case MODE_DIR_SPLIT:
				/* TRUE == TRUE does not mean NON-ZERO == NON-ZERO */
				if (!panel->norev != !textline->size) {
					panel->norev = !panel->norev;
					win_panel_opt();
				}
				break;
			case MODE_P_LOCATE:
			case MODE_S_LOCATE:
				if (textline == &line_cmd)
					next_mode = MODE_SPECIAL_RETURN;
				else
					locate_update();
				break;
			}
			if (next_mode)
				break;
		}

		if (next_mode == MODE_SPECIAL_QUIT)
			err_exit("Normal exit");

		if (next_mode == MODE_SPECIAL_RETURN) {
			next_mode = 0;
			break;
		}
	}

	clex_mode = clex_mode->previous;
	win_heading();
	if (panel != clex_mode->panel) {
		panel = clex_mode->panel;
		pan_adjust(panel);		/* screen size might have changed */
		win_panel();
	}
	if (textline != clex_mode->textline) {
		textline = clex_mode->textline;
		edit_adjust();
		win_edit();
	}
}

static void
menu_prepare(void)
{
	/* leave cursor position unchanged */
	panel = panel_mainmenu.pd;
	pan_adjust(panel);
	textline = 0;
}

static void
cx_menu_pick(void)
{
	if (panel_mainmenu.pd->curs >= 0)
		(*tab_mainmenu[panel_mainmenu.pd->curs].fn)();
	if (next_mode == 0)
		next_mode = MODE_SPECIAL_RETURN;
}

static void
cx_menu_na(void)
{
	win_remark("ctrl-G works only in the file panel");
}

void
cx_version(void)
{
	win_remark("Welcome to CLEX " VERSION " !");
}

/*
 * err_exit() is the only exit function that terminates CLEX main
 * process. It is used for normal (no error) termination as well.
 */
void
err_exit(const char *format, ...)
{
	va_list argptr;

	/*
	 * all cleanup functions used here:
	 *  - must not call err_exit()
	 *  - must not require initialization
	 */
	xterm_title_restore();
	if (display.curses)
		curses_stop();
	tty_reset();

	fputs("\nTerminating CLEX: ",stdout);
	va_start(argptr,format);
	vprintf(format,argptr);
	va_end(argptr);
	putchar('\n');

	tty_pgrp_reset();
	exit(0);
	/* NOTREACHED */
}
