/*
*
* $Id: queue.c,v 1.3 2001/06/02 07:10:45 konst Exp $
*
* Copyright (C) 2000 Denis V. Dmitrienko <denis@null.net>
* Copyright (C) 2000 Bill Soudan <soudan@kde.org>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or (at
* your option) any later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
* USA
*
*/

#include <stdlib.h>
#include <time.h>

#include "queue.h"
#include "list.h"

void icq_UDPQueueNew(ICQLINK *link)
{
  link->icq_UDPQueue = list_new();
  link->icq_UDPExpireInterval = 15; /* expire interval = 15 sec */
}

void icq_UDPQueuePut(ICQLINK *link, icq_Packet *p, int attempt)
{
  icq_UDPQueueItem *ptr = (icq_UDPQueueItem*)malloc(sizeof(icq_UDPQueueItem));
#ifdef QUEUE_DEBUG
  printf("icq_UDPQueuePut(seq=0x%04X, cmd=0x%04X)\n", icq_PacketReadUDPOutSeq1(p),
	 icq_PacketReadUDPOutCmd(p));
#endif
  ptr->attempts = attempt;
  ptr->expire = time(0L)+link->icq_UDPExpireInterval;
  ptr->pack = p;
#ifdef QUEUE_DEBUG
  printf("enqueuing queueitem %p\n", ptr);
#endif
  list_enqueue(link->icq_UDPQueue, ptr);
}

icq_Packet *icq_UDPQueueGet(ICQLINK *link)
{
  icq_UDPQueueItem *ptr = (icq_UDPQueueItem*)list_first(link->icq_UDPQueue);
  icq_Packet *pack = 0L;
  if(ptr)
  {
    pack = ptr->pack;
    list_remove(link->icq_UDPQueue, (list_node*)ptr);
  }
#ifdef QUEUE_DEBUG
  if(pack)
    printf("icq_UDPQueueGet(cmd=0x%04X)\n", icq_PacketReadUDPOutCmd(pack));
#endif
  return pack;
}

icq_Packet *icq_UDPQueuePeek(ICQLINK *link)
{
  icq_UDPQueueItem *ptr = (icq_UDPQueueItem*)list_first(link->icq_UDPQueue);
  if(ptr)
    return ptr->pack;
  else
    return 0L;
}

void _icq_UDPQueueItemFree(void *p)
{
  icq_UDPQueueItem *pitem=(icq_UDPQueueItem *)p;

#ifdef QUEUE_DEBUG
  printf("_icq_UDPQueueItemFree(%p)\n", p);
#endif

  if (pitem->pack)
    icq_PacketDelete(pitem->pack);

  free(p);
}

/* Frees the queue and dispose it */
void icq_UDPQueueDelete(ICQLINK *link)
{
#ifdef QUEUE_DEBUG
  printf("icq_UDPQueueDelete\n");
#endif
  if(link->icq_UDPQueue)
  {
    list_delete(link->icq_UDPQueue, _icq_UDPQueueItemFree);
    link->icq_UDPQueue = 0;
  }
}

/* Only frees the queue */
void icq_UDPQueueFree(ICQLINK *link)
{
#ifdef QUEUE_DEBUG
  printf("icq_UDPQueueFree\n");
#endif
  if(link->icq_UDPQueue)
    list_free(link->icq_UDPQueue, _icq_UDPQueueItemFree);
}

int icq_UDPQueueFindSeq(void *p, va_list data)
{
  int seq=va_arg(data, int);
  return icq_PacketReadUDPOutSeq1(((icq_UDPQueueItem *)p)->pack) == seq;
}

void icq_UDPQueueDelSeq(ICQLINK *link, WORD seq)
{
  icq_UDPQueueItem *ptr;
#ifdef QUEUE_DEBUG
  printf("icq_UDPQueueDelSeq(seq=0x%04X", seq);
#endif
  ptr = list_traverse(link->icq_UDPQueue, icq_UDPQueueFindSeq, seq);
  if(ptr)
  {
#ifdef QUEUE_DEBUG
    printf(", cmd=0x%04X",icq_PacketReadUDPOutCmd(ptr->pack));
#endif
    list_remove(link->icq_UDPQueue, ptr);
    _icq_UDPQueueItemFree(ptr);
  }
#ifdef QUEUE_DEBUG
  printf(")\n");
#endif
}

long icq_UDPQueueInterval(ICQLINK *link)
{
  long interval;
  icq_UDPQueueItem *ptr = (icq_UDPQueueItem*)list_first(link->icq_UDPQueue);
  if(ptr)
  {
    interval = ptr->expire - time(0L);
    return interval>=0?interval:0;
  }
  return -1;
}
