/*
*
* $Id: list.c,v 1.3 2001/06/02 07:10:45 konst Exp $
*
* Copyright (C) 2000 Denis V. Dmitrienko <denis@null.net>
* Copyright (C) 2000 Bill Soudan <soudan@kde.org>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or (at
* your option) any later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
* USA
*
*/

/*
 * linked list functions
 */

#include <stdlib.h>
#include <stdio.h>

#include "list.h"

list *list_new()
{
  list *plist=(list *)malloc(sizeof(list));

  plist->head=0;
  plist->tail=0;
  plist->count=0;

  return plist;
}

/* Frees all list nodes and list itself */
void list_delete(list *plist, void (*item_free_f)(void *))
{
  list_free(plist, item_free_f);
  free(plist);
}

/* Only frees the list nodes */
void list_free(list *plist, void (*item_free_f)(void *))
{
  list_node *p=plist->head;

#ifdef LIST_TRACE
  printf("list_free(%p)\n", plist);
  list_dump(plist);
#endif

  while(p)
  {
    list_node *ptemp=p;

    p=p->next;
    (*item_free_f)((void *)ptemp->item);
    list_remove_node(plist, ptemp);
  }
}

void list_insert(list *plist, list_node *pnode, void *pitem)
{
  list_node *pnew=(list_node *)malloc(sizeof(list_node));
  pnew->item=pitem;

#ifdef LIST_TRACE
  printf("inserting %x (node=%x) into list %x\n", pitem, pnew, plist);
#endif
 
  plist->count++;

  /* null source node signifies insert at end of list */
  if(!pnode) 
  {
    pnew->previous=plist->tail;
    pnew->next=0;
    if(plist->tail)
      plist->tail->next=pnew;
    plist->tail=pnew;

    if(!plist->head)
      plist->head=pnew;
  }
  else
  {
    pnew->previous=pnode->previous;
    pnew->next=pnode;

    if(pnew->previous)
      pnew->previous->next=pnew;

    if(pnew->next)
      pnode->previous=pnew;
	
    if(plist->head==pnode)
      plist->head=pnew;
  }

#ifdef LIST_TRACE
  list_dump(plist);
#endif
}

void *list_remove_node(list *plist, list_node *p)
{
  void *pitem;

  if(!p)
    return 0;

#ifdef LIST_TRACE
  printf("removing %x (node=%x) from list %x\n", p->item, p, plist);
#endif

  plist->count--;

  if(p->next)
    p->next->previous=p->previous;

  if(p->previous)
    p->previous->next=p->next;

  if(plist->head==p)
    plist->head=p->next;

  if(plist->tail==p)
    plist->tail=p->previous;

  p->next=0;
  p->previous=0;

#ifdef LIST_TRACE
  list_dump(plist);
#endif

  pitem=p->item;

  free(p);

  return pitem;
}

void *list_traverse(list *plist, int (*item_f)(void *, va_list), ...)
{
  list_node *i=plist->head;
  int f=0;
  va_list ap;

#ifdef LIST_TRACE
  printf("list_traverse(%p)\n", plist);
  list_dump(plist);
#endif
  va_start(ap, item_f);

  /* call item_f for each item in list until end of list or item 
   * function returns 0 */
  while(i && !f)
  {
    list_node *pnext=i->next;

    if(!(f=(*item_f)(i->item, ap)))
      i=pnext;
  }
  va_end(ap);
  if (i)
    return i->item;
  else
    return 0;
}

int list_dump(list *plist)
{
  list_node *p=plist->head;

  printf("list %x { head=%x, tail=%x, count=%d }\ncontents: ",
	 (int)plist, (int)plist->head, (int)plist->tail, plist->count);

  while(p)
  {
    printf("%x, ", (int)p->item);
    p=p->next;
  }
  printf("end\n");

  return 0;
}

void *list_first(list *plist)
{
  if(plist->head)
    return plist->head->item;
  else
    return 0;
}

void *list_last(list *plist)
{
  if(plist->tail)
    return plist->tail->item;
  else
    return 0;
}

void *list_at(list *plist, int num)
{
  list_node *ptr = plist->head;
  while(ptr && num)
  {
    num--;
    ptr = ptr->next;
  }
  if(!num)
    return ptr->item;
  else
    return 0L;
}

list_node *list_find(list *plist, void *pitem)
{
  list_node *p=plist->head;

  while(p)
  {
    if(p->item==pitem)
      return p;
    p=p->next;
  }
  return 0;
}

void *list_remove(list *plist, void *pitem)
{
  list_node *p=list_find(plist, pitem);

  if(p)
    return list_remove_node(plist, p);
  else
    return 0;
}
