/*
 * calmwm - the calm window manager
 *
 * Copyright (c) 2004 Marius Aamodt Eriksen <marius@monkey.org>
 * All rights reserved.
 *
 * $Id: search.c,v 1.17 2004/08/19 08:37:16 marius Exp $
 */

#include "headers.h"
#include "calmwm.h"

#define SearchMask (KeyPressMask|ExposureMask)

static int  _strsubmatch(char *, char *);

void
search_init(struct screen_ctx *sc)
{
	sc->searchwin = XCreateSimpleWindow(G_dpy, sc->rootwin, 0, 0,
	    1, 1, 1, sc->blackpixl, sc->whitepixl);
}

/*
 * Input: list of items,
 * Output: choose one
 * so, exactly like menus
 */

struct menu *
search_start(struct menu_q *menuq,
    void (*match)(struct menu_q *, struct menu_q *, char *), 
    void (*rank)(struct menu_q *resultq, char *search), char *prompt)
{
	struct screen_ctx *sc = screen_current();
	int x, y, dx, dy, fontheight,
	    focusrevert, mutated, xmax, ymax, warp, added, beobnoxious = 0;
	XEvent e;
	char searchstr[MENU_MAXENTRY];
	char dispstr[MENU_MAXENTRY*2];
	char promptstr[MENU_MAXENTRY];
	Window focuswin;
	struct menu *mi = NULL;
	struct menu_q resultq;
	char chr;
	enum ctltype ctl;
	size_t len;
	u_int n;

	if (prompt == NULL)
		prompt = "search";

	TAILQ_INIT(&resultq);

	xmax = DisplayWidth(G_dpy, sc->which);
	ymax = DisplayHeight(G_dpy, sc->which);

	xu_ptr_getpos(sc->rootwin, &x, &y);

	searchstr[0] = '\0';

	snprintf(promptstr, sizeof(promptstr), "%s>", prompt);
	dy = fontheight = G_font->ascent + G_font->descent + 1;
	dx = XTextWidth(G_font, promptstr, strlen(promptstr));

	XMoveResizeWindow(G_dpy, sc->searchwin, x, y, dx, dy);
	XSelectInput(G_dpy, sc->searchwin, SearchMask);
	XMapRaised(G_dpy, sc->searchwin);

	/* TODO: eventually, the mouse should be able to select
	 * results as well.  Right now we grab it only to set a fancy
	 * cursor. */
	if (xu_ptr_grab(sc->searchwin, 0, G_cursor_question) < 0) {
		XUnmapWindow(G_dpy, sc->searchwin);
		return (NULL);
	}

	XGetInputFocus(G_dpy, &focuswin, &focusrevert);
	XSetInputFocus(G_dpy, sc->searchwin,
	    RevertToPointerRoot, CurrentTime);

	for (;;) {
		XMaskEvent(G_dpy, SearchMask, &e);

		switch (e.type) {
		case KeyPress:
			if (input_keycodetrans(e.xkey.keycode, &ctl, &chr, 1) < 0)
				continue;

			added = mutated = 0;

			switch (ctl) {
			case CTL_ERASEONE:
				if ((len = strlen(searchstr)) > 0)
					searchstr[len - 1] = '\0';
				mutated = 1;
				break;
			case CTL_UP:
				mi = TAILQ_LAST(&resultq, menu_q);
				if (mi == NULL)
					break;

				TAILQ_REMOVE(&resultq, mi, resultentry);
				TAILQ_INSERT_HEAD(&resultq, mi, resultentry);
				break;
			case CTL_DOWN:
				mi = TAILQ_FIRST(&resultq);
				if (mi == NULL)
					break;

				TAILQ_REMOVE(&resultq, mi, resultentry);
				TAILQ_INSERT_TAIL(&resultq, mi, resultentry);
				break;
			case CTL_RETURN:
				/* This is just picking the match the
				 * cursor is over. */
				if (strlen(searchstr) == 0)
					goto out;

				if ((mi = TAILQ_FIRST(&resultq)) != NULL)
					goto found;
				else
					goto out;
			case CTL_ABORT:
				goto out;
			default:
				break;
			}

			if (chr != '\0') {
				char str[2];

				str[0] = chr;
				str[1] = '\0';
				mutated = 1;
				added =
				    strlcat(searchstr, str, sizeof(searchstr));
			}

			beobnoxious = 0;
			if (mutated && strlen(searchstr) > 0) {
				(*match)(menuq, &resultq, searchstr);
				beobnoxious = TAILQ_EMPTY(&resultq);
				if (!beobnoxious && rank != NULL)
					(*rank)(&resultq, searchstr);
			} else if (mutated)
				TAILQ_INIT(&resultq);

		case Expose:
			snprintf(dispstr, sizeof(dispstr), "%s%s%s",
			    promptstr, searchstr,
			    beobnoxious ? " [FAILING]" : "");
			dx = XTextWidth(G_font, dispstr, strlen(dispstr));
			dy = fontheight;

			TAILQ_FOREACH(mi, &resultq, resultentry) {
				dx = MAX(dx,
				    XTextWidth(G_font, mi->text,
					MIN(strlen(mi->text), MENU_MAXENTRY)));
				dy += fontheight;
			}

			/*
			 * Calculate new geometry.
			 *
			 * XXX - put this into a util function -- it's
			 * used elsewhere, too.
			 */
			warp = 0;
			if (x < 0) {
				x = 0;
				warp = 1;
			}
			if (x + dx >= xmax) {
				x = xmax - dx;
				warp = 1;
			}

			if (y < 0) {
				y = 0;
				warp = 1;
			}
			if (y + dy >= ymax) {
				y = ymax - dy;
				warp = 1;
			}

			if (warp)
				xu_ptr_setpos(sc->rootwin, x, y);

			XClearWindow(G_dpy, sc->searchwin);
			XMoveResizeWindow(G_dpy, sc->searchwin, x, y, dx, dy);

			XDrawString(G_dpy, sc->searchwin, sc->gc, 0,
			    G_font->ascent + 1,
			    dispstr, strlen(dispstr));
			n = 1;
			TAILQ_FOREACH(mi, &resultq, resultentry) {
				XDrawString(G_dpy, sc->searchwin, sc->gc, 0,
				    n*fontheight + G_font->ascent + 1,
				    mi->text,
				    MIN(strlen(mi->text), MENU_MAXENTRY));
				n++;
			}

			if (n > 1)
				XFillRectangle(G_dpy, sc->searchwin, sc->gc,
				    0, fontheight, dx, fontheight);
			break;
		}
	}

 out:
	/* (if no match) */
	xu_ptr_ungrab();
	XSetInputFocus(G_dpy, focuswin, focusrevert, CurrentTime);
 found:
	XUnmapWindow(G_dpy, sc->searchwin);

	return (mi);
}

void
search_match_client(struct menu_q *menuq, struct menu_q *resultq, char *search)
{
	struct menu_q tier2;
	struct winname *wn;
	struct client_ctx *curcc = client_current();
	struct menu *mi;

	TAILQ_INIT(resultq);
	TAILQ_INIT(&tier2);

	/*
	 * 1. Look through labels.
	 * 
	 * 2. Look at title history, from present to past.
	 *
	 * XXX - this needs to be ranked better.
	 */

	TAILQ_FOREACH(mi, menuq, entry) {
		struct client_ctx *cc = mi->ctx;
		if (cc == curcc)
			continue;

		if (cc->label != NULL && _strsubmatch(search, cc->label)) {
			cc->matchname = cc->label;
			TAILQ_INSERT_TAIL(resultq, mi, resultentry);
		} else {
			TAILQ_FOREACH_REVERSE(wn, &cc->nameq, winname_q, entry)
				if (_strsubmatch(search, wn->name)) {
					cc->matchname = wn->name;
					TAILQ_INSERT_TAIL(&tier2,
					    mi, resultentry);
					break;
				}
		}
	}

	while ((mi = TAILQ_FIRST(&tier2)) != NULL) {
		TAILQ_REMOVE(&tier2, mi, resultentry);
		TAILQ_INSERT_TAIL(resultq, mi, resultentry);
	}
}

void
search_match_text(struct menu_q *menuq, struct menu_q *resultq, char *search)
{
	struct menu *mi;

	TAILQ_INIT(resultq);

	TAILQ_FOREACH(mi, menuq, entry)
		if (_strsubmatch(search, mi->text))
			TAILQ_INSERT_TAIL(resultq, mi, resultentry);
}

void
search_rank_text(struct menu_q *resultq, char *search)
{
	return;
}

static int
_strsubmatch(char *sub, char *str)
{
	size_t len, sublen;
	u_int n;

	len = strlen(str);
	sublen = strlen(sub);

	if (sublen > len)
		return (0);

	for (n = 0; n <= len - sublen; n++)
		if (strncasecmp(sub, str + n, sublen) == 0)
			return (1);

	return (0);
}
