/* 
   cadaver, command-line DAV client
   Copyright (C) 1999-2001, Joe Orton <joe@manyfish.co.uk>, 
   except where otherwise indicated.
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <sys/types.h>

#include <sys/time.h>
#include <sys/stat.h>

#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <time.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif 
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <errno.h>

#ifdef NEED_SNPRINTF_H
#include "snprintf.h"
#endif

#include "i18n.h"

#include "getopt.h"
#include "getpass.h"

#ifdef ENABLE_NETRC
#include "netrc.h"
#endif

#include <ne_request.h>
#include <ne_auth.h>
#include <ne_basic.h>
#include <ne_string.h>
#include <ne_uri.h>
#include <ne_socket.h>
#include <ne_locks.h>
#include <ne_alloc.h>
#include <ne_redirect.h>

#include "common.h"

#include "cadaver.h"
#include "cmdline.h"
#include "commands.h"
#include "options.h"
#include "utils.h"

#define DEFAULT_NAMESPACE "http://webdav.org/cadaver/custom-properties/"

#ifdef ENABLE_NETRC
static netrc_entry *netrc_list; /* list of netrc entries */
#endif

ne_session *session;
ne_lock_session *lock_session;
static nssl_context *ssl_context;

static char *progname, *lockstore; /* argv[0] */

char *server_hostname, *proxy_hostname;
int server_port, proxy_port;
char *server_username = NULL, *server_password = NULL;

/* Global options */

char *path, /* current working collection */
    *old_path; /* previous working collection */
/* TODO: use dynamically allocated memory for paths, maybe */

int tolerant; /* tolerate DAV-enabledness failure */

int use_expect = 0;

int have_connection = 0; /* true when we are connected to the server */
int dav_collection;

/* Current output state */
static enum out_state {
    out_none, /* not doing anything */
    out_incommand, /* doing a simple command */
    out_transfer_start, /* transferring a file, not yet started */
    out_transfer_plain, /* doing a plain ... transfer */
    out_transfer_pretty /* doing a pretty progress bar transfer */
} out_state;   

/* Protoypes */

static RETSIGTYPE quit_handler(int signo);

static void close_connection(void);
static void open_connection(const char *url);

static void execute_version(void);

static void transfer_progress(void *ud, off_t progress, off_t total);
static void connection_status(void *ud, ne_conn_status status, 
			      const char *info);
static int supply_creds_server(void *userdata, const char *realm, int attempt,
			       char *username, char *password);
static int supply_creds_proxy(void *userdata, const char *realm, int attempt,
			      char *username, char *password);

#define CMD_VARY 9999

/* Separate structures for commands and command names. */
/* DON'T FORGET TO ADD A NEW COMMAND ALIAS WHEN YOU ADD A NEW COMMAND */
const struct command commands[] = {
/* C1: connected, 1-arg function C2: connected, 2-arg function
 * U0: disconnected, 0-arg function. */
#define C1(x,c,h) { cmd_##x, true, 1, 1, parmscope_remote, execute_##x,c,h }
#define U0(x,h) { cmd_##x, false, 0, 0, parmscope_none, execute_##x,#x,h }
#define UO1(x,c,h) { cmd_##x, false, 0, 1, parmscope_none, execute_##x,c,h }
#define C2M(x,c,h) { cmd_##x, true, 2, CMD_VARY, parmscope_remote, multi_##x,c,h }
#define C1M(x,c,h) { cmd_##x, true, 1, CMD_VARY, parmscope_remote, multi_##x,c,h }
    { cmd_ls, true, 0, 1, parmscope_remote, execute_ls, 
      N_("ls [path]"), N_("List contents of current [or other] collection") },
    C1(cd, N_("cd path"), N_("Change to specified collection")),
    { cmd_pwd, true, 0, 0, parmscope_none, execute_pwd,
      "pwd", N_("Display name of current collection") },
    { cmd_put, true, 1, 2, parmscope_none, execute_put, 
      N_("put local [remote]"), N_("Upload local file") },
    { cmd_get, true, 1, 2, parmscope_none, execute_get, 
      N_("get remote [local]"), N_("Download remote resource") },
    C1M(mget, N_("mget remote..."), N_("Download many remote resources")),
    { cmd_mput, true, 1, CMD_VARY, parmscope_local, multi_mput, 
      N_("mput local..."), N_("Upload many local files") },
    C1(edit, N_("edit resource"), N_("Edit given resource")),
    C1M(less, N_("less remote..."), N_("Display remote resource through pager")), 
    C1M(mkcol, N_("mkcol remote..."), N_("Create remote collection(s)")), 
    C1M(cat, N_("cat remote..."), N_("Display remote resource(s)")), 
    C1M(delete, N_("delete remote..."), N_("Delete non-collection resource(s)")),
    C1M(rmcol, N_("rmcol remote..."), N_("Delete remote collections and ALL contents")),
    C2M(copy, N_("copy source... dest"), N_("Copy resource(s) from source to dest")), 
    C2M(move, N_("move source... dest"), N_("Move resource(s) from source to dest")),
    
/* DON'T FORGET TO ADD A NEW COMMAND ALIAS WHEN YOU ADD A NEW COMMAND */
    
    C1(lock, N_("lock resource"), N_("Lock given resource")),
    C1(unlock, N_("unlock resource"), N_("Unlock given resource")),
    C1(discover, N_("discover resource"), N_("Display lock information for resource")),
    C1(steal, N_("steal resource"), N_("Steal lock token for resource")),
    { cmd_showlocks, true, 0, 0, parmscope_none, execute_showlocks,
      "showlocks", N_("Display list of owned locks") },
    
#if 0
    C1(propedit, "propedit resource", "Enter property editor for resource"),
#endif
    C1(propnames, "propnames res", "Names of properties defined on resource"),

    { cmd_chexec, true, 2, 2, parmscope_none, execute_chexec,
      N_("chexec [+|-] remote"), N_("Change isexecutable property of resource") },
    
    { cmd_propget, true, 1, 2, parmscope_none, execute_propget,
      N_("propget res [propname]"), 
      N_("Retrieve properties of resource") },
    { cmd_propdel, true, 2, 2, parmscope_none, execute_propdel,
      N_("propdel res propname"), 
      N_("Delete property from resource") },
    { cmd_propset, true, 3, 3, parmscope_none, execute_propset,
      N_("propset res propname value"),
      N_("Set property on resource") },

    { cmd_set, false, 0, 2, parmscope_none, execute_set, 
      N_("set [option] [value]"), N_("Set an option, or display options") },
    { cmd_open, false, 1, 1, parmscope_none, open_connection, 
      "open URL", N_("Open connection to given URL") },
    { cmd_close, true, 0, 0, parmscope_none, close_connection, 
      "close", N_("Close current connection") },
    { cmd_echo, false, 1, CMD_VARY, parmscope_remote, execute_echo, 
      "echo", NULL },
    { cmd_quit, false, 0, 1, parmscope_none, NULL, "quit", N_("Exit program") },
    /* Unconnected operation, 1 mandatory argument */
    { cmd_unset, false, 1, 2, parmscope_none, execute_unset, 
      N_("unset [option] [value]"), N_("Unsets or clears value from option.") },
    /* Unconnected operation, 0 arguments */
    UO1(lcd, N_("lcd [directory]"), N_("Change local working directory")), 
    { cmd_lls, false, 0, CMD_VARY, parmscope_local, execute_lls, 
      N_("lls [options]"), N_("Display local directory listing") },
    U0(lpwd, N_("Print local working directory")),
    { cmd_logout, true, 0, 0, parmscope_none, execute_logout, "logout",
      N_("Logout of authentication session") },
    UO1(help, N_("help [command]"), N_("Display help message")), 
    U0(version, NULL),
    { cmd_unknown, 0 } /* end-of-list marker, DO NOT move */

/* DON'T FORGET TO ADD A NEW COMMAND ALIAS WHEN YOU ADD A NEW COMMAND. */

#undef C1
#undef U0
#undef UO1
#undef C2M
#undef C1M
};    

static void usage(void)
{
    printf(_(
"Usage: %s [OPTIONS] http://hostname[:port]/path\n"
"  Port defaults to 80, path defaults to '/'\n"
"Options:\n"
"  -e, --expect100  Enable sending of `Expect: 100-continue' header.\n"
"     *** Warning: For Apache servers, use only with version 1.3.9 and later.\n"
"  -t, --tolerant   Allow cd/open into non-WebDAV enabled collection.\n"
"  -V, --version    Display version information.\n"
"  -h, --help       Display this help message.\n"
"Please send bug reports and feature requests to <cadaver@webdav.org>\n"), progname);
}

static void execute_version(void)
{
    printf(PACKAGE " " VERSION "\n%s\n", ne_version_string());
}

static void close_connection(void)
{
    finish_locking(lockstore);
    ne_session_destroy(session);
    sock_destroy_ssl_context(ssl_context);
    have_connection = false;
    NE_FREE(path);
    NE_FREE(old_path);
    printf(_("Connection to `%s' closed.\n"), server_hostname);
}

/* Sets the current collection to the given path.  Returns zero on
 * success, non-zero if newpath is an untolerated non-WebDAV
 * collection. */
int set_path(const char *newpath)
{
    int is_coll = (getrestype(newpath) == resr_collection);
    if (is_coll || tolerant) {
	if (!is_coll) {
	    dav_collection = false;
	    printf(_("Ignored error: %s not WebDAV-enabled:\n%s\n"), newpath,
		   ne_get_error(session));
	} else {
	    dav_collection = true;
	}
	return 0;
    } else {
	printf(_("Could not access %s (not WebDAV-enabled?):\n%s\n"), newpath,
	       ne_get_error(session));
	return 1;
    }
}

static void redirect_notify(void *userdata, const char *src, const char *dest)
{
    switch (out_state) {
    case out_none:
	printf("Redirected to %s\n", dest);
	break;
    case out_incommand:
	printf("redirected to %s...", dest);
	break;
    case out_transfer_start:
    case out_transfer_plain:
    case out_transfer_pretty:
	/* FIXME */
	break;
    }
}

static int privkey_prompt(void *userdata, const char *filename, 
			  char *buf, int buflen)
{
    static char prompt[BUFSIZ];
    int len;

    snprintf(prompt, BUFSIZ, _("Enter private key password for `%s': "), filename);
    
    do {
	char *pass = fm_getpassword(prompt);
	if (pass == NULL) {
	    return -1;
	}
	strncpy(buf, pass, buflen);
	/* zero it out the buffer now */
	memset(pass, 0, PASSWORDLEN);
	len = strlen(buf);
	if (len < 4) {
	    printf(_("Private key password must be 4 or more characters!\n"));
	}
    } while (len < 4);
 
    return 0;
}

static int setup_ssl(void)
{
    char *cc = get_option(opt_cert);
    char *key = get_option(opt_certkey);
    
    if (ne_set_secure(session, 1)) {
	printf(
	    _("SSL support has not be compiled into this application.\n"
	      "URLs beginning with \"https:\" cannot be used!\n"));
	return -1;
    } 
    
    /* Set the client cert too if we have one. */
    if (cc != NULL) {
	
	/* Use our private key password prompt. */
	sock_set_key_prompt(ssl_context, privkey_prompt, NULL);

	if (sock_set_client_cert(ssl_context, cc, key)) {
	    printf(_("Could not use client certificate `%s' (wrong password)?\n"),
		   cc);
	    return -1;
	} else {
	    printf(_("Using client certificate `%s'.\n"), cc);
	    ne_set_secure_context(session, ssl_context);
	}
    }

    return 0;
}

/* FIXME: Leaky as a bucket */
static void open_connection(const char *url)
{
    char *proxy_host = get_option(opt_proxy), *pnt;
    ne_server_capabilities caps;
    struct uri uri = {0};
    int ret;

    if (have_connection) {
	close_connection();
    } else {
	NE_FREE(path);
	NE_FREE(old_path);
    }

    session = ne_session_create();
    ssl_context = sock_create_ssl_context();
    ne_redirect_register(session, NULL, redirect_notify, NULL);

    init_locking(lockstore);
    
    ne_set_progress(session, transfer_progress, NULL);
    ne_set_status(session, connection_status, NULL);

    NE_FREE(server_hostname);
	
    /* Single argument: see whether we have a path or scheme */
    if (strchr(url, '/') == NULL) {
	/* No path, no scheme -> just a hostname */
	pnt = strchr(url, ':');
	if (pnt != NULL) {
	    *pnt++ = '\0';
	    uri.port = atoi(pnt);
	} else {
	    uri.port = 80;
	}
	uri.host = ne_strdup(url);
    } else {
	/* Parse the URL */
	if (uri_parse(url, &uri, NULL) || uri.host == NULL) {
	    printf(_("Could not parse URL `%s'\n"), url);
	    return;
	}

	if (uri.scheme == NULL || strcasecmp(uri.scheme, "http") == 0) {
	    if (uri.port == -1) {
		uri.port = 80;
	    }
	} else if (strcasecmp(uri.scheme, "https") == 0) {
	    if (uri.port == -1) {
		uri.port = 443;
	    }
	    /* Set up SSL stuff. */
	    if (setup_ssl()) {
		return;
	    }
	} else {
	    printf(_("Unrecognized URL scheme `%s'.\n"), uri.scheme);
	    uri_free(&uri);
	    return;
	}
    }

    if (uri.path == NULL) {
	uri.path = ne_strdup("/");
    } else {
	if (!uri_has_trailing_slash(uri.path)) {
	    CONCAT2(pnt, uri.path, "/");
	    free(uri.path);
	    uri.path = pnt;
	}
    }

    /* Get the proxy details */
    if (proxy_host != NULL) {
	if (get_option(opt_proxy_port) != NULL) {
	    proxy_port = atoi((char *)get_option(opt_proxy_port));
	} else {
	    proxy_port = 8080;
	}
	proxy_hostname = proxy_host;
    }

    server_hostname = uri.host;
    server_port = uri.port;

#ifdef ENABLE_NETRC
    {
	netrc_entry *found;
	found = search_netrc(netrc_list, server_hostname);
	if (found != NULL) {
	    if (found->account && found->password) {
		server_username = found->account;
		server_password = found->password;
	    }
	}
    }
#endif /* ENABLE_NETRC */
    have_connection = false;
    /* Default to '/' if no path given */
#if 0
    if (open_path != NULL) {
	if (uri_has_trailing_slash(open_path)) {
	    use_path = ne_strdup(open_path);
	} else if (open_path[0] == '/') {
	    CONCAT2(use_path, open_path, "/");
	} else {
	    CONCAT3(use_path, "/", open_path, "/");
	}
    } else {
	use_path = ne_strdup("/");
    }
#endif

    ne_set_expect100(session, use_expect);
    ne_set_useragent(session, PACKAGE "/" VERSION);
    ne_set_server_auth(session, supply_creds_server, NULL);
    ne_set_proxy_auth(session, supply_creds_proxy, NULL);
    
    /* FIXME: returns here LEAK */
    if (proxy_host) {
	if (ne_session_proxy(session, proxy_hostname, proxy_port) != NE_OK) {
	    printf(_("Could not resolve proxy server hostname `%s'.\n"),
		    proxy_hostname);
	    return;
	}
    }

    if (ne_session_server(session, server_hostname, server_port) != NE_OK) {
	printf(_("Could not resolve server hostname `%s'.\n"), server_hostname);
	return;
    }
    
    ret = ne_options(session, uri.path, &caps);
    
    switch (ret) {
    case NE_OK:
	have_connection = true;
	path = NULL;
	if (set_path(uri.path)) {
	    close_connection();
	} else {
	    path = uri.path;
	}
	break;
    case NE_CONNECT:
	printf(_("Could not connect to remote host.\n"));
	break;
    default:
	/* FIXME: This is NOT a "could not open connection" error */
	printf(_("Could not contact server:\n%s\n"),
	       ne_get_error(session));
	break;
    }

}
       
/* Sets proxy server from hostport argument */    
static void set_proxy(const char *str)
{
    char *hostname = ne_strdup(str), *pnt;

    pnt = strchr(hostname, ':');
    if (pnt != NULL) {
	*pnt++ = '\0';
    }
    set_option(opt_proxy, (void *)hostname);
    set_option(opt_proxy_port, pnt);
}

static void parse_args(int argc, char **argv)
{
    static const struct option opts[] = {
	{ "version", no_argument, NULL, 'V' },
	{ "help", no_argument, NULL, 'h' },
	{ "expect100", no_argument, NULL, 'e' },
	{ "proxy", required_argument, NULL, 'p' },
	{ "tolerant", no_argument, NULL, 't' },
	{ 0, 0, 0, 0 }
    };
    int optc;
    while ((optc = getopt_long(argc, argv, "ehtp:V", opts, NULL)) != -1) {
	switch (optc) {
	case 'h': usage(); exit(-1);
	case 'V': execute_version(); exit(-1);
	case 'e': use_expect = true; break;
	case 'p': set_proxy(optarg); break;
	case 't': tolerant = 1; break;
	case '?': 
	default:
	    printf(_("Try `%s --help' for more information.\n"), progname);
	    exit(-1);
	}
    }
    if (optind == (argc-1)) {
	open_connection(argv[optind]);
#ifdef HAVE_ADD_HISTORY
	{ 
	    char *run_cmd;
	    CONCAT2(run_cmd, "open ", argv[optind]);
	    add_history(run_cmd);
	    free(run_cmd);
	}
#endif
    } else if (optind < argc) {
	usage();
	exit(-1);
    }
}

static char * read_command(void)
{
    static char prompt[BUFSIZ];
    if (path) {
	snprintf(prompt, BUFSIZ, "dav:%s%c ", path,
		  dav_collection?'>':'?');
    } else {
	sprintf(prompt, "dav:!> ");
    }
    return readline(prompt); 
}

static int execute_command(const char *line)
{
    const struct command *cmd;
    char **tokens;
    int argcount, ret = 0;
    tokens = parse_command(line, &argcount);
    if (argcount == 0) {
	free(tokens);
	return 0;
    }
    argcount--;
    cmd = get_command(tokens[0]);
    if (cmd == NULL) {
	printf(_("Unrecognised command. Type 'help' for a list of commands.\n"));
    } else if (argcount < cmd->min_args) {
	printf(_("The `%s' command requires %d argument%s"),
		tokens[0], cmd->min_args, cmd->min_args==1?"":"s");
	if (cmd->short_help) {
	    printf(_(":\n  %s : %s\n"), cmd->call, cmd->short_help);
	} else {
	    printf(".\n");
	}
    } else if (argcount > cmd->max_args) {
	if (cmd->max_args) {
	    printf(_("The `%s' command takes at most %d argument%s"), tokens[0],
		    cmd->max_args, cmd->max_args==1?"":"s");
	} else {
	    printf(_("The `%s' command takes no arguments"), tokens[0]);
	}	    
	if (cmd->short_help) {
	    printf(_(":\n" "  %s : %s\n"), cmd->call, cmd->short_help);
	} else {
	    printf(".\n");
	}
    } else if (!have_connection && cmd->needs_connection) {
	printf(_("The `%s' command can only be used when connected to the server.\n"
		  "Try running `open' first (see `help open' for more details).\n"), 
		  tokens[0]);
    } else if (cmd->id == cmd_quit) {
	ret = -1;
    } else {
	/* Cast away */
	void (*take0)(void)=cmd->handler, 
	    (*take1)(const char *)=cmd->handler, 
	    (*take2)(const char *, const char *)=cmd->handler,
	    (*take3)(const char *, const char *, const char *)=cmd->handler,
	    (*takeV)(int, const char **)=cmd->handler;
	/* with a nod in the general direction of apache */
	switch (cmd->max_args) {
	case 0: (*take0)(); break;
	case 1: /* tokens[1]==NULL if argcount==0 */
	    (*take1)(tokens[1]); break; 
	case 2: 
	    if (argcount <=1) {
		(*take2)(tokens[1], NULL);
	    } else {
		(*take2)(tokens[1], tokens[2]);
	    }
	    break;
	case 3:
	    (*take3)(tokens[1], tokens[2], tokens[3]);
	    break;
	case CMD_VARY:
	    (*takeV)(argcount, (const char **) &tokens[1]);
	default:
	    break;
	}
    }
    split_string_free(tokens);
    return ret;
}

static RETSIGTYPE quit_handler(int sig)
{
    /* Reinstall handler */
    if (child_running) {
	/* The child gets the signal anyway... it can deal with it.
	 * Proper way is probably to ignore signals while child is
	 * running? */
	signal(sig, quit_handler);
	return;
    } else {
	printf(_("Terminated by signal %d.\n"), sig);
	if (have_connection) {
	    close_connection();
	}
	exit(-1);
    }
}

void init_signals(void)
{
    signal(SIGPIPE, SIG_IGN);
    signal(SIGTERM, quit_handler);
    signal(SIGABRT, quit_handler);
    signal(SIGQUIT, quit_handler);
    signal(SIGINT, quit_handler);
}

static void init_netrc(void)
{
#ifdef ENABLE_NETRC
    char *netrc;
    CONCAT2(netrc, getenv("HOME"), "/.netrc");
    netrc_list = parse_netrc(netrc);
#endif
}

static void init_rcfile(void)
{
    char *rcfile, buf[BUFSIZ];
    struct stat st;
    FILE *f;
    CONCAT2(rcfile, getenv("HOME"), "/.cadaverrc");
    if (stat(rcfile, &st) != 0) {
	NE_DEBUG(DEBUG_FILES, "No rcfile.\n");
    } else {
	f = fopen(rcfile, "r");
	if (f == NULL) {
	    printf(_("Could not read rcfile %s: %s\n"), rcfile, 
		   strerror(errno));
	} else {
	    for (;;) {
		if (fgets(buf, BUFSIZ, f) != NULL) {
		    STRIP_EOL(buf);
		    execute_command(buf);
		} else {
		    break;
		}
	    }
	    fclose(f);
	}
    }
    free(rcfile);
}


#ifdef HAVE_LIBREADLINE

#define COMPLETION_CACHE_EXPIRE 10 /* seconds */

#ifndef HAVE_RL_COMPLETION_MATCHES
/* readline <4.2 compatibility. */
#define rl_completion_matches completion_matches
#define rl_filename_completion_function filename_completion_function
#endif

static char *remote_completion(const char *text, int state)
{
    static struct resource *reslist, *current;
    static int len;
    static time_t last_fetch;
    static char *last_path;

    char *name;
    
    if (state == 0) {
	/* Check to see if we should refresh the dumb cache.
	 * or, initialize the local cache of remote filenames
	 * The remote resource list persists until a completion
	 * in a new context is requested or the cache expires.
	 */

	/* TODO get cache expire time from config, currently from cadaver.h
	 * TODO cache and fetch on deep/absolute paths: (path: /a/b/, text: c/d)
	 */
	if (last_fetch < (time(NULL) - COMPLETION_CACHE_EXPIRE) 
	    || !last_path 
	    || strcmp(path, last_path) != 0) {

	    if (last_path != NULL) {
		free(last_path);
	    }

	    if (reslist != NULL) { 
		free_resource_list(reslist); 
	    }

	    /* Hide the connection status */
	    ne_set_status(session, NULL, NULL);
	    if (fetch_resource_list(session, path, 1, 0, &reslist) != NE_OK) {
		reslist = NULL;
	    }
	    /* Restore the session connection printing */
	    ne_set_status(session, connection_status, NULL);

	    last_path = ne_strdup(path);
	}

	current = reslist;
	len = strlen(text);
	time(&last_fetch);
    }

    while (current) {
	/* Massage the absolute URI to a URI relative to our path */
	/* Copy & paste & search & replace from ls.c */
	if (uri_has_trailing_slash(current->uri)) {
	    current->uri[strlen(current->uri)-1] = '\0';
	}

	name = strrchr(current->uri, '/');
	if (name != NULL && strlen(name+1) > 0) {
	    name++;
	} else {
	    name = current->uri;
	}
	name = uri_unescape(name);

	if (strncmp(text, name, len) == 0) {
	    current = current->next;
	    return ne_strdup(name);
	}

	current = current->next;
    }
    
    return NULL;
}

static char **completion(const char *text, int start, int end)
{
    char **matches = NULL;
    char *sep = strchr(rl_line_buffer, ' ');

    if (start == 0) {
	matches = rl_completion_matches(text, command_generator);
    }
    else if (sep != NULL) {
	char *cname = ne_strndup(rl_line_buffer, sep - rl_line_buffer);
	const struct command *cmd;
	cname[sep - rl_line_buffer] = '\0';
	cmd = get_command(cname);
	if (cmd != NULL) { 
	    switch (cmd->scope) {
	    case parmscope_none:
		break;
	    case parmscope_local:
		matches = rl_completion_matches(text, 
						rl_filename_completion_function);
		break;
	    case parmscope_option:
		/* TODO */
		break;
	    case parmscope_remote:
		matches = rl_completion_matches(text, remote_completion);
		break;
	    }
	}
	free(cname);
    }		    
    return matches;
}

#endif /* HAVE_LIBREADLINE */

void output(enum output_type t, const char *fmt, ...)
{
    va_list params;
    if (t == o_finish) {
	switch (out_state) {
	case out_transfer_plain:
	    printf("] ");
	    break;
	default:
	    putchar(' ');
	    break;
	}
    }
    va_start(params, fmt);
    vfprintf(stdout, fmt, params);
    va_end(params);
    fflush(stdout);
    switch (t) { 
    case o_start:
	out_state = out_incommand;
	break;
    case o_transfer:
	out_state = out_transfer_start;
	break;
    case o_finish:
	out_state = out_none;
	break;
    }
}

static void init_readline(void)
{
#ifdef HAVE_LIBREADLINE
    rl_readline_name = "cadaver";
    rl_attempted_completion_function = completion;
#endif /* HAVE_LIBREADLINE */
}

#ifndef HAVE_LIBREADLINE
char *readline(const char *prompt)
{
    static char buf[256];
    char *ret;
    if (prompt) {
	printf("%s", prompt);
    }
    ret = fgets(buf, 256, stdin);
    if (ret) {
	STRIP_EOL(buf);
	return ne_strdup(buf);
    } else {
	return NULL;
    }
}
#endif

static void init_options(void)
{
    char *lockowner, *tmp;
    char *user = getenv("USER"), *hostname = getenv("HOSTNAME");
    
    if (user && hostname) {
	/* set this here so they can override it */
	CONCAT4(lockowner, "mailto:", user, "@", hostname);
	set_option(opt_lockowner, lockowner);
    } else {
	set_option(opt_lockowner, NULL);
    }

    set_option(opt_editor, NULL);
    set_option(opt_namespace, ne_strdup(DEFAULT_NAMESPACE));
    set_bool_option(opt_overwrite, 1);
    lockdepth = NE_DEPTH_INFINITE;
    lockstore = ne_lockscope_exclusive;

    /* This is what Markus Kahn says we should do. */
    if ((tmp = getenv("LC_ALL")) ||
	(tmp = getenv("LC_CTYPE")) ||
	(tmp = getenv("LANG"))) {
	if (strstr(tmp, "UTF-8")) {
	    int val = 1;
	    set_option(opt_utf8, &val);
	}
    }

}

int main(int argc, char *argv[])
{
    int ret = 0;
    char *home = getenv("HOME");

    path = NULL;
    have_connection = false;
    progname = argv[0];

#ifdef ENABLE_NLS
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);
#endif /* ENABLE_NLS */

    ne_debug_init(stderr, 0);
    if (!home) {
	/* Show me the way to go home... */
	printf(_("Environment variable $HOME needs to be set!\n"));
	return -1;
    }

    sock_init();

    CONCAT2(lockstore, home, "/.davlocks");

    /* Options before rcfile, so rcfile settings can
     * override defaults */
    init_options();
    init_netrc();

    init_signals();

    init_rcfile();
    
    parse_args(argc, argv);

    init_readline();

    while (ret == 0) {
	char *cmd;
	cmd = read_command();
	if (cmd == NULL) {
	    /* Is it safe to do this... they just closed stdin, so
	     * is it bad to write to stdout? */
	    putchar('\n');
	    ret = 1;
	} else {
#ifdef HAVE_ADD_HISTORY
	    if (strcmp(cmd, "") != 0) add_history(cmd);
#endif
	    ret = execute_command(cmd);
	    free(cmd);
	}
    }
    if (have_connection) {
	close_connection();
    }
    finish_locking(lockstore);

    return 0;
}

static void connection_status(void *ud, ne_conn_status status, const char *info)
{
    if (get_bool_option(opt_quiet)) {
	return;
    }
    switch (out_state) {
    case out_none:
	switch (status) {
	case ne_conn_namelookup:
	    printf(_("Looking up hostname... "));
	    break;
	case ne_conn_connecting:
	    printf(_("Connecting to server... "));
	    break;
	case ne_conn_connected:
	    printf(_("connected.\n"));
	    break;
	case ne_conn_secure:
	    printf(_("Using secure connection: %s\n"), info);
	    break;
	}
	break;
    case out_incommand:
	/* fall-through */
    case out_transfer_start:
	switch (status) {
	case ne_conn_namelookup:
	case ne_conn_secure:
	    /* should never happen */
	    break;
	case ne_conn_connecting:
	    printf(_(" (reconnecting..."));
	    break;
	case ne_conn_connected:
	    printf(_("done)"));
	    break;
	}
	break;
    case out_transfer_plain:
	switch (status) {
	case ne_conn_namelookup:
	case ne_conn_secure:
	    break;
	case ne_conn_connecting:
	    printf(_("] reconnecting: "));
	    break;
	case ne_conn_connected:
	    printf(_("okay ["));
	    break;
	}
	break;
    case out_transfer_pretty:
	switch (status) {
	case ne_conn_namelookup:
	case ne_conn_secure:
	    break;
	case ne_conn_connecting:
	    printf(_("\rTransfer timed out, reconnecting... "));
	    break;
	case ne_conn_connected:
	    printf(_("okay."));
	    break;
	}
	break;	
    }
    fflush(stdout);
}

/* From ncftp.
   This function is (C) 1995 Mike Gleason, (mgleason@NcFTP.com)
 */
static void 
sub_timeval(struct timeval *tdiff, struct timeval *t1, struct timeval *t0)
{
    tdiff->tv_sec = t1->tv_sec - t0->tv_sec;
    tdiff->tv_usec = t1->tv_usec - t0->tv_usec;
    if (tdiff->tv_usec < 0) {
	tdiff->tv_sec--;
	tdiff->tv_usec += 1000000;
    }
}

/* Smooth progress bar.
 * Doesn't update the bar more than once every 100ms, since this 
 * might give flicker, and would be bad if we are displaying on
 * a slow link anyway.
 */
static void pretty_progress_bar(off_t progress, off_t total)
{
    int len, n;
    double pc;
    static struct timeval last_call = {0};
    struct timeval this_call;
    
    if (total < 0)
	return;

    if (progress < total && gettimeofday(&this_call, NULL) == 0) {
	struct timeval diff;
	sub_timeval(&diff, &this_call, &last_call);
	if (diff.tv_sec == 0 && diff.tv_usec < 100000) {
	    return;
	}
	last_call = this_call;
    }
    if (progress == 0 || total == 0) {
	pc = 0;
    } else {
	pc = (double)progress / total;
    }
    len = pc * 30;
    printf(_("\rProgress: ["));
    for (n = 0; n<30; n++) {
	putchar((n<len-1)?'=':
		 (n==(len-1)?'>':' '));
    }
    printf(_("] %5.1f%% of %ld bytes"), pc*100, total);
    fflush(stdout);
}

static void transfer_progress(void *ud, off_t progress, off_t total)
{
    switch (out_state) {
    case out_none:
    case out_incommand:
	/* Do nothing */
	return;
    case out_transfer_start:
	if (isatty(STDOUT_FILENO) && total > 0) {
	    out_state = out_transfer_pretty;
	    putchar('\n');
	    pretty_progress_bar(progress, total);
	} else {
	    out_state = out_transfer_plain;
	    printf(" [.");
	}
	break;
    case out_transfer_pretty:
	if (total > 0) {
	    pretty_progress_bar(progress, total);
	}
	break;
    case out_transfer_plain:
	putchar('.');
	fflush(stdout);
	break;
    }
}

static int supply_creds(const char *prompt, const char *realm, const char *hostname,
			char *username, char *password)
{
    char *tmp;

    switch (out_state) {
    case out_transfer_pretty:
	putchar('\n');
	/*** fall-through ***/
    case out_none:
	break;
    case out_incommand:
	/* fall-through */
    case out_transfer_start:
	putchar(' ');
	break;
    case out_transfer_plain:
	printf("] ");
	break;
    }
    printf(prompt, realm, hostname);
    
    tmp = readline(_("Username: "));
    if (tmp == NULL) {
	printf(_("\rAuthentication aborted!\n"));
	return -1;
    } else if (strlen(tmp) >= NE_ABUFSIZ) {
	printf(_("\rUsername too long (>%d)\n"), NE_ABUFSIZ);
	free(tmp);
	return -1;
    }

    strcpy(username, tmp);
    free(tmp);

    tmp = fm_getpassword(_("Password: "));
    if (tmp == NULL) {
	printf(_("Authentication aborted!\n"));
	return -1;
    } else if (strlen(tmp) >= NE_ABUFSIZ) {
	printf(_("\rPassword too long (>%d)\n"), NE_ABUFSIZ);
	memset(tmp, strlen(tmp), 0);
	return -1;
    }
    
    strcpy(password, tmp);
	
    switch (out_state) {
    case out_transfer_start:
    case out_incommand:
	printf(_("Retrying:"));
	fflush(stdout);
	break;
    case out_transfer_plain:
	printf(_("Retrying ["));
	fflush(stdout);
	break;
    default:
	break;
    }
    return 0;
}

static int supply_creds_server(void *userdata, const char *realm, int attempt,
			       char *username, char *password)
{
    if (attempt > 1)
	return -1;

    /* Pass back netrc values if we have them. */
    if (server_username != NULL && server_password != NULL) {
	strncpy(username, server_username, NE_ABUFSIZ);
	strncpy(password, server_password, NE_ABUFSIZ);
	return 0;
    }

    return supply_creds(
	_("Authentication required for %s on server `%s':\n"), realm,
	server_hostname, username, password);
}

static int supply_creds_proxy(void *userdata, const char *realm, int attempt,
			      char *username, char *password) 
{
    if (attempt > 1)
	return -1;

    return supply_creds(
	_("Authentication required for %s on proxy server `%s':\n"), realm,
	proxy_hostname, username, password);
}

