/*
    ettercap -- ncurses interface for data injector and filtering form

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "include/ec_main.h"

#ifdef HAVE_NCURSES  // don't compile if ncurses interface is not supported

#include <string.h>
#include <ncurses.h>
#include <unistd.h>
#ifdef HAVE_CTYPE_H
   #include <ctype.h>
#endif
#ifdef HAVE_FORM
   #include <form.h>
#endif



#include "include/ec_interface.h"
#include "include/ec_interface_sniff_data.h"
#include "include/ec_decodedata.h"
#include "include/ec_error.h"
#include "include/ec_filterdrop.h"
#ifdef HAVE_FORM
   #include "include/ec_interface_form.h"
#endif

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif


#define BOTTOM_COLOR 1        // color schemes
#define TITLE_COLOR  2
#define MAIN_COLOR   3
#define POINT_COLOR  4
#define SEL_COLOR    5
#define HELP_COLOR   6
#define SNIFF_COLOR  7


// protos...

void Interface_Inject_Redraw(void);
int Interface_Inject_Run(u_char *inject_data, char proto, char *app);
int Interface_Inject_Filter(DROP_FILTER *filter);

// global variables

extern WINDOW *main_window;
extern WINDOW *data_source_win, *data_dest_win, *data_source, *data_dest, *win_pointer;

extern int W_MAINX1, W_MAINY1, W_MAINX2, W_MAINY2;
extern int W_BOTTOMY2;



//---------------------------


void Interface_Inject_Redraw(void)
{

   Interface_Sniff_Data_Redraw();

#ifdef DEBUG
   Debug_msg("Interface_Inject_Redraw");
#endif


   doupdate();
}



int Interface_Inject_Run(u_char *inject_data, char proto, char *app)
{
   WINDOW *inject_window, *i_win;
   int dimY = 10;
   int dimX = 60;
   char inject_sequence[MAX_INJECT];
   int len;

#ifdef DEBUG
   Debug_msg("Interface_Inject_Run -- %c %s", proto, app);
#endif

   i_win = newwin(dimY+2, dimX+2, W_BOTTOMY2/2 - dimY/2, W_MAINX2/2 - dimX/2);
   inject_window = newwin(dimY, dimX, W_BOTTOMY2/2 - dimY/2 +1, W_MAINX2/2 - dimX/2 +1);
   wbkgdset(i_win, COLOR_PAIR(HELP_COLOR));
   wattron(i_win, A_BOLD);
   box(i_win,ACS_VLINE,ACS_HLINE);
   mvwprintw(i_win,  0, 2, "Type characters to be injected (max %d):", MAX_INJECT);
   wbkgdset(inject_window, COLOR_PAIR(BOTTOM_COLOR));
   wmove(inject_window, 0, 0);
   echo();
   scrollok(inject_window, TRUE);
   keypad(inject_window, TRUE);
   curs_set(TRUE);
   wnoutrefresh(i_win);
   wnoutrefresh(inject_window);
   doupdate();

   mvwgetnstr(inject_window, 1, 0, inject_sequence, MAX_INJECT);

#ifdef DEBUG
   Debug_msg("Interface_Inject_Run -- inject_sequence len -- [%d]", strlen(inject_sequence));
#endif

   noecho();
   curs_set(FALSE);
   delwin(i_win);
   delwin(inject_window);
   doupdate();

   len = FilterDrop_strescape(inject_data, inject_sequence);

#ifdef DEBUG
   Debug_msg("Interface_Inject_Run -- inject_data len -- [%d]", len);
#endif

   return len;

}


#ifdef HAVE_FORM


int Interface_Inject_Filter(DROP_FILTER *filter)
{
   WINDOW *w;
   FORM *form;
   FIELD *f[12];
   int finished = 0, c;
   unsigned n = 0;
   int status = 0;

#ifdef DEBUG
   Debug_msg("Interface_Inject_Filter");
#endif

   refresh();

   f[n++] = make_label(0, 15, "Define a filter");
   f[n++] = make_label(1, 0, "Search :");
   f[n++] = make_field(2, 0, 4, MAX_FILTER/4, FALSE);
   f[n++] = make_label(7, 0, "Action (Drop/Replace/Log) :");
   f[n++] = make_field(7, 28, 1, 7, FALSE);
   f[n++] = make_label(9, 0, "Replace :");
   f[n++] = make_field(10, 0, 4, MAX_FILTER/4, FALSE);
   f[n++] = (FIELD *)0;


   form = new_form(f);
   display_form(form);
   w = form_win(form);

   if (win_pointer == data_dest_win)
   {
      char type[2];
      sprintf(type, "%c",  ARP_Filter_Source.type);
      set_field_buffer(f[2], 0, ARP_Filter_Source.search);
      set_field_buffer(f[6], 0, ARP_Filter_Source.replace);
      set_field_buffer(f[4], 0, type);
   }
   else
   {
      char type[2];
      sprintf(type, "%c",  ARP_Filter_Dest.type);
      set_field_buffer(f[2], 0, ARP_Filter_Dest.search);
      set_field_buffer(f[6], 0, ARP_Filter_Dest.replace);
      set_field_buffer(f[4], 0, type);
   }

   curs_set(1);

   form_driver(form, REQ_OVL_MODE);

   finished = get_form_data(form, w);

   erase_form(form);
   free_form(form);

   if (finished == 2)
   {
      memcpy(&filter->search, field_buffer(f[2], 0), MAX_FILTER);
      trim_buffer(filter->search, ' ');
      memcpy(&filter->replace, field_buffer(f[6], 0), MAX_FILTER);
      trim_buffer(filter->replace, ' ');

      filter->slen = strlen(filter->search);
      filter->rlen = strlen(filter->replace);

      memcpy(&filter->type, field_buffer(f[4], 0), 1);
      filter->type = toupper(filter->type);
      if (filter->type == ' ') filter->type = 0;
      status = 1;
   }

   for (c = 0; f[c] != 0; c++)
      free_field(f[c]);

   curs_set(0);

   return status;
}

#else    // DOESN'T HAVE_FORM

int Interface_Inject_Filter(DROP_FILTER *filter)
{
   WINDOW *filter_window, *f_win;
   int dimY = 10;
   int dimX = 60;
   char type[7];

#ifdef DEBUG
   Debug_msg("Interface_Inject_Filter -- NO FORM");
#endif

   f_win = newwin(dimY+2, dimX+2, W_BOTTOMY2/2 - dimY/2, W_MAINX2/2 - dimX/2);
   filter_window = newwin(dimY, dimX, W_BOTTOMY2/2 - dimY/2 +1, W_MAINX2/2 - dimX/2 +1);
   wbkgdset(f_win, COLOR_PAIR(HELP_COLOR));
   wattron(f_win, A_BOLD);
   box(f_win,ACS_VLINE,ACS_HLINE);
   mvwprintw(f_win,  0, 2, "Define a filter:", MAX_INJECT);
   wbkgdset(filter_window, COLOR_PAIR(BOTTOM_COLOR));
   wmove(filter_window, 0, 0);
   echo();
   scrollok(filter_window, TRUE);
   keypad(filter_window, TRUE);
   curs_set(TRUE);

   wprintw(filter_window, "Search : ");
      wnoutrefresh(f_win);
      wnoutrefresh(filter_window);
      doupdate();
   wgetnstr(filter_window, filter->search, MAX_FILTER);
   filter->slen = strlen(filter->search);

   wprintw(filter_window, "\nAction (Drop/Replace/Log) : ");
      wnoutrefresh(f_win);
      wnoutrefresh(filter_window);
      doupdate();
   wgetnstr(filter_window, type, 7);
   filter->type = toupper(type[0]);

   if (filter->type == 'R')
   {
      wprintw(filter_window, "\nReplace : ");
         wnoutrefresh(f_win);
         wnoutrefresh(filter_window);
         doupdate();
      wgetnstr(filter_window, filter->replace, MAX_FILTER);
      filter->rlen = strlen(filter->replace);
   }

   noecho();
   curs_set(FALSE);
   delwin(f_win);
   delwin(filter_window);
   doupdate();

   return 1;
}

#endif   // HAVE_FORM


#endif

/* EOF */
