/*
 * Copyright (C) 2000-2001 Peter J Jones (pjones@pmade.org)
 * All Rights Reserved
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * Class for working with subcommands
 */

#ifndef __Command_hh__
#define __Command_hh__

// Standard Includes
#include <vector>
#include <string>
#include <cctype>

// Local Includes
#include "Option.hh"

// namespace stuff
using namespace std;

//# class Command
/** Class for working with subcommands

	This class contains code for working with subcommands.
	Subcommands are when a application can perform various
	tasks, and each task needs it's own command line options.
**/
class Command {
  public:
	  /// ctor
	  Command() : 
		  m_have_short_options(false),
		  m_have_long_options(false)
		  {}
	  /// get the name of this command
	  string get_name() const
		{ return m_name; }
	  /// set the name of this subcommand
	  void set_name(string name)
		{ m_name = name; }
	  /// get the list of aliases
	  vector<string> &get_aliases()
		{ return m_aliases; }
	  /// find out if this command has long options
	  bool has_long_options()
		{ return m_have_long_options; }
	  /// find out if this command has short options
	  bool has_short_options()
		{ return m_have_short_options; }
	  /// add an alias for this command
	  void add_alias(string alias)
		{ m_aliases.push_back(alias); }
	  /// add a option to this subcommand
	  void add_option(Option &option);
	  /// get the vector of options
	  vector<Option> &get_options()
		{ return m_options; }

	  string get_varname();
	  /// set the varname
	  void set_varname(string &varname)
		{ m_varname = varname; }

	  string get_description();
	  /// set the description
	  void set_description(string description)
		{ m_description = description; }

	  /// struct for comparing commands
	  struct Cmp {
		  bool operator()(Command const &c1, Command const &c2) {
			  return c1.get_name() < c2.get_name();
		  }
	  };
  private:
	  vector<Option>	m_options;
	  vector<string>	m_aliases;
	  string			m_description;
	  string			m_name;
	  string			m_varname;
	  bool				m_have_short_options;
	  bool				m_have_long_options;
};

#endif
