/***************************************************************************
    plugin_katexmltools.cpp
    $Id: plugin_katexmltools.cpp,v 1.5 2002/02/26 22:00:27 jowenn Exp $
    List elements, attributes, attribute values and entities allowed by DTD.
    Needs a DTD in XML format, as produced by dtdparse.
    Based on katehtmltools.
    copyright            : (C) 2001,2002 by Daniel Naber
    email                : daniel.naber@t-online.de
 ***************************************************************************/

/***************************************************************************
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 ***************************************************************************/

/*
  FIXME: 

 -Correctly support more than one view
 -Don't use selection(), i.e. make it faster
 -See the "fixme"'s in the code

  TODO:
   
 -Currently only one DTD can be used for all open documents
 -Check the doctype to load the meta DTD (use Qt 3.0 RegExp)
 -Support for more than one namespace at the same time (e.g. XSLT + XSL-FO)?
  =>This could also be handled in the XSLT DTD fragment, as described in the XSLT 1.0 spec,
  but then at <xsl:template match="/"><html> it will only show you HTML elements!
 
 -Option to insert empty element in <empty/> form
 -Show expanded entities with QChar::QChar(int rc) + unicode font
 -Can we (in a reliable way) suggest elements for an completely emtpy document?
 -Check the language, e.g. "there are no possible sub-elements"
 -Behave as expected when text is marked (e.g. build tags around marked text?)
 -Don't ignore entities defined in the document's prologue

 -Maybe only read the meta DTD file once, then store the resulting QMap on disk (using QDataStream)?
  We'll then have to compare time_of_cache_file <-> time_of_meta_dtd.
 
 -Try to use libxml
 -New function?: build example file for DTD (in new file)
 -New function?: "Check well-formedness"? validation is not possible with Qt 2.x and 3.0beta5,
  but well-formedness also cannot be checked automatically, at least useful errors seems
  to be given
*/

#include "plugin_katexmltools.h"
#include "plugin_katexmltools.moc"

#include <qdatetime.h>
#include <qdom.h>
#include <qfile.h>
#include <qinputdialog.h>
#include <qlayout.h>
#include <qlistbox.h>
#include <qprogressdialog.h>
#include <qpushbutton.h>
#include <qregexp.h>
#include <qstring.h>

#include <kaction.h>
#include <kbuttonbox.h>
#include <kcursor.h>
#include <kdebug.h>
#include <kfiledialog.h>
#include <kglobal.h>
#include <kinstance.h>
#include <kio/job.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kstddirs.h>

extern "C"
{
    void* init_katexmltoolsplugin()
    {
        KGlobal::locale()->insertCatalogue("katexmltools"); 
        return new KatePluginFactory;
    }
}

KatePluginFactory::KatePluginFactory()
{
    s_instance = new KInstance("kate");
}

KatePluginFactory::~KatePluginFactory()
{
    delete s_instance;
}

QObject* KatePluginFactory::createObject( QObject* parent, const char* name, const char*, const QStringList & )
{
    return new PluginKateXMLTools( parent, name );
}

KInstance* KatePluginFactory::s_instance = 0L;

PluginKateXMLTools::PluginKateXMLTools(QObject* parent, const char* name)
    : Kate::Plugin (parent, name)
{
    //kdDebug() << "PluginKateXMLTools constructor called" << endl;
    
    m_ready = false;        // are we ready to give advice? If not, let the user load meta DTD first
    m_dtd_string = QString();
    m_url_string = QString(); 

    // "SGML support" only means case-insensivity, because HTML is case-insensitive up to version 4:
    m_sgml_support = true;      // TODO: make this an run-time option (maybe automatically set)

    m_max_check = 500;           // maximum cursor key movements (fixme: find a better solution)
    m_max_check_elements = 100;  // maximum elements to check to find parent element (fixme: find a better solution)

    // FIXME: initialize m_elements_list, m_attributes_list and m_attributevalues_list, but 
    // it works without?!?!
}

PluginKateXMLTools::~PluginKateXMLTools()
{
    //kdDebug() << "xml tools descructor 1..." << endl;
}

Kate::PluginView *PluginKateXMLTools::createView(Kate::MainWindow *win)
{
    // TODO: doesn't this have to be deleted?
    Kate::PluginView *view = new Kate::PluginView (this, win);
    (void) new KAction ( i18n("Insert XML..."), CTRL + Key_Return, this, 
        SLOT(slotGetInput()), view->actionCollection(), "xml_tool" );
    (void) new KAction ( i18n("Insert Entity..."), Key_F10, this,
        SLOT(slotGetEntity()), view->actionCollection(), "xml_tool_entity" );
    (void) new KAction ( i18n("Close Element"), Key_F11, this,
        SLOT(slotCloseElement()), view->actionCollection(), "xml_tool_close_element" );
    (void) new KAction ( i18n("Assign Meta DTD..."), 0, this,
        SLOT(getDTD()), view->actionCollection(), "xml_tool_assign" );
    view->setXML("plugins/katexmltools/ui.rc");
    return view;
}

/** Load the meta DTD. In case of success return tur and set the 'ready'
  * flag to true, to show that this plugin is ready to give hints about the DTD.
  */
void PluginKateXMLTools::getDTD()
{

    // testing the event filter:
    //Kate::View *kv = myApp->getViewManager()->getActiveView();
    //kv->installEventFilter( this );   // gets only the special keys like ESC, Backspace, ...
    //m_pluginview->installEventFilter( this );

    /* TODO:
    // Guess the meta DTD by looking and the doctype's public identifier:
    Kate::View *kv = myApp->getViewManager()->getActiveView();
    if( kv ) {
        uint line, col;
        kv->cursorPositionReal(&line, &col);
        kv->setCursorPositionReal(0, 0);
        uint count = 0;
        // check the start of the document for a doctype
        // TODO: remove comments
        do {
            kv->shiftCursorRight();
            count++;
        } while( count < 1000 );
        QString doc_start = kv->getDoc()->selection();
        int length;
        QRegExp re("<!DOCTYPE\\s+.*\\s+PUBLIC\\s+\"", false);
        int match = re.match(doc_start, 0, &length);
        kdDebug() << "match: " << match << ", length: " << length << endl;
        kv->setCursorPositionReal(line, col);
    }
    */

    // Load the Meta DTD
        
    // Start where the supplied XML-DTDs are installed by default:
    if( m_url_string.isNull() ) {
        m_url_string = KGlobal::dirs()->findResourceDir("data", "katexmltools/");
        m_url_string = m_url_string + "katexmltools/";
    }
    
    KURL url = KFileDialog::getOpenURL(m_url_string, "*.xml",
        0, i18n("Assign Meta DTD in XML format"));
    if( url.isEmpty() ) {
        return;
    }
    m_url_string = url.url();    // remember directory for next time
    m_dtd_string = "";

    // TODO: loading+parsing takes 1 second on a fast computer for the KDE docbook DTD,
    // so show waiting mouse pointer
    KIO::Job *job = KIO::get(url);
    connect(job, SIGNAL(result(KIO::Job *)), this, SLOT(slotFinished(KIO::Job *)));
    connect(job, SIGNAL(data(KIO::Job *, const QByteArray &)), this, SLOT(slotData(KIO::Job *, const QByteArray &)));
}

void PluginKateXMLTools::slotFinished(KIO::Job *job)
{
    if( job->error() ) {
        //kdDebug() << "XML Plugin error: DTD in XML format (" << filename << ") could not be loaded" << endl;
        job->showErrorDialog(0);
    } else if ( static_cast<KIO::TransferJob *>(job)->isErrorPage() ) {
        // catch failed loading loading via http:
        KMessageBox::error(0, i18n("The file '%1' could not be opened. "
            "The server returned an error.").arg(m_url_string),
            i18n("XML Plugin Error"));
    } else {
        analyzeDTD();
    }
}

void PluginKateXMLTools::slotData(KIO::Job *, const QByteArray &data)
{
    m_dtd_string += QString(data);
}

void PluginKateXMLTools::analyzeDTD()
{
    QApplication::setOverrideCursor(KCursor::waitCursor());

    QDomDocument doc("dtd_in_xml");
    if ( !doc.setContent( m_dtd_string ) || doc.doctype().name() != "dtd" ) {
        //kdDebug() << "XML Plugin error: DTD in XML format (" << filename << ") could not be parsed" << endl;
        QApplication::restoreOverrideCursor();
        KMessageBox::error(0, i18n("The file '%1' could not be parsed. "
            "Please check that the file is a valid XML file of this type:\n"
            "-//Norman Walsh//DTD DTDParse V2.0//EN\n"
            "You can produce such files with dtdparse.").arg(m_url_string),
            i18n("XML Plugin Error"));
        return;
    }

    uint list_length = 0;
    list_length += doc.elementsByTagName("entity").count();
    list_length += doc.elementsByTagName("element").count();
    // count this twice, as it will be iterated twice (TODO: optimize that?):
    list_length += doc.elementsByTagName("attlist").count() * 2;

    QProgressDialog progress( i18n("Analyzing meta DTD..."), i18n("Cancel"), list_length,
                                0, "progress", TRUE );
    progress.setMinimumDuration(400);
    progress.setProgress(0);
    
    // Get information from meta DTD and put it in Qt data structures:
    if( ! getEntities(&doc, &progress) ) {
        QApplication::restoreOverrideCursor();
        return;
    }
    if( ! getAllowedElements(&doc, &progress) ) {
        QApplication::restoreOverrideCursor();
        return;
    }
    if( ! getAllowedAttributes(&doc, &progress) ) {
        QApplication::restoreOverrideCursor();
        return;
    }
    if( ! getAllowedAttributeValues(&doc, &progress) ) {
        QApplication::restoreOverrideCursor();
        return;
    }

    progress.setProgress(list_length);    // just to make sure the dialog disappears

    m_ready = true;
    QApplication::restoreOverrideCursor();
}

/** Insert a closing tag for the nearest not-closed parent element.
  */
void PluginKateXMLTools::slotCloseElement()
{
    Kate::View *kv=myApp->getViewManager()->getActiveView();
    if( ! kv ) {
        return;
    }
    QString parent_element = getParentElement(*kv);
    if( ! parent_element.isEmpty() ) {
        kv->insertText("</" + parent_element + ">");
    }
}

void PluginKateXMLTools::slotGetEntity()
{
    slotGetInput(entities);
}

void PluginKateXMLTools::slotGetInput()
{
    slotGetInput(none);
}

/** ...
  */
void PluginKateXMLTools::slotGetInput(Mode mode)
{
    Kate::View *kv=myApp->getViewManager()->getActiveView();
    if( ! kv ) {
        return;
    }

    if( ! m_ready ) {
        int ret = KMessageBox::questionYesNo(0,
        i18n("You need to assign a meta DTD to the current file before you can "
            "use the XML plugin. Several common meta DTDs are part of the XML plugin. "
            "You can also produce your own meta DTDs with a program called dtdparse. "
            "Do you want to assign a meta DTD now?"),
            i18n("Assign Meta DTD"));
        if( ret == KMessageBox::Yes ) {
            getDTD();
            if( ! m_ready ) {        // FIXME: ready will always be false b/c of the slot stuff in getDTD()
                return;
            }
        } else {
            return;
        }
    }
    
    QString name;
    QString instruction;
    QStringList allowed = QStringList();

    // get char on the left of the cursor:
    kv->shiftCursorLeft();
    QString left_ch = kv->getDoc()->selection();
    kv->shiftCursorRight();

    if( left_ch == "&" || mode == entities ) {
        allowed = getEntitiesFast("");
        instruction = i18n("Select an entity:");
        mode = entities;
    } else if( insideTag(*kv) != "" ) {
        if( insideAttribute(*kv) != "" ) {
            QString current_element = insideTag(*kv);     // TODO: make more elegant
            QString current_attribute = insideAttribute(*kv);     // TODO: make more elegant
            allowed = getAllowedAttributeValuesFast(current_element, current_attribute);
            instruction = i18n("Select an attribute value for %1/%2:").arg(current_element).arg(current_attribute);
            if( allowed.count() == 0 ) {
                allowed.append(i18n("(no predefined values available)"));
                mode = none;
            } if( allowed.count() == 1 && 
                (allowed[0] == "CDATA" || allowed[0] == "ID" || allowed[0] == "IDREF" ||
                allowed[0] == "IDREFS" || allowed[0] == "ENTITY" || allowed[0] == "ENTITIES" ||
                allowed[0] == "NMTOKEN" || allowed[0] == "NMTOKENS") ) {
                // these must not be taken literally, e.g. don't insert the string "CDATA"
                allowed[0] = i18n("(allowed type: %1)").arg(allowed[0]);
                mode = none;
            } else if( allowed.count() == 1 && allowed[0] == "__NONE" ) {
                // TODO: implement this in a better way
                allowed[0]  = i18n("(unknown element '%1' or attribute '%2')").arg(current_element).arg(current_attribute);
                mode = none;
            } else {
                mode = attributevalues;
            }
        } else {
            QString current_element = insideTag(*kv);     // TODO: make more elegant
            allowed = getAllowedAttributesFast(current_element);
            instruction = i18n("Select an attribute for &lt;%1&gt;:").arg(current_element);
            if( allowed.count() == 0 ) {
                allowed.append(i18n("(no attributes available)"));
                mode = none;
            } else if( allowed.count() == 1 && allowed[0] == "__NONE" ) {
                // TODO: implement this in a better way
                allowed[0] = i18n("(unknown element '%1')").arg(current_element);
                mode = none;
            } else {
                mode = attributes;
            }
        }
    } else {
        QString parent_element = getParentElement(*kv);
        allowed = getAllowedElementsFast(parent_element);
        instruction = i18n("Select a sub-element for &lt;%1&gt;:").arg(parent_element);
        if( allowed.count() == 0 ) {
            allowed.append(i18n("(there are no possible sub-elements)"));
            mode = none;
        } else if( allowed.count() == 1 && allowed[0] == "__NONE" ) {
            // TODO: implement this in a better way
            allowed[0] = i18n("(unknown parent element '%1')").arg(parent_element);
            mode = none;
        } else {
            mode = elements;
        }
    }

    QString text = KatePrompt(i18n("XML Plugin"), instruction, allowed,
        (QWidget *)myApp->getViewManager()->getActiveView(), kv);

    if( !text.isEmpty() && mode != none && !text.startsWith("(allowed") ) {
        if( mode == entities ) {
            QString entity = "";
            if( left_ch.at(0) != '&' ) {
                entity = "&";
            }
            entity += text + ";";
            kv->insertText(entity);
        } else if( mode == attributevalues ) {
            // Remove what's currently in the quotes:
            // find left quote:
            QString ch;
            uint line, col;
            uint backspace_count = 0;
            do {    
                kv->shiftCursorLeft();
                ch = kv->getDoc()->selection();
                kv->cursorPositionReal(&line, &col);
            } while( !isQuote(ch.left(1)) && (line > 0 || col > 0) );
            kv->cursorRight();    // don't mark anything
            // find right quote:
            uint move_right_line = line;
            do {
                kv->shiftCursorRight();
                ch = kv->getDoc()->selection();
                kv->cursorPositionReal(&line, &col);
                if( move_right_line != line ) {
                    // quote not yet closed (well, not on this line)
                    backspace_count++;
                    break;
                }
                move_right_line = line;
                backspace_count++;
            } while( !isQuote(ch.right(1)) && (line > 0 || col > 0) && backspace_count < m_max_check );
            // fixme: find a better solution for situation "quote not closed and near end of document":
            if( backspace_count < (m_max_check - 1) ) {
                backspace_count--;
                kv->cursorLeft();
                while( backspace_count > 0 ) {
                    kv->backspace();
                    backspace_count--;
                }
            }
            // replace by new attribute value:
            kv->insertText(text);
        } else if( mode == attributes ) {
            // add space if inserted directly after element name:
            QString attr = "";
            if( !left_ch.at(0).isSpace() ) {
                attr = " ";
            }
            attr += text + "=\"\"";
            // add space if inserted directly before other attribute:
            kv->shiftCursorRight();
            QString right_ch = kv->getDoc()->selection();
            kv->cursorLeft();
            if( !right_ch.at(0).isSpace() && right_ch.at(0) != '/' && right_ch.at(0) != '>') {
                attr += " ";
            }
            kv->insertText(attr);
            // place cursor inside attribute value:
            kv->cursorLeft();
        } else if( mode == elements ) {
            // note: some elements can be empty, but don't have to (e.g. xsl:apply-templates)
            QString tags = "";
            tags += "<" + text + ">";
            tags += "</" + text + ">";
            kv->insertText(tags);
            // place cursor in between elements:
            uint place_count = 0;
            while( place_count < (text.length()+3) ) {
                kv->cursorLeft();
                place_count++;
            }
      }
  }

}

/** Show a modal dialog with a list, an OK and a Cancel button.
  */
QString PluginKateXMLTools::KatePrompt(QString strTitle, QString strPrompt, 
    QStringList list, QWidget *widget, Kate::View * /* kv */)
{
    /* testing the code completion stuff: */
    /*
    QValueList<KTextEditor::CompletionEntry> comp_list;
    KTextEditor::CompletionEntry entry;
    QStringList::Iterator it;
    for( it = list.begin(); it != list.end(); ++it ) {
        entry.text = (*it);
        entry.comment = "blah";
        comp_list << entry;
    }
    kv->showCompletionBox(comp_list, 0);
    return "";
    */
   
    SelectDialog dialog(widget, strTitle, strPrompt, list);
    dialog.exec();
    QString text = dialog.selection();
    if( text.isNull() ) {
        text = "";
    }
    return text;
}

/** Check if cursor is inside a tag, that is
  * if "<" occurs before ">" occurs (on the left side of the cursor).
  * Return the tag name, return "" if we cursor is outside a tag.
  */
QString PluginKateXMLTools::insideTag(Kate::View &kv)
{
    uint line = 0, col = 0;
    kv.cursorPositionReal(&line, &col);
    int y = line;    // another variable because uint <-> int
    do {
        QString line_str = kv.getDoc()->textLine(y);
        for( uint x = col; x > 0; x-- ) {
            QString ch = line_str.mid(x-1, 1);
            if( ch == ">" ) {   // cursor is outside tag
                return "";
            } else if( ch == "<" ) {
                QString tag;
                // look for white space on the right to get the tag name
                for( uint z = x; z <= line_str.length() ; z++ ) {
                    ch = line_str.mid(z-1, 1);
                    if( ch.at(0).isSpace() || ch == "/" || ch == ">" ) {
                        return tag.right(tag.length()-1);
                    } else if( z == line_str.length() ) {   // end of line
                        tag += ch;
                        return tag.right(tag.length()-1);
                    } else {
                        tag += ch;
                    }
                }
            }
        }
        y--;
        col = kv.getDoc()->textLine(y).length();
    } while( y >= 0 );

    return "";
}

/** Check if cursor is inside an attribute value, that is
  * if '="' is on the left, and if it's nearer than "<" or ">".
  * Return the attribute name or "" if we're outside an attribute 
  * value.
  * Note: only call when insideTag() == true.
  * TODO: allow whitespace around "="
  */
QString PluginKateXMLTools::insideAttribute(Kate::View &kv)
{
    uint line = 0, col = 0;
    kv.cursorPositionReal(&line, &col);
    int y = line;    // another variable because uint <-> int
    uint x = 0;
    QString line_str = "";
    QString ch = "";
    do {
        line_str = kv.getDoc()->textLine(y);
        for( x = col; x > 0; x-- ) {
            ch = line_str.mid(x-1, 1);
            QString ch_left = line_str.mid(x-2, 1);
            // TODO: allow whitespace
            if( isQuote(ch) && ch_left == "=" ) {
                break;
            } else if( isQuote(ch) && ch_left != "=" ) {
                return "";
            } else if( ch == "<" || ch == ">" ) {
                return "";
            }
        }
        y--;
        col = kv.getDoc()->textLine(y).length();
    } while( !isQuote(ch) );

    // look for next white space on the left to get the tag name
    QString attr = "";
    for( int z = x; z >= 0; z-- ) {
        ch = line_str.mid(z-1, 1);
        if( ch.at(0).isSpace() ) {
            break;
        } else if( z == 0 ) {   // start of line == whitespace
            attr += ch;
            break;
        } else {
            attr = ch + attr;
        }
    }

    return attr.left(attr.length()-2);
}

/** Find the parent element for the current cursor position. That is,
  * go left and find the first opening element that's not closed yet,
  * ignoring empty elements.
  * Examples: If cursor is at "X", the correct parent element is "p":
  * <p> <a x="xyz"> foo <i> test </i> bar </a> X
  * <p> <a x="xyz"> foo bar </a> X
  * <p> foo <img/> bar X
  * <p> foo bar X
  * WARNING (TODO): behaviour undefined if called when cursor is
  * /inside/ an element!!
  */
QString PluginKateXMLTools::getParentElement(Kate::View &kv)
{

    int nesting_level = 1;

    bool in_tag = false;
    QString tag = "";

    uint line = 0, col = 0;
    kv.cursorPositionReal(&line, &col);
    int y = line;    // another variable because uint <-> int
    do {
        QString line_str = kv.getDoc()->textLine(y);
        for( int x = col; x > 0; x-- ) {
            QString ch = line_str.mid(x-1, 1);
            if( ch == ">" ) {
                in_tag = true;
                tag = "";
            } else if( ch == "<" ) {
                in_tag = false;
                if( isOpeningTag("<"+tag+">") ) {
                    nesting_level--;
                } else if( isClosingTag("<"+tag+">") ) {
                    nesting_level++;
                }
                if( nesting_level <= 0 ) {
                    // cut of everything after the element name:
                    uint elem_count = 0;
                    while( ! tag.at(elem_count).isSpace() && elem_count < tag.length() ) {
                        elem_count++;
                    }
                    QString element = tag.left(elem_count);
                    return element;
                }
                // "else": empty tags can be ignored
            } else if( in_tag == true ) {
                tag = ch + tag;        // we're moving left!
            }
        }
        y--;
        col = kv.getDoc()->textLine(y).length();
    } while( y >= 0 );
    // TODO: limit line_count? show waiting mouse pointer?!

    return QString();
}

/** Return true if the tag is neither a closing tag
  * nor an empty tag, nor a comment, nor processing instruction.
  */
bool PluginKateXMLTools::isOpeningTag(QString tag)
{
    if( !isClosingTag(tag) && !isEmptyTag(tag) && 
        !tag.startsWith("<?") && !tag.startsWith("<!") ) {
        return true;
    } else {
        return false;
    }
}

/** Return true if the tag is a closing tag. Return false
  * if the tag is an opening tag or an empty tag (!)
  */
bool PluginKateXMLTools::isClosingTag(QString tag)
{
    if( tag.startsWith("</") ) {
        return true;
    } else {
        return false;
    }
}

bool PluginKateXMLTools::isEmptyTag(QString tag)
{
    if( tag.right(2) == "/>" ) {
        return true;
    } else {
        return false;
    }
}

/** Return true if ch is a single or double quote
  */
bool PluginKateXMLTools::isQuote(QString ch)
{
    if( ch == "\"" || ch == "'" ) {
        return true;
    } else {
        return false;
    }
}

// ========================================================================
// DOM stuff:

/** Iterate through the XML to get a mapping which sub-elements are allowed for
  * all elements.
  */
bool PluginKateXMLTools::getAllowedElements(QDomDocument *doc, QProgressDialog *progress)
{

    m_elements_list.clear();
    // We only display a list, i.e. we pretend that the content model is just
    // a set, so we use a map. This is necessay e.g. for xhtml 1.0's head element, 
    // which would otherwise display some elements twice.
    QMap<QString,bool> subelement_list;    // the bool is not used
    
    QDomNodeList list = doc->elementsByTagName("element");
    uint list_length = list.count();    // speedup (really!)

    for( uint i = 0; i < list_length; i++ ) {
        if( progress->wasCancelled() ) {
            return false;
        }
        progress->setProgress(progress->progress()+1);
        qApp->processEvents();

        subelement_list.clear();
        QDomNode node = list.item(i);
        QDomElement elem = node.toElement();
        
        if( !elem.isNull() ) {

            // Enter the expanded content model, which may also include stuff not allowed.
            // We do not care if it's a <sequence-group> or whatever.
            QDomNodeList content_model_list = elem.elementsByTagName("content-model-expanded");
            QDomNode content_model_node = content_model_list.item(0);
            QDomElement content_model_elem = content_model_node.toElement();
            if( ! content_model_elem.isNull() ) {
                // check for <pcdata/>:
                QDomNodeList pcdata_list = content_model_elem.elementsByTagName("pcdata");
                if( pcdata_list.count() > 0 ) {
                    subelement_list[i18n("(allowed type: PCDATA)")] = true;
                }
                // check for other sub elements:
                QDomNodeList sub_list = content_model_elem.elementsByTagName("element-name");
                uint sub_list_length = sub_list.count();
                for( uint l = 0; l < sub_list_length; l++ ) {
                    QDomNode sub_node = sub_list.item(l);
                    QDomElement sub_elem = sub_node.toElement();
                    if( !sub_elem.isNull() ) {
                        subelement_list[sub_elem.attribute("name")] = true;
                    }
                }
            }

            // Now remove the elements not allowed (e.g. <a> is explicitely not allowed in <a> 
            // in the HTML 4.01 Strict DTD):
            QDomNodeList exclusions_list = elem.elementsByTagName("exclusions");
            if( exclusions_list.length() > 0 ) {    // sometimes there are no exclusions (e.g. in XML DTDs there are never exclusions)
                QDomNode exclusions_node = exclusions_list.item(0);
                QDomElement exclusions_elem = exclusions_node.toElement();
                if( ! exclusions_elem.isNull() ) {
                    QDomNodeList sub_list = exclusions_elem.elementsByTagName("element-name");
                    uint sub_list_length = sub_list.count();
                    for( uint l = 0; l < sub_list_length; l++ ) {
                        QDomNode sub_node = sub_list.item(l);
                        QDomElement sub_elem = sub_node.toElement();
                        if( !sub_elem.isNull() ) {
                            QMap<QString,bool>::Iterator it = subelement_list.find(sub_elem.attribute("name"));
                            if( it != subelement_list.end() ) {
                                subelement_list.remove(it);
                            }
                        }
                    }
                }
            }

            // turn the map into a list:
            QStringList subelement_list_tmp;
            QMap<QString,bool>::Iterator it;
            for( it = subelement_list.begin(); it != subelement_list.end(); ++it ) {
                subelement_list_tmp.append(it.key());
            }
            m_elements_list.insert(elem.attribute("name"), subelement_list_tmp);

        }
        
    } // end iteration over all <element> nodes
    return true;
}

/** Check which elements are allowed inside a parent element. This returns
  * a list of allowed elements, but it doesn't care about order or if only a certain
  * number of occurences is allowed.
  */
QStringList PluginKateXMLTools::getAllowedElementsFast(QString parent_element)
{
    if( m_sgml_support ) {
        // find the matching element, ignoring case:
        QMap<QString,QStringList>::Iterator it;
        for( it = m_elements_list.begin(); it != m_elements_list.end(); ++it ) {
            if( it.key().lower() == parent_element.lower() ) {
                return it.data();
            }
        }
    } else {
        if( m_elements_list.contains(parent_element) ) {
            return m_elements_list[parent_element];
        }
    }
    return QStringList("__NONE");
}

/** Iterate through the XML to get a mapping which attributes are allowed inside 
  * all elements.
  */
bool PluginKateXMLTools::getAllowedAttributes(QDomDocument *doc, QProgressDialog *progress)
{
    m_attributes_list.clear();
    QStringList allowed_attributes;
    QDomNodeList list = doc->elementsByTagName("attlist");
    uint list_length = list.count();

    for( uint i = 0; i < list_length; i++ ) {
        if( progress->wasCancelled() ) {
            return false;
        }
        progress->setProgress(progress->progress()+1);
        qApp->processEvents();
        allowed_attributes.clear();
        QDomNode node = list.item(i);
        QDomElement elem = node.toElement();
        if( !elem.isNull() ) {
            // Enter the list of <attribute>:
            QDomNodeList attribute_list = elem.elementsByTagName("attribute");
            uint attribute_list_length = attribute_list.count();
            for( uint l = 0; l < attribute_list_length; l++ ) {
                QDomNode attribute_node = attribute_list.item(l);
                QDomElement attribute_elem = attribute_node.toElement();
                if( ! attribute_elem.isNull() ) {
                    allowed_attributes.append(attribute_elem.attribute("name"));
                }
            }
            m_attributes_list.insert(elem.attribute("name"), allowed_attributes);
        }
    }
    return true;
}

/** Check which attributes are allowed for an element.
  */
QStringList PluginKateXMLTools::getAllowedAttributesFast(QString element)
{
    if( m_sgml_support ) {
        // find the matching element, ignoring case:
        QMap<QString,QStringList>::Iterator it;
        for( it = m_attributes_list.begin(); it != m_attributes_list.end(); ++it ) {
            if( it.key().lower() == element.lower() ) {
                return it.data();
            }
        }
    } else {
        if( m_attributes_list.contains(element) ) {
            return m_attributes_list[element];
        }
    }
    return QStringList("__NONE");
}

/** Iterate through the XML to get a mapping which attribute values are allowed
  * for all attributes inside all elements.
  */
bool PluginKateXMLTools::getAllowedAttributeValues(QDomDocument *doc, QProgressDialog *progress)
{
    m_attributevalues_list.clear();                        // 1 element : n possible attributes
    QMap<QString,QStringList> attributevalues_tmp;        // 1 attribute : n possible values
    QDomNodeList list = doc->elementsByTagName("attlist");
    uint list_length = list.count();

    for( uint i = 0; i < list_length; i++ ) {
        if( progress->wasCancelled() ) {
            return false;
        }
        progress->setProgress(progress->progress()+1);
        qApp->processEvents();
        
        attributevalues_tmp.clear();
        QDomNode node = list.item(i);
        QDomElement elem = node.toElement();
        if( !elem.isNull() ) {
            // Enter the list of <attribute>:
            QDomNodeList attribute_list = elem.elementsByTagName("attribute");
            uint attribute_list_length = attribute_list.count();
            for( uint l = 0; l < attribute_list_length; l++ ) {
                QDomNode attribute_node = attribute_list.item(l);
                QDomElement attribute_elem = attribute_node.toElement();
                if( ! attribute_elem.isNull() ) {
                    QString value = attribute_elem.attribute("value");
                    attributevalues_tmp.insert(attribute_elem.attribute("name"), QStringList::split(QRegExp(" "), value));
                }
            }
            m_attributevalues_list.insert(elem.attribute("name"), attributevalues_tmp);
        }
    }
    return true;
}

/** Check which attributes values are allowed for an attribute in an element
  * (the element is necessary because e.g. "href" inside <a> could be different
  * to an "href" inside <link>):
  */
QStringList PluginKateXMLTools::getAllowedAttributeValuesFast(QString element, QString attribute)
{
    // Direct access would be faster than iteration of course but not always correct, 
    // because we need to be case-insensitive.
    if( m_sgml_support ) {
        // first find the matching element, ignoring case:
        QMap< QString,QMap<QString,QStringList> >::Iterator it;
        for( it = m_attributevalues_list.begin(); it != m_attributevalues_list.end(); ++it ) {
            if( it.key().lower() == element.lower() ) {
                QMap<QString,QStringList> attr_vals = it.data();
                QMap<QString,QStringList>::Iterator it_v;
                // then find the matching attribute for that element, ignoring case:
                for( it_v = attr_vals.begin(); it_v != attr_vals.end(); ++it_v ) {
                    if( it_v.key().lower() == attribute.lower() ) {
                        return(it_v.data());
                    }
                }
            }
        }
    } else {
        if( m_attributevalues_list.contains(element) ) {
            QMap<QString,QStringList> attr_vals = m_attributevalues_list[element];
            if( attr_vals.contains(attribute) ) {
                return attr_vals[attribute];
            }
        }
    }
    // no predefined values available:
    return QStringList("__NONE");
}

/** Iterate through the XML to get a mapping of all entity names and their expanded 
  * version, e.g. nbsp => &#160;. Parameter entities are ignored.
  */
bool PluginKateXMLTools::getEntities(QDomDocument *doc, QProgressDialog *progress)
{
    m_entity_list.clear();
    QDomNodeList list = doc->elementsByTagName("entity");
    uint list_length = list.count();
    
    for( uint i = 0; i < list_length; i++ ) {
        if( progress->wasCancelled() ) {
            return false;
        }
        progress->setProgress(progress->progress()+1);
        qApp->processEvents();
        QDomNode node = list.item(i);
        QDomElement elem = node.toElement();
        if( !elem.isNull() 
            && elem.attribute("type") != "param" ) { // TODO: what's cdata <-> gen ?
            QDomNodeList expanded_list = elem.elementsByTagName("text-expanded");
            QDomNode expanded_node = expanded_list.item(0);
            QDomElement expanded_elem = expanded_node.toElement();
            if( ! expanded_elem.isNull() ) {
                QString exp = expanded_elem.text();
                // TODO: support more than one &#...; in the expanded text
                /* TODO include do this when the unicode font problem is solved:
                if( exp.contains(QRegExp("^&#x[a-zA-Z0-9]+;$")) ) {
                    // hexadecimal numbers, e.g. "&#x236;"
                    uint end = exp.find(";");
                    exp = exp.mid(3, end-3);
                    exp = QChar();
                } else if( exp.contains(QRegExp("^&#[0-9]+;$")) ) {
                    // decimal numbers, e.g. "&#236;"
                    uint end = exp.find(";");
                    exp = exp.mid(2, end-2);
                    exp = QChar(exp.toInt());
                }
                */
                m_entity_list.insert(elem.attribute("name"), exp);
            } else {
                m_entity_list.insert(elem.attribute("name"), QString());
            }
        }
    }
    return true;
}

/** Get a list of all (non-parameter) entities that start with a certain string.
  */
QStringList PluginKateXMLTools::getEntitiesFast(QString start)
{
    QStringList entities;
    QMap<QString,QString>::Iterator it;
    for( it = m_entity_list.begin(); it != m_entity_list.end(); ++it ) {
        if( (*it).startsWith(start) ) {
            QString str = it.key();
            /* TODO 
            if( !it.data().isEmpty() ) {
                //str += " -- " + it.data();
                QRegExp re("&#(\\d+);");
                if( re.search(it.data()) != -1 ) {
                    uint ch = re.cap(1).toUInt();
                    str += " -- " + QChar(ch).decomposition();
                }
                //kdDebug() << "####" << it.data() << endl;
            }
            */
            entities.append(str);
            // TODO: later use a table view
        }
    }
    return entities;
}

// ========================================================================
// Dialog stuff:

/** The dialog with a list and an OK and a Cancel button.
  */
SelectDialog::SelectDialog(QWidget *parent, QString caption, QString prompt, QStringList list) :
  KDialog(parent, 0, TRUE)
{
    QPushButton *btnCancel, *btnOk;
    QBoxLayout *box = new QVBoxLayout(this, 2, 0);

    setCaption(caption);

    QLabel *instruction = new QLabel(this);
    instruction->setText(prompt);
    box->addWidget(instruction);

    m_Selection = new QString();
    m_ListBox = new QListBox(this);
    // Try to set a full unicode font if available, to show the expanded entities:
    /*
    // TODO: This doesn't work, even with "Arial Unicode MS" installed. Either it's not
    // recognized as a unicode font or the other fonts are recognized as unicode fonts
    // and are preferred. Using QFont("Arial Unicode MS") directly works.
    QFont font = QFont();
    font.setCharSet(QFont::Unicode);
    m_ListBox->setFont(font);
    kdDebug() << "####" << font.charSet() << endl;
    */
    box->addWidget(m_ListBox, 100);

    connect(m_ListBox, SIGNAL(selected(int)), this, SLOT(slotSelect()));

    KButtonBox *butbox = new KButtonBox(this);
    btnOk = butbox->addButton(i18n("OK"));
    btnOk->setDefault(TRUE);
    connect(btnOk, SIGNAL(clicked()), this, SLOT(slotSelect()));

    btnCancel = butbox->addButton(i18n("Cancel"));
    connect(btnCancel, SIGNAL(clicked()), this, SLOT(slotCancel()));
    butbox->layout();
    box->addWidget(butbox);

    // Sort list case-insensitive. This looks strange but using a QMap
    // is even suggested by the Qt documentation.
    QMap<QString,QString> map_list;
    for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it ) {
        QString str = *it;
        if( map_list.contains(str.lower()) ) {
            // do not override a previous value, e.g. "Auml" and "auml" are two different
            // entities, but they should be sorted next to each other.
            // TODO: currently it's undefined if e.g. "A" or "a" comes first, it depends on
            // the meta DTD (really? it seems to work okay?!?)
            map_list[str.lower()+"_"] = str;
        } else {
            map_list[str.lower()] = str;
        }
    }
    list.clear();
    QMap<QString,QString>::Iterator it;
    // Qt doc: "the items are alphabetically sorted [by key] when iterating over the map":
    for( it = map_list.begin(); it != map_list.end(); ++it ) {
        list.append(it.data());
    }
    m_ListBox->insertStringList(list);

    // TODO?: better pre-select (not always just the first):
    m_ListBox->setSelected(0, TRUE);
    m_ListBox->setFocus();

    // TODO: use previous width/height and x/y position
    resize(170, 300);
    box->activate();
}

SelectDialog::~SelectDialog()
{
    if( m_Selection ) {
        delete m_Selection;
    }
}

QString SelectDialog::selection()
{
    return *m_Selection;
}

void SelectDialog::slotSelect()
{
    *m_Selection = m_ListBox->currentText();
    accept();
}

void SelectDialog::slotCancel()
{
    reject();
}
