/***************************************************************************
                          kdtimer.cpp  -  description
                             -------------------
    begin                : Tue Apr 2 2002
    copyright            : (C) 2002 by Erik Johansson
    email                : erre@telia.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qdatetime.h>
#include <qtimer.h>

// Local includes
#include "kdtimer.h"
		
KDTimer::KDTimer( const QDateTime *fdate, short int ff, short int fr )
{
	init( &ff, &fr );
	*finishDate = *fdate;
	calcFormat[FINISH] = ff;
	roundOff[FINISH] = fr;
	
	// No halfway date
	delete halfDate;
	halfDate = 0;
	useHalfWay = false;
}

KDTimer::KDTimer( const QDateTime *fdate, short int ff, short int fr, const QDateTime *hdate )
{
	init( &ff, &fr );
	*finishDate = *fdate;
	calcFormat[FINISH] = ff;
	roundOff[FINISH] = fr;
	
	// Use halfway date, settings = finish
	*halfDate = * hdate;
	useHalfWay = true;
	calcFormat[HALF] = ff;
	roundOff[HALF] = fr;
}

KDTimer::KDTimer( const QDateTime *fdate, short int ff, short int fr, const QDateTime *hdate, short int hf, short int hr)
{
	init( &ff, &fr, &hf, &hr );
	*finishDate = *fdate;
	calcFormat[FINISH] = ff;
	roundOff[FINISH] = fr;
	
	// Use halfway date and settings
	*halfDate = *hdate;
	useHalfWay = true;
	calcFormat[HALF] = hf;
	roundOff[HALF] = hr;
}

/**
* KDTimer::init( *format, *round, *format = 0, *round = 0 )
* Author: Erik Johansson <erre@telia.com> 2002-04-03
* Init some member vars and get memory for the dates
* Make sure the format (ff and hf) is within boundaries (KD::NONE (0) - KD::ALL (127) ) else set it to KD::ALL
* Make sure the round off (fr and hr) is within boundaries (KD::DOWN (0) - KD::MATH (3) ) else set it to KD::DOWN
* hf and hr are only checked if != 0
**/
void KDTimer::init( short int *ff, short int *fr, short int *hf = 0, short int *hr = 0 )
{
	finishDate = new QDateTime();
	halfDate = new QDateTime();
	calcDateTime = new QDateTime();	
	calcDate = new QDate();
	calcTime = new QTime();
	
	calcingDate = false;
	calcingTime = false;

	dateTimer = new QTimer();
	timeTimer = new QTimer();
	checkTimer = new QTimer();
	connect( dateTimer, SIGNAL( timeout() ), this, SLOT( slotCalcDate() ) );
	connect( timeTimer, SIGNAL( timeout() ), this, SLOT( slotCalcTime() ) );
	connect( checkTimer, SIGNAL( timeout() ), this, SLOT( slotCheckDateTime() ) );
	
	// Check boundaries
	if ( *ff > KD::ALL || *ff < KD::NONE )
		*ff = KD::ALL;
	if ( *fr > KD::MATH || *fr < KD::DOWN )
		*fr = KD::DOWN;		

	if ( hf != 0 )
	{
		if ( *hf > KD::ALL || *hf < KD::NONE )
			*hf = KD::ALL;
	}
	if ( hr != 0 )
	{
		if ( *hr > KD::MATH || *hr < KD::DOWN )
			*hr = KD::DOWN;		
	}
}

const STimeLeft * KDTimer::start()
{
	*calcDate = QDate::currentDate();
	*calcTime = QTime::currentTime();
	*calcDateTime = QDateTime::currentDateTime();
	roundOff[CURRENT] = roundOff[FINISH];
	calcFormat[CURRENT] = calcFormat[FINISH];
	slotCheckDateTime();
	if ( !timeLeft.finish )	
	{
		slotCalcDate();
		slotCalcTime();						
		checkTimer->start( 1000, false );	// 1 sec
	}
	return &timeLeft;
}
	
KDTimer::~KDTimer()
{
	delete finishDate;
	delete halfDate;
	delete calcDateTime;
	delete calcDate;
	delete calcTime;
	delete dateTimer;
	delete timeTimer;
	delete	 checkTimer;
}

void KDTimer::slotCheckDateTime()
{
	// If we have reached to or past finish
	if ( !timeLeft.finish && ( QDateTime::currentDateTime() >= *finishDate ) )
	{
		emit changeSettings(2);
		dateTimer->stop();
		timeTimer->stop();
		checkTimer->stop();
		timeLeft.setDateValue(0);
		timeLeft.setTimeValue(0);
		timeLeft.finish = true;	
	}

	// If we have reached to or past half way	
	if ( useHalfWay && !timeLeft.half && !timeLeft.finish && ( QDateTime::currentDateTime() >= *halfDate ) )
	{
		roundOff[CURRENT] = roundOff[HALF];
		calcFormat[CURRENT] = calcFormat[HALF];		
		timeTimer->stop();
		slotCalcTime();			
		emit changeSettings(1);					
		dateTimer->stop();
		slotCalcDate();
		timeLeft.half = true;
	}
}

void KDTimer::years( short int r )
{
	const QDate cDate = *calcDate;
	const QDate fDate = finishDate->date();
	
	// Get the number of WHOLE years left and save it to timeLeft.years.
	if ( cDate.month() < fDate.month() )
		timeLeft.years = fDate.year() - cDate.year();
	else if ( cDate.month() > fDate.month() )
		timeLeft.years = ( (fDate.year() - cDate.year() - 1) >= 0 ) ? fDate.year() - cDate.year() - 1 : 0;
	else if ( cDate.month() == fDate.month() )
	{
		if ( cDate.day() == fDate.day() )
		{
			if ( *calcTime <= finishDate->time() )
				timeLeft.years = fDate.year() - cDate.year();
			else if ( *calcTime > finishDate->time()  )
				timeLeft.years = ( (fDate.year() - cDate.year() - 1) >= 0 ) ? fDate.year() - cDate.year() - 1 : 0;
		}
		else if ( cDate.day() < fDate.day() )
			timeLeft.years = fDate.year() - cDate.year();
		else if ( cDate.day() > fDate.day() )
			timeLeft.years = ( (fDate.year() - cDate.year() - 1) >= 0 ) ? fDate.year() - cDate.year() - 1 : 0;			
	}
	
	// Update calcDate by adding timeLeft.years to it.
	calcDate->setYMD( cDate.year() + timeLeft.years, cDate.month(), cDate.day() );
	
	// Round of timeLeft.years according to r.
	switch ( r )
	{
		case KD::UP:
			if ( *calcDate != finishDate->date() )
				timeLeft.years += 1;
			break;
		case KD::MATH:
			if ( *calcDate != finishDate->date() )
			{
				months( KD::NONE, -2 );  // Call months(), asking it not to round off and not to update calcDate.
				if ( timeLeft.months >= 6 ) // If there is more then or equal to 6 months left round up
					timeLeft.years += 1;
				timeLeft.months = 0; // reset months
			}
			break;
		case KD::DOWN: 	// fall through
		default:
			break;
	}
}

void KDTimer::months( short int r, const short int f = -1 )
{
	ULONG months = 0;
	// Check if we can call years() to calc years left
	if ( f >= 0 && f < KD::YEARS )
	{
		years( KD::NONE );
		timeLeft.months = timeLeft.years * 12;
		timeLeft.years = 0;
		months = timeLeft.months;
	}
	
	const QDate cDate = *calcDate;
	const QDate fDate = finishDate->date();
	
	// Calc WHOLE months left and ADD it timeLeft.months
	if ( cDate.day() < fDate.day() )
	{
		if ( cDate.year() == fDate.year() )
			timeLeft.months += fDate.month() - cDate.month();
		else if ( cDate.year() < fDate.year() )
			timeLeft.months += 12 - cDate.month() + fDate.month();
	}
	else if ( cDate.day() > fDate.day() )
	{
		if ( cDate.year() == fDate.year() )
			timeLeft.months += ( ( fDate.month() - cDate.month() - 1 ) >= 0 ) ? fDate.month() - cDate.month() - 1 : 0;
		else if ( cDate.year() < fDate.year() )
			timeLeft.months += ( ( 11 - cDate.month() + fDate.month() ) >= 0 ) ? 11 - cDate.month() + fDate.month() : 0;
	}
	else if ( cDate.day() == fDate.day() ) // The same day, check time
	{
		if ( *calcTime <= finishDate->time() )
		{
			if ( cDate.year() == fDate.year() )
				timeLeft.months += fDate.month() - cDate.month();
			else if ( cDate.year() < fDate.year() )
				timeLeft.months += 12 - cDate.month() + fDate.month();
		}
		else if ( *calcTime > finishDate->time()  )
		{
			if ( cDate.year() == fDate.year() )
				timeLeft.months += ( ( fDate.month() - cDate.month() - 1 ) >= 0 ) ? fDate.month() - cDate.month() - 1 : 0;
			else if ( cDate.year() < fDate.year() )
				timeLeft.months += ( ( 11 - cDate.month() + fDate.month() ) >= 0 ) ? 11 - cDate.month() + fDate.month() : 0;
		}
	}

	if ( f != -2) // Do not update calcDate if f == -2 (se years() round off)
	{
		// Months has already been added to calcDate by years() or is 0	
		if ( ( cDate.month() + ( timeLeft.months - months) ) > 12 )
			calcDate->setYMD( cDate.year() + 1, (cDate.month() + timeLeft.months - months - 12), cDate.day() );
		else
			calcDate->setYMD( cDate.year(), (cDate.month() + timeLeft.months - months), cDate.day() );
	}
		
	// Round off timeLeft.months according to r
	switch ( r )
	{
		case KD::UP:
			if ( *calcDate != finishDate->date() )
				timeLeft.months += 1;
			break;
		case KD::MATH:
			if ( *calcDate != finishDate->date() )
			{
				days( KD::NONE, -2 ); // Call days(), asking it not to round off and not to update calcDate
				if ( timeLeft.days >= 15 ) // If there is more then or equal to 15 days left round up
					timeLeft.months += 1;
				timeLeft.days = 0; // reset days
			}
			break;
		case KD::DOWN: 	// fall through
		default:
			break;
	}		
}

void KDTimer::weeks( short int r )
{
	ULONG temp = calcDate->daysTo( finishDate->date() );
	if ( *calcTime > finishDate->time() )
		temp = ( temp == 0 ) ? 0 : temp - 1;
	timeLeft.weeks = (temp - (temp % 7)) / 7;
	*calcDate = calcDate->addDays( timeLeft.weeks * 7 );
	
	switch ( r )
	{
		case KD::UP:
			if ( *calcDate != finishDate->date() )
				timeLeft.weeks += 1;
			break;
		case KD::MATH:
			if ( *calcDate != finishDate->date() )
			{
				days( KD::NONE, -2 ); // Call days(), asking it not to round off and not to update calcDate
				if ( timeLeft.days > 3 ) // If there is more then 3 days left round up
					timeLeft.weeks += 1;
				timeLeft.days = 0; // reset days
			}
			break;
		case KD::DOWN: 	// fall through
		default:
			break;
	}	
}

void KDTimer::days( short int r, const short int f = -1 )
{
	ULONG temp = calcDate->daysTo( finishDate->date() );
	if ( *calcTime > finishDate->time() )
		temp = ( temp == 0 ) ? 0 : temp - 1;
	timeLeft.days = temp;
	
	if ( f != -2 ) // f == -2 do NOT update calcDate.
		*calcDate = calcDate->addDays( timeLeft.days );
		
	switch ( r )
	{
		case KD::UP:
			if ( *calcDate != finishDate->date() )
				timeLeft.days += 1;
			break;
		case KD::MATH:
			if ( *calcDate != finishDate->date() )
			{
				hours( KD::NONE, -2 );
				if ( timeLeft.hours >= 12 ) // If there is more then or equal to 12 hours left round up
					timeLeft.days += 1;
				timeLeft.hours = 0; // reset hours
			}
			break;
		case KD::DOWN: 	// fall through
		default:
			break;
	}		
}

void KDTimer::hours( short int r, const short int f = -1 )
{
	if ( f >= 0 && f < KD::DAYS )
	{
		days( KD::NONE ); // Call days() asking it not to round off, but to update calcDate.
		timeLeft.hours = timeLeft.days * 24;
		timeLeft.days = 0;	
		*calcDateTime = QDateTime( *calcDate, *calcTime );  // Update calcDateTime with the new calcDate
	}
	long temp = calcDateTime->secsTo( *finishDate );
	timeLeft.hours += (temp - (temp % 3600)) / 3600;
	
	if ( f != -2 ) // Update calcDateTime with temp, NOT with timeLeft.hours (may already been updated)
		*calcDateTime = calcDateTime->addSecs( temp - (temp % 3600) );
		
	switch ( r )
	{
		case KD::UP:
			if ( *calcDateTime != *finishDate )
				timeLeft.hours += 1;
			break;
		case KD::MATH:
			if ( *calcDateTime != *finishDate )
			{
				mins( KD::NONE, -2 );
				if ( timeLeft.mins >= 30 ) // If there is more then or equal to 30 mins left round up
					timeLeft.hours += 1;
				timeLeft.mins = 0; // reset mins
			}
			break;
		case KD::DOWN: 	// fall through
		default:
			break;
	}		
}

void KDTimer::mins( short int r, const short int f = -1 )
{
	if ( f < 0 && f < KD::HOURS )
	{
		hours ( KD::NONE, f );
		timeLeft.mins = timeLeft.hours * 60;
		timeLeft.hours = 0;
	}
	ULONG temp = calcDateTime->secsTo( *finishDate );
	timeLeft.mins += (temp - (temp % 60)) / 60;
	if ( f != -2 )
		*calcDateTime = calcDateTime->addSecs( timeLeft.mins * 60 );
		
	switch ( r )
	{
		case KD::UP:
			if ( *calcDateTime != *finishDate )
				timeLeft.mins += 1;
			break;
		case KD::MATH:
			if ( *calcDateTime != *finishDate )
			{
				secs( -2 ); // Call secs, dont update calcDateTime
				if ( timeLeft.secs >= 30 ) // If there is more then or equal to 30 secs left round up
					timeLeft.mins += 1;
				timeLeft.secs = 0; // reset secs
			}
			break;
		case KD::DOWN: 	// fall through
		default:
			break;
	}			
}

void KDTimer::secs( const short int f = -1 )
{
	timeLeft.secs = calcDateTime->secsTo( *finishDate );
	if ( f != -2 )
		*calcDateTime = calcDateTime->addSecs( timeLeft.secs );
}

void KDTimer::slotCalcDate()
{
	if ( !calcingTime && ( calcFormat[CURRENT] >= KD::DAYS ) )
	{
		calcingDate = true; // We have started calcing date (can't calc time at the same time)
		timeLeft.setDateValue( 0 );  // Reset
		*calcDate = QDate::currentDate();
		*calcTime = QTime::currentTime();
		short int format = calcFormat[CURRENT];
		short int round = roundOff[CURRENT];
		
		const unsigned short RECALC = 60000; // Next update 1min
		// Calc years left		
		if ( format > KD::YEARS )
		{
			years( KD::NONE );
			format -= KD::YEARS;
		}
		else if ( format == KD::YEARS )		// Only calc years, recalc once/min
		{
			years( round );
			format -= KD::YEARS;
			dateTimer->start( RECALC, true );
		}
		// Calc months left
		if ( format > KD::MONTHS )
		{
			months( KD::NONE, calcFormat[CURRENT] );
			format -= KD::MONTHS;
		}	
		else if ( format == KD::MONTHS )		// Calc months (maybe years), recalc once/min
		{
			months( round, calcFormat[CURRENT] );
			format -= KD::MONTHS;
			dateTimer->start( RECALC, true );
		}
		// Calc weeks left
		if ( format > KD::WEEKS )
		{
			weeks( KD::NONE );
			format -= KD::WEEKS;
		}
		else if ( format == KD::WEEKS ) 	// Calc weeks (maybe years & months), recalc once/min
		{
			weeks( round );
			format -= KD::WEEKS;
			dateTimer->start( RECALC, true );
		}
		// Calc days left
		if ( format > KD::DAYS )
		{
			days( KD::NONE );
			format -= KD::DAYS;
		}
		else if ( format == KD::DAYS )	// Calc days (maybe years, months & weeks), recalc once/min
		{
			days( round );
			format -= KD::DAYS;
			dateTimer->start( RECALC, true );
		}
		
		if ( format == 0 ) // no more
			emit timeLeftChanged();
		else
			dateTimer->start( 3000, true ); // every 3s
		
		calcingDate = false;
	}
	else if ( calcingTime && ( calcFormat[CURRENT] >= KD::DAYS ) ) // Currently calcing time, retry in 10ms
	{
		dateTimer->start( 10, true );
	}
}	

void KDTimer::slotCalcTime()
{
	if ( !calcingDate ) // We are note calcing date at the moment
	{
		calcingTime = true;
		*calcTime = QTime::currentTime();		
		if ( calcFormat[CURRENT] >= KD::DAYS )  // *calcDate should already be set by slotCalcDate()
			*calcDateTime = QDateTime( *calcDate, *calcTime );
		else // *calcDate not set by slotCalcDate()
		{
			*calcDate = QDate::currentDate();
			*calcDateTime = QDateTime( *calcDate, *calcTime );
		}			
		// Set all but hour, min and sec bit to 0.
		// caclFormat[CURRENT] = 0XXX XYYY where X = 0 or 1, Y = 0 or 1
		// 0XXX XYYY & 7 = 0XXX XYYY & 0000 0111 = 0000 0YYY
		short int format = calcFormat[CURRENT] & 7;
		
		short int round = roundOff[CURRENT];
		timeLeft.setTimeValue(0); // reset
		
		// Calc hours left
		if ( format > KD::HOURS )
		{
			hours( KD::NONE, calcFormat[CURRENT] );
			format -= KD::HOURS;
		}
		else if ( format == KD::HOURS )
		{
			hours( round, calcFormat[CURRENT] );
			format -= KD::HOURS;
			timeTimer->start( 60000, true );			// Next update 1min
		}
		// Calc mins left
		if ( format > KD::MINS )
		{
			mins( KD::NONE, calcFormat[CURRENT] );
			format -= KD::MINS;
		}
		else if ( format == KD::MINS )
		{
			mins( round, calcFormat[CURRENT] );
			format -= KD::MINS;
			timeTimer->start( 10000, true );		// Next update 10s				
		}
		// Calc secs left
		if ( format == KD::SECS )
		{
			secs();
			format -= KD::SECS;
			timeTimer->start( 1000, true );		// Next update 1s
		}
		
		if ( format == 0 )
			emit timeLeftChanged();
		
		calcingTime = false;
	}
	else
	{
		timeTimer->start(30, true); // Retry in 30ms
	}
}
