/************************************************************************/
/*									*/
/*  Layout of a document.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	"docLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Default leading in terms of the size of the dominant font in a	*/
/*  paragraph.								*/
/*									*/
/************************************************************************/

#   define LINEDISTFAC	5

/************************************************************************/
/*									*/
/*  Initialize a layout job.						*/
/*									*/
/************************************************************************/

void docPsInitLayoutJob(	LayoutJob *	lj )
    {
    lj->ljChangedRectanglePixels= (DocumentRectangle *)0;
    lj->ljAdd= (AppDrawingData *)0;
    lj->ljBd= (BufferDocument *)0;
    lj->ljChangedItem= (BufferItem *)0;

    docInitLayoutPosition( &(lj->ljPosition) );

    lj->ljLayoutScreen.slStartRow= (START_ROW)0;
    lj->ljLayoutScreen.slStartParagraph= (START_SCREEN_PARAGRAPH)0;
    lj->ljLayoutScreen.slScreenFrame= (SCREEN_FRAME)0;
    lj->ljLayoutScreen.slLayoutLine= (LAYOUT_SCREEN_LINE)0;

    lj->ljPlaceScreen.slStartRow= (START_ROW)0;
    lj->ljPlaceScreen.slStartParagraph= (START_SCREEN_PARAGRAPH)0;
    lj->ljPlaceScreen.slScreenFrame= (SCREEN_FRAME)0;
    lj->ljPlaceScreen.slLayoutLine= (LAYOUT_SCREEN_LINE)0;

    return;
    }

void docPsCleanLayoutJob(	LayoutJob *	lj )
    {
    return;
    }

void docLayoutInitBlockFrame(	BlockFrame *	bf )
    {
    bf->bfX0Twips= 0;
    bf->bfX1Twips= 0;
    bf->bfY0Twips= 0;
    bf->bfY1Twips= 0;

    docInitNotesReservation( &(bf->bfNotesReservation) );
    }

void docInitNotesReservation(	NotesReservation *	nr )
    {
    nr->nrFirstFootnote= -1;
    nr->nrFootnoteCount= 0;

    nr->nrFtnsepHeight= 0;
    nr->nrFootnoteHeight= 0;
    }

/************************************************************************/
/*									*/
/*  Continue to lay out the text on a subsequent page.			*/
/*									*/
/************************************************************************/

void docLayoutSectColumnTop(	BufferItem *		bodySectBi,
				BufferDocument *	bd,
				LayoutPosition *	lpTop,
				BlockFrame *		bf )
    {
    docBlockFrameTwips( bf, bodySectBi, bd, lpTop->lpPage, lpTop->lpColumn );

    lpTop->lpPageYTwips= bf->bfY0Twips;
    lpTop->lpAtTopOfColumn= 1;

    return;
    }

void docLayoutToNextColumn(	BufferItem *		bi,
				BufferDocument *	bd,
				LayoutPosition *	lpTop,
				BlockFrame *		bf )
    {
    while ( bi				&&
	    bi->biLevel != DOClevSECT	)
	{ bi= bi->biParent;	}

    if  ( ! bi )
	{ XDEB(bi); return;	}

    lpTop->lpPage++;

    docLayoutSectColumnTop( bi, bd, lpTop, bf );

    return;
    }

/************************************************************************/
/*									*/
/*  Calculate the height of a series of lines in a paragraph.		*/
/*									*/
/************************************************************************/

static void docPsAboveParagraph(	BufferItem *			bi )
    {
    int				above= 0;
    const BorderProperties *	bp= (const BorderProperties *)0;

    above= bi->biParaSpaceBeforeTwips;

    if  ( bi->biParaInTable			&&
	  bi->biNumberInParent == 0		)
	{
	const BufferItem *	rowBi= bi->biParent->biParent;
	const CellProperties *	cp;
	int			col= bi->biParent->biNumberInParent;

	if  ( col >= rowBi->biRowCellCount )
	    { LLDEB(col,rowBi->biRowCellCount);	}
	else{
	    cp= &(rowBi->biRowCells[col]);

	    if  ( cp->cpTopBorder.bpStyle != DOCbsNONE )
		{
		above += cp->cpTopBorder.bpSpacingTwips;
		/*  Word seems not to allocate this
		above += cp->cpTopBorder.bpWidthTwips;
		*/
		}
	    }

	/*  Nicer but Word does not do this.
	above += ( bi->biParaLeadingTwips+ 1 )/ 2;
	*/
	}

    if  ( bi->biParaTopBorder.bpStyle != DOCbsNONE )
	{
	/*  Word seems not to allocate this
	above += bi->biParaTopBorder.bpWidthTwips;
	*/
	above += bi->biParaTopBorder.bpSpacingTwips;
	bp= &(bi->biParaTopBorder);
	}
    else{
	if  ( bi->biParaBoxBorder.bpStyle != DOCbsNONE )
	    {
	    BufferItem *	prevBi= (BufferItem *)0;

	    if  ( bi->biNumberInParent > 0 )
		{
		prevBi= bi->biParent->biChildren[
					    bi->biNumberInParent- 1];
		}
	    if  ( ! prevBi					||
		  prevBi->biParaBoxBorder.bpStyle == DOCbsNONE	)
		{
		/*  Word seems not to allocate this
		above += bi->biParaBoxBorder.bpWidthTwips;
		*/
		above += bi->biParaBoxBorder.bpSpacingTwips;
		bp= &(bi->biParaBoxBorder);
		}
	    }
	}

    bi->biParaSpaceAboveLinesTwips= above;
    bi->biParaBorderAboveParagraph= bp;

    return;
    }

static void docPsBelowParagraph(	BufferItem *			bi )
    {
    int				below= 0;
    const BorderProperties *	bp= (const BorderProperties *)0;

    if  ( bi->biParaBottomBorder.bpStyle != DOCbsNONE )
	{
	below += bi->biParaLeadingTwips;
	below += bi->biParaBottomBorder.bpSpacingTwips;
	/*  Word seems not to allocate this
	below += bi->biParaBottomBorder.bpWidthTwips;
	*/
	bp= &(bi->biParaBottomBorder);
	}
    else{
	if  ( bi->biParaBoxBorder.bpStyle != DOCbsNONE )
	    {
	    BufferItem *	nextBi= (BufferItem *)0;

	    if  ( bi->biNumberInParent <
				bi->biParent->biChildCount- 1 )
		{
		nextBi= bi->biParent->biChildren[
					    bi->biNumberInParent+ 1];
		}

	    if  ( ! nextBi					||
		  nextBi->biParaBoxBorder.bpStyle == DOCbsNONE	)
		{
		below += bi->biParaLeadingTwips;
		below += bi->biParaBoxBorder.bpSpacingTwips;
		/*  Word seems not to allocate this
		below += bi->biParaBoxBorder.bpWidthTwips;
		*/
		bp= &(bi->biParaBoxBorder);
		}
	    }
	}

    if  ( bi->biParaInTable					&&
	  bi->biNumberInParent == bi->biParent->biChildCount- 1	)
	{
	const BufferItem *	rowBi= bi->biParent->biParent;
	const CellProperties *	cp;
	int			col= bi->biParent->biNumberInParent;

	/*  Nicer but Word does not do this.
	below += bi->biParaLeadingTwips/ 2;
	*/

	if  ( col >= rowBi->biRowCellCount )
	    { LLDEB(col,rowBi->biRowCellCount);	}
	else{
	    cp= &(rowBi->biRowCells[col]);

	    if  ( cp->cpBottomBorder.bpStyle != DOCbsNONE )
		{
		below += cp->cpBottomBorder.bpSpacingTwips;
		/*  Word seems not to allocate this
		below += cp->cpBottomBorder.bpWidthTwips;
		*/
		}
	    }
	}

    below += bi->biParaSpaceAfterTwips;

    bi->biParaSpaceBelowLinesTwips= below;
    bi->biParaBorderBelowParagraph= bp;

    return;
    }

/************************************************************************/
/*									*/
/*  Determine the 'majority' font of a paragraph, and get the font	*/
/*  extents for that font.						*/
/*									*/
/*  1)  Note that subscript/superscript is NOT taken into account.	*/
/*									*/
/************************************************************************/

int docPsParagraphLineExtents(	const AppPhysicalFontList *	apfl,
				BufferItem *			bi )
    {
    const TextParticule *	tp= bi->biParaParticules;

    int				size;
    int				paraAscent= 0;
    int				paraDescent= 0;
    int				i;

    static int *		counts;
    int *			fresh;

    int				found;
    int				foundCount;

    fresh= (int *)realloc( counts, apfl->apflCount* sizeof(int) );
    if  ( ! fresh )
	{ LXDEB(apfl->apflCount,fresh); return -1;	}
    counts= fresh;

    for ( i= 0; i < apfl->apflCount; i++ )
	{ counts[i]= 0;	}

    for ( i= 0; i < bi->biParaParticuleCount; tp++, i++ )
	{
	if  ( tp->tpKind != DOCkindTEXT		&&
	      tp->tpKind != DOCkindTAB		&&
	      tp->tpKind != DOCkindOBJECT	)
	    { continue;	}

	if  ( tp->tpPhysicalFont < 0			||
	      tp->tpPhysicalFont >= apfl->apflCount	)
	    { LLDEB(tp->tpPhysicalFont,apfl->apflCount); continue;	}

	counts[tp->tpPhysicalFont] += tp->tpStrlen+ 1;
	}

    found= -1;
    foundCount= 0;
    for ( i= 0; i < apfl->apflCount; i++ )
	{
	if  ( counts[i] > foundCount )
	    { found= i; foundCount= counts[i];	}
	}

    if  ( found >= 0 )
	{
	AfmFontInfo *	afi;
	int		encoding;

	afi= docPsPrintGetAfi( &encoding, apfl, found );

	size= 10* apfl->apflFonts[found].apfAttribute.taFontSizeHalfPoints;

	paraAscent= ( size* afi->afiFontBBox.abbTop+ 500 ) / 1000;
	paraDescent= paraAscent- size;
	}
    else{ /* LDEB(found); */ size= 200;	}

    bi->biParaAscentTwips= paraAscent;
    bi->biParaDescentTwips= paraDescent;

    bi->biParaLeadingTwips= size/ LINEDISTFAC;

    docPsAboveParagraph( bi );
    docPsBelowParagraph( bi );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert the correct page number in a page header/footer.		*/
/*									*/
/************************************************************************/

int docLayoutExternalItem(	ExternalItem *			ei,
				DocumentRectangle *		drChanged,
				int				page,
				int				y0Twips,
				BufferDocument *		bd,
				const BufferItem *		sectBi,
				AppDrawingData *		add,
				LAYOUT_EXTERNAL			layoutExternal,
				DOC_CLOSE_OBJECT		closeObject )
    {
    if  ( page != ei->eiPageFormattedFor )
	{
	RecalculateFields		rf;
	int				y1Twips;

	docInitRecalculateFields( &rf );

	rf.rfBd= bd;
	rf.rfVoidadd= (void *)add;
	rf.rfCloseObject= closeObject;
	rf.rfUpdateFlags=
		FIELDdoDOC_FORMATTED|FIELDdoDOC_COMPLETE|FIELDdoPAGE_NUMBER;
	rf.rfFieldsUpdated= 0;


	if  ( docRecalculateTextLevelFieldsInExternalItem( &rf, ei,
							    sectBi, page ) )
	    { LDEB(page); return -1;	}

	y1Twips= y0Twips;

	if  ( layoutExternal 					&&
	      (*layoutExternal)( &y1Twips, ei, page, y0Twips,
					bd, add, drChanged )	)
	    { LDEB(1); return -1;	}

	ei->eiPageFormattedFor= page;
	ei->eiY0UsedTwips= y0Twips;
	ei->eiY1UsedTwips= y1Twips;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Derive the frame for a paragraph from the page rectangle and the	*/
/*  paragraph properties.						*/
/*									*/
/*  For paragraphs inside a table cell, geometry is derived from the	*/
/*  table column.							*/
/*									*/
/*  1)  The block starts below what has been reserved for the page	*/
/*	header.								*/
/*  2)  The block starts above what has been reserved for the page	*/
/*	footer.								*/
/*									*/
/************************************************************************/

void docBlockFrameTwips(	BlockFrame *		bf,
				BufferItem *		bi,
				const BufferDocument *	bd,
				int			page,
				int			column )
    {
    ExternalItem *		ei;
    int				inHeaderFooter;

    const DocumentProperties *	dp= &(bd->bdProperties);
    const SectionProperties *	sp;
    const DocumentGeometry *	dg;

    BufferItem *		sectBi;

    while( bi				&&
	   bi->biLevel != DOClevSECT	)
	{ bi= bi->biParent;	}

    if  ( ! bi )
	{ XDEB(bi); return;	}

    sp= &(bi->biSectProperties);
    dg= &(sp->spDocumentGeometry);

    bf->bfX0Twips= dg->dgLeftMarginTwips;
    bf->bfX1Twips= dg->dgPageWideTwips- dg->dgRightMarginTwips;

    docInitNotesReservation( &(bf->bfNotesReservation) );

    switch( bi->biInExternalItem )
	{
	case DOCinBODY:

	case DOCinFOOTNOTE:
	case DOCinENDNOTE:

	case DOCinFTNSEP:
	case DOCinFTNSEPC:
	case DOCinFTNCN:
	case DOCinAFTNSEP:
	case DOCinAFTNSEPC:
	case DOCinAFTNCN:

	    inHeaderFooter= docWhatPageHeader( &ei, bi, page, dp );

	    /*  1 Reserved!!  */
	    if  ( ei && ei->eiItem )
		{ bf->bfY0Twips= ei->eiY1ReservedTwips;		}
	    else{ bf->bfY0Twips= dg->dgTopMarginTwips;		}

	    inHeaderFooter= docWhatPageFooter( &ei, bi, page, dp );

	    /*  2 Reserved!!  */
	    if  ( ei && ei->eiItem )
		{ bf->bfY1Twips= ei->eiY0ReservedTwips; }
	    else{ bf->bfY1Twips= dg->dgPageHighTwips- dg->dgBottomMarginTwips; }

	    /*
	    appDebug( "PAGE %3d BLOCK Y: %5d..%5d %s\n",
			    page, bf->bfY0Twips, bf->bfY1Twips,
			    docExternalKindStr( bi->biInExternalItem ) );
	    */

	    break;

	case DOCinSECT_HEADER:
	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:

	case DOCinSECT_FOOTER:
	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    if  ( column != 0 )
		{ LDEB(column);	}

	    sectBi= bd->bdItem.biChildren[bi->biNumberInParent];
	    ei= docSectionHeaderFooter( sectBi, bi->biInExternalItem );
	    bf->bfY0Twips= ei->eiY0ReservedTwips;
	    bf->bfY1Twips= ei->eiY1ReservedTwips;
	    break;

	default:
	    LDEB(bi->biInExternalItem);
	}

    return;
    }

void docParagraphFrameTwips(	ParagraphFrame *	pf,
				const BlockFrame *	bf,
				int			bottom,
				int			stripHigh,
				BufferItem *		paraBi )
    {
    const NotesReservation *	nrBf= &(bf->bfNotesReservation);
    int				footnoteHeight;

    footnoteHeight= nrBf->nrFtnsepHeight+ nrBf->nrFootnoteHeight;

    pf->pfX0GeometryTwips= bf->bfX0Twips;
    pf->pfX1GeometryTwips= bf->bfX1Twips;

    pf->pfPageHigh= bf->bfY1Twips- bf->bfY0Twips;

    pf->pfStripHigh= stripHigh;
    pf->pfPageY1Twips= bf->bfY1Twips- footnoteHeight;

    if  ( bottom > 0			&&
	  bottom <= pf->pfPageY1Twips	)
	{ pf->pfPageY1Twips= bottom;	}

    if  ( paraBi->biParaInTable )
	{
	int			col0;
	int			col1;
	const BufferItem *	cellBi= paraBi->biParent;
	const BufferItem *	rowBi= cellBi->biParent;
	const RowProperties *	rp= &(rowBi->biRowProperties);

	col0= col1= cellBi->biNumberInParent;

	if  ( col1 >= rowBi->biRowCellCount )
	    { LLDEB(col1,rowBi->biRowCellCount);	}
	else{
	    const CellProperties *	cp1= rp->rpCells+ col1;

	    if  ( cp1->cpLeftInMergedRow )
		{
		while( col1 < rowBi->biRowCellCount- 1	&&
		       cp1[1].cpMergedWithLeft		)
		    { col1++; cp1++;	}
		}

	    pf->pfX1GeometryTwips= bf->bfX0Twips+ cp1->cpRightBoundaryTwips;

	    if  ( col0 == 0 )
		{ pf->pfX0GeometryTwips += rowBi->biRowLeftIndentTwips;	}
	    else{
		CellProperties *	cpp;

		cpp= rp->rpCells+ col0- 1;
		pf->pfX0GeometryTwips += cpp->cpRightBoundaryTwips;
		}
	    }

	pf->pfX0GeometryTwips += rowBi->biRowHalfGapWidthTwips;
	pf->pfX1GeometryTwips -= rowBi->biRowHalfGapWidthTwips;
	}

    pf->pfX0TextLinesTwips= pf->pfX0GeometryTwips+
					paraBi->biParaLeftIndentTwips;
    pf->pfX1TextLinesTwips= pf->pfX1GeometryTwips-
					paraBi->biParaRightIndentTwips;

    pf->pfX0FirstLineTwips= pf->pfX0TextLinesTwips+
					paraBi->biParaFirstIndentTwips;

    if  ( paraBi->biInExternalItem == DOCinBODY		||
	  paraBi->biInExternalItem == DOCinENDNOTE	||
	  paraBi->biInExternalItem == DOCinAFTNSEP	)
	{ pf->pfHasBottom= 1;	}
    else{ pf->pfHasBottom= 0;	}

    return;
    }

