/************************************************************************/
/*									*/
/*  Layout of a document.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	<appImage.h>
#   include	"docLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Initialisation for paragraph layout.				*/
/*									*/
/************************************************************************/

static int docLayoutOpenParaFonts(	BufferItem *			paraBi,
					AppDrawingData *		add,
					const DocumentProperties *	dp )
    {
    int				part;
    TextParticule *		tp;

    const DocumentFontList *	dfl= &(dp->dpFontList);

    tp= paraBi->biParaParticules;
    for ( part= 0; part < paraBi->biParaParticuleCount; tp++, part++ )
	{
	if  ( tp->tpPhysicalFont < 0 )
	    {
	    tp->tpPhysicalFont= appOpenDocumentFont( add, dfl,
							tp->tpTextAttribute );

	    if  ( tp->tpPhysicalFont < 0 )
		{
		APP_DEB(appDebug( "\"%.*s\"\n",
		    (int)tp->tpStrlen, paraBi->biParaString+ tp->tpStroff ));
		LLDEB(part,tp->tpPhysicalFont);
		return -1;
		}
	    }
	}

    return 0;
    }

static int docLayoutParaInit(	BufferItem *			paraBi,
				AppDrawingData *		add,
				const DocumentProperties *	dp )
    {
    if  ( docLayoutOpenParaFonts( paraBi, add, dp ) )
	{ LDEB(1); return -1;	}

    if  ( docPsParagraphLineExtents( &(add->addPhysicalFontList), paraBi ) )
	{ LDEB(1); return -1;	}

    paraBi->biParaLineCount= 0;

    return 0;
    }

static int docPsPlaceParaInit(	BufferItem *			paraBi,
				AppDrawingData *		add,
				const DocumentProperties *	dp )
    {
    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*  Claim space above a particular line.				*/
/*									*/
/*  1)  Cope with the output of jade: Ignore enormous space before's	*/
/*	in footers.							*/
/*									*/
/************************************************************************/

static void docPsAboveLine(	LayoutPosition *		lp,
				int				fromLinePos,
				const BufferItem *		paraBi,
				int				part,
				const DocumentGeometry *	dg,
				AppDrawingData *		add )
    {
    double		xfac= add->addMagnifiedPixelsPerTwip;

    int			spaceAboveLineTwips= 0;
    int			spaceAboveLinePixels= 0;

    /*  1  */
    if  ( part == 0					&&
	  ! fromLinePos					&&
	  paraBi->biInExternalItem == DOCinBODY		)
	{
	int		isFirst= 0;

	if  ( paraBi->biParaInTable )
	    {
	    if  ( paraBi->biNumberInParent == 0 )
		{ isFirst= 1;	}
	    }
	else{
	    if  ( lp->lpPageYTwips == dg->dgTopMarginTwips )
		{ isFirst= 1;	}
	    }

	spaceAboveLineTwips= paraBi->biParaSpaceAboveLinesTwips;

	if  ( ! isFirst )
	    { spaceAboveLineTwips += paraBi->biParaSpaceBeforeTwips;	}

	spaceAboveLinePixels= TWIPStoPIXELS( xfac, spaceAboveLineTwips );
	}

    lp->lpPageYTwips += spaceAboveLineTwips;
    /* still at top */

    return;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*  Claim space below a particular line.				*/
/*									*/
/*  Also decide whether it will fit in the current formatting frame.	*/
/*									*/
/************************************************************************/

static int docPsBelowLine(	const LayoutPosition *		lpTop,
				LayoutPosition *		lp,
				const TextLine *		tl,
				const BufferItem *		paraBi,
				int				past,
				const ParagraphFrame *		pf,
				const NotesReservation *	nrLine,
				AppDrawingData *		add )
    {
    int			spaceBelowLineTwips= 0;
    int			lineBottom;
    int			lineHeight;
    int			frameBottom;

    int			isLast= 0;
    int			footnoteHeight;

    if  ( past == paraBi->biParaParticuleCount )
	{
	if  ( paraBi->biParaInTable )
	    {
	    if  ( paraBi->biNumberInParent ==
					paraBi->biParent->biChildCount- 1 )
		{ isLast= 1;	}
	    }

	if  ( ! isLast )
	    {
	    spaceBelowLineTwips= paraBi->biParaSpaceBelowLinesTwips;
	    spaceBelowLineTwips += paraBi->biParaSpaceAfterTwips;
	    }
	}

    lineBottom= lp->lpPageYTwips+ tl->tlLineHeightTwips+ spaceBelowLineTwips;
    lineHeight= lineBottom- lpTop->lpPageYTwips;

    footnoteHeight= nrLine->nrFtnsepHeight+ nrLine->nrFootnoteHeight;
    frameBottom= pf->pfPageY1Twips- footnoteHeight;

    if  ( pf->pfHasBottom				&&
	  lineBottom > frameBottom			&&
	  lineHeight < pf->pfPageHigh			&&
	  ( pf->pfStripHigh < 0			||
	    lineHeight < pf->pfStripHigh	)	)
	{
	lp->lpPageYTwips= pf->pfPageY1Twips+ 1;
	lp->lpAtTopOfColumn= 0;
	return 1;
	}

    lp->lpPageYTwips += tl->tlLineSpacingTwips;
    lp->lpAtTopOfColumn= 0;

    if  ( past == paraBi->biParaParticuleCount /* && ! isLast */ )
	{
	spaceBelowLineTwips= paraBi->biParaSpaceBelowLinesTwips;

	lp->lpPageYTwips += spaceBelowLineTwips;
	lp->lpAtTopOfColumn= 0;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*  Place a line on the page without reformatting its text. Just the	*/
/*  position changes, not the layout of the text inside the line.	*/
/*									*/
/************************************************************************/

static int docPsPlace_Line(	TextLine *			resTl,
				NotesReservation *		pNrLine,
				const BufferItem *		paraBi,
				int				part,
				ParticuleData *			pd,
				const ParagraphLayoutContext *	plc,
				const ParagraphFrame *		pf,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				const ScreenLayout *		sl )
    {
    BufferDocument *		bd= plc->plcBd;
    AppDrawingData *		add= plc->plcAdd;

    int				accepted;
    int				res;
    int				i;

    TextParticule *		tp= paraBi->biParaParticules+ part;

    TextLine			workTl;
    TextLine *			tl= &(workTl);

    LayoutPosition		lp;
    const SectionProperties *	sp;
    const DocumentGeometry *	dg;

    const int			fromLinePos= 0;

    NotesReservation		nrLine;

    lp= *lpTop;
    sp= &(paraBi->biParent->biParent->biParent)->biSectProperties;
    dg= &(sp->spDocumentGeometry);

    workTl= *resTl;
    /*
    tl->tlStroff= tp->tpStroff;
    tl->tlFirstParticule= part;
    */

    docInitNotesReservation( &nrLine );

    /*  1  */
    docPsAboveLine( &lp, fromLinePos, paraBi, part, dg, add );

    tl->tlTopPosition= lp;
    accepted= tl->tlParticuleCount;

    if  ( accepted < 1 )
	{ LDEB(accepted); docListItem(0,paraBi); return -1; }

    if  ( docLayoutCollectParaFootnoteHeight( &nrLine,
					    tl->tlTopPosition.lpPage,
					    tl->tlTopPosition.lpColumn,
					    bd, paraBi, tl->tlFirstParticule,
					    tl->tlFirstParticule+ accepted ) )
	{ LDEB(1); return -1;	}

    res= docPsBelowLine( lpTop, &lp, tl, paraBi, part+ accepted,
							pf, &nrLine, add );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ *lpBottom= lp; return 0; }

    if  ( sl && sl->slLayoutLine					&&
	  (*sl->slLayoutLine)( tl, paraBi, part, accepted, add,
							pf, pd ) < 0	)
	{ LDEB(accepted); return -1;	}

    tl->tlX0Pixels= pf->pfX0Pixels;
    tl->tlX1Pixels= pf->pfX1Pixels;

    for ( i= 0; i < accepted; tp++, i++ )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    DocumentField *	df;

	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

	    df->dfPage= lp.lpPage;
	    }
	}

    *lpBottom= lp;
    *pNrLine= nrLine;
    *resTl= *tl;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*  Place the lines on the page without reformatting their text. Just	*/
/*  the positions of the lines change, not the layout of the text	*/
/*  inside the line.							*/
/*									*/
/************************************************************************/

static int docPsPlaceLines(	ParagraphLayoutPosition *	plp,
				BlockFrame *			bf,
				const ParagraphLayoutContext *	plc,
				BufferItem *			paraBi )
    {
    ParagraphStripPosition *	psp= &(plp->plpProgress);
    int				part= psp->pspPart;

    int				done= 0;
    int				line= psp->pspLine;

    TextLine *			tl= paraBi->biParaLines+ line;

    ParticuleData *		pd= (ParticuleData *)0;

    LayoutPosition		lp;

    lp= plp->plpPos;

    while( part < paraBi->biParaParticuleCount )
	{
	NotesReservation	nrLine;
	int			accepted;
	TextLine		boxLine;

	DocumentRectangle *	drChanged= plc->plcChangedRectanglePixels;

	docInitNotesReservation( &nrLine );

	if  ( line >= paraBi->biParaLineCount )
	    { done= paraBi->biParaParticuleCount- part; break;	}

	boxLine= *tl;

	accepted= docPsPlace_Line( &boxLine, &nrLine,
				    paraBi, part, pd, plc,
				    &(plp->plpFormattingFrame),
				    &lp, &lp, &(plc->plcScreenLayout) );

	if  ( accepted < 0 )
	    { LLDEB(line,accepted); return -1; }

	docLayoutReserveNoteHeight( &(plp->plpFormattingFrame), bf, &nrLine );

	if  ( drChanged	)
	    {
	    DocumentRectangle	drTl;
	    DocumentRectangle	drBox;

	    drTl.drX0= tl->tlX0Pixels;
	    drTl.drX1= tl->tlX1Pixels;
	    drTl.drY0= TL_TOP_PIXELS( plc->plcAdd, tl );
	    drTl.drY1= TL_BELOW_PIXELS( plc->plcAdd, tl );

	    drBox.drX0= boxLine.tlX0Pixels;
	    drBox.drX1= boxLine.tlX1Pixels;
	    drBox.drY0= TL_TOP_PIXELS( plc->plcAdd, &boxLine );
	    drBox.drY1= TL_BELOW_PIXELS( plc->plcAdd, &boxLine );

	    if  ( drTl.drY0 != drBox.drY0	||
		  drTl.drX0 != drBox.drX0	)
		{
		docUnionRectangle( drChanged, drChanged, &drTl );
		docUnionRectangle( drChanged, drChanged, &drBox );
		}
	    }

	if  ( accepted == 0 )
	    { break;	}

	*tl= boxLine;

	part += accepted; done += accepted; line++; tl++;

	if  ( boxLine.tlHasPageBreak )
	    {
	    lp.lpPageYTwips= plp->plpFormattingFrame.pfPageY1Twips+ 1;
	    lp.lpAtTopOfColumn= 0;
	    break;
	    }
	}

    psp->pspLine= line;
    plp->plpPos= lp;

    return done;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*									*/
/*  1)  Cope with the output of sgmls: Ignore enormous space before's	*/
/*	in footers.							*/
/*									*/
/************************************************************************/

static int docPsLayout_Line(	TextLine *			resTl,
				NotesReservation *		pNrLine,
				int				fromLinePos,
				const BufferItem *		paraBi,
				int				part,
				ParticuleData *			pd,
				const ParagraphLayoutContext *	plc,
				const ParagraphFrame *		pf,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				const ScreenLayout *		sl )
    {
    BufferDocument *		bd= plc->plcBd;
    AppDrawingData *		add= plc->plcAdd;

    const DocumentProperties *	dp= &(bd->bdProperties);
    int				accepted;
    int				res;
    int				i;

    TextParticule *		tp= paraBi->biParaParticules+ part;

    TextLine			workTl;
    TextLine *			tl= &(workTl);

    LayoutPosition		lp;
    const SectionProperties *	sp;
    const DocumentGeometry *	dg;

    NotesReservation		nrLine;

    lp= *lpTop;
    sp= &(paraBi->biParent->biParent->biParent)->biSectProperties;
    dg= &(sp->spDocumentGeometry);

    docInitTextLine( tl );
    tl->tlStroff= tp->tpStroff;
    tl->tlFirstParticule= part;

    docInitNotesReservation( &nrLine );

    /*  1  */
    docPsAboveLine( &lp, fromLinePos, paraBi, part, dg, add );

    tl->tlTopPosition= lp;
    accepted= docLayoutLineBox( tl, paraBi, part,
			    dp->dpTabIntervalTwips, &dp->dpFontList,
			    &(add->addPhysicalFontList), tp, pd, pf );

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    if  ( docLayoutCollectParaFootnoteHeight( &nrLine,
					tl->tlTopPosition.lpPage,
					tl->tlTopPosition.lpColumn,
					bd, paraBi, tl->tlFirstParticule,
					tl->tlFirstParticule+ accepted ) )
	{ LDEB(1); return -1;	}

    res= docPsBelowLine( lpTop, &lp, tl, paraBi, part+ accepted,
							pf, &nrLine, add );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ *lpBottom= lp; return 0; }

    /* above
    tl->tlStroff= tp->tpStroff;
    tl->tlFirstParticule= part;
    */
    tl->tlStrlen=
	    tp[accepted-1].tpStroff+ tp[accepted-1].tpStrlen- tp->tpStroff;
    tl->tlParticuleCount= accepted;

    if  ( sl && sl->slLayoutLine					&&
	  (*sl->slLayoutLine)( tl, paraBi, part, accepted, add,
							pf, pd ) < 0	)
	{ LDEB(accepted); return -1;	}

    tl->tlX0Pixels= pf->pfX0Pixels;
    tl->tlX1Pixels= pf->pfX1Pixels;

    for ( i= 0; i < accepted; tp++, i++ )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    DocumentField *	df;

	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

	    df->dfPage= lp.lpPage;
	    }
	}

    *lpBottom= lp;
    *pNrLine= nrLine;
    *resTl= *tl;

    return accepted;
    }

static int docLayoutLines(	ParagraphLayoutPosition *	plp,
				BlockFrame *			bf,
				const ParagraphLayoutContext *	plc,
				BufferItem *			paraBi )
    {
    ParagraphStripPosition *	psp= &(plp->plpProgress);
    int				part= psp->pspPart;

    int				done= 0;
    int				line= psp->pspLine;


    ParticuleData *		pd;

    LayoutPosition		lp;

    lp= plp->plpPos;

    if  ( docPsClaimParticuleData( paraBi, &pd ) )
	{ LDEB(paraBi->biParaParticuleCount); return -1;	}

    while( part < paraBi->biParaParticuleCount )
	{
	int			accepted;
	TextLine		boxLine;
	TextLine *		tl;
	const int		fromLinePos= 0;

	NotesReservation	nrLine;

	DocumentRectangle *	drChanged= plc->plcChangedRectanglePixels;

	docInitNotesReservation( &nrLine );

	accepted= docPsLayout_Line( &boxLine, &nrLine,
				fromLinePos, paraBi, part, pd, plc,
				&(plp->plpFormattingFrame), &lp, &lp,
				&(plc->plcScreenLayout) );

	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	if  ( accepted == 0 )
	    { break;	}

	if  ( drChanged	)
	    {
	    DocumentRectangle	drBox;

	    drBox.drX0= boxLine.tlX0Pixels;
	    drBox.drX1= boxLine.tlX1Pixels;
	    drBox.drY0= TL_TOP_PIXELS( plc->plcAdd, &boxLine );
	    drBox.drY1= TL_BELOW_PIXELS( plc->plcAdd, &boxLine );

	    docUnionRectangle( drChanged, drChanged, &drBox );
	    }

	docLayoutReserveNoteHeight( &(plp->plpFormattingFrame), bf, &nrLine );

	if  ( line >= paraBi->biParaLineCount )
	    {
	    tl= docInsertTextLine( paraBi, -1 );
	    if  ( ! tl )
		{ XDEB(tl); return -1;		}
	    }
	else{
	    tl= paraBi->biParaLines+ line;
	    }

	*tl= boxLine;

	part += accepted; done += accepted; line++;

	if  ( boxLine.tlHasPageBreak )
	    {
	    lp.lpPageYTwips= plp->plpFormattingFrame.pfPageY1Twips+ 1;
	    lp.lpAtTopOfColumn= 0;
	    break;
	    }
	}

    psp->pspLine= line;
    plp->plpPos= lp;

    return done;
    }

static int docLayoutParaItem(	BufferItem *			paraBi,
				BlockFrame *			bf,
				LayoutJob *			lj,
				const ParagraphLayoutContext *	plc )
    {
    ParagraphLayoutJob		plj;

    docPsBeginParagraphLayoutProgress( &plj,
			paraBi->biNumberInParent, 0, 0,
			paraBi->biNumberInParent+ 1,
			&(lj->ljPosition) );

    if  ( docLayoutParagraphs( plc, paraBi->biParent, bf, lj, &plj ) )
	{ LDEB(1); return -1;	}

    lj->ljPosition= plj.pljPos.plpPos;

    if  ( paraBi->biParaLineCount < 1 )
	{ LDEB(paraBi->biParaLineCount); docListItem(0,paraBi); return -1; }

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adjust the position of the top of a section, depending on the kind	*/
/*  of break								*/
/*									*/
/*  1)  Note that even and odd are swapped as from the users point of	*/
/*	view page numbers count from one.				*/
/*  2)  As this routine is only called for the body of the document,	*/
/*	page wraps can be performed unconditionally.			*/
/*									*/
/************************************************************************/

static void docLayoutPlaceSectTop(	BufferItem *		sectBi,
					BlockFrame *		bf,
					LayoutJob *		lj )
    {
    if  ( sectBi->biSectBreakKind != DOCsbkNONE		||
	  ( sectBi->biParent			&&
	    sectBi->biNumberInParent == 0	)	)
	{
	if  ( sectBi->biNumberInParent == 0 )
	    {
	    lj->ljPosition.lpPage= 0;
	    lj->ljPosition.lpColumn= 0;

	    docLayoutSectColumnTop( sectBi, lj->ljBd, &(lj->ljPosition), bf );
	    }
	else{
	    if  ( BF_HAS_FOOTNOTES( bf )				&&
		  docLayoutFootnotesForColumn( bf, &(lj->ljPosition), lj ) )
		{ LDEB(1); return;	}

	    docLayoutToNextColumn( sectBi, lj->ljBd, &(lj->ljPosition), bf );
	    }

	/*  1  */
	if  ( ( sectBi->biSectBreakKind == DOCsbkEVEN	&&
		! ( lj->ljPosition.lpPage % 2 )		)	||
	      ( sectBi->biSectBreakKind == DOCsbkODD	&&
		( lj->ljPosition.lpPage % 2 )		)	)
	    {
	    docLayoutToNextColumn( sectBi, lj->ljBd, &(lj->ljPosition), bf );
	    }
	}

    sectBi->biTopPosition= lj->ljPosition;

    if  ( sectBi->biNumberInParent == 0		&&
	  sectBi->biParent			)
	{ sectBi->biParent->biTopPosition= lj->ljPosition; }

    return;
    }

/************************************************************************/
/*									*/
/*  Adjust the bottom of an item to changes inside.			*/
/*									*/
/************************************************************************/

static void docPsPlaceAdjustBottom(	int *			pChanged,
					BufferItem *		bi,
					const LayoutPosition *	lp,
					AppDrawingData *	add,
					DocumentRectangle *	drChanged )
    {
    int			changed= 0;

    int			oldY1Pixels;
    int			newY1Pixels;

    oldY1Pixels= BI_BELOW_PIXELS( add, bi );

    if  ( bi->biBelowPosition.lpPage != lp->lpPage		||
	  bi->biBelowPosition.lpPageYTwips != lp->lpPageYTwips	)
	{ bi->biBelowPosition= *lp; changed= 1; }

    newY1Pixels= LP_YPIXELS( add, lp );

    if  ( oldY1Pixels < newY1Pixels )
	{
	if  ( drChanged					&&
	      drChanged->drY1 < newY1Pixels -1	)
	    { drChanged->drY1=  newY1Pixels -1;	}
	}

    if  ( oldY1Pixels > newY1Pixels )
	{
	if  ( drChanged							&&
	      drChanged->drY1 < oldY1Pixels- 1	)
	    { drChanged->drY1=  oldY1Pixels- 1;	}
	}

    if  ( changed )
	{ *pChanged= changed;	}

    return;
    }

static void docPsLayoutAdjustBottom(	int *			pChanged,
					BufferItem *		bi,
					const LayoutPosition *	lp,
					AppDrawingData *	add,
					DocumentRectangle *	drChanged )
    {
    docPsPlaceAdjustBottom( pChanged, bi, lp, add, drChanged );

    *pChanged= 1;
    }

/************************************************************************/
/*									*/
/*  Place successive items, after the predecessor.			*/
/*									*/
/************************************************************************/

static int docLayoutPlaceChildren(
				BufferItem *			parentBi,
				int				from,
				BlockFrame *			bf,
				LayoutJob *			lj,
				const ParagraphLayoutContext *	plc )
    {
    int				i;
    int				changed;
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;

    BufferDocument *		bd= lj->ljBd;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const NotesProperties *	npEndnotes= &(dp->dpEndnoteProperties);

    for ( i= from; i < parentBi->biChildCount; i++ )
	{
	BufferItem *	child= parentBi->biChildren[i];

	child->biTopPosition= lj->ljPosition;

	switch( child->biLevel )
	    {
	    case DOClevSECT:
		docLayoutPlaceSectTop( child, bf, lj );

		if  ( docLayoutPlaceChildren( child, 0, bf, lj, plc ) )
		    { LDEB(1); return -1;	}

		if  ( child->biInExternalItem == DOCinBODY		&&
		      npEndnotes->npPosition == DPftnPOS_SECT_END	)
		    {
		    if  ( docLayoutEndnotesForSection(
					    child->biNumberInParent, bf, lj ) )
			{ LDEB(child->biNumberInParent); return -1;	}
		    }

		break;

	    case DOClevDOC:
	    rowAsGroup:
		if  ( docLayoutPlaceChildren( child, 0, bf, lj, plc ) )
		    { LDEB(1); return -1;	}
		break;

	    case DOClevCELL:
		{
		ParagraphLayoutJob	plj;

		docPsBeginParagraphLayoutProgress( &plj,
						0, 0, 0, child->biChildCount,
						&(lj->ljPosition) );

		if  ( docLayoutParagraphs( plc, child, bf, lj, &plj ) )
		    { LDEB(1); return -1;	}

		lj->ljPosition= plj.pljPos.plpPos;
		}
		break;

	    case DOClevROW:
		if  ( ! child->biRowHasTableParagraphs )
		    { goto rowAsGroup;	}

		if  ( docLayoutRowItem( child, bf, lj, plc ) )
		    { LDEB(1); return -1;	}

		break;

	    case DOClevPARA:
		if  ( child->biParaLineCount == 0 )
		    { break;	}

		if  ( docLayoutParaItem( child, bf, lj, plc ) )
		    { LDEB(1); return -1;	}

		break;

	    default:
		LDEB(parentBi->biLevel); return -1;
	    }

	docPsPlaceAdjustBottom( &changed, child,
				    &(lj->ljPosition), lj->ljAdd, drChanged );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Redo the layout of a section because one of its external items	*/
/*  changed size.							*/
/*									*/
/************************************************************************/

static int docRedoBodyLayout(		BufferItem *		bodyBi,
					const LayoutJob *	ljRef )
    {
    LayoutJob			bodyLj;

    if  ( bodyBi->biInExternalItem != DOCinBODY )
	{ LDEB(bodyBi->biInExternalItem);	}

    bodyLj= *ljRef;
    bodyLj.ljAdd= ljRef->ljAdd;
    bodyLj.ljBd= ljRef->ljBd;
    bodyLj.ljChangedItem= bodyBi;
    bodyLj.ljPosition= bodyBi->biTopPosition;

    if  ( docLayoutItemAndParents( bodyBi, &bodyLj ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

static int docRedoBodyItemLayout(	BufferItem *		bi,
					LayoutJob *		lj )

    {
    BufferDocument *		bd= lj->ljBd;

    ExternalItem *		ei;
    BufferItem *		bodyBi;

    if  ( lj->ljChangedItem->biInExternalItem != bi->biInExternalItem )
	{
	LDEB(lj->ljChangedItem->biInExternalItem);
	LDEB(bi->biInExternalItem);
	return 0;
	}

    if  ( docGetExternalItem( &ei, &bodyBi, bd, bi ) )
	{ LDEB(bi->biInExternalItem); return -1; }
    if  ( ! bodyBi )
	{ bodyBi= &(bd->bdItem);	}

    if  ( lj->ljPosition.lpPageYTwips == ei->eiY1UsedTwips )
	{ return 0;	}

    if  ( docRedoBodyLayout( bodyBi, lj ) )
	{ LDEB(bi->biInExternalItem); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adjust the document rectangle to the changed layout of the document	*/
/*									*/
/************************************************************************/

static void docLayoutAdjustDocumentRectangle( LayoutJob *		ljRef )
    {
    AppDrawingData *		add= ljRef->ljAdd;
    DocumentRectangle *		drChanged= ljRef->ljChangedRectanglePixels;

    int				extendRectangle;
    int				y1;

    y1= ljRef->ljPosition.lpPage* add->addPageStepPixels+
			add->addPaperRect.drY1- add->addPaperRect.drY0;

    extendRectangle= add->addBackRect.drY1 != y1;

    if  ( drChanged					&&
	  extendRectangle				&&
	  drChanged->drY1 < add->addBackRect.drY1 )
	{ drChanged->drY1=  add->addBackRect.drY1; }

    add->addBackRect.drY1= y1;
    add->addDocRect.drY1= y1- add->addBottomMarginPixels;

    if  ( drChanged					&&
	  extendRectangle				&&
	  drChanged->drY1 < add->addBackRect.drY1 )
	{ drChanged->drY1=  add->addBackRect.drY1; }

    return;
    }

/************************************************************************/
/*									*/
/*  Fixup routine that is called when the first child item does not	*/
/*  have the same top position as its parent.				*/
/*									*/
/************************************************************************/

static int docPsFixupParentGeometry(	BufferItem *		bi,
					BufferItem *		biParent )
    {
    LayoutPosition *	lpBi= &(bi->biTopPosition);
    LayoutPosition *	lpPa= &(biParent->biTopPosition);

    if  ( lpPa->lpPage != lpBi->lpPage			||
	  lpPa->lpPageYTwips != lpBi->lpPageYTwips	)
	{
	SSDEB(docLevelStr(biParent->biLevel),docLevelStr(bi->biLevel));
	LLDEB(lpPa->lpPage,lpBi->lpPage);
	LLDEB(lpPa->lpPageYTwips,lpBi->lpPageYTwips);

	biParent->biTopPosition= *lpBi;

	return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adjust the geometry of a parent item to changes in a child.		*/
/*									*/
/************************************************************************/

static int docAdjustParentGeometry(	BufferItem *		bi,
					BlockFrame *		bf,
					LayoutJob *		lj )
    {
    BufferDocument *		bd= lj->ljBd;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const NotesProperties *	npEndnotes= &(dp->dpEndnoteProperties);

    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    BufferItem *		parentBi;
    int				from;

    ParagraphLayoutContext	plc;

    plc.plcChangedRectanglePixels= lj->ljChangedRectanglePixels;
    plc.plcAdd= lj->ljAdd;
    plc.plcBd= lj->ljBd;

    plc.plcScreenLayout= lj->ljPlaceScreen;
    plc.plcStartParagraph= docPsPlaceParaInit;
    plc.plcLayoutLines= docPsPlaceLines;
    plc.plcAdjustBottom= docPsPlaceAdjustBottom;

    parentBi= bi->biParent;

    if  ( bi->biLevel == DOClevPARA )
	{
	ParagraphLayoutJob	plj;
	int			advanced= 0;

	docPsInitParagraphLayoutJob( &plj );

	docPsBeginParagraphLayoutProgress( &plj,
				    bi->biNumberInParent+ 1, 0, 0,
				    parentBi->biChildCount,
				    &(lj->ljPosition) );

	docPsAdvanceParagraphLayout( &advanced,
				    &(plj.pljPos0.plpProgress),
				    &(plj.pljPos0.plpProgress),
				    &(plj.pljPos.plpProgress),
				    lj->ljPosition.lpPage, parentBi );

	if  ( bi->biParaStartsOnNewPage && ! bi->biParaInTable )
	    {
	    BufferItem *	bbi= bi;

	    while( bbi->biNumberInParent == 0 )
		{
		if  ( ! bbi->biParent )
		    { break;	}

		bbi->biParent->biTopPosition= bi->biTopPosition;
		bbi= bbi->biParent;
		}
	    }

	if  ( plj.pljPos0.plpProgress.pspPara >= bi->biNumberInParent+ 1 )
	    {
	    if  ( plj.pljPos0.plpProgress.pspPara > bi->biNumberInParent+ 1 )
		{ LLDEB(plj.pljPos0.plpProgress.pspPara,bi->biNumberInParent); }

	    from= bi->biNumberInParent+ 1;
	    }
	else{
	    from= plj.pljPos0.plpProgress.pspPara;

	    if  ( from == 0 )
		{
		lj->ljPosition= parentBi->biChildren[from]->biTopPosition;
		}
	    else{
		lj->ljPosition=
			    parentBi->biChildren[from- 1]->biBelowPosition;
		}
	    }
	}
    else{ from= bi->biNumberInParent+ 1;	}

    while( parentBi )
	{
	int		changed= 0;

	switch( parentBi->biLevel )
	    {
	    case DOClevDOC:
		if  ( bi->biNumberInParent == 0			&&
		      docPsFixupParentGeometry( bi, parentBi )	)
		    { LDEB(from); from= 0; }

		if  ( from <= parentBi->biChildCount- 1 )
		    {
		    docLayoutPlaceChildren( parentBi, from, bf, lj, &plc );
		    }
		else{ lj->ljPosition= bi->biBelowPosition;		}

		if  ( parentBi->biInExternalItem == DOCinBODY		&&
		      npEndnotes->npPosition == DPftnPOS_DOC_END	)
		    {
		    if  ( docLayoutEndnotesForDocument( bf, lj ) )
			{ LDEB(1); return -1;	}
		    }

		break;

	    case DOClevSECT:
		if  ( bi->biNumberInParent == 0			&&
		      docPsFixupParentGeometry( bi, parentBi )	)
		    { LDEB(from); from= 0; }

		if  ( from <= parentBi->biChildCount- 1 )
		    {
		    docLayoutPlaceChildren( parentBi, from, bf, lj, &plc );
		    }
		else{ lj->ljPosition= bi->biBelowPosition;		}

		if  ( parentBi->biInExternalItem == DOCinBODY		&&
		      npEndnotes->npPosition == DPftnPOS_SECT_END	)
		    {
		    if  ( docLayoutEndnotesForSection(
				    parentBi->biNumberInParent, bf, lj ) )
			{ LDEB(1); return -1;	}
		    }

		break;


	    parentRowAsGroup:
		if  ( bi->biNumberInParent == 0			&&
		      docPsFixupParentGeometry( bi, parentBi )	)
		    { LDEB(from); from= 0; }

		if  ( from <= parentBi->biChildCount- 1 )
		    {
		    docLayoutPlaceChildren( parentBi, from, bf, lj, &plc );
		    }
		else{ lj->ljPosition= bi->biBelowPosition;		}

		break;

	    case DOClevCELL:
		if  ( bi->biNumberInParent == 0			&&
		      docPsFixupParentGeometry( bi, parentBi )	)
		    { LDEB(from); from= 0; }

		if  ( from <= parentBi->biChildCount- 1 )
		    {
		    ParagraphLayoutJob	plj;

		    docPsInitParagraphLayoutJob( &plj );

		    docPsBeginParagraphLayoutProgress( &plj,
					    from, 0, 0, parentBi->biChildCount,
					    &(lj->ljPosition) );

		    if  ( docLayoutParagraphs( &plc, parentBi, bf, lj, &plj ) )
			{ LDEB(1); return -1;	}

		    lj->ljPosition= plj.pljPos.plpPos;
		    }
		else{ lj->ljPosition= bi->biBelowPosition;		}

		break;

	    case DOClevROW:
		if  ( ! parentBi->biRowHasTableParagraphs )
		    { goto parentRowAsGroup;	}

		if  ( parentBi->biNumberInParent > 0	&&
		      parentBi->biParent		&&
		      parentBi->biRowKeepOnOnePage	)
		    {
		    BufferItem *	pp= parentBi->biParent;
		    BufferItem *	ch;

		    ch= pp->biChildren[parentBi->biNumberInParent- 1];

		    parentBi->biTopPosition= ch->biBelowPosition;
		    lj->ljPosition= ch->biBelowPosition;
		    }

		lj->ljPosition= parentBi->biTopPosition;

		if  ( docLayoutRowItem( parentBi, bf, lj, &plc ) )
		    { LDEB(1); return -1;	}

		break;

	    default:
		LDEB(parentBi->biLevel); return -1;
	    }

	docPsPlaceAdjustBottom( &changed, parentBi,
				    &(lj->ljPosition), lj->ljAdd, drChanged );

	if  ( ! changed )
	    { break;	}

	bi= parentBi; parentBi= bi->biParent;
	from= bi->biNumberInParent+ 1;
	}

    if  ( bi->biInExternalItem == DOCinBODY	&&
	  BF_HAS_FOOTNOTES( bf )		)
	{
	if  ( docLayoutFootnotesForColumn( bf, &(lj->ljPosition), lj ) )
	    { LDEB(1); return -1;	}
	}

    if  ( ! parentBi )
	{
	switch( bi->biInExternalItem )
	    {
	    case DOCinBODY:
		docLayoutAdjustDocumentRectangle( lj );
		break;

	    case DOCinSECT_HEADER:
	    case DOCinFIRST_HEADER:
	    case DOCinLEFT_HEADER:
	    case DOCinRIGHT_HEADER:
	    case DOCinSECT_FOOTER:
	    case DOCinFIRST_FOOTER:
	    case DOCinLEFT_FOOTER:
	    case DOCinRIGHT_FOOTER:

		if  ( bi->biLevel != DOClevSECT )
		    { LLDEB(bi->biLevel,DOClevSECT); return -1;	}
		if  ( lj->ljChangedItem->biInExternalItem == DOCinBODY )
		    { break; }

		if  ( docRedoBodyItemLayout( bi, lj ) )
		    { LDEB(1); return -1;	}

		break;

	    case DOCinFOOTNOTE:
	    case DOCinENDNOTE:

		if  ( bi->biLevel != DOClevSECT )
		    { LLDEB(bi->biLevel,DOClevSECT); return -1;	}
		if  ( lj->ljChangedItem->biInExternalItem == DOCinBODY )
		    { break; }

		if  ( docRedoBodyItemLayout( bi, lj ) )
		    { LDEB(1); return -1;	}

		break;

	    case DOCinFTNSEP:
	    case DOCinFTNSEPC:
	    case DOCinFTNCN:
	    case DOCinAFTNSEP:
	    case DOCinAFTNSEPC:
	    case DOCinAFTNCN:

		if  ( bi->biLevel != DOClevSECT )
		    { LLDEB(bi->biLevel,DOClevSECT); return -1;	}
		if  ( lj->ljChangedItem->biInExternalItem == DOCinBODY )
		    { break; }

		if  ( docRedoBodyItemLayout( bi, lj ) )
		    { LDEB(1); return -1;	}

		break;

	    default:
		LDEB(bi->biInExternalItem);
		break;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Do the layout of a section.						*/
/*									*/
/************************************************************************/

static int docLayoutSectItem(		BufferItem *		sectBi,
					BlockFrame *		bf,
					LayoutJob *		lj )
    {
    BufferDocument *		bd= lj->ljBd;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const NotesProperties *	npEndnotes= &(dp->dpEndnoteProperties);

    DocumentNote *		dn;

    const SectionProperties *	sp= &(sectBi->biSectProperties);
    const DocumentGeometry *	dgSect= &(sp->spDocumentGeometry);

    int				i;

    /**/

    if  ( docSectHeaderFooterPrelayout( sectBi, lj ) )
	{ LDEB(1); return -1;	}

    /**/

    dn= bd->bdNotes;
    for ( i= 0; i < bd->bdNoteCount; dn++, i++ )
	{
	ExternalItem *	ei= (&dn->dnExternalItem);

	if  ( dn->dnSectNr != sectBi->biNumberInParent )
	    { continue;	}
	if  ( ! ei->eiItem )
	    { continue;	}

	if  ( docExternalItemPrelayout( ei, dgSect, lj ) )
	    { LDEB(1); return -1;	}
	}

    /**/

    docLayoutPlaceSectTop( sectBi, bf, lj );

    for ( i= 0; i < sectBi->biChildCount; i++ )
	{
	if  ( docLayoutItemImplementation( sectBi->biChildren[i], bf, lj ) )
	    { LDEB(i); return -1;	}
	}

    /**/

    if  ( sectBi->biInExternalItem == DOCinBODY		&&
	  npEndnotes->npPosition == DPftnPOS_SECT_END	)
	{
	if  ( docLayoutEndnotesForSection( sectBi->biNumberInParent, bf, lj ) )
	    { LDEB(sectBi->biNumberInParent); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Do the layout of the document root item.				*/
/*									*/
/************************************************************************/

static int docLayoutDocItem(		BufferItem *		docBi,
					BlockFrame *		bf,
					LayoutJob *		lj )
    {
    BufferDocument *		bd= lj->ljBd;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const NotesProperties *	npEndnotes= &(dp->dpEndnoteProperties);
    const DocumentGeometry *	dgDoc= &(dp->dpGeometry);

    int				i;

    if  ( docExternalItemPrelayout( &(bd->bdEiFtnsep), dgDoc, lj ) )
	{ LDEB(1); return -1;	}
    if  ( docExternalItemPrelayout( &(bd->bdEiFtnsepc), dgDoc, lj ) )
	{ LDEB(1); return -1;	}
    if  ( docExternalItemPrelayout( &(bd->bdEiFtncn), dgDoc, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docExternalItemPrelayout( &(bd->bdEiAftnsep), dgDoc, lj ) )
	{ LDEB(1); return -1;	}
    if  ( docExternalItemPrelayout( &(bd->bdEiAftnsepc), dgDoc, lj ) )
	{ LDEB(1); return -1;	}
    if  ( docExternalItemPrelayout( &(bd->bdEiAftncn), dgDoc, lj ) )
	{ LDEB(1); return -1;	}

    for ( i= 0; i < docBi->biChildCount; i++ )
	{
	if  ( docLayoutItemImplementation( docBi->biChildren[i], bf, lj ) )
	    { LDEB(1); return -1;	}
	}

    if  ( docBi->biInExternalItem == DOCinBODY		&&
	  npEndnotes->npPosition == DPftnPOS_DOC_END	)
	{
	if  ( docLayoutEndnotesForDocument( bf, lj ) )
	    { LDEB(1); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Do the layout of a document item.					*/
/*									*/
/*  This is the main entry poin of the formatter.			*/
/*									*/
/************************************************************************/

int docLayoutItemImplementation(	BufferItem *		bi,
					BlockFrame *		bf,
					LayoutJob *		lj )
    {
    BufferDocument *		bd= lj->ljBd;
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    int				i;
    int				y1;

    ParagraphLayoutContext	plc;

    plc.plcChangedRectanglePixels= lj->ljChangedRectanglePixels;
    plc.plcAdd= lj->ljAdd;
    plc.plcBd= lj->ljBd;

    plc.plcScreenLayout= lj->ljLayoutScreen;
    plc.plcStartParagraph= docLayoutParaInit;
    plc.plcLayoutLines= docLayoutLines;
    plc.plcAdjustBottom= docPsLayoutAdjustBottom;

    y1= BI_BELOW_PIXELS( lj->ljAdd, bi )- 1;

    i= BI_TOP_PIXELS( lj->ljAdd, bi );

    if  ( drChanged				&&
	  drChanged->drY0 > i	)
	{ drChanged->drY0=  i;	}

    bi->biTopPosition= lj->ljPosition;

    if  ( drChanged )
	{
	i= BI_TOP_PIXELS( lj->ljAdd, bi );

	if  ( drChanged->drY0 > i )
	    { drChanged->drY0=  i;			}
	if  ( drChanged->drY1 < y1 )
	    { drChanged->drY1=  y1;	}
	}

    switch( bi->biLevel )
	{
	case DOClevDOC:

	    if  ( docLayoutDocItem( bi, bf, lj ) )
		{ LDEB(1); return -1;	}
	    break;

	rowAsGroup:
	sectAsGroup:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docLayoutItemImplementation( bi->biChildren[i], bf, lj ) )
		    { LDEB(1); return -1;	}
		}
	    break;

	case DOClevCELL:
	    {
	    ParagraphLayoutJob	plj;

	    docPsInitParagraphLayoutJob( &plj );

	    docPsBeginParagraphLayoutProgress( &plj,
					    0, 0, 0, bi->biChildCount,
					    &(lj->ljPosition) );

	    if  ( docLayoutParagraphs( &plc, bi, bf, lj, &plj ) )
		{ LDEB(1); return -1;	}

	    lj->ljPosition= plj.pljPos.plpPos;

	    docFieldRefreshFlags( bi, bd );
	    }
	    break;

	case DOClevSECT:
	    if  ( ! bi->biParent )
		{ goto sectAsGroup;	}

	    if  ( docLayoutSectItem( bi, bf, lj ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{ goto rowAsGroup;	}

	    docFieldRefreshFlags( bi, bd );

	    if  ( docLayoutRowItem( bi, bf, lj, &plc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevPARA:
	    docFieldRefreshFlags( bi, bd );

	    if  ( docLayoutParaItem( bi, bf, lj, &plc ) )
		{ LDEB(1); return -1;	}

	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    bi->biBelowPosition= lj->ljPosition;

    y1= BI_BELOW_PIXELS( lj->ljAdd, bi )- 1;

    if  ( drChanged		&&
	  drChanged->drY1 < y1	)
	{ drChanged->drY1=  y1;	}

    return 0;
    }

int docLayoutItemAndParents(	BufferItem *		bi,
				LayoutJob *		lj )
    {
    BufferDocument *		bd= lj->ljBd;
    BlockFrame			bf;

    docLayoutInitBlockFrame( &bf );

    if  ( bi->biLevel != DOClevDOC )
	{
	docBlockFrameTwips( &bf, bi, bd,
			    lj->ljPosition.lpPage, lj->ljPosition.lpColumn );

	if  ( bi->biInExternalItem == DOCinBODY		&&
	      docPrevParagraph( bi )			)
	    {
	    DocumentPosition		dpHere;

	    if  ( docFirstPosition( &dpHere, bi ) )
		{ LDEB(1); return -1;	}

	    if  ( docCollectFootnotesForColumn( &bf, &dpHere, lj ) )
		{ LDEB(lj->ljPosition.lpPage); return -1;	}
	    }
	}

    if  ( docLayoutItemImplementation( bi, &bf, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docAdjustParentGeometry( bi, &bf, lj ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Finish formatting a paragraph, either because additional lines are	*/
/*  to be added to the paragraph, or because its tail is to be shifted	*/
/*  (up or) down after some lines have been reformatted.		*/
/*									*/
/************************************************************************/

static int docLayoutFinishParaAdjust(
				ParagraphLayoutPosition *	plp,
				BlockFrame *			bf,
				const ParagraphLayoutContext *	plc,
				const LayoutJob *		lj,
				BufferItem *			paraBi )
    {
    int				done= 0;
    ParagraphStripPosition *	psp= &(plp->plpProgress);

    while( psp->pspPart < paraBi->biParaParticuleCount )
	{
	int	accepted;

	accepted= (*plc->plcLayoutLines)( plp, bf, plc, paraBi );

	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	plp->plpProgress.pspPart += accepted;
	done += accepted;

	/*  1  */
	if  ( psp->pspPart < paraBi->biParaParticuleCount		&&
	      ( accepted > 0 || ! plp->plpPos.lpAtTopOfColumn )		)
	    {
	    if  ( BF_HAS_FOOTNOTES( bf )				&&
		  docLayoutFootnotesForColumn( bf, &(plp->plpPos), lj )	)
		{ LDEB(1); return -1;	}

	    docLayoutToNextColumn( paraBi, plc->plcBd, &(plp->plpPos), bf );

	    if  ( psp->pspPart == 0 )
		{ paraBi->biTopPosition= plp->plpPos;	}
	    }
	}

    return done;
    }

/************************************************************************/
/*									*/
/*  Adjust the layout of a paragraph after a modification.		*/
/*									*/
/*  0)  Do not start with the first line of the page.			*/
/*  1)  Initialisation for paragraph formatting.			*/
/*  2)  Recalculate the layout of the first line that was changed.	*/
/*  3)  Did not fit on this page.. Try the next one.			*/
/*  4)  Layout subsequent lines of the paragraph, until the layout	*/
/*	process reaches the point outside the modified region where it	*/
/*	starts to cut lines at the same point where they were		*/
/*	originally cut.							*/
/*  5)  More lines than the paragraph originally contained are needed.	*/
/*	Just layout the rest.						*/
/*  6)  We are beyond the end of the modified region, and the new line	*/
/*	is to start where the old line starts. Just place the		*/
/*	subsequent lines from here.					*/
/*  7)  Recalculate the layout of a subsequent line that was changed.	*/
/*  8)  Remove superfluous lines from the administration.		*/
/*  9)  If the paragraph has changed height.. Adjust the redraw region.	*/
/*  10) Adjust the geometry of the rest of the document to the changes.	*/
/*									*/
/************************************************************************/

/*  3  */
static int docPsLayoutOneLine(	TextLine *			tl,
				BufferItem *			paraBi,
				ParagraphLayoutPosition *	plp,
				ParticuleData *			pd,
				const ParagraphLayoutContext *	plc,
				LayoutJob *			lj,
				DocumentRectangle *		drChanged,
				BlockFrame *			bf )
    {
    AppDrawingData *		add= plc->plcAdd;
    BufferDocument *		bd= plc->plcBd;

    ParagraphStripPosition *	psp= &(plp->plpProgress);

    const int			fromLinePos= 1;
    int				accepted;

    NotesReservation		nrLine;

    TextLine			boxLine;

    if  ( drChanged )
	{
	int	x0Pixels= TL_TOP_PIXELS( add, tl );
	int	x1Pixels= TL_BELOW_PIXELS( add, tl );

	if  ( drChanged->drY0 > x0Pixels	)
	    { drChanged->drY0=  x0Pixels;	}

	if  ( drChanged->drY1 < x1Pixels	)
	    { drChanged->drY1=  x1Pixels;	}
	}

    docInitNotesReservation( &nrLine );

    /*  2,7  */
    accepted= docPsLayout_Line( &boxLine, &nrLine, fromLinePos,
				paraBi, psp->pspPart, pd, plc,
				&(plp->plpFormattingFrame),
				&(plp->plpPos), &(plp->plpPos),
				&(lj->ljLayoutScreen) );

    if  ( accepted < 0 )
	{ LDEB(accepted); return -1;	}

    /*  3  */
    if  ( accepted == 0 )
	{
	if  ( paraBi->biInExternalItem != DOCinBODY	&&
	      paraBi->biInExternalItem != DOCinENDNOTE	&&
	      paraBi->biInExternalItem != DOCinAFTNSEP	)
	    { LDEB(paraBi->biInExternalItem);	}

	if  ( BF_HAS_FOOTNOTES( bf )					&&
	      docLayoutFootnotesForColumn( bf, &(plp->plpPos), lj )	)
	    { LDEB(1); return -1;	}

	docLayoutToNextColumn( paraBi, bd, &(plp->plpPos), bf );

	docInitNotesReservation( &nrLine );

	accepted= docPsLayout_Line(
			    &boxLine, &nrLine, fromLinePos,
			    paraBi, psp->pspPart, pd, plc,
			    &(plp->plpFormattingFrame),
			    &(plp->plpPos), &(plp->plpPos),
			    &(lj->ljLayoutScreen) );

	if  ( accepted < 1 )
	    { LDEB(accepted); return -1;	}
	}

    docLayoutReserveNoteHeight( &(plp->plpFormattingFrame), bf, &nrLine );

    if  ( drChanged )
	{
	int	x1= LP_YPIXELS( add, &(plp->plpPos) );

	if  ( drChanged->drY1 < x1	)
	    { drChanged->drY1=  x1;	}
	}

    *tl= boxLine;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Redo the layout of a paragraph after an editing action.		*/
/*									*/
/*  To avoid screen flashing, a minimalistic approach is used: As	*/
/*  little of the layout is redone, and an administration is kept of	*/
/*  what has been changed, in order to redraw a minimal screen		*/
/*  rectangle.								*/
/*									*/
/************************************************************************/

int docPsAdjustParaLayout(	BufferItem *		paraBi,
				int			fromLine,
				int			stroffShift,
				int			stroffUpto,
				LayoutJob *		lj )
    {
    BufferDocument *		bd= lj->ljBd;
    const DocumentProperties *	dp= &(bd->bdProperties);
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    AppDrawingData *		add= lj->ljAdd;

    TextLine			boxLine;
    TextLine *			tl;
    int				oneMore;

    TextParticule *		tp;

    ParticuleData *		pd;

    int				accepted;
    int				off;

    const int			bottomTwips= -1;
    const int			stripHigh= -1;

    int				paragraphHeightChanged;
    int				firstLineHeightChanged;

    ParagraphLayoutContext	plcL;
    ParagraphLayoutContext	plcP;

    ParagraphLayoutPosition	plp;

    BlockFrame			bf;

#   if 0
    LDEB(444);
    lj->ljPosition= paraBi->biTopPosition;
    if  ( docLayoutItemAndParents( paraBi, lj ) )
	{ LDEB(1); return -1;	}
    return 0;
#   endif

    plcL.plcChangedRectanglePixels= lj->ljChangedRectanglePixels;
    plcL.plcAdd= lj->ljAdd;
    plcL.plcBd= lj->ljBd;

    plcP.plcChangedRectanglePixels= lj->ljChangedRectanglePixels;
    plcP.plcAdd= lj->ljAdd;
    plcP.plcBd= lj->ljBd;

    plcL.plcScreenLayout= lj->ljLayoutScreen;
    plcL.plcStartParagraph= docLayoutParaInit;
    plcL.plcLayoutLines= docLayoutLines;
    plcL.plcAdjustBottom= docPsLayoutAdjustBottom;

    plcP.plcScreenLayout= lj->ljPlaceScreen;
    plcP.plcStartParagraph= docPsPlaceParaInit;
    plcP.plcLayoutLines= docPsPlaceLines;
    plcP.plcAdjustBottom= docPsPlaceAdjustBottom;

    if  ( docPsClaimParticuleData( paraBi, &pd ) )
	{ LDEB(paraBi->biParaParticuleCount); return -1;	}

    /*  1  */
    if  ( docLayoutOpenParaFonts( paraBi, add, dp ) )
	{ LDEB(1); return -1;	}

    /*  0  */
    tl= paraBi->biParaLines+ fromLine;
    oneMore= 0;
    if  ( fromLine > 0							&&
	  tl[-1].tlTopPosition.lpPage != tl[ 0].tlTopPosition.lpPage	)
	{ fromLine--; tl--; oneMore= 1;	}
    tp= paraBi->biParaParticules+ tl->tlFirstParticule;

    plp.plpProgress.pspPara= paraBi->biNumberInParent;
    plp.plpProgress.pspLine= fromLine;
    plp.plpProgress.pspPart= tl->tlFirstParticule;
    plp.plpPos= tl->tlTopPosition;

    lj->ljPosition= tl->tlTopPosition;

    docBlockFrameTwips( &bf, paraBi, bd,
					tl[ 0].tlTopPosition.lpPage,
					tl[ 0].tlTopPosition.lpColumn );

    if  ( paraBi->biInExternalItem == DOCinBODY	)
	{
	DocumentSelection	dsLine;

	docLineSelection( &dsLine, paraBi, fromLine );

	if  ( docCollectFootnotesForColumn( &bf, &(dsLine.dsBegin), lj ) )
	    { LDEB(lj->ljPosition.lpPage); return -1;	}
	}

    docParagraphFrameTwips( &(plp.plpFormattingFrame), &bf,
					bottomTwips, stripHigh, paraBi );

    if  ( docPsParagraphLineExtents( &(add->addPhysicalFontList), paraBi ) )
	{ LDEB(1); return -1;	}

    if  ( lj->ljLayoutScreen.slScreenFrame )
	{
	(*lj->ljLayoutScreen.slScreenFrame)( &(plp.plpFormattingFrame),
								add, paraBi );
	}

    if  ( lj->ljLayoutScreen.slStartParagraph				&&
	  (*lj->ljLayoutScreen.slStartParagraph)( paraBi, add, bd )	)
	{ LDEB(1); return -1;	}

    if  ( drChanged )
	{
	int	y0= TL_TOP_PIXELS( add, tl );

	if  ( drChanged->drX0 > add->addBackRect.drX0 )
	    { drChanged->drX0=  add->addBackRect.drX0;	}
	if  ( drChanged->drX1 < add->addBackRect.drX1 )
	    { drChanged->drX1=  add->addBackRect.drX1;	}

	if  ( drChanged->drY0 > y0 )
	    { drChanged->drY0=  y0;	}
	}

    /*  2,3  */
    boxLine= *tl;
    accepted= docPsLayoutOneLine( &boxLine, paraBi, &plp, pd, &plcL,
							lj, drChanged, &bf );

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    firstLineHeightChanged= 0;
    if  ( boxLine.tlTopPosition.lpPage != tl->tlTopPosition.lpPage	||
	  boxLine.tlTopPosition.lpPageYTwips !=
				    tl->tlTopPosition.lpPageYTwips	||
	  boxLine.tlLineHeightTwips != tl->tlLineHeightTwips		)
	{ firstLineHeightChanged= 1;	}

    *tl= boxLine;

    if  ( tl->tlHasPageBreak )
	{
	if  ( paraBi->biInExternalItem != DOCinBODY )
	    { LDEB(paraBi->biInExternalItem);			}
	else{
	    if  ( BF_HAS_FOOTNOTES( &bf )				&&
		  docLayoutFootnotesForColumn( &bf, &(plp.plpPos), lj )	)
		{ LDEB(1); return -1;	}

	    docLayoutToNextColumn( paraBi, bd, &(plp.plpPos), &bf );
	    }
	}

    off= tl->tlStroff+ tl->tlStrlen;
    plp.plpProgress.pspLine++; tl++;
    plp.plpProgress.pspPart += accepted; tp += accepted; pd += accepted;

    /*  4  */
    while( plp.plpProgress.pspPart < paraBi->biParaParticuleCount )
	{
	/*  5  */
	if  ( plp.plpProgress.pspLine >= paraBi->biParaLineCount )
	    {
	    accepted= docLayoutFinishParaAdjust( &plp, &bf, &plcL, lj, paraBi );

	    if  ( accepted < 1 )
		{ LDEB(accepted); return -1;	}

	    break;
	    }

	/*  6  */
	if  ( ! oneMore							&&
	      tl->tlStroff + stroffShift == off && off >= stroffUpto	)
	    {
	    int		ll;
	    int		partShift;

	    partShift= plp.plpProgress.pspPart- tl->tlFirstParticule;

	    for ( ll= plp.plpProgress.pspLine;
					ll < paraBi->biParaLineCount; ll++ )
		{
		paraBi->biParaLines[ll].tlFirstParticule += partShift;
		paraBi->biParaLines[ll].tlStroff += stroffShift;
		}

	    accepted= docLayoutFinishParaAdjust( &plp, &bf, &plcP, lj, paraBi );
	    if  ( accepted < 1 )
		{ LDEB(accepted); return -1;	}

	    break;
	    }

	/*  7,3  */
	boxLine= *tl;
	accepted= docPsLayoutOneLine( &boxLine, paraBi, &plp, pd, 
						&plcL, lj, drChanged, &bf );

	if  ( accepted < 1 )
	    { LDEB(accepted); return -1;	}

	*tl= boxLine;

	if  ( tl->tlHasPageBreak )
	    {
	    if  ( paraBi->biInExternalItem != DOCinBODY )
		{ LDEB(paraBi->biInExternalItem);		}
	    else{
		if  ( BF_HAS_FOOTNOTES( &bf )				&&
		      docLayoutFootnotesForColumn( &bf, &(plp.plpPos), lj ) )
		    { LDEB(1); return -1;	}

		docLayoutToNextColumn( paraBi, lj->ljBd, &(plp.plpPos), &bf );
		}
	    }

	off= tl->tlStroff+ tl->tlStrlen;
	plp.plpProgress.pspLine++; tl++; oneMore= 0;
	plp.plpProgress.pspPart += accepted; tp += accepted; pd += accepted;
	}

    /*  8  */
    if  ( plp.plpProgress.pspLine < paraBi->biParaLineCount )
	{
	docDeleteLines( paraBi, plp.plpProgress.pspLine,
			    paraBi->biParaLineCount- plp.plpProgress.pspLine );
	}

    if  ( plp.plpProgress.pspPart != paraBi->biParaParticuleCount )
	{
	LLDEB(plp.plpProgress.pspPart,paraBi->biParaParticuleCount);
	return -1;
	}
    if  ( plp.plpProgress.pspLine !=  paraBi->biParaLineCount )
	{
	LLDEB(plp.plpProgress.pspLine,paraBi->biParaLineCount);
	return -1;
	}

    /*  9  */
    paragraphHeightChanged= 0;

    docPsPlaceAdjustBottom( &paragraphHeightChanged, paraBi,
					&(plp.plpPos), lj->ljAdd, drChanged );

    lj->ljPosition= plp.plpPos;

    if  ( firstLineHeightChanged || paragraphHeightChanged )
	{
	if  ( firstLineHeightChanged		&&
	      paraBi->biNumberInParent > 0	)
	    {
	    paraBi= paraBi->biParent->biChildren[paraBi->biNumberInParent- 1];
	    lj->ljPosition= paraBi->biBelowPosition;
	    }

	/*  10  */
	if  ( docAdjustParentGeometry( paraBi, &bf, lj ) )
	    { LDEB(1); return -1;	}
	}

    return 0;
    }

