%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\begin{code}
module ArgumentDefaults ( get_default_flag,
                        ) where
import Monad ( liftM, liftM2 )
import Maybe ( catMaybes )

import DarcsArguments
import RepoPrefs

get_default_flag :: String -> DarcsOption -> IO [DarcsFlag]
\end{code}

\paragraph{defaults}\label{defaults}

Default values for darcs commands can be configured on a per-repository
basis by editing (and possibly creating) the \verb!_darcs/prefs/defaults!
file.  Each line of this file has the following form:
\begin{verbatim}
COMMAND FLAG VALUE
\end{verbatim}
where \verb!COMMAND! is either the name of the command to which the default
applies, or \verb!ALL! to indicate that the default applies to all commands
accepting that flag.  The \verb!FLAG! term is the name of the long argument
option without the ``\verb!--!'', i.e.\ \verb!verbose! rather than
\verb!--verbose!.  Finally, the \verb!VALUE! option can be omitted if the
flag is one such as \verb!verbose! that doesn't involve a value.
Each line only takes one flag.  To set multiple defaults for the same
command (or for \verb!ALL! commands), use multiple lines.

\begin{tabular}{ll}
{\tt \verb!~/.darcs/defaults!} & {\tt provides defaults for this user account} \\
{\tt \verb!project/_darcs/pref/defaults!} & {\tt provides defaults for one project, overrules changes per user} \\
\end{tabular}

For example, if your system clock is bizarre, you could instruct darcs to
always ignore the file modification times by adding the following line to
your \verb!_darcs/prefs/defaults! file.  (Note that this would have to be
done for each repository!)
\begin{verbatim}
ALL ignore-times
\end{verbatim}

If you never want to run a test when recording to a particular repository
(but still want to do so when running check on that repo), and like to name
all your patches ``Stupid patch'', you could use the following:
\begin{verbatim}
record no-test
record patch-name Stupid patch
\end{verbatim}

There are some options which are meant specifically for use in
\verb!_darcs/prefs/defaults!. One of them is \verb!--disable!. As the name
suggests, this option will disable every command that got it as argument. So,
if you are afraid that you could damage your repositories by inadvertent use of
a command like amend-record, add the following line to
\verb!_darcs/prefs/defaults!:
\begin{verbatim}
amend-record disable
\end{verbatim}

Also, a global preferences file can be created with the name
\verb!.darcs/defaults! in your home directory. Options present there will
be added to the repository-specific preferences.
If they conflict with repository-specific options, the repository-specific
ones will take precedence.

\begin{code}
get_default_flag com flag = do
    defs <- get_defaults
    case gda com flag defs of
      [] -> return $ gda "ALL" flag defs
      f -> return $ f

gda :: String -> DarcsOption -> [(String,String,String)] -> [DarcsFlag]
gda com opt all_defs = gda' $ filter (\ (c,_,_)-> c == com) all_defs
    where gda' defs = concatMap (choose_option opt) defs
          choose_option (DarcsNoArgOption _ fls c _) (_,f,_)
              | f `elem` fls = [c]
          choose_option (DarcsArgOption _ fls c _ _) (_,f,d)
              | f `elem` fls = [c d]
          choose_option (DarcsMultipleChoiceOption os) p =
              concatMap (\o -> choose_option o p) os
          choose_option _ _ = []

default_content :: IO [String] -> IO [(String,String,String)]
default_content = liftM (catMaybes . map (doline.words))
    where doline (c:a:r) = Just (c, drop_dashdash a, unwords r)
          doline _ = Nothing
          drop_dashdash ('-':'-':a) = a
          drop_dashdash a = a

get_defaults :: IO [(String,String,String)]
get_defaults = default_content  $
               liftM2 (++) (get_preflist "defaults") (get_global "defaults")
\end{code}

