/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: util.c,v 1.6 2001/11/07 02:03:37 martine Exp $
 */

#include <gtk/gtk.h>
#include <glib.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>

#include "util.h"

#include "../pixmaps/question.xpm"
#include "../pixmaps/goat.xpm"
#include "../pixmaps/web.xpm"

char*
timet_to_ljdate(time_t unixtime) {
	char *ljdate = g_new(char, 20);
	struct tm *time = localtime(&unixtime);
	snprintf(ljdate, 20, "%04d-%02d-%02d %02d:%02d:%02d",
			 time->tm_year+1900, time->tm_mon+1, time->tm_mday,
			 time->tm_hour, time->tm_min, time->tm_sec);
	ljdate[19] = 0;
	return ljdate;
}

void hash_print_cb(gpointer key, gpointer value, gpointer data) {
	printf("  %p, %p:", key, value);
	if (key > (gpointer)0x8000000 || value > (gpointer)0x8000000)
		printf("%s -> %s\n", (char*)key, (char*)value);
	else
		printf(" [x]\n");
}

void hash_print(GHashTable* hash) {
	g_print("hash %p:\n", hash);
	g_hash_table_foreach(hash, hash_print_cb, NULL);
	g_print("done.\n");
}

gint free_all_hash_cb(gpointer key, gpointer val, gpointer user_data) {
	g_free(key); g_free(val);
	return TRUE;
}

void hash_destroy(GHashTable* hash) {
	if (hash == NULL) return;
	g_hash_table_foreach_remove(hash, free_all_hash_cb, NULL);
	g_hash_table_destroy(hash);
}

void default_column_clicked(GtkCList *list, gint col, gpointer d) {
	gtk_clist_set_sort_column(list, col);
	gtk_clist_sort(list);
}

GtkWidget* lj_table_new(int rows, int cols) {
	GtkWidget *table;
	table = gtk_table_new(rows, cols, FALSE);
	return table;
}
void lj_table_label(GtkTable *table, int row, char *text) {
	GtkWidget *label = gtk_label_new(text);
	gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.5f);
	gtk_table_attach(table, GTK_WIDGET(label), 
			0, 1, row, row+1, GTK_FILL, 0, 2, 2);
}
void lj_table_content(GtkTable *table, int row, GtkWidget *content) {
	gtk_table_attach(table, GTK_WIDGET(content), 
			1, 2, row, row+1, GTK_EXPAND|GTK_FILL, 0, 2, 2);
}
void lj_table_fillrow(GtkTable *table, int row, GtkWidget *content) {
	gtk_table_attach(table, GTK_WIDGET(content), 
			0, 2, row, row+1, GTK_EXPAND|GTK_FILL, 0, 2, 2);
}

char* field_strdup_printf(char *fmt, char* fields[]) {
	GString *out;
	char *src = fmt;
	int index;
	char *ret;

	out = g_string_sized_new(256);

	while (*src != 0) {
		/* scan forward for a "%x" item */
		for ( ; *src != 0; src++) {
			if ((*src == '%') && (isdigit((int)*(src+1))))
				break;
			g_string_append_c(out, *src);
		}

		if (*src == 0) break;

		/* now read the %x item */
		src++;
		if (*src == 0) break; /* damnit, a % with no following char. */
		if (*src >= '0' && *src <= '9') {
			index = *src - '0';
			if (fields[index]) 
				g_string_append(out, fields[index-1]); /* 1-based index. */
		} else if (*src == '%') {
			/* %% -> % */
			g_string_append_c(out, *src);
		}
		src++;
	}
	ret = out->str;
	g_string_free(out, FALSE);
	return ret;
}

void
lj_dialog_init(GtkWidget *dlg, GtkWidget *parent, gchar *title, int width, int height) {
	if (parent)
		gtk_window_set_transient_for(GTK_WINDOW(dlg), GTK_WINDOW(parent));
	gtk_window_set_title(GTK_WINDOW(dlg), title);
	if (width > 0) {
		if (height > 0) {
			gtk_window_set_default_size(GTK_WINDOW(dlg), width, height);
		} else {
			gtk_window_set_default_size(GTK_WINDOW(dlg), width, 0.618*width);
		}
	}
	lj_win_set_icon(dlg);

	/* make window fixed-size and modal */
	gtk_window_set_policy(GTK_WINDOW(dlg), FALSE, TRUE, TRUE);
	gtk_window_set_modal(GTK_WINDOW(dlg), TRUE);	

	/* sort of a hack, to make the buttons justify the way i like. */
	GTK_BOX(GTK_DIALOG(dlg)->action_area)->homogeneous = 0;
}

GtkWidget*
lj_dialog_new(GtkWidget *parent, gchar *title, int width, int height) {
	GtkWidget *dlg;
	dlg = gtk_dialog_new();
	lj_dialog_init(dlg, parent, title, width, height);
	return dlg;
}

void
lj_dialog_set_contents(GtkWidget *dlg, GtkWidget *contents) {
	GtkWidget *vbox;
	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
	gtk_box_pack_start(GTK_BOX(vbox), contents, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), vbox, TRUE, TRUE, 0);
	gtk_widget_show_all(GTK_DIALOG(dlg)->vbox);
}

void
lj_dialog_add_button(GtkWidget *dlg, GtkWidget *button) {
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->action_area), button, FALSE, FALSE, 0);
}

GtkWidget*
lj_dialog_buttonbox_new() {
	return gtk_hbox_new(FALSE, 5);
}
void
lj_dialog_buttonbox_add(GtkWidget *box, GtkWidget *button) {
	gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 0);
}

void
lj_dialog_set_contents_buttonbox(GtkWidget *dlg, GtkWidget *contents, GtkWidget *buttonbox) {
	GtkWidget *vbox, *hbox;

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_end(GTK_BOX(hbox), buttonbox, FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), contents, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), vbox, TRUE, TRUE, 0);
	gtk_widget_show_all(GTK_DIALOG(dlg)->vbox);
}

void
lj_dialog_add_close(GtkWidget *dlg) {
	GtkWidget *button;

	button = gtk_button_new_with_label("  Close  ");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", 
			GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dlg));
	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dlg)->action_area), button, FALSE, FALSE, 0);
	gtk_widget_show_all(GTK_DIALOG(dlg)->action_area);
}

GtkWidget*
lj_dialog_add_okcancel(GtkWidget *dlg, char *okcaption, 
		GtkSignalFunc okfunc, gpointer data) {
	GtkWidget *ok;
	GtkWidget *button;

	button = gtk_button_new_with_label("  Cancel  ");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", 
			GTK_SIGNAL_FUNC(gtk_widget_hide), GTK_OBJECT(dlg));
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", 
			GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dlg));
	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dlg)->action_area), button, FALSE, FALSE, 0);

	ok = button = gtk_button_new_with_label(okcaption ? okcaption : "  OK  ");
	gtk_signal_connect(GTK_OBJECT(button), "clicked", 
			GTK_SIGNAL_FUNC(okfunc), data);
	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dlg)->action_area), button, FALSE, FALSE, 0);

	gtk_widget_show_all(GTK_DIALOG(dlg)->action_area);
	return ok;
}

static gint confirm_destroy(GtkWidget *w, gpointer d) {
	gtk_main_quit();
	return TRUE;
}

static void
ok_clicked(GtkWidget *w, int *ret) {
	GtkWidget *dlg = gtk_widget_get_ancestor(w, gtk_dialog_get_type());
 	*ret = TRUE;
	gtk_widget_destroy(dlg);
}

int lj_confirm(GtkWidget *parent, char *title, char *msg) {
	GtkWidget *dlg, *hbox;
	int ret = FALSE;

	dlg = lj_dialog_new(parent, title, -1, -1);
	gtk_signal_connect(GTK_OBJECT(dlg), "destroy",
			GTK_SIGNAL_FUNC(confirm_destroy), NULL);

	hbox = gtk_hbox_new(FALSE, 10); {
		GdkPixmap *pix; GdkBitmap *bit;
		GtkWidget *pixmap;

		pix = gdk_pixmap_colormap_create_from_xpm_d(parent->window, 
				NULL, &bit, NULL, question_xpm);
		pixmap = gtk_pixmap_new(pix, bit);
		gtk_box_pack_start(GTK_BOX(hbox), pixmap, FALSE, FALSE, 0);

		gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(msg), TRUE, TRUE, 0);
	}
	lj_dialog_set_contents(dlg, hbox);

	lj_dialog_add_okcancel(dlg, NULL, ok_clicked, &ret);
	/* ret remains on the stack the entire time the dialog is present. */

	gtk_widget_show(dlg);

	gtk_main();
	/* will return when the dialog is closed;
	 * if they click "ok", then ret will be TRUE. */

	return ret;
}

void lj_win_set_icon(GtkWidget *win) {
	static GdkPixmap *pix = NULL;
	static GdkBitmap *bit;

	gtk_widget_realize(win);
	if (pix == NULL) 
		pix = gdk_pixmap_create_from_xpm_d(win->window, &bit, 
				NULL, goat_xpm);
	gdk_window_set_icon(win->window, (GdkWindow*)NULL, pix, bit);
}

void string_replace(char **dest, char *src) {
	if (*dest) g_free(*dest);
	*dest = src;
}

void web_pixmap(GtkWidget *w, GdkPixmap **p, GdkBitmap **b) {
	static GdkPixmap *pix = NULL;
	static GdkBitmap *bit = NULL;
	if (pix == NULL) {
		pix = gdk_pixmap_create_from_xpm_d(w->window, &bit, NULL, web_xpm);
	}
	*p = pix; *b = bit;
}

GtkWidget* web_button(GtkWidget *w, char *caption) {
	GtkWidget *hbox, *button;
	GdkPixmap *pix; GdkBitmap *bit;
	GtkWidget *pixmap, *label;

	web_pixmap(w, &pix, &bit);
	pixmap = gtk_pixmap_new(pix, bit);

	label = gtk_label_new(caption);

	hbox = gtk_hbox_new(FALSE, 5); 
	gtk_box_pack_start(GTK_BOX(hbox), pixmap, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	button = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button), hbox);
	return button;
}

void hex_to_color(char *buf, gdouble color[4]) {
	int r, g, b;
	sscanf(buf, "#%2x%2x%2x", &r, &g, &b);
	color[0] = (double)r/255.0;
	color[1] = (double)g/255.0;
	color[2] = (double)b/255.0;
	color[3] = 0.0;
}

void hex_to_gdkcolor(char *buf, GdkColor *c) {
	int r, g, b;
	sscanf(buf, "#%2x%2x%2x", &r, &g, &b);
	c->red   = r * 256;
	c->green = g * 256;
	c->blue  = b * 256;
}

void color_to_hex(gdouble color[4], char* buf) {
	char hextable[] = { '0', '1', '2', '3', '4', '5', '6', '7',
						'8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };
	int i;

	buf[0] = '#';
	buf[7] = '\0';
	for (i=0; i<3; i++) {
		int val = 255 * color[i];
		buf[i*2+1] = hextable[val/16];
		buf[i*2+2] = hextable[val%16];
	}
}

gpointer
clist_data_from_selection(GtkCList *l) {
	gint row;
	
	if (l->selection == NULL)
		return NULL;

	row = (gint) l->selection->data;
	return gtk_clist_get_row_data(l, row);
}

GtkWidget*
scroll_wrap(GtkWidget *w) {
	GtkWidget *scroll;

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), 
			GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(scroll), w);
	return scroll;
}

void
geometry_save(GtkWidget *win, geometry *geom) {
	if (win->window) {
		gdk_window_get_position(win->window, &geom->x, &geom->y);
		gdk_window_get_size(win->window, &geom->width, &geom->height);
	}
}

void
geometry_load(GtkWidget *win, geometry *geom) {
	if (geom->width == 0)
		return;

	gtk_widget_set_uposition(win, geom->x, geom->y);
	gtk_window_set_default_size(GTK_WINDOW(win), geom->width, geom->height);
}

static void
geometry_save_event(GtkWidget *win, GdkEvent *e, geometry *geom) {
	geometry_save(win, geom);
}
void geometry_tie(GtkWidget *win, geometry *geom) {
	/* load the existing geometry for this window */
	geometry_load(win, geom);
	/* and track the new geometry */
	gtk_signal_connect(GTK_OBJECT(win), "configure-event",
			GTK_SIGNAL_FUNC(geometry_save_event), geom);
}

char*
urlencode(unsigned char *string) {
	int escapecount = 0;
	unsigned char *src, *dest;
	unsigned char *newstr;
	
	char hextable[] = { '0', '1', '2', '3', '4', '5', '6', '7',
					'8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };
	
	if (string == NULL) return NULL;

	for (src = string; *src != 0; src++) 
		if (!isalnum(*src)) escapecount++;
	
	newstr = g_new(char, strlen(string) - escapecount + (escapecount * 3) + 1);
	
	src = string;
	dest = newstr;
	while (*src != 0) {
		if (!isalnum(*src)) {
			*dest++ = '%';
			*dest++ = hextable[*src >> 4];
			*dest++ = hextable[*src & 0x0F];
			src++;
		} else {
			*dest++ = *src++;
		}
	}
	*dest = 0;

	return newstr;
}

unsigned char* 
urldecode(char *string) {
	int destlen = 0;
	unsigned char *src, *dest;
	unsigned char *newstr;
	
	if (string == NULL) return NULL;

	for (src = string; *src != 0; src++) {
		if (*src == '%') { src+=2; } /* FIXME: this isn't robust. should check
										the next two chars for 0 */
		destlen++;
	}
	
	newstr = g_new(char, destlen + 1);
	src = string;
	dest = newstr;
	while (*src != 0) {
		if (*src == '%') {
			char h = toupper(src[1]);
			char l = toupper(src[2]);
			int vh, vl;
			vh = isalpha(h) ? (10+(h-'A')) : (h-'0');
			vl = isalpha(l) ? (10+(l-'A')) : (l-'0');
			*dest++ = ((vh<<4)+vl);
			src += 3;
		} else if (*src == '+') {
			*dest++ = ' ';
			src++;
		} else {
			*dest++ = *src++;
		}
	}
	*dest = 0;

	return newstr;
}

char*
util_skipline(char *text) {
	while (*text != 0) {
		if (*text == '\n') {
			text++;
			break;
		}
		text++;
	}
	return text;
}

char*
util_getline(char *text) {
	int len;
	char *str;

	for (len = 0; text[len] != 0; len++)
		if (text[len] == '\r' || text[len] == '\n') break;

	str = g_new(char, len+1);
	memcpy(str, text, len+1);
	str[len] = 0;
	return str;
}

void lj_messagebox(GtkWidget *parent, char *title, char *message) {
	GtkWidget *dlg, *box;
	dlg = lj_dialog_new(parent, title, -1, -1);
	
	box = gtk_vbox_new(FALSE, 10); {
		GtkWidget *label;
		gtk_container_set_border_width(GTK_CONTAINER(box), 10);

		label = gtk_label_new(message);
		gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

		gtk_box_pack_start(GTK_BOX(box), label, TRUE, TRUE, 0);
	}
	lj_dialog_set_contents(dlg, box);
	lj_dialog_add_close(dlg);
	gtk_widget_show(dlg);

}

