/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: lj.c,v 1.8 2001/11/16 01:13:22 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>

#include "network.h"
#include "dotconf.h"
#include "gtkspell.h"
#include "meta.h"
#include "util.h"
#include "security.h"
#include "login.h"
#include "lj.h"
#include "menu.h"

void lj_spell_restart(ljwin *ljw);

void 
lj_update_title(ljwin *ljw) {
	if (conf.loginok) {
		char *title;

		if (conf.usejournal == NULL) {
			title = g_strdup_printf("%s's LiveJournal", conf.fullname);
		} else {
			title = g_strdup_printf("%s's LiveJournal (using %s)", conf.fullname, conf.usejournal);
		}
		gtk_window_set_title(GTK_WINDOW(ljw->win), title);
		g_free(title);
	} else {
		gtk_window_set_title(GTK_WINDOW(ljw->win), PROGRAMNAME " " VERSION);
	}
}

GHashTable*
lj_build_post(ljwin *ljw) {
	time_t curtime_time_t = time(NULL);
	struct tm *curtime = localtime(&curtime_time_t);
	GHashTable *request;
	gchar *subject = NULL, *text;

	text = gtk_editable_get_chars(GTK_EDITABLE(ljw->eentry), 0, -1);
	if (text == NULL) return NULL;
	subject = gtk_editable_get_chars(GTK_EDITABLE(ljw->esubject), 0, -1);

	request = net_request_new("postevent");

	g_hash_table_insert(request, g_strdup("subject"), g_strdup(subject));
	g_hash_table_insert(request, g_strdup("event"), g_strdup(text));
	g_hash_table_insert(request, g_strdup("year"), 
			g_strdup_printf("%d", curtime->tm_year+1900));
	g_hash_table_insert(request, g_strdup("mon"), 
			g_strdup_printf("%d", curtime->tm_mon+1));
	g_hash_table_insert(request, g_strdup("day"), 
			g_strdup_printf("%d", curtime->tm_mday));
	g_hash_table_insert(request, g_strdup("hour"), 
			g_strdup_printf("%d", curtime->tm_hour));
	g_hash_table_insert(request, g_strdup("min"), 
			g_strdup_printf("%d", curtime->tm_min));
	if (conf.postmode == POSTMODE_GUI) {
		metamgr_append_to_request(METAMGR(ljw->metamgr), request);
		security_append_to_request(&ljw->security, request);
	}
	return request;
}

static gboolean 
do_post(ljwin *ljw) {
	GHashTable *request, *result;

	request = lj_build_post(ljw);

	result = net_request_run(ljw->win, "Sumbitting entry...", request);
	hash_destroy(request);

	if (net_request_succeeded(result)) {
		hash_destroy(result);
		return TRUE;
	} 
	hash_destroy(result);
	return FALSE;
}
		
static void 
submit_entry(GtkWidget *w, ljwin *ljw) {
	if (do_post(ljw)) {
		gtk_editable_delete_text(GTK_EDITABLE(ljw->eentry), 0, -1);
		gtk_editable_delete_text(GTK_EDITABLE(ljw->esubject), 0, -1);
		if (!conf.savemetadata) {
			metamgr_clear(METAMGR(ljw->metamgr));
		}
		if (conf.revertusejournalafterpost) {
			gtk_check_menu_item_set_active(
					GTK_CHECK_MENU_ITEM(ljw->musejournal_defaultjournal),
					TRUE);
			lj_update_title(ljw);
		}
	}
}

static void 
do_logout(ljwin *ljw) {
	string_replace(&conf.usejournal, NULL);
	lj_update_title(ljw);
	conf.loginok = FALSE;
	if (conf.pickws) {
		g_list_foreach(conf.pickws, (GFunc)g_free, NULL);
		g_list_free(conf.pickws);
		conf.pickws = NULL;
	}
	if (conf.friendgroups) {
		GList *l;
		for (l = conf.friendgroups; l != NULL; l = l->next) {
			friendgroup *fg = l->data;
			g_free(fg->name);
			g_free(fg);
		}
		g_list_free(conf.friendgroups);
		conf.friendgroups = NULL;
	}
}

void 
lj_do_loginout(ljwin *ljw, int autolog) {
	if (conf.loginok) {
		do_logout(ljw);
		menu_update(ljw, NULL);
		lj_update_title(ljw);
	}
	login_dlg_run(ljw, autolog); /* this calls menu_update() */
	lj_update_title(ljw);
	gtk_widget_grab_focus(ljw->eentry);
}

static GtkWidget*
make_subject_bar(ljwin *ljw) {
	GtkWidget *hbox;
	GtkWidget *secmgr;

	hbox = gtk_hbox_new(FALSE, 5); 
	gtk_box_pack_start(GTK_BOX(hbox), 
			gtk_label_new("Subject:"), FALSE, FALSE, 0);

	ljw->esubject = gtk_entry_new();
	gtk_widget_set_usize(ljw->esubject, 100, -1); 
	/* usize is just the minimum size... */
	gtk_box_pack_start(GTK_BOX(hbox), ljw->esubject, TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX(hbox), 
			gtk_label_new("Security:"), FALSE, FALSE, 0);

	ljw->security.type = conf.defaultsecurity;
	secmgr = secmgr_new(&ljw->security);
	gtk_box_pack_start(GTK_BOX(hbox), secmgr, FALSE, FALSE, 0);

	return hbox;
}

static GtkWidget*
make_main_entry(ljwin *ljw) {
	GtkWidget *scroll;

	ljw->eentry = gtk_text_new(NULL, NULL);
	gtk_text_set_editable(GTK_TEXT(ljw->eentry), TRUE);
	gtk_text_set_word_wrap(GTK_TEXT(ljw->eentry), TRUE);
	gtk_signal_connect(GTK_OBJECT(ljw->eentry), "activate",
		GTK_SIGNAL_FUNC(submit_entry), ljw);

	scroll = gtk_scrolled_window_new(NULL, NULL); 
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
			GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(scroll), ljw->eentry);

	return scroll;
}

static GtkWidget*
make_submit_bar(ljwin *ljw) {
	GtkWidget *box;
	
	box = gtk_hbox_new(FALSE, 5); 

	ljw->metamgr = metamgr_new(GTK_BOX(ljw->mainvbox));
	gtk_box_pack_start(GTK_BOX(box), ljw->metamgr, TRUE, TRUE, 0);

	ljw->bsubmit = gtk_button_new_with_label("  Submit  ");
	gtk_signal_connect(GTK_OBJECT(ljw->bsubmit), "clicked",
		GTK_SIGNAL_FUNC(submit_entry), ljw);
	gtk_box_pack_end(GTK_BOX(box), ljw->bsubmit, FALSE, FALSE, 0);

	return box;
}

static GtkWidget*
make_app_contents(ljwin *ljw) {
	ljw->mainvbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(ljw->mainvbox), 5);

	gtk_box_pack_start(GTK_BOX(ljw->mainvbox), 
			make_subject_bar(ljw), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(ljw->mainvbox), 
			make_main_entry(ljw), TRUE, TRUE, 0);
	gtk_box_pack_end(GTK_BOX(ljw->mainvbox), 
			make_submit_bar(ljw), FALSE, FALSE, 0);
	
	return ljw->mainvbox;
}

void 
lj_run() {
	ljwin ljw_actual = {0}, *ljw = &ljw_actual;
	GtkWidget *vbox;

	ljw->win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(ljw->win), FALSE, TRUE, TRUE);
	gtk_window_set_default_size(GTK_WINDOW(ljw->win), 250, 150);
	geometry_tie(ljw->win, &conf.mainwin_geom);

	lj_update_title(ljw);
	lj_win_set_icon(ljw->win);

	gtk_signal_connect(GTK_OBJECT(ljw->win), "delete-event",
		GTK_SIGNAL_FUNC(gtk_main_quit), NULL);

	vbox = gtk_vbox_new(FALSE, 0); 

	gtk_box_pack_start(GTK_BOX(vbox), 
			menu_make_bar(ljw), FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), make_app_contents(ljw), 
			TRUE, TRUE, 0);

	gtk_container_add(GTK_CONTAINER(ljw->win), vbox);

	lj_spell_restart(ljw);
	gtk_widget_show_all(vbox);

	menu_update(ljw, NULL);

	gtk_widget_show(ljw->win);
	while (gtk_events_pending())
		gtk_main_iteration();

	/* autolog if username and password known. */
	lj_do_loginout(ljw, (conf.username && conf.password));
	gtk_main();
}

void 
lj_spell_restart(ljwin *ljw) {
	char *args[20];
	char *buf = g_strdup(conf.spellcheck);
	int len = strlen(buf);
	int arg = 0;
	int i;

	args[arg++] = buf;
	for (i = 0; i < len; i++) {
		if (buf[i] == ' ') {
			buf[i] = 0;
			if (buf[i+1] != 0) {
				args[arg++] = buf+i+1;
			}
		}
	}
	args[arg] = NULL;

	if (gtkspell_running()) {
		gtkspell_stop();
		gtkspell_uncheck_all(GTK_TEXT(ljw->eentry));
	}
	if (conf.usespellcheck) {
		if (gtkspell_start(NULL, args) < 0) {
			fprintf(stderr, 
					"gtkspell_start() failed; spellchecking disabled.\n");
		} else {
			gtkspell_attach(GTK_TEXT(ljw->eentry));
			gtkspell_check_all(GTK_TEXT(ljw->eentry));
		}
	}

	g_free(buf);
}
