/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: friends.c,v 1.5 2001/08/25 20:59:28 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <time.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>

#include <libhalfgnome/halfgnome.h>
#include <libhalfgnome/spawn.h>

#include "friends.h"
#include "network.h"
#include "dotconf.h"
#include "util.h"
#include "friends.h"
#include "friendedit.h"

#include "friendgroups.h"

#include "../pixmaps/doublearrow.xpm"
#include "../pixmaps/leftarrow.xpm"
#include "../pixmaps/rightarrow.xpm"

#include "../pixmaps/community.xpm"
#include "../pixmaps/user.xpm"

/* jesus, this whole file is really ugly. :( */
typedef struct _friend_dlg friend_dlg;

struct _friend_dlg {
	GtkWidget *win;
	GtkWidget *friendlist;
	GtkWidget *badd, *bedit, *bremove, *bweb;

	GtkWidget *filesel;

	GList *friends;
	guint filter;

	GdkPixmap *darrow,  *larrow,  *rarrow,  *icon_comm,  *icon_user;
	GdkBitmap *darrowm, *larrowm, *rarrowm, *icon_commm, *icon_userm; 
};

enum {
	FRIEND_COL_TYPE,
	FRIEND_COL_CONN,
	FRIEND_COL_USERNAME,
	FRIEND_COL_NAME
};

#define FILTER_FRIEND_MY   (1 << 0)
#define FILTER_FRIEND_OF   (1 << 1)
#define FILTER_FRIEND_BOTH (1 << 2)
#define FILTER_USER        (1 << 3)
#define FILTER_COMMUNITY   (1 << 4)

static void export_cb(GtkWidget *w, friend_dlg *fdlg);

gint friend_compare_username(gconstpointer a, gconstpointer b) {
	const friend *fa = a;
	const friend *fb = b;

	return strcmp(fa->username, fb->username);
}

static friend*
friend_exists(GList *friends, char *username) {
	GList *l;
	for (l = friends; l != NULL; l = l->next) {
		if (strcmp(((friend*)l->data)->username, username) == 0)
			return l->data;
	}
	return NULL;
}

static int 
find_friend_row(friend_dlg *dlg, gchar *username) {
	int i;
	gchar *text;
	for (i = 0; i < GTK_CLIST(dlg->friendlist)->rows; i++) {
		gtk_clist_get_text(GTK_CLIST(dlg->friendlist), 
				i, FRIEND_COL_USERNAME, &text);
		if (strcasecmp(text, username) == 0) 
			return i;
	}
	return -1;
}

static void
update_pixmap(friend_dlg *dlg, int row) {
	friend *f;

	f = gtk_clist_get_row_data(GTK_CLIST(dlg->friendlist), row);

	switch (f->conn) {
		case FRIEND_MY:
			gtk_clist_set_pixmap(GTK_CLIST(dlg->friendlist), row, 
					FRIEND_COL_CONN, 
					dlg->rarrow, dlg->rarrowm);
			break;
		case FRIEND_OF:
			gtk_clist_set_pixmap(GTK_CLIST(dlg->friendlist), row, 
					FRIEND_COL_CONN, 
					dlg->larrow, dlg->larrowm);
			break;
		case FRIEND_BOTH:
			gtk_clist_set_pixmap(GTK_CLIST(dlg->friendlist), row, 
					FRIEND_COL_CONN, 
					dlg->darrow, dlg->darrowm);
			break;
	}
}

static void
update_color(friend_dlg *fdlg, int row) {
	friend *f;
	GtkStyle *newstyle;
	GdkColor c;
	int i;

	f = gtk_clist_get_row_data(GTK_CLIST(fdlg->friendlist), row);

	if (f->conn & FRIEND_MY) {
		newstyle = gtk_style_copy(gtk_widget_get_style(fdlg->friendlist));
		for (i = 0; i < 5; i++) newstyle->bg_pixmap[i] = NULL;
		hex_to_gdkcolor(f->foreground, &c);
		memcpy(&newstyle->fg[GTK_STATE_NORMAL], &c, sizeof(GdkColor));
		hex_to_gdkcolor(f->background, &c);
		memcpy(&newstyle->base[GTK_STATE_NORMAL], &c, sizeof(GdkColor));
	} else {
		newstyle = gtk_widget_get_style(fdlg->friendlist);
	}
		
	gtk_clist_set_cell_style(GTK_CLIST(fdlg->friendlist), 
			row, FRIEND_COL_USERNAME, newstyle);
}

static int 
modify_friend_clist(friend_dlg *dlg, int row, int bfriend, int bfriendof) {
	friend *f;

	f = gtk_clist_get_row_data(GTK_CLIST(dlg->friendlist), row);

	if (bfriend == -1)
		bfriend = f->conn & FRIEND_MY;
	if (bfriendof == -1)
		bfriendof = f->conn & FRIEND_OF;

	f->conn = (bfriend ? FRIEND_MY : 0) | 
		(bfriendof ? FRIEND_OF : 0);

	update_pixmap(dlg, row);
	update_color(dlg, row);
	return f->conn;
}

static friend*
friend_new(char *username, char *name, char *fg, char *bg, int conn, guint mask, char *type) {
	friend *f;
	f = g_new0(friend, 1);

	f->username = g_strdup(username);
	f->name = g_strdup(name);
	f->conn = conn;
	if (fg)
		strcpy(f->foreground, fg);
	if (bg)
		strcpy(f->background, bg);
	f->groupmask = mask;

	if (type) {
		if (strcmp(type, "community") == 0) 
			f->type = FRIEND_TYPE_COMMUNITY;
	} else {
		f->type = FRIEND_TYPE_USER;
	}
	return f;
}

static void
friend_free(gpointer d) {
	friend *f = d;
	g_free(f->username);
	g_free(f->name);
	g_free(f);
}

static void
add_friend_clist(friend_dlg *fdlg, friend *f) {
	int row;
	gchar *insertrow[4] = {NULL, NULL, NULL, NULL};

	insertrow[FRIEND_COL_USERNAME] = f->username;
	insertrow[FRIEND_COL_NAME] = f->name;

	row = gtk_clist_append(GTK_CLIST(fdlg->friendlist), insertrow);
	gtk_clist_set_row_data(GTK_CLIST(fdlg->friendlist), row, f);
	if (f->type == FRIEND_TYPE_COMMUNITY) {
		gtk_clist_set_pixmap(GTK_CLIST(fdlg->friendlist), row, 
				FRIEND_COL_TYPE, 
				fdlg->icon_comm, fdlg->icon_commm);
	} else {
		gtk_clist_set_pixmap(GTK_CLIST(fdlg->friendlist), row, 
				FRIEND_COL_TYPE, 
				fdlg->icon_user, fdlg->icon_userm);
	}
	update_pixmap(fdlg, row);
	update_color(fdlg, row);
}

static void 
load_pixmaps(friend_dlg *dlg) {
	gtk_widget_realize(dlg->win);
	dlg->darrow = gdk_pixmap_create_from_xpm_d(dlg->win->window, 
			&dlg->darrowm, NULL, doublearrow_xpm);
	dlg->rarrow = gdk_pixmap_create_from_xpm_d(dlg->win->window, 
			&dlg->rarrowm, NULL, rightarrow_xpm);
	dlg->larrow = gdk_pixmap_create_from_xpm_d(dlg->win->window, 
			&dlg->larrowm, NULL, leftarrow_xpm);

	dlg->icon_comm = gdk_pixmap_create_from_xpm_d(dlg->win->window, 
			&dlg->icon_commm, NULL, community_xpm);
	dlg->icon_user = gdk_pixmap_create_from_xpm_d(dlg->win->window, 
			&dlg->icon_userm, NULL, user_xpm);
}

static void
update_clist_from_friends(friend_dlg *dlg) {
	GtkCList *clist = GTK_CLIST(dlg->friendlist);
	GList *l;
	friend *f;

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	for (l = dlg->friends; l != NULL; l = l->next) {
		f = l->data;
		if ((dlg->filter & FILTER_FRIEND_MY)
				&& (f->conn == FRIEND_MY)) {
			continue;
		} else if ((dlg->filter & FILTER_FRIEND_OF)
				&& (f->conn == FRIEND_OF)) {
			continue;
		} else if ((dlg->filter & FILTER_FRIEND_BOTH)
				&& (f->conn == FRIEND_BOTH)) {
			continue;
		} else if ((dlg->filter & FILTER_USER)
				&& (f->type == FRIEND_TYPE_USER)) {
			continue;
		} else if ((dlg->filter & FILTER_COMMUNITY)
				&& (f->type == FRIEND_TYPE_COMMUNITY)) {
			continue;
		}
		add_friend_clist(dlg, f);
	}
	gtk_clist_set_column_width(clist, FRIEND_COL_USERNAME, 
			gtk_clist_optimal_column_width(clist, FRIEND_COL_USERNAME));
	gtk_clist_unselect_all(GTK_CLIST(dlg->friendlist));
	gtk_clist_thaw(clist);
}

static void
filter_cb(GtkWidget *w, friend_dlg *dlg) {
	int filter;

	filter = GPOINTER_TO_INT(gtk_object_get_user_data(GTK_OBJECT(w)));
	if (GTK_CHECK_MENU_ITEM(w)->active) {
		dlg->filter |= filter;
	} else {
		dlg->filter &= (~filter);
	}
	update_clist_from_friends(dlg);
}

static void
filter_showall_cb(GtkWidget *w, friend_dlg *dlg) {
	dlg->filter = 0;
	update_clist_from_friends(dlg);
}

static GtkWidget*
filter_menu_item(friend_dlg *dlg, GtkWidget *content, int filter) {
	GtkWidget *item;
	item = gtk_check_menu_item_new();
	gtk_container_add(GTK_CONTAINER(item), content);
	gtk_object_set_user_data(GTK_OBJECT(item), GINT_TO_POINTER(filter));
	if (dlg->filter & filter)
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
	gtk_signal_connect(GTK_OBJECT(item), "activate",
			GTK_SIGNAL_FUNC(filter_cb), dlg);
	return item;
}

static GtkWidget*
filter_menu_item_with_image(friend_dlg *dlg, char *text, int filter, 
		GdkPixmap *pix, GdkBitmap *bit) {
	GtkWidget *hbox, *pixmap, *label;

	hbox = gtk_hbox_new(FALSE, 5);
	label = gtk_label_new(text);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
	pixmap = gtk_pixmap_new(pix, bit);
	gtk_box_pack_end(GTK_BOX(hbox), pixmap, FALSE, FALSE, 0);
	return filter_menu_item(dlg, hbox, filter);
}

static void
filter_popup(GtkWidget *w, GdkEventButton *eb, friend_dlg *dlg) {
	GList *l;
	friend *f;
	int sum = 0, fsum = 0, fosum = 0, bsum = 0, csum = 0, usum = 0;
	GtkWidget *menu, *item;
	char buf[100];

	for (l = dlg->friends; l != NULL; l = l->next) {
		f = l->data;
		switch (f->conn) {
			case FRIEND_MY: fsum++; break;
			case FRIEND_OF: fosum++; break;
			case FRIEND_BOTH: bsum++; break;
		}
		switch (f->type) {
			case FRIEND_TYPE_COMMUNITY: csum++; break;
			case FRIEND_TYPE_USER: usum++; break;
		}
		sum++;
	}

	menu = gtk_menu_new();

	if (fsum > 0) {
		sprintf(buf, "Hide %d Friend%s", fsum, (fsum == 1 ? "" : "s"));
		item = filter_menu_item_with_image(dlg, buf, FILTER_FRIEND_MY, 
				dlg->rarrow, dlg->rarrowm);
		gtk_menu_append(GTK_MENU(menu), item);
	}

	if (fosum > 0) {
		sprintf(buf, "Hide %d Friend Of%s", fosum, (fosum == 1 ? "" : "s"));
		item = filter_menu_item_with_image(dlg, buf, FILTER_FRIEND_OF,
				dlg->larrow, dlg->larrowm);
		gtk_menu_append(GTK_MENU(menu), item);
	}

	if (bsum > 0) {
		sprintf(buf, "Hide %d Friend Both%s", bsum, (bsum == 1 ? "" : "s"));
		item = filter_menu_item_with_image(dlg, buf, FILTER_FRIEND_BOTH,
				dlg->darrow, dlg->darrowm);
		gtk_menu_append(GTK_MENU(menu), item);
	}

	gtk_menu_append(GTK_MENU(menu), gtk_menu_item_new());

	if (usum > 0 && csum > 0) {
		if (usum > 0) {
			sprintf(buf, "Hide %d User%s", usum, (usum == 1 ? "" : "s"));
			item = filter_menu_item_with_image(dlg, buf, FILTER_USER,
					dlg->icon_user, dlg->icon_userm);
			gtk_menu_append(GTK_MENU(menu), item);
		}

		if (csum > 0) {
			sprintf(buf, "Hide %d Communit%s", csum, (csum == 1 ? "y" : "ies"));
			item = filter_menu_item_with_image(dlg, buf, FILTER_COMMUNITY,
					dlg->icon_comm, dlg->icon_commm);
			gtk_menu_append(GTK_MENU(menu), item);
		}

		gtk_menu_append(GTK_MENU(menu), gtk_menu_item_new());
	}
	
	sprintf(buf, "Show All %d", sum);
	item = gtk_menu_item_new_with_label(buf);
	gtk_signal_connect(GTK_OBJECT(item), "activate",
			GTK_SIGNAL_FUNC(filter_showall_cb), dlg);
	gtk_menu_append(GTK_MENU(menu), item);

	gtk_widget_show_all(menu);
	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
		eb->button, eb->time);
}


static int 
load_friends(friend_dlg *dlg) {
	GHashTable *request, *result;
	char key[30];		 /* sprintf buffer */
	int i, friend_count, friendof_count;
	char *username;
	char *name;
	char *fg, *bg;
	char *value;
	char *type;
	guint mask;
	friend *f;
	
	request = net_request_new("getfriends");

	/* hooray for extending the protocol! */
	g_hash_table_insert(request, g_strdup("includefriendof"), 
			g_strdup("1"));
	
	result = net_request_run(dlg->win, "Loading Friends...", request);
	hash_destroy(request);

	if (!net_request_succeeded(result)) {
		hash_destroy(result);
		return -1;
	}

	/* the new idea is to just load all of this into the glist,
	 * then update the clist in one swell foop. */

	friend_count = atoi(g_hash_table_lookup(result, "friend_count"));
	for (i = 1; i <= friend_count; i++) {
		sprintf(key, "friend_%d_user", i);
		username = g_hash_table_lookup(result, key);
		
		sprintf(key, "friend_%d_name", i);
		name = g_hash_table_lookup(result, key);

		sprintf(key, "friend_%d_type", i);
		type = g_hash_table_lookup(result, key);

		sprintf(key, "friend_%d_fg", i);
		fg = g_hash_table_lookup(result, key);
		sprintf(key, "friend_%d_bg", i);
		bg = g_hash_table_lookup(result, key);

		sprintf(key, "friend_%d_groupmask", i);
		value = g_hash_table_lookup(result, key);
		if (value) {
			mask = atoi(value);
		} else {
			mask = FRIEND_GROUP_ALLFRIENDS;
		}

		f = friend_new(username, name, fg, bg, FRIEND_MY, mask, type);
		dlg->friends = g_list_insert_sorted(dlg->friends, f, friend_compare_username);
	}	   

	friendof_count = atoi(g_hash_table_lookup(result, "friendof_count"));
	for (i = 1; i <= friendof_count; i++) {
		sprintf(key, "friendof_%d_user", i);
		username = g_hash_table_lookup(result, key);

		f = friend_exists(dlg->friends, username);
		if (f) {
			f->conn |= FRIEND_OF;
		} else {
			sprintf(key, "friendof_%d_name", i);
			name = g_hash_table_lookup(result, key);

			sprintf(key, "friendof_%d_type", i);
			type = g_hash_table_lookup(result, key);

			f = friend_new(username, name, NULL, NULL, FRIEND_OF, 0, type);
			dlg->friends = g_list_insert_sorted(dlg->friends, f, friend_compare_username);
		}
	}	   

	dlg->filter = 0;

	hash_destroy(result);
	return 0;
}

static void 
row_selected(GtkCList *list, gint row, gint col, GdkEventButton *event, gpointer data) {
	friend_dlg *dlg = (friend_dlg *) data;
	friend *f;

	f = gtk_clist_get_row_data(list, row);
	if (f == NULL) return;

	if (f->conn & FRIEND_MY || list->selection == NULL) {
		gtk_widget_set_sensitive(GTK_WIDGET(dlg->bedit), 
				list->selection != NULL);
		gtk_widget_set_sensitive(GTK_WIDGET(dlg->bremove), 
				list->selection != NULL);
	}
	gtk_widget_set_sensitive(GTK_WIDGET(dlg->bweb), list->selection != NULL);
}

static void
editadd_friend(friend_dlg *fdlg, gboolean edit) {
	int row = -1;
	GtkCList *list = GTK_CLIST(fdlg->friendlist);
	friend *toedit = NULL, *newf;
	
	if (list->selection != NULL) {
		row = (int)list->selection->data;

		toedit = gtk_clist_get_row_data(list, row);
		if (!edit && toedit->conn & FRIEND_MY) {
			/* if they're already a friend, don't "add" them. */
			toedit = NULL;
		}
	}

	if (edit && (!toedit || !(toedit->conn & FRIEND_MY)))
		return; /* not a friend, nothing to edit. */

	/* make the dialog, and fill in that info */
	newf = friend_edit_dlg_run(fdlg->win, edit, toedit);

	if (newf == NULL) return;

	if (newf != toedit) {
		/* new friend */
		if (!friend_exists(fdlg->friends, newf->username)) {
			fdlg->friends = g_list_insert_sorted(fdlg->friends, newf, friend_compare_username);
			add_friend_clist(fdlg, newf);
		} else {
			/* they added a friend that already exists.  bah. */
			int row;

			row = find_friend_row(fdlg, newf->username);
			if (row >= 0) {
				modify_friend_clist(fdlg, row, TRUE, -1);
			} 
			friend_free(newf);
		}
	} else {
		/* modify friend */
		modify_friend_clist(fdlg, row, TRUE, -1);
		gtk_clist_unselect_row(GTK_CLIST(fdlg->friendlist), row, 0);
		gtk_clist_select_row(GTK_CLIST(fdlg->friendlist), row, 0);
	}
}

static void 
add_clicked(GtkWidget *w, friend_dlg *fdlg) {
	editadd_friend(fdlg, FALSE);
}

static void 
edit_clicked(GtkWidget *w, friend_dlg *fdlg) {
	editadd_friend(fdlg, TRUE);
}

static void 
remove_clicked(GtkWidget *w, friend_dlg *fdlg) {
	GHashTable *request, *result;
	GtkCList *list = GTK_CLIST(fdlg->friendlist);
	char key[50];
	gchar *user;
	int selrow;

	if (list->selection == NULL)
		return;

	selrow = (int)list->selection->data;

	request = net_request_new("editfriends");

	/* making a request key like "editfriend_delete_<username>" */
	gtk_clist_get_text(GTK_CLIST(list), selrow, FRIEND_COL_USERNAME, &user);
	snprintf(key, sizeof(key), "editfriend_delete_%s", user);
	g_hash_table_insert(request, g_strdup(key), g_strdup("1"));

	result = net_request_run(fdlg->win, "Deleting Friend...", request);
	hash_destroy(request);

	if (net_request_succeeded(result)) {
		if (modify_friend_clist(fdlg, selrow, FALSE, -1) == 0) {
			/* the friend is no more! */
			friend *f;
			f = gtk_clist_get_row_data(GTK_CLIST(fdlg->friendlist), selrow);
			fdlg->friends = g_list_remove(fdlg->friends, f);
			friend_free(f);
			gtk_clist_remove(GTK_CLIST(fdlg->friendlist), selrow);
		} else {
			/* force the buttons on the side to update! */
			gtk_clist_unselect_row(GTK_CLIST(fdlg->friendlist), selrow, 0);
			gtk_clist_select_row(GTK_CLIST(fdlg->friendlist), selrow, 0);
		}
	}

	hash_destroy(result);
}

static void 
web_clicked(GtkWidget *w, friend_dlg *fdlg) {
	char *user;
	char *spawn;

	GtkCList *list = GTK_CLIST(fdlg->friendlist);
	int selrow;

	if (list->selection == NULL)
		return;

	selrow = (int)list->selection->data;
	gtk_clist_get_text(GTK_CLIST(list), selrow, FRIEND_COL_USERNAME, &user);
	spawn = g_strdup_printf("%s/userinfo.bml?user=%s", conf.ljserver, user);
	halfgnome_spawn_url(spawn);
	g_free(spawn);
}

static gint 
type_compare(GtkCList *list, gconstpointer p1, gconstpointer p2) {
	const GtkCListRow *r1 = p1;
	const GtkCListRow *r2 = p2;
	/* you'll note that none of this is docmented in any GTK manual. */
	friend *f1 = r1->data;
	friend *f2 = r2->data;

	if (f1 == NULL || f2 == NULL) return 0;
	if (f1->type > f2->type) return 1;
	else if (f1->type < f2->type) return -1;
	return 0;
}
static gint 
conn_compare(GtkCList *list, gconstpointer p1, gconstpointer p2) {
	const GtkCListRow *r1 = p1;
	const GtkCListRow *r2 = p2;
	/* you'll note that none of this is docmented in any GTK manual. */
	friend *f1 = r1->data;
	friend *f2 = r2->data;

	if (f1 == NULL || f2 == NULL) return 0;
	if (f1->conn > f2->conn) return 1;
	else if (f1->conn < f2->conn) return -1;
	return 0;
}

static void 
column_clicked(GtkCList *list, gint col, friend_dlg *fdlg) {
	if (col == list->sort_column) {
		/* clicked on an already sorted column; switch sort order. */
		if (list->sort_type == GTK_SORT_ASCENDING) {
			gtk_clist_set_sort_type(list, GTK_SORT_DESCENDING);
		} else {
			gtk_clist_set_sort_type(list, GTK_SORT_ASCENDING);
		}
	} else {
		gtk_clist_set_sort_type(list, GTK_SORT_ASCENDING);
		gtk_clist_set_sort_column(list, col);
		if (col == FRIEND_COL_TYPE) {
			gtk_clist_set_compare_func(list, type_compare);
		} else if (col == FRIEND_COL_CONN) {
			gtk_clist_set_compare_func(list, conn_compare);
		} else {
			/* this is also not docmented. 
			 * NULL resets it to the default compare function. */
			gtk_clist_set_compare_func(list, NULL); 
		}
	}
	gtk_clist_sort(list);
}

static gint
button_press_idle_cb(gpointer d) {
	gdk_threads_enter();
	editadd_friend(d, TRUE);
	gdk_threads_leave();
	return FALSE;
}
	
static gint
button_press_event(GtkCList *list, GdkEventButton *be, friend_dlg *fdlg) {
	if (be->type == GDK_2BUTTON_PRESS) {
		/* work around sawfish brokenness? */
		gtk_idle_add(button_press_idle_cb, fdlg);
		return 1;
	}
	return 0;
}

static gboolean
key_press_event(GtkCList *list, GdkEventKey *e, friend_dlg *fdlg) {
	int sel = -1;
	friend *f;
	int i;

	if (list->selection)
		sel = (int)list->selection->data;

	if (sel) {
		f = gtk_clist_get_row_data(list, sel);
		if (f != NULL && f->username[0] == e->keyval) {
			/* advance to next name starting with this letter.
			 * if we're at the end of the matches, 
			 * we fall through to the "select first matching name" code */
			if (sel+1 < list->rows) {
				f = gtk_clist_get_row_data(list, sel+1);
				if (f->username[0] == e->keyval) {
					gtk_clist_unselect_row(list, sel, 0);
					gtk_clist_select_row(list, sel+1, 0);
					gtk_clist_moveto(list, sel+1, 0, 0.25, 0);
					return 1;
				}
			}
		}
	}

	/* if we get here, then we just want to select the first matching name */
	for (i = 0; i < list->rows; i++) {
		f = gtk_clist_get_row_data(list, i);
		if (f->username[0] == e->keyval) {
			if (sel)
				gtk_clist_unselect_row(list, sel, 0);
			gtk_clist_select_row(list, i, 0);
			gtk_clist_moveto(list, i, 0, 0.25, 0);
			return 1;
		}
	}

	return 0;
}

static GtkWidget*
friends_clist_create(friend_dlg *fdlg) {
	static char* titles[] = { "Type", "Link", "Username", "Name" };
	GtkCList *list;

	list = GTK_CLIST(gtk_clist_new_with_titles(4, titles));
	gtk_clist_set_auto_sort(list, TRUE);
	gtk_clist_set_sort_column(list, FRIEND_COL_USERNAME);
	gtk_clist_set_column_width(list, FRIEND_COL_CONN, 20);
	gtk_clist_set_column_width(list, FRIEND_COL_USERNAME, 
			gdk_string_width(GTK_WIDGET(list)->style->font, 
				"aaaaaaaaaaaaa"));
	gtk_clist_set_column_justification(list, 
			FRIEND_COL_TYPE, GTK_JUSTIFY_CENTER);
	gtk_clist_set_column_justification(list, 
			FRIEND_COL_CONN, GTK_JUSTIFY_CENTER);
	gtk_signal_connect(GTK_OBJECT(list), "select-row", row_selected, fdlg);
	gtk_signal_connect(GTK_OBJECT(list), "unselect-row", row_selected, fdlg);
	gtk_signal_connect(GTK_OBJECT(list), "click-column", 
			GTK_SIGNAL_FUNC(column_clicked), fdlg);
	gtk_signal_connect(GTK_OBJECT(list), "button-press-event",
			GTK_SIGNAL_FUNC(button_press_event), fdlg);
	gtk_signal_connect(GTK_OBJECT(list), "key-press-event",
			GTK_SIGNAL_FUNC(key_press_event), fdlg);
	return GTK_WIDGET(list);
}


static GtkWidget*
friends_list_create(friend_dlg *fdlg) {
	GtkWidget *scroll;

	scroll = gtk_scrolled_window_new (NULL, NULL); {
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), 
				GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

		fdlg->friendlist = friends_clist_create(fdlg);

		gtk_container_add(GTK_CONTAINER(scroll), fdlg->friendlist);
	}
	return scroll;
}

static void
friendgroups_cb(GtkWidget *w, friend_dlg *fdlg) {
	friendgroups_dialog_new(fdlg->win, fdlg->friends);
}

static GtkWidget*
make_buttonbox(friend_dlg *fdlg) {
	GtkWidget *box;

	fdlg->badd = gtk_button_new_with_label(" Add... ");
	gtk_signal_connect(GTK_OBJECT(fdlg->badd), "clicked",
			GTK_SIGNAL_FUNC(add_clicked), fdlg);

	fdlg->bedit = gtk_button_new_with_label(" Edit... ");
	gtk_widget_set_sensitive(fdlg->bedit, FALSE);
	gtk_signal_connect(GTK_OBJECT(fdlg->bedit), "clicked",
			GTK_SIGNAL_FUNC(edit_clicked), fdlg);

	fdlg->bremove = gtk_button_new_with_label(" Remove ");
	gtk_widget_set_sensitive(fdlg->bremove, FALSE);
	gtk_signal_connect(GTK_OBJECT(fdlg->bremove), "clicked",
			GTK_SIGNAL_FUNC(remove_clicked), fdlg);

	fdlg->bweb = web_button(fdlg->win, "User Info...");
	gtk_widget_set_sensitive(fdlg->bweb, FALSE);
	gtk_signal_connect(GTK_OBJECT(fdlg->bweb), "clicked",
			GTK_SIGNAL_FUNC(web_clicked), fdlg);

	box = lj_dialog_buttonbox_new();
	lj_dialog_buttonbox_add(box, fdlg->badd);
	lj_dialog_buttonbox_add(box, fdlg->bedit);
	lj_dialog_buttonbox_add(box, fdlg->bremove);
	lj_dialog_buttonbox_add(box, fdlg->bweb);
	return box;
}

static void 
friend_dialog_destroy_cb(GtkWidget *w, friend_dlg *fdlg) {
	GList *l;
	for (l = fdlg->friends; l != NULL; l = l->next) {
		friend_free(l->data);
	}
	if (fdlg->friends)
		g_list_free(fdlg->friends);
	gtk_main_quit();
}

void 
friends_dialog(GtkWidget *mainwin) {
	friend_dlg fdlg_actual = {0}, *fdlg = &fdlg_actual;

	if (load_friends(fdlg) < 0) {
		return;
	}
	
	fdlg->win = lj_dialog_new(mainwin, "Edit Friends", 350, -1);
	geometry_tie(fdlg->win, &conf.friends_geom);
	gtk_signal_connect(GTK_OBJECT(fdlg->win), "destroy",
			GTK_SIGNAL_FUNC(friend_dialog_destroy_cb), fdlg);

	load_pixmaps(fdlg);

	lj_dialog_set_contents_buttonbox(fdlg->win,
			friends_list_create(fdlg),
			make_buttonbox(fdlg));

	{
		GtkWidget *button;
		GtkWidget *hbox = GTK_DIALOG(fdlg->win)->action_area;

		button = gtk_button_new_with_label("  Friend Groups...  ");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				GTK_SIGNAL_FUNC(friendgroups_cb), fdlg);
		gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

		button = gtk_button_new_with_label("  Filter...  ");
		gtk_signal_connect(GTK_OBJECT(button), 
				"button-press-event",
				GTK_SIGNAL_FUNC(filter_popup), fdlg);
		gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

		button = gtk_button_new_with_label("  Export...  ");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				GTK_SIGNAL_FUNC(export_cb), fdlg);
		gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	}
	lj_dialog_add_close(fdlg->win);

	update_clist_from_friends(fdlg);

	gtk_widget_show(fdlg->win);
	gtk_main();
}

static void
export_do_cb(GtkWidget *w, friend_dlg *fdlg) {
	char *filename = 
		gtk_file_selection_get_filename(GTK_FILE_SELECTION(fdlg->filesel));
	FILE *fout;
	friend *f;
	GList *l;

	if ((fout = fopen(filename, "w")) == NULL) {
		fprintf(stderr, "lj: Unable to open %s: %s\n", 
				filename, strerror(errno));
		return;
	}
	for (l = fdlg->friends; l != NULL; l = l->next) {
		f = l->data;

		fprintf(fout, "%c-%c %s\n", 
				(f->conn & FRIEND_OF ? '<' : ' '),
				(f->conn & FRIEND_MY ? '>' : ' '),
				f->username);
	}

	fclose(fout);
}

static void
suggest_cb(GtkWidget *w, GtkWidget *entry) {
	char buf[50];
	time_t curtime;
	struct tm *date;
	
	time(&curtime);
	date = localtime(&curtime);

	sprintf(buf, "friends.%04d-%02d-%02d", 
			date->tm_year+1900, date->tm_mon+1, date->tm_mday);

	gtk_entry_set_text(GTK_ENTRY(entry), buf);
}

static void
export_cb(GtkWidget *w, friend_dlg *fdlg) {
	GtkWidget *fsel;
	GtkWidget *bstamp;

	fdlg->filesel = fsel = gtk_file_selection_new("Select output file");
	gtk_window_set_transient_for(GTK_WINDOW(fsel), GTK_WINDOW(fdlg->win));
	gtk_window_set_modal(GTK_WINDOW(fsel), TRUE);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fsel)->ok_button),
			"clicked", GTK_SIGNAL_FUNC(export_do_cb), fdlg);

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fsel)->ok_button),
			"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer)fsel);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fsel)->cancel_button),
			"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer)fsel);

	bstamp = gtk_button_new_with_label("  Suggest Filename  ");
	gtk_box_pack_start(GTK_BOX(GTK_FILE_SELECTION(fsel)->action_area),
			bstamp, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(bstamp), "clicked",
			GTK_SIGNAL_FUNC(suggest_cb), 
			GTK_FILE_SELECTION(fsel)->selection_entry);
	gtk_widget_show(bstamp);

	gtk_widget_show(fsel);
	lj_win_set_icon(fsel);
}

